<?php
/*
  Plugin Name: ReDi Restaurant Booking
  Description: Online Reservation plugin for Restaurants. It's a fully automated system that verifies available seats in restaurant, accepts and cofrirms reservation immideately.
  Version: 22.0919
  Author: پرشین اسکریپت
  Author URI: https://www.persianscript.ir
  Text Domain: redi-booking
  Domain Path: /lang
*/

if (!defined('REDI_BOOKING_PLUGIN_URL')) {
    define('REDI_BOOKING_PLUGIN_URL', plugin_dir_url(__FILE__));
}
if (!defined('REDI_BOOKING_TEMPLATE')) {
    define('REDI_BOOKING_TEMPLATE', plugin_dir_path(__FILE__) . 'templates' . DIRECTORY_SEPARATOR);
}

require_once('redi.php');

define('REDI_TOTALDATES', 7);

if (!class_exists('ReDiBooking')) {
    
	if ( ! class_exists( 'CustomFieldsSaveTo' ) ) {
		class CustomFieldsSaveTo {
			const WPOptions = 'options';
			const API = 'api';
		}
	}

    if (!class_exists('EmailContentType')) {
        class EmailContentType
        {
            const Canceled = 'Canceled';
            const Confirmed = 'Confirmed';
        }
    }

    class ReDiBooking
    {
        public $version = '22.0919';
        /**
         * @var string The options string name for this plugin
         */
        private $optionsName = 'redi_booking_options';
        private $apiKeyOptionName = 'redi_booking_options_ApiKey';
        private static $name = 'REDI_BOOKING';
        private $options = array();
        private $ApiKey;
        private $redi;
        private $weekday = array('Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday');
        private $styles = array();
        private $table_name = 'redi_restaurant_booking_v2';


        function filter_timeout_time($time)
        {
            $time = 30; //new number of seconds
            return $time;
        }

        public function __construct()
        {
            $this->_name = self::$name;
            //Initialize the options
            $this->get_options();

            $this->ApiKey = isset($this->options['ID']) ? $this->options['ID'] : null;

            $this->redi = new ReDiBookingApi($this->ApiKey);
            //Actions
            add_action('init', array(&$this, 'init_sessions'));
            add_action('admin_menu', array(&$this, 'redi_booking_admin_menu_link'));

            $this->page_title = 'booking';
            $this->content = '[redibooking]';
            $this->page_name = $this->_name;
            $this->page_id = '0';

            register_activation_hook(__FILE__, array($this, 'activate'));
            register_deactivation_hook(__FILE__, array($this, 'deactivate'));
            register_uninstall_hook(__FILE__, array('ReDiBooking', 'uninstall'));

            add_action('wp_ajax_redi_booking_submit', array(&$this, 'redi_booking_ajax'));
            add_action('wp_ajax_nopriv_redi_booking_submit', array($this, 'redi_booking_ajax'));
            add_filter('redi_http_request_timeout', array(&$this, 'filter_timeout_time'));
            add_filter('redi-booking-email-content', array($this, 'redi_booking_get_email_content'));

			add_action('wp_enqueue_scripts', array(&$this, 'remove_blockui_scripts'), 50000);            
            add_action('redi-booking-confirm-reservation', array($this, 'confirm_reservation'));
            add_action('redi-booking-send-confirmation-email', array($this, 'send_confirmation_email'));
            add_shortcode('redibooking', array(&$this, 'shortcode'));


            $this->create_database();
        }

        function create_database()
        {
			global $wpdb;
			require_once ABSPATH . 'wp-admin/includes/upgrade.php';
			$table_name = $this->table_name;
            
            if ( !$wpdb->get_var( "show tables like '$table_name'" ) ) 
            {
				$sql = "CREATE TABLE " . $table_name . "(
					id INT NOT NULL AUTO_INCREMENT,				
					reservation_number INT,
					name VARCHAR(50),					
					phone VARCHAR(20),
					email VARCHAR(50),
					date_from VARCHAR(30),
                    date_to VARCHAR(30),
                    language VARCHAR(30),
                    currenttime DATETIME,
                    prepayment VARCHAR(30),
					guests INT,
					comments TEXT,
					UNIQUE KEY id (id)
				);";
				dbDelta( $sql );

				$sql = "CREATE TABLE " . $table_name . '_custom_fields' . "(
					id INT NOT NULL AUTO_INCREMENT,				
					reservation_number INT,
					name VARCHAR(100),					
					value VARCHAR(1000),
					type VARCHAR(100),
					UNIQUE KEY id (id)
				);";
                dbDelta( $sql );            
            }
		}

		function remove_blockui_scripts()
		{
            // Remove blockui scripts as they blocks screen when calendar is shown
            // Reproduced when woocommerce plugin is active
            if (function_exists('is_woocommerce'))
            {
                # Only load CSS and JS on Woocommerce pages   
                if (!is_woocommerce() && !is_cart() && !is_checkout()) 
                {             
                    wp_dequeue_script("jquery-blockui");
                    wp_deregister_script("jquery-blockui");
                }
            }
		}        

        function language_files($mofile, $domain)
        {
            if ($domain === 'redi-booking') {
                $full_file = WP_PLUGIN_DIR . '/redi-restaurant-booking/lang/' . $domain . '-' . get_locale() . '.mo';
                $generic_file = WP_PLUGIN_DIR . '/redi-restaurant-booking/lang/' . $domain . '-' . substr(get_locale(),
                        0, 2) . '.mo';
                if (file_exists($full_file)) {
                    return $full_file;
                }
                if (file_exists($generic_file)) {
                    return $generic_file;
                }
            }

            return $mofile;
        }

        function ReDiBooking()
        {
            $this->__construct();
        }


        function plugin_get_version()
        {
            $plugin_data = get_plugin_data(__FILE__);
            $plugin_version = $plugin_data['Version'];

            return $plugin_version;
        }

        /**
         * Retrieves the plugin options from the database.
         * @return array
         */
        function get_options()
        {
            if (!$options = get_option($this->optionsName)) {
                
                $options = get_option('wp_' . $this->optionsName);
            }
     
            $this->options = $options;
        }

        private function register()
        {
            //Gets email and sitename from config
            $new_account = $this->redi->createUser(array('Email' => get_option('admin_email'), 'Source' => 'EnvatoMarket'));

            if (isset($new_account['ID']) && !empty($new_account['ID'])) {
                $this->ApiKey = $this->options['ID'] = $new_account['ID'];
                $this->redi->setApiKey($this->options['ID']);
                $place = $this->redi->createPlace(array(
                    'place' => array(
                        'Name' => get_bloginfo('name'),//get from site name
                        'City' => 'city',
                        'Country' => 'country',
                        'Address' => 'Address line 1',
                        'Email' => get_option('admin_email'),
                        'EmailCC' => '',
                        'Phone' => '[areacode] [number]',
                        'WebAddress' => get_option('siteurl'),
                        'Lang' => str_replace('_', '-', get_locale()),
                        'MinTimeBeforeReservation' => 24, // hour
                        'DescriptionShort' => '',
                        'DescriptionFull' => '',
                        'Catalog' => (int)false
                    )
                ));

                if (isset($place['Error'])) {
                    return $place;
                }

                $placeID = (int)$place['ID'];

                $category = $this->redi->createCategory($placeID,
                    array('category' => array('Name' => 'Restaurant')));

                if (isset($category['Error'])) {
                    return $category;
                }

                $categoryID = (int)$category['ID'];
                $service = $this->redi->createService($categoryID,
                    array('service' => array('Name' => 'Person', 'Quantity' => 10)));

                if (isset($service['Error'])) {
                    return $service;
                }

                foreach ($this->weekday as $value) {
                    $times[$value] = array('OpenTime' => '12:00', 'CloseTime' => '00:00');
                }
                $this->redi->setServiceTime($categoryID, $times);

                $this->saveAdminOptions();
            }


            return $new_account;
        }

        /**
         * Saves the admin options to the database.
         */
        function saveAdminOptions()
        {
            return update_option($this->optionsName, $this->options);
        }

        function display_errors($errors, $admin = false)
        {
            if (isset($errors['Error'])) {
                foreach ((array)$errors['Error'] as $error) {
                    echo '<div class="error"><p>' . $error . '</p></div>';
                }
            }
            //WP-errors
            if (isset($errors['Wp-Error']) && $admin) {
                foreach ((array)$errors['Wp-Error'] as $error_key => $error) {
                    foreach ((array)$error as $err) {
                        echo '<div class="error"><p>' . $error_key . ' : ' . $err . '</p></div>';
                    }
                }
            }
        }

        /**
         * Adds settings/options page
         */
        function redi_booking_admin_options_page()
        {
            $errors = array();

            if ($this->ApiKey == null) { 

                $return = $this->register();
                $this->display_errors($return, true);
            }

            if ($this->ApiKey == null) {

                $errors['Error'] = array(
                    esc_html_e('ReDi Restaurant Booking plugin could not get an API key from the reservationdiary.eu server when it activated.' .
                        '<br/> You can try to fix this by going to the ReDi Restaurant Booking "options" page. ' .
                        '<br/>This will cause ReDi Restaurant Booking plugin to retry fetching an API key for you. ' .
                        '<br/>If you keep seeing this error it usually means that server where you host your web site can\'t connect to our reservationdiary.eu server. ' .
                        '<br/>You can try asking your WordPress host to allow your WordPress server to connect to api.reservationdiary.eu' .
                        '<br/>In case you can not solve this problem yourself, please contact us directly by <a href="mailo:info@reservationdiary.eu">info@reservationdiary.eu</a>',
                        'redi-booking')
                );
                $this->display_errors($errors, true);
                die;
            }
            $places = $this->redi->getPlaces();

            if (isset($places['Error'])) {
                $this->display_errors($places, true);
                die;
            }
            $placeID = $places[0]->ID;

            $categories = $this->redi->getPlaceCategories($placeID);

            $categoryID = $categories[0]->ID;

            if (isset($_POST['action']) && $_POST['action'] == 'cancel') {
                if (isset($_POST['id'])) {
                    $params = array(
                        'ID' => self::GetPost('id'),
                        'Lang' => str_replace('_', '-', get_locale()),
                        'Reason' => mb_substr(self::GetPost('Reason'), 0, 250),
                        'CurrentTime' => date('Y-m-d H:i', current_time('timestamp')),
                        'Version' => self::plugin_get_version()
                    );

                    
                    $cancel = $this->redi->cancelReservation($params);
                    
                    if (isset($cancel['Error'])) {
                        $errors[] = $cancel['Error'];
                    } else {
                        $cancel_success = esc_html_e('Reservation has been successfully canceled.',
                            'redi-booking');
                    }

                } else {
                    $errors[] = esc_html__('Reservation number is required', 'redi-booking');
                }
            }
            $settings_saved = false;
            if (isset($_POST['submit'])) {
                $form_valid = true;
                $services = (int)self::GetPost('services');
                $minPersons = (int)self::GetPost('MinPersons');
                $maxPersons = (int)self::GetPost('MaxPersons');
                $largeGroupsMessage = self::GetPost('LargeGroupsMessage');
                $maxTime = self::GetPost('MaxTime');
                $thanks = self::GetPost('Thanks', 0);
                $alternativeTimeStep = self::GetPost('AlternativeTimeStep', 30);
                $MinTimeBeforeReservation = self::GetPost('MinTimeBeforeReservation');
                $dateFormat = self::GetPost('DateFormat');

				$manualReservation          = self::GetPost('ManualReservation', 0);
                //validation
                if ($minPersons >= $maxPersons) {
                    $errors[] = esc_html__('Min Persons should be lower than Max Persons', 'redi-booking');
                    $form_valid = false;
                }

                $reservationTime = (int)self::GetPost('ReservationTime');
                if ($reservationTime <= 0) {
                    $errors[] = esc_html__('Reservation time should be greater than 0', 'redi-booking');
                    $form_valid = false;
                }
                $place = array(
                    'place' => array(
                        'Name' => self::GetPost('Name'),
                        'City' => self::GetPost('City'),
                        'Country' => self::GetPost('Country'),
                        'Address' => self::GetPost('Address'),
                        'Email' => self::GetPost('Email'),
                        'EmailCC' => self::GetPost('EmailCC'),
                        'Phone' => self::GetPost('Phone'),
                        'WebAddress' => self::GetPost('WebAddress'),
                        'Lang' => self::GetPost('Lang'),
                        'MinTimeBeforeReservation' => self::GetPost('MinTimeBeforeReservation'),
                        'Catalog' => (int)false,
                        'DateFormat' => self::GetPost('DateFormat'),
                        'DescriptionShort' => '',
                        'DescriptionFull' => '',
                        'MaxTimeBeforeReservation' => $maxTime,
                        'Version' => $this->version,
                        'TableFeature' => self::GetPost('TableFeature', 0)
                    )
                );

                if (empty($place['place']['Country'])) {
                    $errors[] = esc_html__('Country is required', 'redi-booking');
                    $form_valid = false;
                }
				$this->options[ 'customfields_save_to' ] = CustomFieldsSaveTo::API;

				for ( $i = 0; $i != REDI_CUSTOM_FIELDS; $i++ ) {
					$field_id       = 'field_' . $i . '_id';
                    $field_name = 'field_' . $i . '_name';
                    $field_type = 'field_' . $i . '_type';
                    $field_required = 'field_' . $i . '_required';
                    $field_message = 'field_' . $i . '_message';

					$$field_id = self::GetPost( $field_id );

					$$field_name = self::GetPost( $field_name );

					$$field_type = self::GetPost( $field_type );
					$$field_required = ( self::GetPost( $field_required ) === 'on' );

					$$field_message = self::GetPost( $field_message );

					if ( empty( $$field_name ) && isset( $$field_id ) && $$field_id > 0 ) { //name is empty so delete this field
						$this->redi->deleteCustomField( self::lang(), $placeID, $$field_id );
					} else {
						//new or update
						if ( isset( $$field_id ) && $$field_id > 0 ) {
							$this->redi->updateCustomField( self::lang(), $placeID, $$field_id, array(
								'customfield' => array(
									'Name'     => $$field_name,
									'Message'  => $$field_message,
									'Required' => $$field_required ? 'true' : 'false',
									'Type'     => $$field_type
								)
							) );
						} else {
							$this->redi->saveCustomField( self::lang(), $placeID, array(
								'customfield' => array(
									'Name'     => $$field_name,
									'Message'  => $$field_message,
									'Required' => $$field_required ? 'true' : 'false',
									'Type'     => $$field_type
								)
							) );
                        }
                    }
                }

                if ($form_valid) {
                    $settings_saved = true;
                    $serviceTimes = self::GetServiceTimes();

                    $this->options['Thanks'] = $thanks;
                    $this->options['AlternativeTimeStep'] = $alternativeTimeStep;
                    $this->options['services'] = $services;
                    $this->options['MinTimeBeforeReservation'] = $MinTimeBeforeReservation;
                    $this->options['DateFormat'] = $dateFormat;
                    $this->options['ReservationTime'] = $reservationTime;
                    $this->options['MinPersons'] = $minPersons;
                    $this->options['MaxPersons'] = $maxPersons;
                    $this->options['LargeGroupsMessage'] = $largeGroupsMessage;
                    $this->options['MaxTime'] = $maxTime;
					$this->options['ManualReservation']        = $manualReservation;

                    $placeID = self::GetPost('Place');
                    $categories = $this->redi->getPlaceCategories($placeID);
                    if (isset($categories['Error'])) {
                        $errors[] = $categories['Error'];
                        $settings_saved = false;
                    }
                    $categoryID = $categories[0]->ID;
                    $this->options['OpenTime'] = self::GetPost('OpenTime');
                    $this->options['CloseTime'] = self::GetPost('CloseTime');

                    $getServices = $this->redi->getServices($categoryID);
                    if (isset($getServices['Error'])) {
                        $errors[] = $getServices['Error'];
                        $settings_saved = false;
                    }
                    if (count($getServices) != $services) {
                        if (count($getServices) > $services) {
                            //delete
							$diff = count( $getServices ) - $services;
							
							$cancel = $this->redi->deleteServices( $categoryID, $diff );
							if ( isset( $cancel['Error'] ) ) {
								$errors[]       = $cancel['Error'];
								$settings_saved = false;
							}
							$cancel = array();
                        } else {
                            //add
                            $diff = $services - count($getServices);

                            $cancel = $this->redi->createService($categoryID,
                                array(
                                    'service' => array(
                                        'Name' => 'Person',
                                        'Quantity' => $diff
                                    )
                                ));
                            if (isset($cancel['Error'])) {
                                $errors[] = $cancel['Error'];
                                $settings_saved = false;
                            }
                            $cancel = array();
                        }
                    }

                    $this->saveAdminOptions();

                    if (is_array($serviceTimes) && count($serviceTimes)) {
                        $cancel = $this->redi->setServiceTime($categoryID, $serviceTimes);
                        if (isset($cancel['Error'])) {
                            $errors[] = $cancel['Error'];
                            $settings_saved = false;
                        }
                        $cancel = array();
                    }
                    $cancel = $this->redi->setPlace($placeID, $place);
                    if (isset($cancel['Error'])) {
                        $errors[] = $cancel['Error'];
                        $settings_saved = false;
                    }
                    $cancel = array();
                }

                $places = $this->redi->getPlaces();
                if (isset($places['Error'])) {
                    $errors[] = $places['Error'];
                    $settings_saved = false;
                }
            }

            $this->options = get_option($this->optionsName);

            if ($settings_saved || !isset($_POST['submit'])) {
                $thanks = $this->GetOption('Thanks', 0);
                $manualReservation = $this->GetOption( 'ManualReservation', 0);
                $minPersons = $this->GetOption('MinPersons', 1);
                $maxPersons = $this->GetOption('MaxPersons', 10);
                $alternativeTimeStep = $this->GetOption('AlternativeTimeStep', 30);
                $largeGroupsMessage = $this->GetOption('LargeGroupsMessage', '');
                $maxTime = $this->GetOption('MaxTime', 1);

                $getServices = $this->redi->getServices($categoryID);
                if (isset($getServices['Error'])) {
                    $errors[] = $getServices['Error'];
                }

                $reservationTime = $this->getReservationTime();

				$custom_fields_save_to = $this->GetOption( 'customfields_save_to' );

				if ( $custom_fields_save_to === null || $custom_fields_save_to === CustomFieldsSaveTo::WPOptions ) {
					// Get custom fields from wordpress options
					$custom_fields = array();
					for ( $i = 0; $i != REDI_CUSTOM_FIELDS; $i ++ ) {
                    $field_name = 'field_' . $i . '_name';
                    $field_type = 'field_' . $i . '_type';
                    $field_required = 'field_' . $i . '_required';
                    $field_message = 'field_' . $i . '_message';

						$custom_fields[ $i ] = (object) array(
							'Id'       => 0,
							'Name'     => $this->GetOption( $field_name ),
							'Type'     => $this->GetOption( $field_type ),
							'Message'  => $this->GetOption( $field_message ),
							'Required' => $this->GetOption( $field_required )
						);
					}
				} else {
					$custom_fields = $this->redi->getCustomField( self::lang(), $placeID );
                }
            }

            if (!$settings_saved && isset($_POST['submit'])) {
                $alternativeTimeStep = self::GetPost('AlternativeTimeStep');
            }

            require_once(REDI_BOOKING_TEMPLATE . 'admin.php');
            require_once(REDI_BOOKING_TEMPLATE . 'advanced.php');

            wp_register_style('redi_booking_admin', REDI_BOOKING_PLUGIN_URL . 'css/redi-admin.css');
            wp_enqueue_style('redi_booking_admin');
        }

        private function GetOption($name, $default = null)
        {
            return isset($this->options[$name]) ? $this->options[$name] : $default;
        }

        private static function GetPost($name, $default = null, $post = null)
        {
            if ($post) {
                return isset($post[$name]) ? $post[$name] : $default;
            }

            return isset($_POST[$name]) ? $_POST[$name] : $default;
        }

        function GetServiceTimes()
        {
            $serviceTimes = array();
            foreach ($_POST['OpenTime'] as $key => $value) {
                if (self::set_and_not_empty($value)) {
                    $serviceTimes[$key]['OpenTime'] = $value;
                }
            }
            foreach ($_POST['CloseTime'] as $key => $value) {
                if (self::set_and_not_empty($value)) {
                    $serviceTimes[$key]['CloseTime'] = $value;
                }
            }

            return $serviceTimes;
        }

        function ajaxed_admin_page($placeID, $categoryID, $settings_saved = false)
        {
            require_once(plugin_dir_path(__FILE__) . 'languages.php');
            $places = $this->redi->getPlaces();
            $getServices = $this->redi->getServices($categoryID);

            if (!isset($_POST['submit']) || $settings_saved) {

                $serviceTimes = $this->redi->getServiceTime($categoryID); //goes to template 'admin'
                $serviceTimes = json_decode(json_encode($serviceTimes), true);
                $place = $this->redi->getPlace($placeID); //goes to template 'admin'

            } else {
                $place = array(
                    'Name' => self::GetPost('Name'),
                    'City' => self::GetPost('City'),
                    'Country' => self::GetPost('Country'),
                    'Address' => self::GetPost('Address'),
                    'Email' => self::GetPost('Email'),
                    'EmailCC' => self::GetPost('EmailCC'),
                    'Phone' => self::GetPost('Phone'),
                    'WebAddress' => self::GetPost('WebAddress'),
                    'Lang' => self::GetPost('Lang'),
                    'MinTimeBeforeReservation' => self::GetPost('MinTimeBeforeReservation'),
                    'Catalog' => (int)false,
                    'DateFormat' => self::GetPost('DateFormat')
                );
                $serviceTimes = self::GetServiceTimes();
            }
            require_once('countrylist.php');
            require_once(REDI_BOOKING_TEMPLATE . 'admin_ajaxed.php');
        }

        function init_sessions()
        {
            if (!session_id()) {
                session_start();
            }

            if (function_exists('load_plugin_textdomain')) {
                add_filter('load_textdomain_mofile', array($this, 'language_files'), 10, 2);
                load_plugin_textdomain('redi-booking', false, 'redi-booking/lang');
                load_plugin_textdomain('redi-booking-errors', false,
                    'redi-booking/lang');
            }
        }

        /**
         * @desc Adds the options subpanel
         */
        function redi_booking_admin_menu_link()
        {
            $icon = 'dashicons-groups';
            add_menu_page('Redi Restaurant Booking',
                'Redi Restaurant Booking',
                'manage_options',
                'redi-booking',
                array(&$this, 'redi_booking_admin_options_page'), $icon, 21 );
        }

        static function install()
        {
        }

        public function activate()
        {
            delete_option($this->_name . '_page_title');
            add_option($this->_name . '_page_title', $this->page_title, '', 'yes');

            delete_option($this->_name . '_page_name');
            add_option($this->_name . '_page_name', $this->page_name, '', 'yes');

            delete_option($this->_name . '_page_id');
            add_option($this->_name . '_page_id', $this->page_id, '', 'yes');

            $the_page = get_page_by_title($this->page_title);

            if (!$the_page) {
                // Create post object
                $_p = array();
                $_p['post_title'] = $this->page_title;
                $_p['post_content'] = $this->content;
                $_p['post_status'] = 'publish';
                $_p['post_type'] = 'page';
                $_p['comment_status'] = 'closed';
                $_p['ping_status'] = 'closed';
                $_p['post_category'] = array(1); // the default 'Uncategorized'
                // Insert the post into the database
                $this->page_id = wp_insert_post($_p);
            } else {
                // the plugin may have been previously active and the page may just be trashed...
                $this->page_id = $the_page->ID;

                //make sure the page is not trashed...
                $the_page->post_status = 'publish';
                $this->page_id = wp_update_post($the_page);
            }

            delete_option($this->_name . '_page_id');
            add_option($this->_name . '_page_id', $this->page_id);

            if ($this->ApiKey == null) {
                $this->register();
            }

        }

        private static function set_and_not_empty($value)
        {
            return (isset($value) && !empty($value));
        }

        public function deactivate()
        {
            $this->deletePage();
            $this->deleteOptions();
        }

        public static function uninstall()
        {
            self::deletePage(true);
            self::deleteOptions();
        }

        private function deletePage($hard = false)
        {
            $id = get_option(self::$name . '_page_id');
            if ($id && $hard == true) {
                wp_delete_post($id, true);
            } elseif ($id && $hard == false) {
                wp_delete_post($id);
            }
        }

        private function deleteOptions()
        {
            delete_option(self::$name . '_page_title');
            delete_option(self::$name . '_page_name');
            delete_option(self::$name . '_page_id');
        }

        function getCalendarDateFormat($format)
        {
            switch ($format) {
                case 'MM-dd-yyyy':
                    return 'mm-dd-yy';

                case 'dd-MM-yyyy':
                    return 'dd-mm-yy';

                case 'yyyy.MM.dd':
                    return 'yy.mm.dd';

                case 'MM.dd.yyyy':
                    return 'mm.dd.yy';

                case 'dd.MM.yyyy':
                    return 'dd.mm.yy';

                case 'yyyy/MM/dd':
                    return 'yy/mm/dd';

                case 'MM/dd/yyyy':
                    return 'mm/dd/yy';

                case 'dd/MM/yyyy':
                    return 'dd/mm/yy';
            }

            return 'yy-mm-dd';
        }

        function getPHPDateFormat($format)
        {
            switch ($format) {
                case 'MM-dd-yyyy':
                    return 'm-d-Y';

                case 'dd-MM-yyyy':
                    return 'd-m-Y';

                case 'yyyy.MM.dd':
                    return 'Y.m.d';

                case 'MM.dd.yyyy':
                    return 'm.d.Y';

                case 'dd.MM.yyyy':
                    return 'd.m.Y';

                case 'yyyy/MM/dd':
                    return 'Y/m/d';

                case 'MM/dd/yyyy':
                    return 'm/d/Y';

                case 'dd/MM/yyyy':
                    return 'd/m/Y';
            }

            return 'Y-m-d';
        }

        public function shortcode($attributes)
        {
            if (is_array($attributes) && is_array($this->options)) {
                $this->options = array_merge($this->options, $attributes);
            }

            ob_start();
            wp_enqueue_script('jquery');
            wp_enqueue_style('jquery_ui');
			
            //calendar localization
            if(!wp_script_is('jquery-ui-datepicker-localization')) {
                wp_register_script('jquery-ui-datepicker-localization', REDI_BOOKING_PLUGIN_URL . 'lib/datetimepicker/js/jquery.ui.i18n.all.min.js');
			}

            if(!wp_style_is('jquery-ui-custom-style')) {
                wp_register_style('jquery-ui-custom-style',
                    REDI_BOOKING_PLUGIN_URL . 'css/custom-theme/jquery-ui-1.8.18.custom.css','jquery_ui', '1.8.18');
            }
            wp_enqueue_style('jquery-ui-custom-style');
            wp_enqueue_script('jquery-ui-datepicker');
			wp_enqueue_script('jquery-ui-datepicker-localization');

            //DateTime parsing library
            if(!wp_script_is('moment')) {
                wp_register_script('moment', REDI_BOOKING_PLUGIN_URL . 'lib/moment/moment.js');
            }
            wp_enqueue_script('moment');

            wp_register_script('redi_booking_frontend_custom_fields', REDI_BOOKING_PLUGIN_URL . 'js/redi-booking-frontend-custom-fields.js');
            wp_register_script('redi_booking_frontend_vars', REDI_BOOKING_PLUGIN_URL . 'js/redi-booking-frontend-vars.js');
            wp_register_script('redi_booking_frontend_forms', REDI_BOOKING_PLUGIN_URL . 'js/redi-booking-frontend-forms.js', array('jquery'));

            $time_format = get_option('time_format');
            $time_format_s = explode(':',$time_format);

            if (isset($time_format_s[0]) && in_array($time_format_s[0], array('g','h'))) :
                $time_format = 'h:mm tt';
            else:
                $time_format = 'HH:mm';
            endif;

            wp_localize_script('redi_booking_frontend_forms',
                'redi_booking',
                array( // URL to wp-admin/admin-ajax.php to process the request
                    'ajaxurl' => admin_url('admin-ajax.php'),
                    'id_missing' => esc_html__('Reservation number can\'t be empty', 'redi-booking'),
                    'name_missing' => esc_html__('Name can\'t be empty', 'redi-booking'),
                    'email_missing' => esc_html__('Email can\'t be empty', 'redi-booking'),
                    'phone_missing' => esc_html__('Phone can\'t be empty', 'redi-booking'),
                    'reason_missing' => esc_html__('Reason can\'t be empty', 'redi-booking'),
                ));

            wp_register_style('redi_booking_styles', REDI_BOOKING_PLUGIN_URL . 'css/restaurant.css');
            wp_enqueue_style('redi_booking_styles');

            wp_register_style('google_fonts_sans_narrow', add_query_arg(array(
                'family' => 'PT+Sans+Narrow:400,700',
                'subset' => 'latin,cyrillic'
            ), "//fonts.googleapis.com/css"), array(), null);

            wp_enqueue_style('google_fonts_sans_narrow');

	        wp_register_style('font-awesome', REDI_BOOKING_PLUGIN_URL . 'lib/font-awesome-4.7.0/css/font-awesome.min.css',null, '4.7.0');
	        wp_enqueue_style('font-awesome');

            $apiKeyId = (int)$this->GetOption('apikeyid');

            if ($apiKeyId) {
                $this->ApiKey = $this->GetOption('apikey' . $apiKeyId, $this->ApiKey);

                $check = get_option($this->apiKeyOptionName . $apiKeyId);
                if ($check != $this->ApiKey) { // update only if changed
                    //Save Key if newed
                    update_option($this->apiKeyOptionName . $apiKeyId, $this->ApiKey);
                }
                $this->redi->setApiKey($this->ApiKey);
            }
            if ($this->ApiKey == null) {
                $this->display_errors(array(
                    'Error' => '<div class="error"><p>' . esc_html_e('Online reservation service is not available at this time. Try again later or contact us directly.',
                            'redi-booking') . '</p></div>'
                ));

                return;
            }

            //places
            $places = $this->redi->getPlaces();
            if (isset($places['Error'])) {
                $this->display_errors($places, true);
                die;
            }

            if (isset($this->options['placeid'])) {
                $places = array((object)array('ID' => $this->options['placeid']));
            }
            $placeID = $places[0]->ID;

            $categories = $this->redi->getPlaceCategories($placeID);
            if (isset($categories['Error'])) {
                $this->display_errors($categories, true);
                die;
            }

            $categoryID = $categories[0]->ID;
            $time_format = get_option('time_format');
            $date_format_setting = isset($this->options['DateFormat']) ? $this->options['DateFormat'] : null;
            $date_format = $this->getPHPDateFormat($date_format_setting);
            $calendar_date_format = $this->getCalendarDateFormat($date_format_setting);
            $MinTimeBeforeReservation = (int)(isset($this->options['MinTimeBeforeReservation']) && $this->options['MinTimeBeforeReservation'] > 0 ? $this->options['MinTimeBeforeReservation'] : 0) + 1;
            $reservationStartTime = strtotime('+' . $MinTimeBeforeReservation . ' hour',
                current_time('timestamp'));
            $startDate = date($date_format, $reservationStartTime);
            $startDateISO = date('Y-m-d', $reservationStartTime);
            $startTime = mktime(date("G", $reservationStartTime), 0, 0, 0, 0, 0);

            $minPersons = $this->GetOption('MinPersons', 1);
            $maxPersons = $this->GetOption('MaxPersons', 10);
            $largeGroupsMessage = $this->GetOption('LargeGroupsMessage', '');

            $maxTime = $this->GetOption('MaxTime', 1);
            $thanks = $this->GetOption('Thanks');
			$manualReservation = $this->GetOption( 'ManualReservation');
            $time_format_hours = self::dropdown_time_format();
			$custom_fields_save_to = $this->GetOption( 'customfields_save_to' );

			if ($custom_fields_save_to == CustomFieldsSaveTo::WPOptions) {
				// Get custom fields from wordpress options
				$custom_fields = array();
				for ( $i = 0; $i != REDI_CUSTOM_FIELDS; $i ++ ) {
					$field_name     = 'field_' . $i . '_name';
					$field_type     = 'field_' . $i . '_type';
					$field_required = 'field_' . $i . '_required';
					$field_message  = 'field_' . $i . '_message';

					$custom_fields[ $i ] = (object) array(
						'Id'       => 0,
						'Name'     => $this->GetOption( $field_name ),
						'Type'     => $this->GetOption( $field_type ),
						'Message'  => $this->GetOption( $field_message ),
						'Required' => $this->GetOption( $field_required )
					);
				}
			} else {
				$custom_fields = $this->redi->getCustomField( self::lang(), $placeID );
			}

            $hide_clock = false;
            $persons = 1;
            $all_busy = false;
            //pre call
            $categories = $this->redi->getPlaceCategories($placeID);
            $categoryID = $categories[0]->ID;
            $step1 = self::object_to_array(
                $this->step1($categoryID,
                    array(
                        'startDateISO' => $startDateISO,
                        'startTime' => '0:00',
                        'persons' => 1,
                        'lang' => get_locale()
                    )
                )
            );
            $hide_clock = true;
            $dates = array();
            $date_date = $reservationStartTime;
            for ($dates_index = 0; $dates_index != REDI_TOTALDATES; $dates_index++) {
                $dates[] =
                    array(
                        'month' => date_i18n('M', $date_date),
                        'day' => date_i18n('d', $date_date),
                        'weekday' => date_i18n('D', $date_date),
                        'hidden' => date('Y-m-d', $date_date)
                    );
                $date_date = strtotime("+1 day", $date_date);
            }

            $d_dates = $this->redi->getBlockingDates(self::lang(), $categoryID, array(
                'StartTime' => date('Y-m-d'),
                'EndTime' => date('Y-m-d', strtotime('+12 month')) // + 3 months
            ));

            $disabled_dates = array();

            foreach ($d_dates as $date) {
                if ($date->Blocked || !$date->Open) {
                    array_push($disabled_dates,  
                        array("date" => $date->Date,
                    "reason" => $date->Reason));
                }
            }

            $js_locale = get_locale();
            $datepicker_locale = substr($js_locale, 0, 2);
			if (isset($this->options['ManualReservation']) && $this->options['ManualReservation'] == 1)
			{
				$manual = true;
			}
            
            $paymentMode = apply_filters('redi-booking-payment-mode', false);
            
            wp_localize_script('redi_booking_frontend_vars',
                'redi_frontend_vars',
                array( // URL to wp-admin/admin-ajax.php to process the request
                    'plugin_name' => esc_html__('ReDi Restaurant Booking version ' . $this->version),
                    'date_format' => $calendar_date_format,
                    'payment' => $paymentMode,
                    'time_format' => $time_format,
                    'locale' => $js_locale,
                    'datepicker_locale' => $datepicker_locale,
                    'apikeyid' => $apiKeyId,
                    'disabled_dates' => json_encode($disabled_dates, JSON_PRETTY_PRINT)
                ));

            if (array_key_exists('action', $_GET) && 
                ($_GET['action'] === "booking-confirmation" || $_GET['action'] === "booking-payment-confirmation"))
            {
                global $wp;
                $url = home_url($wp->request);
                
                $data = array(
					'quantity' => $_GET['quantity'],
					'id'       => $_GET['id'],
					'lang'     => $_GET['lang'],
                    'startdate' => $_GET['startdate'],
                    'return-url' => $url
				);
                
                if ($_GET['action'] === "booking-confirmation")
                {
                    $redirect = apply_filters('redi-booking-confirmation', $data);
                }

                if ($redirect == null || is_array($redirect))
                {
                    $reservation_id = $_GET['id'];
                    require_once(REDI_BOOKING_TEMPLATE . 'frontend-confirmation.php');
                }
                else
                {
                    self::redirect($redirect);
                }
            }
            else
            {
                wp_enqueue_script('redi_booking_frontend_custom_fields');
                wp_enqueue_script('redi_booking_frontend_vars');
                wp_enqueue_script('redi_booking_frontend_forms');
                require_once(REDI_BOOKING_TEMPLATE . 'frontend.php');
                require_once(REDI_BOOKING_TEMPLATE . 'frontend-cancel.php');
            }

            $out = ob_get_contents();

            ob_end_clean();

            return $out;
        }

        function confirm_reservation($args)
        {
            $response = $this->redi->confirmReservation($args['reservation_id'], $args['params']);

            $content = apply_filters('redi-booking-email-content', array(
                'id' => (int)$args['reservation_id'],
                'type' => EmailContentType::Confirmed,
                'lang' => str_replace('_', '-', self::GetPost('lang'))
            ));
            
            do_action( 'redi-booking-send-confirmation-email', $content );
        }

        function redi_booking_get_email_content($args)
        {
            return $this->redi->getEmailContent(
                $args['id'],
                $args['type'],
                array(
                    'Lang' => $args['lang']
                )
            );
        }

        function send_confirmation_email($emailContent)
        {
            if (!isset($emailContent['Error'])) 
            {
                $blogName = wp_specialchars_decode(get_option('blogname'), ENT_QUOTES);
                $adminEmail = get_option('admin_email');

                wp_mail($emailContent['To'], $emailContent['Subject'], $emailContent['Body'],
                    array(
                        'Content-Type: text/html; charset=UTF-8',
                        'From: '  . $blogName . ' <' . $adminEmail . '>' . "\r\n"
                    ));
            }
        }

		private static function redirect( $url ) {
			$string = '<script type="text/javascript">';
			$string .= 'window.location = "' . $url . '"';
			$string .= '</script>';

			echo $string;
		}

        function dropdown_time_format()
        {
            $time_format = get_option('time_format');
            $time_format_array = str_split($time_format);
            foreach ($time_format_array as $index => $format_char) // some users have G \h i \m\i\n
            {
                if ($format_char === '\\') {
                    $time_format_array[$index] = '';
                    if (isset($time_format_array[$index + 1])) {
                        $time_format_array[$index + 1] = '';
                    }
                }
            }
            $time_format = implode('', $time_format_array);
            $is_am_pm = strpos($time_format, 'g');
            $is_am_pm_lead_zero = strpos($time_format, 'h');

            $is_24 = strpos($time_format, 'G');
            $is_24_lead_zero = strpos($time_format, 'H');

            if ($is_am_pm !== false || $is_am_pm_lead_zero !== false) {
                $a = stripos($time_format, 'a');
                $am_text = '';
                if ($a !== false) {
                    $am_text = $time_format[$a];
                }
                if ($is_am_pm !== false) {
                    return $time_format[$is_am_pm] . ' ' . $am_text;
                }
                if ($is_am_pm_lead_zero !== false) {
                    return $time_format[$is_am_pm_lead_zero] . ' ' . $am_text;
                }
            }
            if ($is_24 !== false) {
                return $time_format[$is_24];
            }
            if ($is_24_lead_zero !== false) {
                return $time_format[$is_24_lead_zero];
            }

            return 'H'; //if no time format found use 24 h with lead zero
        }

        private function step1($categoryID, $post, $placeID = null)
        {
            // convert date to array
            $date = date_parse(self::GetPost('startDateISO', null, $post) . ' 00:00');

            if ($date['error_count'] > 0) {
                echo json_encode(array(
                    'Error' => esc_html_e('Selected date or time is not valid.', 'redi-booking')
                ));
                die;
            }

            $persons = (int)$post['persons'];

			$StartTime = gmdate( 'Y-m-d 00:00', strtotime( $post['startDateISO'] ) ); //CalendarDate + 00:00
			$EndTime   = gmdate( 'Y-m-d 00:00',
					strtotime( '+1 day', strtotime( $post['startDateISO'] ) ) ); //CalendarDate + 1day + 00:00
			$currentTimeISO = date('Y-m-d H:i', current_time('timestamp'));

            $params = array(
                'StartTime' => $StartTime,
                'EndTime' =>  $EndTime,
                'Quantity' => $persons,
                'Lang' => str_replace('_', '-', $post['lang']),
                'CurrentTime' => $currentTimeISO,
                'AlternativeTimeStep' => self::getAlternativeTimeStep($persons)
            );

			$params['ReservationDuration'] = $this->getReservationTime($persons);
			$query = $this->redi->availabilityByDay($categoryID, $params);

            $time_format = get_option('time_format');

            if (isset($query['Error'])) {
                return $query;
            }
            unset($query['debug']);
  
			foreach ($query as $q2) {
				if (isset($q2->Availability)) {
					foreach ($q2->Availability as $q) {
                        
						$q->StartTimeISO = $q->StartTime;
						$q->StartTime = date($time_format, strtotime($q->StartTime));
						$q->EndTime = date($time_format, strtotime($q->EndTime));
					}
				}
			} 

			return $query;
        }

        function save_reservation($params, $reservation)
        {
            if (isset($reservation['Error']))
            {
                return;
            }

            global $wpdb;
            
            $reservParams = $params['reservation'];

            $wpdb->insert( $this->table_name, [	
                'reservation_number' => $reservation['ID'],
                'name'    			 => $reservParams['UserName'],
                'phone'  		     => $reservParams['UserPhone'],
                'email'     		 => $reservParams['UserEmail'],
                'date_from'     	 => $reservParams['StartTime'],
                'date_to'          	 => $reservParams['EndTime'],
                'guests'        	 => $reservParams['Quantity'],
                'comments'           => $reservParams['UserComments'],			
                'prepayment'         => $reservParams['PrePayment'],			
                'currenttime'        => $reservParams['CurrentTime'],			
                'language'           => $reservParams['Lang']	
            ] );

            if (!isset($params['reservation']['Parameters']))
            {
                return;
            }

            $custom_fields = $params['reservation']['Parameters'];

            foreach ($custom_fields as $custom_field)
            {
                $wpdb->insert( $this->table_name . '_custom_fields', [	
                    'reservation_number' => $reservation['ID'],
                    'name'    			 => $custom_field['Name'],
                    'type'  		     => $custom_field['Type'],
                    'value'     		 => $custom_field['Value']
                ] );
            }
        }

        function redi_booking_ajax()
        {
            $apiKeyId = $this->GetPost('apikeyid');
            
            if ($apiKeyId) {
                $this->ApiKey = get_option($this->apiKeyOptionName . $apiKeyId);
                $this->redi->setApiKey($this->ApiKey);
            }

            if (isset($_POST['placeID'])) {
                $placeID = (int)self::GetPost('placeID');
                $categories = $this->redi->getPlaceCategories($placeID);
                if (isset($categories['Error'])) {
                    echo json_encode($categories);
                    die;
                }
                $categoryID = $categories[0]->ID;

            }
            switch ($_POST['get']) {
                case 'step1':
                    echo json_encode($this->step1($categoryID, $_POST, $placeID));
                    break;

                case 'step2':

                    $persons = (int)self::GetPost('persons');
                    // convert to int
                    $startTimeStr = self::GetPost('startTime');

                    // convert to int
                    $startTimeInt = strtotime($startTimeStr, 0);

                    // calculate end time
                    $endTimeInt = strtotime('+' . $this->getReservationTime($persons) . 'minutes', $startTimeInt);

                    // format to ISO
                    $startTimeISO = date('Y-m-d H:i', $startTimeInt);
                    $endTimeISO = date('Y-m-d H:i', $endTimeInt);
                    $currentTimeISO = date('Y-m-d H:i', current_time('timestamp'));

                    $params = array(
                        'StartTime' => $startTimeISO,
                        'EndTime' => $endTimeISO,
                        'Quantity' => $persons,
                        'Alternatives' => 0,
                        'Lang' => str_replace('_', '-', self::GetPost('lang')),
                        'CurrentTime' => $currentTimeISO,
                        'AlternativeTimeStep' => self::getAlternativeTimeStep($persons),
                    );

                    $params['ReservationDuration'] = $this->getReservationTime($persons);
                    $query = $this->redi->query($categoryID, $params);

                    $time_format = get_option('time_format');

                    if (isset($query['Error'])) {
                        echo json_encode($query);
                        die;
                    }

                    unset($query['debug']);

                    foreach ($query as $q) {
                        $q->Select = ($startTimeISO == $q->StartTime && $q->Available);
                        $q->StartTimeISO = $q->StartTime;
                        $q->StartTime = date($time_format, strtotime($q->StartTime));
                        $q->EndTime = date($time_format, strtotime($q->EndTime));
                    }

                    echo json_encode($query[0]);
                    break;

                case 'step3':
                    $persons = (int)self::GetPost('persons');
                    $startTimeStr = self::GetPost('startTime');

                    // convert to int
                    $startTimeInt = strtotime($startTimeStr, 0);

                    // calculate end time
                    $endTimeInt = strtotime('+' . $this->getReservationTime($persons) . 'minutes', $startTimeInt);

                    // format to ISO
                    $startTimeISO = date('Y-m-d H:i', $startTimeInt);
                    $endTimeISO = date('Y-m-d H:i', $endTimeInt);
                    $currentTimeISO = date('Y-m-d H:i', current_time('timestamp'));
                    $comment = '';
					$parameters = array();
                    $custom_fields = array();
                    $custom_fields = $this->redi->getCustomField( self::lang(), $placeID );

                    foreach ($custom_fields as $custom_field) {
                        if (isset($_POST['field_' . $custom_field->Id])) {
                            $parameters[] = array(
                                'Name' => $custom_field->Name,
                                'Text' => $custom_field->Text,
                                'Type' => $custom_field->Type,
                                'Print' => $custom_field->Print ? 'true' : 'false',
                                'Value' =>
                                    $custom_field->Type === 'text' ||  $custom_field->Type === 'dropdown' || $custom_field->Type === 'options' ?
                                        self::GetPost('field_' . $custom_field->Id) : (self::GetPost('field_' . $custom_field->Id) === 'on' ? 'true' : 'false'));
                        }
                    }

                    $comment .= mb_substr(self::GetPost('UserComments', ''), 0, 250);
                    $paymentMode = apply_filters('redi-booking-payment-mode', null);

                    $params = array(
                        'reservation' => array(
                            'StartTime' => $startTimeISO,
                            'EndTime' => $endTimeISO,
                            'Quantity' => $persons,
                            'UserName' => self::GetPost('UserName'),
                            'UserEmail' => self::GetPost('UserEmail'),
                            'UserComments' => $comment,
                            'UserPhone' => self::GetPost('UserPhone'),
                            'Name' => 'Person',
                            'Lang' => str_replace('_', '-', self::GetPost('lang')),
                            'CurrentTime' => $currentTimeISO,
                            'Version' => $this->version,
                            'PrePayment' => $paymentMode ? 'true' : 'false',
                            'Source' => 'HOMEPAGE'
                        )
                    );

					if (isset($this->options['ManualReservation']) && $this->options['ManualReservation'] == 1)
                    {
                        $params['reservation']['ManualConfirmationLevel'] = 100; 
                    }

                    if (!empty($parameters))
                    {
						$params['reservation']['Parameters'] = $parameters;
                    }
                    
                    $reservation = $this->redi->createReservation($categoryID, $params);
                    
                    if (!isset( $reservation['Error'])) 
                    {
                        $this->save_reservation($params, $reservation);

                        if (!$paymentMode) {
                            //call api for content
                            $content = apply_filters('redi-booking-email-content', array(
                                'id' =>  (int)$reservation['ID'],
                                'type' => EmailContentType::Confirmed,
                                'lang' => str_replace('_', '-', self::GetPost('lang'))
                            ));
                            
                            do_action( 'redi-booking-send-confirmation-email', $content );
                            //send
                        }                        
                    }

                    echo json_encode($reservation);
                    
                    break;

                case 'get_place':
                    self::ajaxed_admin_page($placeID, $categoryID, true);
                    break;

                case 'get_custom_fields':
                    echo $this->getCustomFields($placeID);
                    break;
    

                case 'cancel':
                    $params = array(
                        'ID' => self::GetPost('ID'),
                        'personalInformation' => self::GetPost('PersonalInformation'),
                        'Reason' => mb_substr(self::GetPost('Reason'), 0, 250),
                        "Lang" => str_replace('_', '-', self::GetPost('lang')),
                        'CurrentTime' => date('Y-m-d H:i', current_time('timestamp')),
                        'Version' => self::plugin_get_version()
                    );
                    $cancel = $this->redi->cancelReservationByClient($params);
                    echo json_encode($cancel);

                    break;
            }

            die;
        }

        private function getCustomFields($placeID)
        {
            $html = '';
            $custom_fields = $this->redi->getCustomField(self::lang(), $placeID);
            
            foreach ( $custom_fields as $custom_field ) {

                $input_field_type = 'text'; 
                switch ($custom_field->Type) 
                {
                    case 'options': 
                        $input_field_type = 'radio';
                        break;
                    case 'dropdown': 
                        $input_field_type = 'dropdown';
                        break;
                    case 'newsletter':
                    case 'reminder':
                    case 'allowsms':
                    case 'checkbox':
                    case 'gdpr':
                        $input_field_type = 'checkbox';	
                        break;
                }
                
                $html .= '<div class="f_select2"><div class="f_input_box"><span class="selectArrow6';
                if (isset($custom_field->Required) && $custom_field->Required)
                {
                    $html .= ' required';
                }

                if ($input_field_type == 'radio')
                {
                    $html .= ' redi-radio-span';
                }

                $html .= '"></span>';

         
                if ($input_field_type == 'checkbox')
                {
                    $html .= '<span class="checkbox_field"><div class="redi-checkbox"><label><input class="';
                    if ( isset( $custom_field->Required ) && $custom_field->Required ): 
                        $html .= 'field_required';
                    endif; 
                    $html .= '" value="" type="checkbox" name="field_';
                    $html .= $custom_field->Id; 
                    $html .= '" id="field_';
                    $html .= $custom_field->Id;
                    $html .= '"><span class="cr"><i class="cr-icon fa fa-check"></i></span></label></div><label for="field_';
                    $html .= $custom_field->Id; 
                    $html .= '">';
                    $html .= $custom_field->Name;
                    $html .= '</label></span>';
                }
                else if ($input_field_type == 'text')
                {
                    $html .= '<input type="text" placeholder="';
                    $html .= $custom_field->Name;
                    $html .= '" value="" name="field_';
                    $html .= $custom_field->Id;
                    $html .= '" id="field_';
                    $html .= $custom_field->Id; 
                    $html .= '" class="text_field ';
                    
                    if (isset($custom_field->Required) && $custom_field->Required): 
                        $html .= 'field_required';
                    endif;
                    $html .= '"/>';
                }
                else if ($input_field_type == 'radio')
                {
                    $field_values = explode(',', $custom_field->Values);
                    $field_id = 0;

                    $html .= '<label class="redi-radiobutton-label redi-radiobutton-padding">' . $custom_field->Text . '</label><div class="clear-both"></div>';

                    foreach ( $field_values as $field_value ) {
                        if( $field_value ) {
                            $html .= '<div class="redi-radiobutton-padding"><input type="'.$input_field_type.'" value="'.$field_value.'" name="field_'.$custom_field->Id.'" id="field_'.$custom_field->Id.'_'.$field_id.'" class="redi-radiobutton';
                            
                            if( isset( $custom_field->Required ) && $custom_field->Required ) {
                                $html .= ' field_required';
                            } 
                            $html .= '"><label class="redi-radiobutton-label" for="field_'.$custom_field->Id.'_'.$field_id.'">'.$field_value.'</label></div><br/><div class="clear-both"></div>';
                        
                            $field_id++;
                        }
                    }
                }
                else if ($input_field_type == 'dropdown')
                {
                    $field_values = explode( ',', $custom_field->Values );
                    $html .= '<select id="field_'.$custom_field->Id.'" name="field_'.$custom_field->Id.'"';

                    if( isset( $custom_field->Required ) && $custom_field->Required ) {
                        $html .= ' class="field_required"';
                    }                          

                    $html .= '><option value="">' . $custom_field->Text . '</option>';

                    foreach ( $field_values as $field_value ) {
                        if( $field_value ) $html .= '<option value="'.$field_value.'">'.$field_value.'</option>';
                    }

                    $html .= '</select>';
                }

                $html .= '</div></div>';

                if ( isset( $custom_field->Required ) && $custom_field->Required ):
                    $html .= '<div class="clear-both"></div><div id="';
                    $html .= 'field_' . $custom_field->Id . '_message';
                    $html .= '" class="custom_field_required"><span class="validation_error">';
                    $html .= (!empty($custom_field->Message)) ? $custom_field->Message : (esc_html_e('Custom field is required', 'redi-booking'));
                    $html .= '</span></div>';
                endif;

                $html .= '<br/><div class="clear-both"></div>';
            } 
            
            return $html;
        }

        private function getAlternativeTimeStep($persons = 0)
        {
            $filename = plugin_dir_path(__FILE__) . 'alternativetimestep.json';

            if (file_exists($filename) && $persons) {
                $json = json_decode(file_get_contents($filename), true);
                if ($json !== null) {
                    if (array_key_exists($persons, $json)) {
                        return (int)$json[$persons];
                    }
                }
            }

            if (isset($this->options['AlternativeTimeStep']) && $this->options['AlternativeTimeStep'] > 0) 
            {
                return (int)$this->options['AlternativeTimeStep'];
            }

            return 30;
        }

        private function getReservationTime($persons = 0)
        {
            $filename = plugin_dir_path(__FILE__) . 'reservationtime.json';

            if (file_exists($filename) && $persons) {
                $json = json_decode(file_get_contents($filename), true);
                if ($json !== null) {
                    if (array_key_exists($persons, $json)) {
                        return (int)$json[$persons];
                    }
                }
            }

            if (isset($this->options['ReservationTime']) && $this->options['ReservationTime'] > 0) {
                return (int)$this->options['ReservationTime'];
            }

            return 3 * 60;
        }

        private function object_to_array($object)
        {
            return json_decode(json_encode($object), true);
        }
        
        private static function lang()
        {
			return str_replace( '_', '-', get_locale());
		}
    }
}
new ReDiBooking();

register_activation_hook(__FILE__, array('ReDiBooking', 'install'));