<?php
/**
 * API functions for Rating Form
 */
class Rating_Form_API {

	public static function getDefaultAtts() {
		return array(
			'id' => '',
			'post_id' => '',
			'comment_id' => '0',
			'custom_id' => '0',
			'user_id' => '0',
			'term_id' => '0',
			'title' => 'false',
			'score' => 'true',
			'total' => 'true',
			'stats' => 'true',
			'user_stats' => 'false',
			'tooltip' => 'true',
			'result' => 'false',
			'rich_snippet' => 'true',
			'widget' => 'false',
			'comment_review' => 'false',
			'state' => '',
			'custom_id_group' => '0',
			'before_content' => '',
			'after_content' => '',
			'count_type' => '',
			'rates' => __( '%count% rating', 'rating-form' ) . ',' . __( '%count% ratings', 'rating-form' ),
			'txt_score' => __( '%1$s/%2$s', 'rating-form' )
		);
	}

	/**
	 * Initialize attributes
	 *
	 * @param  array $args Rating Form arguments
	 * @return array       arguments
	 */
	public static function initArgs( $args ) {
		$args['id'] = empty($args['id']) ? 0 : intval($args['id']);
		$inputPostID = (!empty($args['post_id']));
		$post_id = get_the_ID() === false ? 0 : get_the_ID();
		$args['post_id'] = $inputPostID ? intval($args['post_id']) : $post_id;
		$args['comment_id'] = empty($args['comment_id']) ? 0 : intval($args['comment_id']);
		$args['custom_id'] = empty($args['custom_id']) ? '0' : trim(strtolower($args['custom_id']));
		$args['user_id'] = empty($args['user_id']) ? 0 : intval($args['user_id']);
		$args['term_id'] = empty($args['term_id']) ? 0 : intval($args['term_id']);
		$args['txt_score'] = htmlentities($args['txt_score']);
		// Attr values to boolean
		$args['title'] = $args['title'] == 'true' ? true : false;
		$args['score'] = $args['score'] == 'true' ? true : false;
		$args['total'] = $args['total'] == 'true' ? true : false;
		$args['stats'] = $args['stats'] == 'true' ? true : false;
		$args['user_stats'] = $args['user_stats'] == 'true' ? true : false;
		$args['tooltip'] = $args['tooltip'] == 'true' ? true : false;
		$args['result'] = $args['result'] == 'true' ? true : false;
		$args['rich_snippet'] = $args['rich_snippet'] == 'true' ? true : false;
		$args['widget'] = array_key_exists("widget", $args) && $args['widget'] == 'true' ? true : false;
		$args['comment_review'] = array_key_exists("comment_review", $args) && $args['comment_review'] == 'true' ? true : false;
		$args['state'] = strtolower($args['state']);
		$args['custom_id_group'] = Rating_Form_API::convertFieldIDArray($args['custom_id_group']);
		$args['user_check'] = empty($args['user_check']) ? false : $args['user_check'] == 'true' ? true : false;
		$args['count_type'] = trim(strtolower($args['count_type']));
		// Initialize empty ID
		$args['comment_id'] = (have_comments() && in_the_loop() && empty($args['comment_id']) && !$args['result']) ? get_comment_ID() : $args['comment_id'];
		// Set post_id to 0 if Term ID or User ID used
		$userExists = get_userdata( $args['user_id'] );
		if ((!empty($args['term_id']) || $userExists) && !$inputPostID) $args['post_id'] = 0;
		// Admin preview - Set post id to 1
		if ( is_admin() && ( !defined( 'DOING_AJAX' ) || !DOING_AJAX ) ) $args['post_id'] = 1;

		return $args;
	}

	/**
	 * Get rating data based on arguments
	 *
	 * @param  array  $args      Rating Form arguments
	 * @return array
	 */
	public static function getData( $args, $settings, $options ) {
		global $wpdb;

		$typeTud = (Rating_Form_Setup::form_types( $settings['type'], 'type' ) == "tud");
		$typeStar = (Rating_Form_Setup::form_types( $settings['type'], 'type' ) == "star");
		$post_id = empty($args['post_id']) ? 0 : $args['post_id'];
		$comment_id = empty($args['comment_id']) ? 0 : $args['comment_id'];
		$term_id = empty($args['term_id']) ? 0 : $args['term_id'];
		$user_id = empty($args['user_id']) ? 0 : $args['user_id'];
		$custom_id = empty($args['custom_id']) ? '0' : $args['custom_id'];
		$result = (array_key_exists("result", $args) && $args['result'] == 'true');
		$excludeBlockedIPs = empty($options['exclude_ip']) ? false : $options['exclude_ip'];
		$checkUser = empty($args['user_check']) ? false : $args['user_check']; // Validate user for existing ratings
		$review_rating = array_key_exists("comment_review", $args) && $args['comment_review'] == 'true' ? 'r' : '';

		$current_user = wp_get_current_user();
		$emptyQuery = array(
			'last_id' => 0,
			'last_date' => 0,
			'average' => 0,
			'up_count' => 0,
			'down_count' => 0,
			'count' => 0,
			'count_percent' => 0
		);

		if ($checkUser) {
			// Check IP or User ID
			$ip = Rating_Form_API::getIP();
			$user = $current_user->ID;
			$user_ip = empty($user) ? 'ip' : 'user';
			$user_ip_val = empty($user) ? $ip : $user;
		}

		$sql = "SELECT ";
		if ($checkUser) {
			$sql .= '* ';
		} else {
			$sql .= "MAX(rate_id) AS last_id, ";
			$sql .= "MAX(date) AS last_date, ";
			$sql .= "0 AS count_percent, ";
			$sql .= "SUM(CASE WHEN rated = '" . $review_rating . "1u' then 1 else 0 end) AS up_count, ";
			$sql .= "SUM(CASE WHEN rated = '" . $review_rating . "1d' then 1 else 0 end) AS down_count, ";
			$sql .= "SUM(CASE WHEN rated REGEXP '^" . $review_rating . "[0-9]+$' then 1 else 0 end) AS count, ";
			$sql .= "(SUM(CASE WHEN rated REGEXP '^" . $review_rating . "[0-9]+$' then rated else 0 end)/";
			$sql .= "SUM(CASE WHEN rated REGEXP '^" . $review_rating . "[0-9]+$' then 1 else 0 end)) AS average ";
		}
		$sql .= "FROM " . $wpdb->prefix . Rating_Form_Setup::TBL_RATING_LOG ." WHERE ";
		$sql .= "post_id = %d AND ";
		$sql .= "comment_id = %d AND ";
		$sql .= "term_id = %d AND ";
		$sql .= "user_id = %d AND ";
		if ( is_array($custom_id) && count($custom_id) > 1 ) {
			$sql .= "custom_id IN('" . implode("','", $custom_id) . "') ";
		} else {
			$sql .= "custom_id = '" . $custom_id . "' ";
		}
		if (!$checkUser && $excludeBlockedIPs) $sql .= "AND ip NOT IN (SELECT ip FROM " . $wpdb->prefix . Rating_Form_Setup::TBL_RATING_BLOCK_LIST . ") ";
		if ($checkUser) {
			$sql .= "AND ".$user_ip." = '" . $user_ip_val . "' ";
			$sql .= 'ORDER by date DESC LIMIT 1;';
		}
		$sqlArgs = array(
			$post_id,
			$comment_id,
			$term_id,
			$user_id
		);

		$query = $wpdb->get_row( $wpdb->prepare($sql, $sqlArgs), ARRAY_A );
	    if ($wpdb->num_rows == 0) return $emptyQuery;
		if ($checkUser) return $query;

		// Check wether args id are being used
		$argsUsed = Rating_Form_API::checkAttrID($args);

		// Stars - Fake ratings
		if ($options['fake_ratings'] && !$argsUsed) {
			// Post Meta
			$pmeta_fake_average = get_post_meta($args['post_id'], 'rf_fake_average', true);
			if (empty($pmeta_fake_average)) $pmeta_fake_average = 0;
			$pmeta_fake_total = get_post_meta($args['post_id'], 'rf_fake_total', true);
			if (empty($pmeta_fake_total)) $pmeta_fake_total = 0;

			// STAR
			$query['average'] = $pmeta_fake_average;
			$query['count'] = $pmeta_fake_total;
		}

		// Thumbs Up / Down - Fake ratings
		if ($options['fake_ratings'] && !$argsUsed) {
			// Post Meta
			$pmeta_fake_up = get_post_meta($args['post_id'], 'rf_fake_up', true);
			if (empty($pmeta_fake_up)) $pmeta_fake_up = 0;
			$pmeta_fake_down = get_post_meta($args['post_id'], 'rf_fake_down', true);
			if (empty($pmeta_fake_down)) $pmeta_fake_down = 0;

			// TUD
			$query['up_count'] = $pmeta_fake_up;
			$query['down_count'] = $pmeta_fake_down;
		}

		// Thumbs Up / Down total
		$thumbsUp = intval($query['up_count']);
		$thumbsDown = intval($query['down_count']);
		$query['tud_count'] = ($thumbsUp + $thumbsDown);

        $countPercent = 0;
        if ($typeStar) {
			$query['average'] = round($query['average'], $options['round']);
			$countPercent = (($query['average'] / $options['max']) * 100);
		}
        else if ($typeTud) {
            $countPercent = $query['tud_count'] == 0 ? 0 : (($thumbsUp / $query['tud_count']) * 100);
			$query['count'] = $query['tud_count'];
        }
		$query['count_percent'] = round($countPercent, $options['round']);
		// Thumbs up - down - 4 -2 = count = 2
		if ($options['up_minus_down']) {
			$upMinusDown = ($thumbsUp - $thumbsDown);
			$query['tud_count'] = $upMinusDown == 0 ? 0 : $upMinusDown;
		}

		// $query['results'] = Rating_Form_API::getResults( $args );
		return $query;
	}

	/**
	 * Retrieve Rating data based on post id / comment_id / custom_id / user_id / term_id attributes
	 * @param $atts - List attributes
	 * @return array rating result
	 */
	public static function getResults( $args ) {//TODO:: combine getData and getResults, sql:innerjoin?
		global $post;
		global $wpdb;

		$ratingData = array();
		$review_rating = '';

		// Attributes
		$args = shortcode_atts(
			array(
				'id' => '',
				'post_id' => '',
				'comment_id' => '0',
				'custom_id' => '0',
				'user_id' => '0',
				'term_id' => '0',
				'state' => '',
				'type' => 'star'
			), $atts );

		// Validate Args
		$args['id'] = strtolower($args['id']);
		$args['post_id'] = empty($args['post_id']) ? (isset($post) ? $post->ID : 0) : intval($args['post_id']);
		$args['comment_id'] = empty($args['comment_id']) ? 0 : intval($args['comment_id']);
		$args['custom_id'] = empty($args['custom_id']) ? '0' : strtolower($args['custom_id']);
		$args['user_id'] = empty($args['user_id']) ? 0 : intval($args['user_id']);
		$args['term_id'] = empty($args['term_id']) ? 0 : intval($args['term_id']);
		$args['state'] = strtolower($args['state']);
		$args['type'] = empty($args['type']) ? 'star' : strtolower($args['type']);
		$isStateReview = ($args['state'] == 'review' ? true : false);//TODO: enable review in rf settings (do not use state=review)

		$sql = "SELECT * FROM " . $wpdb->prefix . Rating_Form_Setup::TBL_RATING_ADD_FORM . " WHERE form_id = %d";
		$this_form_query = $wpdb->get_row( $wpdb->prepare( $sql, $args['id'] ) , ARRAY_A );

		// Options
		$jsonDisplay = empty($this_form_query['display']) ? array() : json_decode($this_form_query['display'], true);
		$max_rating = empty($jsonDisplay['max']) ? 0 : intval($jsonDisplay['max']);
		$round = empty($jsonDisplay['round']) ? 0 : intval($jsonDisplay['round']);

		if ($isStateReview) {
			$review_rating = 'r';
		}

		// Args in SQL
		$argsInSQL = "comment_id = " . $args['comment_id'] .
						" AND custom_id = '" . $args['custom_id'] . "' AND user_id = " . $args['user_id'] .
						" AND term_id = " . $args['term_id'];
		// Average rating
		$sqlAverage = "SELECT ".
				"(SUM(" . ($isStateReview ? "REPLACE(rated, '" . $review_rating . "', '')" : "rated") . ") / COUNT(*)) AS average_rating, " .
				"COUNT(*) AS total " .
				"FROM " . $wpdb->prefix . Rating_Form_Setup::TBL_RATING_LOG . " " .
				"WHERE ".$argsInSQL." AND rated REGEXP '^" . $review_rating . "[0-9]+$' AND post_id = " . $args['post_id'];
		$averageRatingResult = $wpdb->get_row( $sqlAverage, ARRAY_A );
		$ratingData = empty($averageRatingResult) ? $ratingData : $averageRatingResult;

		// Thumbs up rated
		$thumbsUpq = $wpdb->get_var( $wpdb->prepare( " SELECT COUNT(rated) FROM " . $wpdb->prefix . Rating_Form_Setup::TBL_RATING_LOG . " WHERE rated = '" . $review_rating . "1u' AND ".$argsInSQL." AND post_id = %s", $args['post_id'] ) );
		$thumbsUp_num_rows = $wpdb->num_rows;
		$thumbsUp = $thumbsUp_num_rows == 0 ? 0 : $thumbsUpq;

		// Thumbs down rated
		$thumbsDownq = $wpdb->get_var( $wpdb->prepare( " SELECT COUNT(rated) FROM " . $wpdb->prefix . Rating_Form_Setup::TBL_RATING_LOG . " WHERE rated = '" . $review_rating . "1d' AND ".$argsInSQL." AND post_id = %s", $args['post_id'] ) );
		$thumbsDown_num_rows = $wpdb->num_rows;
		$thumbsDown = $thumbsDown_num_rows == 0 ? 0 : $thumbsDownq;

		// Thumbs Up and Down = Total votes
		$tudTotal = ($thumbsUp + $thumbsDown);

		// User rating list
		$typeVal = ($args['type'] == "star" ? " AND rated REGEXP '^" . $review_rating . "[0-9]+$'" : ($args['type'] == "tud" ? " AND (rated = '" . $review_rating . "1u' OR rated = '" . $review_rating . "1d')" : ''));
		$sqlAll = "SELECT * FROM " . $wpdb->prefix . Rating_Form_Setup::TBL_RATING_LOG . " " .
				"WHERE " . $argsInSQL . $typeVal . " AND post_id = " . $args['post_id'];
		$allRatingResult = $wpdb->get_results( $sqlAll, ARRAY_A );

		if ($args['type'] == "star") {
			$ratingData['average_rating'] = empty($ratingData['average_rating']) ? 0 : floatval($ratingData['average_rating']);
			if (!empty($round)) $ratingData['average_rating'] = round($ratingData['average_rating'], $round);
			$ratingData['total'] = empty($ratingData['total']) ? 0 : intval($ratingData['total']);
			if (!empty($max)) $ratingData['max_rating'] = $max_rating;
		} else if ($args['type'] == "tud") {
			$ratingData['up'] = empty($thumbsUp) ? 0 : intval($thumbsUp);
			$ratingData['down'] = empty($thumbsDown) ? 0 : intval($thumbsDown);
			$ratingData['total'] = empty($tudTotal) ? 0 : intval($tudTotal);
        	unset($ratingData['average_rating']);
		}
 		$ratingData['ratings'] = empty($allRatingResult) ? array() : $allRatingResult;

		return $ratingData;
	}

	/**
	 * Get Rating Form Options based on ID
	 * @param $id - Rating Form ID
	 * @return array options
	 */
	public static function optionsByID( $id, $settings = null) {
		$settings = $settings == null ? Rating_Form_API::settingsByID( $id ) : $settings;
		return Rating_Form_API::getOptions( $settings['display'] );
	}

	public static function getOptions( $clm ) {
		$options = empty($clm) ? array() : json_decode($clm, true);
		// Options from field display
		$options['max']              	   = array_key_exists("max", $options) ? $options['max'] : 5;
		$options['round']              	   = array_key_exists("round", $options) ? $options['round'] : 1;
		$options['icon_half']              = '-half hover';
		$options['icon_empty']             = in_array("empty", $options, true) ? '-empty' : '';
		$options['only_thumbs_up']         = in_array("up", $options, true);
		$options['only_thumbs_down']       = in_array("down", $options, true);
		$options['up_minus_down']          = in_array("up_down_total", $options, true);
		$options['edit_rating']            = in_array("edit_rating", $options, true);
		$options['edit_rating_direct']     = in_array("edit_rating_direct", $options, true);
		$options['show_edit_rating_text']  = in_array("edit_rating_text", $options, true);
		$options['exclude_ip']             = in_array("remove_bip_votes", $options, true);
		$options['hide_msg_success']       = in_array("hide_success_msg", $options, true);
		$options['hide_up_count']          = in_array("hide_up_total", $options, true);
		$options['hide_down_count']        = in_array("hide_down_total", $options, true);
		$options['remove_plus_min_chars']  = in_array("remove_up_down_plus_min", $options, true);
		$options['spinner_check']          = (empty($options['spinner']['check']) ? false : $options['spinner']['check']) == "enable" ? true : false;
		$options['spinner_type']           = (empty($options['spinner']['type']) ? false : $options['spinner']['type']);
		$options['spinner_type']           = $options['spinner_check'] ? 'spinner' . ($options['spinner_type'] == 1 ? '' : $options['spinner_type']) : '';
		$options['remove_rs_itemtype']     = in_array("rich_snippet_itemtype_no", $options, true);
		$options['rich_snippet_type']      = empty($options['rich_snippet']['type']) ? 'microdata' : $options['rich_snippet']['type'];
		$options['active']                 = in_array("active", $options, true);
		$options['restrict_ip']        	   = in_array("restrict_ip", $options, true);
		$options['user_logged_in']         = in_array("user_logged_in", $options, true);
		$options['ajax_load']              = in_array("ajax_load", $options, true);
		$options['rtl']                    = in_array("rtl", $options, true);
		$options['redirect_check']         = (empty($options['redirect']['check']) ? false : $options['redirect']['check']) == "enable" ? true : false;
		$options['redirect_url']		   = empty($options['redirect']['url']) ? '' : $options['redirect']['url'];
		$options['redirect_params']		   = empty($options['redirect']['params']) ? '' : $options['redirect']['params'];
		$options['redirect_target']		   = empty($options['redirect']['target']) ? '_blank' : $options['redirect']['target'];
		$options['user_stats_check']       = (empty($options['ustats']['check']) ? false : $options['ustats']['check']) == "enable" ? true : false;
		$options['public_view']            = (empty($options['user_role']['public']) ? false : $options['user_role']['public']) == "1" ? true : false;
	    $options['fake_ratings']           = in_array("fake_ratings", $options, true);
		$options['remove_mobile'] 		   = (empty($options['mobile']['display']) ? false : ($options['mobile']['display'] == 'hide' ? true : false));
		$options['remove_tablet'] 		   = (empty($options['tablet']['display']) ? false : ($options['tablet']['display'] == 'hide' ? true : false));
		$options['hide_after_vote']        = in_array("hide_after_vote", $options, true);
		$options['show_ty_after_vote']     = in_array("show_ty_after_vote", $options, true);
		$options['limitation'] 			   = empty($options['limitation']) ? 1 : intval($options['limitation']);
		$options['title'] 			  	   = empty($options['title']) ? array() : $options['title'];
		$options['tooltip_pos']			   = empty($options['tooltip_pos']) ? 'top' : $options['tooltip_pos'];
		$options['tooltip_pos_wrap']	   = empty($options['tooltip_pos_wrap']) ? 'center' : $options['tooltip_pos_wrap'];
		$options['tooltip_arrow_pos']	   = empty($options['tooltip_arrow_pos']) ? 'bottom' : $options['tooltip_arrow_pos'];
		$options['tooltip_arrow_pos_wrap'] = empty($options['tooltip_arrow_pos_wrap']) ? 'center' : $options['tooltip_arrow_pos_wrap'];

	    return $options;
	}

	public static function settingsByID( $id ) {
        global $wpdb;
    	// Get Rating Form Settings
    	$sql = "SELECT form_id, form_name, type, css, display, date  FROM " . $wpdb->prefix . Rating_Form_Setup::TBL_RATING_ADD_FORM . " WHERE form_id = %s";
    	return $wpdb->get_row( $wpdb->prepare($sql, $id), ARRAY_A );
	}

	public static function getSettings() {
		$option = get_option(Rating_Form_Setup::PAGE_SETTINGS_SLUG);
        return (empty($option) ? array() : $option);
	}

	// Div args ID
	public static function getDivID($args, $cig = false) {
		// Check if comment id exist in WP
		$commentExists = get_comment( $args['comment_id'] );

		// Class Fields IDs
		$divIDargs = '';
		if (!empty($args['post_id'])) $divIDargs .= 'P' . $args['post_id'];
		if (!empty($args['user_id'])) $divIDargs .= 'U' . $args['user_id'];
		if (!empty($args['term_id'])) $divIDargs .= 'T' . $args['term_id'];
		if (!empty($commentExists) && !$args['widget']) $divIDargs .= 'CO' . $commentExists->comment_ID;

		if (!empty($args['custom_id']) && !is_array($args['custom_id']) && !$cig) {
			$divIDargs .= 'CU' . $args['custom_id'];
		}

		if ((!empty($args['custom_id_group']) && is_array($args['custom_id'])) || $cig) {
			$divIDargs .= 'CIG' . $args['custom_id_group'];
		}
		return $divIDargs;
	}
	// Live update shortcode data
	public static function getCustomIDGroupJS( $id, $args ) {
		$html = '';
		$findCIGscript = array();
		$doCIGfunction = array();
		// Responds to array Custom ID Group
		if (!empty($args['custom_id_group'])) {
			if (!is_array($args['custom_id_group'])) {
				$args['custom_id_group'] = array($args['custom_id_group']);
			}

			foreach ($args['custom_id_group'] as $value) {
				$args['custom_id_group'] = $value;
				$RFTOdiv = $id . Rating_Form_API::getDivID($args, true);
				$findCIGscript[] = '[data-rf-id='.$RFTOdiv.'] script';
				$doCIGfunction[] = 'if (typeof '.$RFTOdiv.' == "function") ' . $RFTOdiv . '();'."\r\n";
			}
		}
		$findCIGimplode = count($findCIGscript) ? ',' . implode(',', $findCIGscript) : '';
		// Responds to single Custom ID
		$RFTOdiv = $id . Rating_Form_API::getDivID($args);
		$html .= 'if (jQuery("body").find("[data-rf-id='.$RFTOdiv.'] script'.$findCIGimplode.'").length) {'."\r\n";
		$html .= 'if (typeof '.$RFTOdiv.' == "function") ' . $RFTOdiv . '();'."\r\n";
		foreach ($doCIGfunction as $val) {
			$html .= $val;
		}
		$html .= '}'."\r\n";

		return $html;
	}
	// Round to half
	public static function roundTohalf( $num )
	{
		if($num >= ($half = ($ceil = ceil($num))- 0.5) + 0.25) return $ceil;
		else if($num < $half - 0.25) return floor($num);
		else return $half;
	}
	// Seconds to Format
	public static function formatDateTime( $timestamp )
	{
		$seconds = intval($timestamp) % 60;
	    $minutes = (intval($timestamp) / 60) % 60;
	    $hours = (intval($timestamp) / 3600) % 24;
	    $days = intval(intval($timestamp) / (3600*24));

		return array(
			'seconds' => $seconds,
			'minutes' => $minutes,
			'hours' => $hours,
			'days' => $days,
		);
	}
	public static function getRatingPositions($averageRating) {
		$data['icon_count'] = (Rating_Form_API::roundTohalf($averageRating) >= ceil($averageRating)) ? ceil($averageRating) : floor($averageRating);
		$data['icon_half'] = (Rating_Form_API::roundTohalf($averageRating) > floor($averageRating)) ? ceil($averageRating) : floor($averageRating);
		return $data;
	}

	// Add inline form style
	public static function getInlineStyle( $settings, $removeStyleTag = false ) {
		$content = '';
		$this_form_query = $settings;

		// Options
		$options = empty($this_form_query['display']) ? array() : json_decode($this_form_query['display'], true);
		$remove_inline_style = in_array("remove_inline_style", $options) ? (is_admin() && ( !defined( 'DOING_AJAX' ) || !DOING_AJAX ) ? false : true) : false;

		if (count($this_form_query) > 0 && !empty($this_form_query['css']) && !$remove_inline_style) {
			$arrCss = json_decode($this_form_query['css'], true);
			// type style
			$css_alignment = array_key_exists("alignment", $arrCss) ? $arrCss['alignment'] : '';
			$css_color = array_key_exists("color", $arrCss) ? $arrCss['color'] : '';
			$css_hover_color = array_key_exists("hover_color", $arrCss) ? $arrCss['hover_color'] : '';
			$css_font_size = array_key_exists("font_size", $arrCss) ? $arrCss['font_size'] : '';
			// .def style
			$css_def_color = array_key_exists("def_color", $arrCss) ? $arrCss['def_color'] : '';
			$css_def_font_size = array_key_exists("def_font_size", $arrCss) ? $arrCss['def_font_size'] : '';
			$css_def_background = array_key_exists("def_background", $arrCss) ? $arrCss['def_background'] : '';
			// thumbs up style
			$css_up_color = array_key_exists("up_color", $arrCss) ? $arrCss['up_color'] : '';
			$css_hover_up_color = array_key_exists("hover_up_color", $arrCss) ? $arrCss['hover_up_color'] : '';
			$css_def_up_color = array_key_exists("def_up_color", $arrCss) ? $arrCss['def_up_color'] : '';
			$css_def_up_background = array_key_exists("def_up_background", $arrCss) ? $arrCss['def_up_background'] : '';
			// thumbs down style
			$css_down_color = array_key_exists("down_color", $arrCss) ? $arrCss['down_color'] : '';
			$css_hover_down_color = array_key_exists("hover_down_color", $arrCss) ? $arrCss['hover_down_color'] : '';
			$css_def_down_color = array_key_exists("def_down_color", $arrCss) ? $arrCss['def_down_color'] : '';
			$css_def_down_background = array_key_exists("def_down_background", $arrCss) ? $arrCss['def_down_background'] : '';
			// Custom CSS
			$custom_css = array_key_exists("custom_css", $arrCss) ? $arrCss['custom_css'] : '';

			//GENERAL
			if(!empty($css_alignment)) {
				$content .= '.rating_form_' . $this_form_query['form_id'] . ' {'; // type
				$content .= 'text-align:' . $css_alignment . ';';
				$content .= '}';
			}
			//STAR
			$content .= '.rating_form_' . $this_form_query['form_id'] . ' .rating_form .rf_item {'; // type
			if(!empty($css_color)) $content .= 'color:' . $css_color . ';';
			if(!empty($css_font_size)) $content .= 'font-size:' . $css_font_size . ';'; // TUD
			$content .= '}';
			if (Rating_Form_Setup::form_types($this_form_query['type'], 'type') == "star") {
				if ($this_form_query['type'] == 0) {
					$content .= '.rating_form_' . $this_form_query['form_id'] . ' .rating_form .rf_item img {'; // type custom
					$content .= 'width:' . $css_font_size . ';';
					$content .= 'height:' . $css_font_size . ';';
					$content .= '}';
				}
				if(!empty($css_hover_color)) {
					$content .= '.rating_form_' . $this_form_query['form_id'] . ' .rating_form .rf_item.hover {'; // type hover
					$content .= 'color:' . $css_hover_color . ';';
					$content .= '}';
				}
			}
			$content .= '.rating_form_' . $this_form_query['form_id'] . ' .def {'; // .def
			if(!empty($css_def_background)) $content .= 'background-color:' . $css_def_background . ';';
			if(!empty($css_def_color)) $content .= 'color:' . $css_def_color . ';';
			if(!empty($css_def_font_size)) $content .= 'font-size:' . $css_def_font_size . ';';
			$content .= '}';

			// TUD
			if (Rating_Form_Setup::form_types($this_form_query['type'], 'type') == "tud") {
				if(!empty($css_up_color)) {
					$content .= '.rating_form_' . $this_form_query['form_id'] . ' .rating_form .up_rated {'; // .up_rated
					$content .= 'color:' . $css_up_color . ';';
					$content .= '}';
				}
				if(!empty($css_hover_up_color)) {
					$content .= '.rating_form_' . $this_form_query['form_id'] . ' .rating_form .up_rated.hover {'; // .up_rated.hover
					$content .= 'color:' . $css_hover_up_color . ';';
					$content .= '}';
				}
				$content .= '.rating_form_' . $this_form_query['form_id'] . ' .rating_form .up_rated_txt {'; // .up_rated_txt
				if(!empty($css_def_up_background)) $content .= 'background-color:' . $css_def_up_background . ';';
				if(!empty($css_def_up_color)) $content .= 'color:' . $css_def_up_color . ';';
				$content .= '}';
				if(!empty($css_down_color)) {
					$content .= '.rating_form_' . $this_form_query['form_id'] . ' .rating_form .down_rated {'; // .down_rated
					$content .= 'color:' . $css_down_color . ';';
					$content .= '}';
				}
				if(!empty($css_hover_down_color)) {
					$content .= '.rating_form_' . $this_form_query['form_id'] . ' .rating_form .down_rated.hover {'; // .down_rated.hover
					$content .= 'color:' . $css_hover_down_color . ';';
					$content .= '}';
				}
				$content .= '.rating_form_' . $this_form_query['form_id'] . ' .rating_form .down_rated_txt {'; // .down_rated_txt
				if(!empty($css_def_down_background)) $content .= 'background-color:' . $css_def_down_background . ';';
				if(!empty($css_def_down_color)) $content .= 'color:' . $css_def_down_color . ';';
				$content .= '}';
			}
			// Custom CSS
			if(!empty($custom_css)) {
				$content .= $custom_css;
			}
		}

		$wrap = '';
		if (!empty($content)) {
			if (!$removeStyleTag) $wrap .= '<style type="text/css">';
			$wrap .= $content . "\r\n";
			if (!$removeStyleTag) $wrap .= '</style>';
		}
		return $wrap;
	}

	public static function checkAttrID( $args ) {
		// Check wether args id are being used
		return (!empty($args['comment_id']) || !empty($args['custom_id']) || !empty($args['user_id']) || !empty($args['term_id']));
	}

	public static function addMetaFake($args, $settings, $options) {
		if ($options['fake_ratings'] && !Rating_Form_API::checkAttrID($args)) {
			if (Rating_Form_Setup::form_types($settings['type'], 'type') == "star") {
				if (!metadata_exists('post', $args['post_id'], 'rf_fake_average')) add_post_meta($args['post_id'], 'rf_fake_average', 0);
				if (!metadata_exists('post', $args['post_id'], 'rf_fake_total')) add_post_meta($args['post_id'], 'rf_fake_total', 0);
			} else if (Rating_Form_Setup::form_types($settings['type'], 'type') == "tud") {
				if (!metadata_exists('post', $args['post_id'], 'rf_fake_up')) add_post_meta($args['post_id'], 'rf_fake_up', 0);
				if (!metadata_exists('post', $args['post_id'], 'rf_fake_down')) add_post_meta($args['post_id'], 'rf_fake_down', 0);
			}
		}
	}
	public static function updateMetaRatings($args, $data) {
		$settings = Rating_Form_API::settingsByID($args['id']);

		$rf_real_up 	  = intval($data['up_count']);
		$rf_real_down 	  = intval($data['down_count']);
		$rf_real_average  = $data['average'];
		$rf_real_total 	  = intval($data['count']);
		$rf_up_down_percent = $data['count_percent'];
		// Add/ Update meta values - REAL ratings for: posts, comments, terms, users
		if (!Rating_Form_API::checkAttrID($args)) {
			if (Rating_Form_Setup::form_types($settings['type'], 'type') == "star") {
				if (!empty($rf_real_total)) update_post_meta($args['post_id'], 'rf_total', $rf_real_total);
				if (!empty($rf_real_average)) update_post_meta($args['post_id'], 'rf_average', $rf_real_average);
			} else if (Rating_Form_Setup::form_types($settings['type'], 'type') == "tud") {
				if (!empty($rf_real_up)) update_post_meta($args['post_id'], 'rf_up', $rf_real_up);
				if (!empty($rf_real_down)) update_post_meta($args['post_id'], 'rf_down', $rf_real_down);
				if (!empty($rf_up_down_percent)) update_post_meta($args['post_id'], 'rf_up_down_percent', $rf_up_down_percent);
			}
		} else if (!empty($args['comment_id'])) {
			if (Rating_Form_Setup::form_types($settings['type'], 'type') == "star") {
				if (!empty($rf_real_total)) update_comment_meta($args['comment_id'], 'rf_total', $rf_real_total);
				if (!empty($rf_real_average)) update_comment_meta($args['comment_id'], 'rf_average', $rf_real_average);
			} else if (Rating_Form_Setup::form_types($settings['type'], 'type') == "tud") {
				if (!empty($rf_real_up)) update_comment_meta($args['comment_id'], 'rf_up', $rf_real_up);
				if (!empty($rf_real_down)) update_comment_meta($args['comment_id'], 'rf_down', $rf_real_down);
				if (!empty($rf_up_down_percent)) update_comment_meta($args['comment_id'], 'rf_up_down_percent', $rf_up_down_percent);
			}
		} else if (!empty($args['term_id'])) {
			if (Rating_Form_Setup::form_types($settings['type'], 'type') == "star") {
				if (!empty($rf_real_total)) update_term_meta($args['term_id'], 'rf_total', $rf_real_total);
				if (!empty($rf_real_average)) update_term_meta($args['term_id'], 'rf_average', $rf_real_average);
			} else if (Rating_Form_Setup::form_types($settings['type'], 'type') == "tud") {
				if (!empty($rf_real_up)) update_term_meta($args['term_id'], 'rf_up', $rf_real_up);
				if (!empty($rf_real_down)) update_term_meta($args['term_id'], 'rf_down', $rf_real_down);
				if (!empty($rf_up_down_percent)) update_term_meta($args['term_id'], 'rf_up_down_percent', $rf_up_down_percent);
			}
		} else if (!empty($args['user_id'])) {
			if (Rating_Form_Setup::form_types($settings['type'], 'type') == "star") {
				if (!empty($rf_real_total)) update_user_meta($args['user_id'], 'rf_total', $rf_real_total);
				if (!empty($rf_real_average)) update_user_meta($args['user_id'], 'rf_average', $rf_real_average);
			} else if (Rating_Form_Setup::form_types($settings['type'], 'type') == "tud") {
				if (!empty($rf_real_up)) update_user_meta($args['user_id'], 'rf_up', $rf_real_up);
				if (!empty($rf_real_down)) update_user_meta($args['user_id'], 'rf_down', $rf_real_down);
				if (!empty($rf_up_down_percent)) update_user_meta($args['user_id'], 'rf_up_down_percent', $rf_up_down_percent);
			}
		}
		// Add/ Update meta Custom ID values - REAL ratings for: posts, comments, terms, users
		$meta_key_custom_id = '_cui_' . $args['custom_id'];
		if (!empty($args['post_id'])
				&& !empty($args['custom_id'])
				&& empty($args['comment_id'])
				&& empty($args['term_id'])
				&& empty($args['user_id'])) {
			if (Rating_Form_Setup::form_types($settings['type'], 'type') == "star") {
				if (!empty($rf_real_total)) update_post_meta($args['post_id'], 'rf_total' . $meta_key_custom_id, $rf_real_total);
				if (!empty($rf_real_average)) update_post_meta($args['post_id'], 'rf_average' . $meta_key_custom_id, $rf_real_average);
			} else if (Rating_Form_Setup::form_types($settings['type'], 'type') == "tud") {
				if (!empty($rf_real_up)) update_post_meta($args['post_id'], 'rf_up' . $meta_key_custom_id, $rf_real_up);
				if (!empty($rf_real_down)) update_post_meta($args['post_id'], 'rf_down' . $meta_key_custom_id, $rf_real_down);
				if (!empty($rf_up_down_percent)) update_post_meta($args['post_id'], 'rf_up_down_percent' . $meta_key_custom_id, $rf_up_down_percent);
			}
		} else if (!empty($args['post_id'])
				&& !empty($args['custom_id'])
				&& !empty($args['comment_id'])
				&& empty($args['term_id'])
				&& empty($args['user_id'])) {
			if (Rating_Form_Setup::form_types($settings['type'], 'type') == "star") {
				if (!empty($rf_real_total)) update_comment_meta($args['comment_id'], 'rf_total' . $meta_key_custom_id, $rf_real_total);
				if (!empty($rf_real_average)) update_comment_meta($args['comment_id'], 'rf_average' . $meta_key_custom_id, $rf_real_average);
			} else if (Rating_Form_Setup::form_types($settings['type'], 'type') == "tud") {
				if (!empty($rf_real_up)) update_comment_meta($args['comment_id'], 'rf_up' . $meta_key_custom_id, $rf_real_up);
				if (!empty($rf_real_down)) update_comment_meta($args['comment_id'], 'rf_down' . $meta_key_custom_id, $rf_real_down);
				if (!empty($rf_up_down_percent)) update_comment_meta($args['comment_id'], 'rf_up_down_percent' . $meta_key_custom_id, $rf_up_down_percent);
			}
		} else if (!empty($args['custom_id'])
				&& empty($args['comment_id'])
				&& !empty($args['term_id'])
				&& empty($args['user_id'])) {
			if (Rating_Form_Setup::form_types($settings['type'], 'type') == "star") {
				if (!empty($rf_real_total)) update_term_meta($args['term_id'], 'rf_total' . $meta_key_custom_id, $rf_real_total);
				if (!empty($rf_real_average)) update_term_meta($args['term_id'], 'rf_average' . $meta_key_custom_id, $rf_real_average);
			} else if (Rating_Form_Setup::form_types($settings['type'], 'type') == "tud") {
				if (!empty($rf_real_up)) update_term_meta($args['term_id'], 'rf_up' . $meta_key_custom_id, $rf_real_up);
				if (!empty($rf_real_down)) update_term_meta($args['term_id'], 'rf_down' . $meta_key_custom_id, $rf_real_down);
				if (!empty($rf_up_down_percent)) update_term_meta($args['term_id'], 'rf_up_down_percent' . $meta_key_custom_id, $rf_up_down_percent);
			}
		} else if (!empty($args['custom_id'])
				&& empty($args['comment_id'])
				&& empty($args['term_id'])
				&& !empty($args['user_id'])) {
			if (Rating_Form_Setup::form_types($settings['type'], 'type') == "star") {
				if (!empty($rf_real_total)) update_user_meta($args['user_id'], 'rf_total' . $meta_key_custom_id, $rf_real_total);
				if (!empty($rf_real_average)) update_user_meta($args['user_id'], 'rf_average' . $meta_key_custom_id, $rf_real_average);
			} else if (Rating_Form_Setup::form_types($settings['type'], 'type') == "tud") {
				if (!empty($rf_real_up)) update_user_meta($args['user_id'], 'rf_up' . $meta_key_custom_id, $rf_real_up);
				if (!empty($rf_real_down)) update_user_meta($args['user_id'], 'rf_down' . $meta_key_custom_id, $rf_real_down);
				if (!empty($rf_up_down_percent)) update_user_meta($args['user_id'], 'rf_up_down_percent' . $meta_key_custom_id, $rf_up_down_percent);
			}
		}
	}
	public static function getSuccessMessage( $form_id, $messages, $hide ) {
		// Get translation Messages
		$translatedMsgs = Rating_Form_API::getTranslationForm($form_id, $messages);
		// Success msg
		if (!$hide) {
			$thankyou = '<span class="def thankyou">' . $translatedMsgs['thankyou'] . '</span>'."\r\n";
		}
		return $thankyou;
	}
	public static function getEditRatingMessage( $args ) {
		return ($args['show_edit_rating_text']) ? sprintf(
			Rating_Form_API::getTranslationForm( $args['id'], $args['messages'] )['edit_rating'],
			$args['index'],
			Rating_Form_API::getTitles( $args['type'], $args['titles'] )[$args['index']])
			: '';
	}
	// Update [shortcode] after rating
	public static function updateShortcodeData($args, $average, $options) {
		if (empty($options)) return;
		$settings = Rating_Form_API::settingsByID($args['id']);

		$content = '';
		$updateShortcode = in_array("live_top_ratings", $options, true) ? true : false;
		$updateShortcode2 = in_array("live_rf_total", $options, true) ? true : false;
		$updateShortcode3 = in_array("live_rf_results", $options, true) ? true : false;
		if ($updateShortcode || $updateShortcode2 || $updateShortcode3) {
			$content .= '<script type="text/javascript">'."\r\n";
			$content .= 'jQuery(document).ready(function() {'."\r\n";
			// Live Top Ratings
			if ($updateShortcode) {
				if ( empty($args['custom_id']) && empty($args['comment_id']) && empty($args['term_id']) ) {
					$args_widget = $args;
					$args_widget['result'] = true;
					$args_widget['score'] = false;
					$args_widget['total'] = false;
					$args_widget['title'] = false;
					$args_widget['tooltip'] = false;
					$args_widget['stats'] = false;
					$args_widget['before_content'] = $args['result'] == false ? "" : html_entity_decode($args['before_content']);
					$args_widget['after_content'] = $args['result'] == false ? "" : html_entity_decode($args['after_content']);

					$divRFid_widget = 'RF' . ($args_widget['result'] == true ? 'R' : '') . $args['id'] . Rating_Form_API::getDivID($args);

					$content .= 'jQuery.ajax({'."\r\n";
						$content .= 'type: "POST",'."\r\n";
						$content .= 'url : rating_form_script.ajaxurl,'."\r\n";
						$content .= 'data : { action : "display_rating_form_ajax", args : ' . json_encode($args_widget) . ' },'."\r\n";
						$content .= 'success : function(data) {'."\r\n";

		                $content .= 'var params = {';
		                $content .= '    checks: JSON.parse(data),';
		                $content .= '    parent: jQuery(".rf_top_ratings_widget [data-rf-id=' . $divRFid_widget . ']"),';
		                $content .= '};';
		                $content .= 'rating_form_apply_data(params);';

						if (Rating_Form_Setup::form_types($settings['type'], 'type') == "star") {
							$content .= 'jQuery(".rf_top_ratings_widget [data-rf-id=' . $divRFid_widget . ']").closest("li").find(".rf_avg_rating").text("(' . $average . ')");'."\r\n";
						}
						$content .= '}'."\r\n";
					$content .= '});'."\r\n";
				}
			}
			// Update [rating_form_total]
			if ($updateShortcode2)
				$content .= Rating_Form_API::getCustomIDGroupJS('RFTO', $args);

			// Update [rating_form_results]
			if ($updateShortcode3)
				$content .= Rating_Form_API::getCustomIDGroupJS('RFRS', $args);

			$content .= '});'."\r\n";
			$content .= '</script>'."\r\n";
		}
		return $content;
	}

	// Get rating data based on ID
	public static function getDataByID( $id ) {
		global $wpdb;
		$sql = "SELECT * FROM " . $wpdb->prefix . Rating_Form_Setup::TBL_RATING_LOG . " WHERE rate_id = %d;";
		return $wpdb->get_row( $wpdb->prepare($sql, $id), ARRAY_A );
	}

	public static function getIP() {
		return Rating_Form_API::anonymizeIP( $_SERVER['REMOTE_ADDR'] );
	}

	public static function anonymizeIP( $ip ) {
		$globalSettings = Rating_Form_API::getSettings();
		// Anonymize IP Address
		$isAnonymizeIP = (in_array("anonymize_ip", $globalSettings));
		if ($isAnonymizeIP) {
			// IPv4
			if ( filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4) ) {
				$last_colon = strrpos($ip, '.') + 1;
				$ip = substr($ip, 0, $last_colon) . 0;
			}
			// IPv6
			else if ( filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6) ) {
				$last_colon = strrpos($ip, ':') + 1;
				$ip = substr($ip, 0, $last_colon) . 0;
			}
		}
		return $ip;
	}

	public static function setCookie( $cookieArgs ) {
		if ( !headers_sent() ) {
			setcookie('rf_rating-' . Rating_Form_API::getDivID( $cookieArgs ), $cookieArgs['cookie_amount'], $cookieArgs['cookie_expiration'],
				COOKIEPATH, COOKIE_DOMAIN, false, true);
		}
	}

	// Titles
	public static function getTitles( $type, $titles, $value = null ) {
		global $wpdb;

		$titlesCustom = array();
		$titlesText = array();
		$wpmlTitlesText = array();
		$titles_query = array();

		$titles = empty($titles) ? array() : $titles;

		// Use default titles if Rating Form has no titles assigned
		if (empty($titles)) {
			if (Rating_Form_Setup::form_types($type, 'type') == "star") {
				$titles_query = $wpdb->get_results( "SELECT * FROM " . $wpdb->prefix . Rating_Form_Setup::TBL_RATING_TITLES . " ORDER BY title_id DESC", ARRAY_A );
			} else if (Rating_Form_Setup::form_types($type, 'type') == "tud") {
				$titles_query = $wpdb->get_results( "SELECT * FROM " . $wpdb->prefix . Rating_Form_Setup::TBL_RATING_TITLES . " WHERE position IN(1,2) ORDER BY title_id ASC", ARRAY_A );
			}
		} else {
			// Get titles based on title id
			if (count($titles)) {
				$title_ids = "'" . implode("','", $titles) . "'";
				if (Rating_Form_Setup::form_types($type, 'type') == "star") {
					$sql = "SELECT * FROM " . $wpdb->prefix . Rating_Form_Setup::TBL_RATING_TITLES . " WHERE title_id IN(" . $title_ids . ")";
				} else if (Rating_Form_Setup::form_types($type, 'type') == "tud") {
					$sql = "SELECT * FROM " . $wpdb->prefix . Rating_Form_Setup::TBL_RATING_TITLES . " WHERE title_id IN(" . $title_ids . ") AND (position = 1 || position = 2)";
				}
				$query_titles = $wpdb->get_results( $sql, ARRAY_A  );
				$titlesFound = $wpdb->num_rows;
				if ($titlesFound > 0) $titles_query = $query_titles;
			}
		}

		if (count($titles_query)) {
			foreach ($titles_query as $title_text) {
				$title_text_sslash = __(stripslashes($title_text['text']), 'rating-form');
				$wpmlTitlesText[$title_text['position']]['text'] = $title_text_sslash ;
				$wpmlTitlesText[$title_text['position']]['title_id'] = $title_text['title_id'];
				$titlesText[$title_text['position']] = $title_text_sslash;
				if (!empty($value)) $titlesCustom[] = $title_text[$value];
			}
		}

		// WPML translate strings
		if (has_filter('wpml_translate_single_string')) {
			if (count($wpmlTitlesText)) {
				foreach ($wpmlTitlesText as $key => $value) {
					$def_single = stripslashes(apply_filters('wpml_translate_single_string', $value['text'], 'rating-form', 'Title ID ' . $value['title_id'], 'en'));
					$single = stripslashes(apply_filters('wpml_translate_single_string', $value['text'], 'rating-form', 'Title ID ' . $value['title_id']));

					if ($def_single != $single && $titlesText[$key] != $single) $titlesText[$key] = $single;
				}
			}
		}

		if (empty($titlesCustom)) {
			return $titlesText;
		} else {
			return $titlesCustom;
		}
	}

	public static function getTranslationForm($id, $messages) {
		$array = array();

		// Plugin Rating Form messages
		$pluginTy = __( 'Thank you :)', 'rating-form' );
		$pluginRated = __( 'You already rated', 'rating-form' );
		$pluginLogin = __( 'Login to rate', 'rating-form' );
		$pluginLimit = __( 'Sorry, rating is limited. Try again in %4$d days %3$d hours %2$d minutes %1$d seconds.', 'rating-form' );
		$pluginEditRating = __( 'You find this post %2$s', 'rating-form' );

		// Settings Rating Form messages
		$formTy = empty($messages['ty']) ? $pluginTy : $messages['ty'];
		$formRated = empty($messages['rated']) ? $pluginRated : $messages['rated'];
		$formLogin = empty($messages['login']) ? $pluginLogin : $messages['login'];
		$formLimit = empty($messages['limit']) ? $pluginLimit : $messages['limit'];
		$formEditRating = empty($messages['edit_rating']) ? $pluginEditRating : $messages['edit_rating'];

		$translatedFormTy = $formTy;
		$translatedFormRated = $formRated;
		$translatedFormLogin = $formLogin;
		$translatedFormLimit = $formLimit;
		$translatedFormEditRating = $formEditRating;

		// WPML translate strings
		if (has_filter('wpml_translate_single_string')) {
			// Form Messages of formID
			$translatedFormTy = apply_filters('wpml_translate_single_string', $formTy, 'rating-form', 'Success ID ' . $id);
			$translatedFormRated = apply_filters('wpml_translate_single_string', $formRated, 'rating-form', 'Rated ID ' . $id);
			$translatedFormLogin = apply_filters('wpml_translate_single_string', $formLogin, 'rating-form', 'Login ID ' . $id);
			$translatedFormLimit = apply_filters('wpml_translate_single_string', $formLimit, 'rating-form', 'Limit ID ' . $id);
			$translatedFormEditRating = apply_filters('wpml_translate_single_string', $formEditRating, 'rating-form', 'Edit rating ID ' . $id);
		}

		$array['thankyou'] 		= ($formTy == $translatedFormTy) ? $formTy : $translatedFormTy;
		$array['rated'] 		= ($formRated == $translatedFormRated) ? $formRated : $translatedFormRated;
		$array['login'] 		= ($formLogin == $translatedFormLogin) ? $formLogin : $translatedFormLogin;
		$array['limit'] 			= ($formLimit == $translatedFormLimit) ? $formLimit : $translatedFormLimit;
		$array['edit_rating'] 	= ($formEditRating == $translatedFormEditRating) ? $formEditRating : $translatedFormEditRating;

		return $array;
	}

	/**
	 * Default Rating Form titles translation
	 */
	public static function getDefaultTitles() {
		return array(
			__( 'Very bad!', 'rating-form' ),
			__( 'Bad', 'rating-form' ),
			__( 'Hmmm', 'rating-form' ),
			__( 'Okay', 'rating-form' ),
			__( 'Good!', 'rating-form' ),
			__( 'Very good!', 'rating-form' ),
			__( 'Excellent!', 'rating-form' ),
			__( 'Cool!', 'rating-form' ),
			__( 'Awesome!', 'rating-form' ),
			__( 'Spectaculair!', 'rating-form' ),
			__( 'Like!', 'rating-form' ),
			__( 'Dislike!', 'rating-form' )
		);
	}

	/**
	 * Detect if current device is Tablet
	 *
	 * @return boolean
	 */
	public static function isTablet() {
		$userAgent = $_SERVER['HTTP_USER_AGENT'];
		if ( strpos($userAgent, 'iPad') // iPad
			||(strpos($userAgent, 'Android') && strpos($userAgent, 'Mobile') === false ) // Android
			|| strpos($userAgent, 'windows touch') //windows touch
			|| strpos($userAgent, 'Kindle') // Kindle
			|| strpos($userAgent, 'Silk') // Kindle Amazon
			|| strpos($userAgent, 'firefox tablet') // Firefox tablet
			|| strpos($userAgent, 'WebOS') // Palm
		) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * Deactivate if current device is Mobile
	 * NOTE: WordPress has his own mobile device detection function
	 *
	 * @return boolean is not on device tablet
	 */
	public static function deactivateOnMobile() {
		$tools_options = get_option(Rating_Form_Setup::PAGE_TOOLS_RATING_SLUG);
		// Deactivate on mobile
		$settingsMobileDeactivate = (empty($tools_options['settings']['mobile']['deactivate']) ? 0 : $tools_options['settings']['mobile']['deactivate']);
		$mobileEnable = ($settingsMobileDeactivate == 1 ? true : false);
		return (wp_is_mobile() && Rating_Form_API::isTablet() === false) && $mobileEnable;
	}

	/**
	 * Deactivate if current device is Tablet
	 *
	 * @return boolean is on device tablet
	 */
	public static function deactivateOnTablet() {
		$tools_options = get_option(Rating_Form_Setup::PAGE_TOOLS_RATING_SLUG);
		// Deactivate on tablet
		$settingsTabletDeactivate = (empty($tools_options['settings']['tablet']['deactivate']) ? 0 : $tools_options['settings']['tablet']['deactivate']);
		$tabletEnable = ($settingsTabletDeactivate == 1 ? true : false);
		return Rating_Form_API::isTablet() && $tabletEnable;
	}

	/**
	 * Custom Rating Form ID array converter
	 */
	public static function convertFieldIDArray( $val ) {
		if (is_string($val)) $val = explode(',', $val);

		if (is_array($val) && count($val)) {
			for ($i = 0; $i < count($val); $i++) {
				$val[$i] = trim(strtolower($val[$i]));
			}
			if (count($val) == 1) $val = $val[0];
		}

		return (is_string($val) && empty($val)) ? '0' : $val;
	}

	/**
	 * Check if uesr ID is blocked
	 *
	 * @return boolean			is blocked
	 */
	public static function isBlocked() {
		global $wpdb;
		$current_user = wp_get_current_user();
		$userID = $current_user->ID;
		$user_ip = empty($userID) ? Rating_Form_API::getIP() : $userID;

		$sql = "SELECT ip FROM " . $wpdb->prefix . Rating_Form_Setup::TBL_RATING_BLOCK_LIST . " WHERE ip = %s;";
		$query = $wpdb->get_row( $wpdb->prepare($sql, $user_ip), ARRAY_A );
        return ($wpdb->num_rows > 0);
    }

	/**
	 * Checks whether saved cookie rating exists
	 * @param  array $args		array containing post_id, comment_id, custom_id, user_id, term_id
	 * @return boolean          validate cookie existence
	 */
	public static function cookieCheck( $args ) {
		return isset( $_COOKIE['rf_rating-' . Rating_Form_API::getDivID( $args )] );
	}

	/**
	 * Check IP address saved rating with $args within specified hours
	 * @param  string $ip			ip of voter
	 * @param  array $args    		array containing post_id, comment_id, custom_id, user_id, term_id
	 * @param  int $hours         	ip address time limit
	 * @return boolean            	validate ip address existence
	 */
	public static function ipAddressCheck( $ip, $args ) {
		global $wpdb;

		$globalSettings = Rating_Form_API::getSettings();
		$ip_hours = empty($globalSettings['ip_restrict_time']) ? 24 : $globalSettings['ip_restrict_time'];

		$previous_date = date( 'Y-m-d H:i:s', strtotime( current_time('mysql') ) - ( 3600 * $ip_hours ) );

		$sql = 'SELECT * FROM ' . $wpdb->prefix . Rating_Form_Setup::TBL_RATING_LOG
				. ' WHERE ip = %s AND post_id = %d AND comment_id = %d AND custom_id = %s AND user_id = %d AND term_id = %d AND date >= %s';
		$query = $wpdb->get_results(
			$wpdb->prepare( $sql, $ip, $args['post_id'], $args['comment_id'], $args['custom_id'], $args['user_id'], $args['term_id'], $previous_date )
		);

		return ( count( $query ) > 0 );
	}
}
?>
