<?php
/**
 * Validate Rating Form
 */
class Rating_Form_Validation {
    private $settings = array();
    private $args = array();
    private $errors = array();
    private $custom_id_length = 32;
    private $cid_regex_match = '/^[a-zA-Z0-9\-\_]+$/i';
    private $cid_regex_replace = '/[^a-zA-Z0-9\-\_]/';

    function __construct( $settings, $args ) {
        $this->setSettings( $settings );
        $this->setArguments( $args );
        $this->checkValidation();
    }

    function setSettings( $settings ) {
        $this->settings = $settings;
    }

    function getSettings() {
        return $this->settings;
    }

    function setArguments( $args ) {
        $this->args = $args;
    }

    function getArgument( $str ) {
        return $this->args[$str];
    }

    function checkValidation() {
        $errors = $this->getErrorMessages();
        if ( !$this->validateInputCustomID()
                && !empty($this->getArgument('custom_id'))
                && is_string($this->getArgument('custom_id')) ) {
            $errors[] = $this->getInvalidCustomIDMessage();
        }

        if ( !$this->validateLengthCustomID() ) {
            $errors[] = $this->getCustomIDLengthMessage();
        }

        if ( !$this->rfExists() ) {
            $errors[] = $this->getNotExistsMessage();
        }

        if ( !$this->postExists() && !is_admin() ) {
            $errors[] = $this->getPostNotFoundMessage();
        }

        if ( !$this->userExists() && !empty($this->getArgument('user_id')) ) {
            $errors[] = $this->getUserNotFoundMessage();
        }

        if (!$this->getArgument('result') && Rating_Form_API::isBlocked()) {
            $errors[] = $this->getBlockedMessage();
        }

        $this->setErrorMessages($errors);
    }

    // Settings

    function rfExists() {
        return count( $this->getSettings() );
    }

    function postExists() {
        return (get_post($this->getArgument('post_id')) != null);
    }

    function userExists() {
        return get_userdata( $this->getArgument('user_id') );
    }

    // Messages
    function getInvalidCustomIDMessage() {
        $valid = preg_replace($this->cid_regex_replace, '', $this->getArgument('custom_id'));
        $invalid = preg_replace('/[' . $valid . ']/i', '', $this->getArgument('custom_id'));
        return sprintf(
            __( 'Custom ID has invalid characters. Allowed: a-z, A-Z, 0-9,'.
            ' - (dash), _ (underscore)<br><strong>Invalid:</strong> %1$s<br>'.
            '<strong>Valid:</strong> %2$s', 'rating-form' ) . "<br>",
            $invalid,
            $valid
        );
    }

    function getCustomIDLengthMessage() {
        return __('Custom ID: maximum character length is 32', 'rating-form');
    }

    function getNotExistsMessage() {
        return sprintf( __( 'Rating Form ID %d does not exist', 'rating-form' ), $this->getArgument('id') );
    }

    function getPostNotFoundMessage() {
        return sprintf( __( 'Post ID %d does not exists', 'rating-form' ), $this->getArgument('post_id') );
    }

    function getUserNotFoundMessage() {
        return sprintf( __( 'User ID %d does not exists', 'rating-form' ), $this->getArgument('user_id') );
    }

    function getBlockedMessage() {
        return __( 'You are banned. You cannot vote.', 'rating-form' );
    }

    function validateInputCustomID() {
        return preg_match( $this->cid_regex_match, $this->getArgument('custom_id'));
    }

    function validateLengthCustomID() {
        return (strlen($this->getArgument('custom_id')) <= $this->custom_id_length);
    }

    function isValid() {
        return count($this->getErrorMessages()) === 0;
    }

    function errorMessages() {
        $errors = $this->getErrorMessages();
        $html = '<div class="rf_validation">';
        $html .= '<h4>Rating Form ' . $this->getArgument('id') . '</h4>';
        $html .= '<ul>';
        for ($i = 0; $i < count($errors); $i++) {
            $html .= '<li>' . $errors[$i] . '</li>';
        }
        $html .= '</ul>';
        $html .= '</div>';
        echo $html;
    }

    function setErrorMessages($errors) {
        $this->errors = $errors;
    }

    function getErrorMessages() {
        return $this->errors;
    }
}
?>
