<?php //phpcs:ignore
namespace PRAD_PRO\includes;

defined( 'ABSPATH' ) || exit;

/**
 * License Class
 */
class License {

	private $ITEM_ID     = 58712;
	private $NAME        = 'WowAddons Pro';
	private $SERVER_URL  = 'https://account.wpxpo.com';
	private $VERSION     = PRAD_PRO_VER;
	private $SLUG        = 'product-addons-pro/product-addons-pro.php';
	private $DATE_FORMAT = '';

	/**
	 * Constructor.
	 *
	 * Initializes the class by setting up necessary properties or actions.
	 */
	public function __construct() {
		$this->DATE_FORMAT = $this->DATE_FORMAT;
		add_action( 'admin_init', array( $this, 'edd_license_updater' ) );
		add_action( 'wp_ajax_edd_prad_activate_license', array( $this, 'edd_activate_license' ) );
		add_action( 'wp_ajax_edd_prad_get_license_data', array( $this, 'edd_get_license_data' ) );
		add_action( 'wp_ajax_edd_prad_deactivate_license', array( $this, 'edd_deactivate_license' ) );
	}

	/**
	 * Initialize the EDD License Updater.
	 *
	 * @return void
	 */
	public function edd_license_updater() {
		$this->edd_check_license();
		if ( ! class_exists( 'EDD_SL_Plugin_Updater' ) ) {
			require_once PRAD_PRO_PATH . 'includes/updater/EDD_SL_Plugin_Updater.php';
		}

		$license_key = trim( get_option( 'edd_prad_license_key' ) );
		$edd_updater = new \EDD_SL_Plugin_Updater(
			$this->SERVER_URL,
			$this->SLUG,
			array(
				'version' => $this->VERSION,
				'license' => $license_key,
				'item_id' => $this->ITEM_ID,
				'author'  => $this->NAME,
				'url'     => home_url(),
				'beta'    => false,
			)
		);
	}

	/**
	 * Retrieve and return the EDD license data as a JSON response.
	 *
	 * This method checks the current license status, retrieves the stored
	 * license data from the options table, formats it, and returns it in a
	 * JSON success response.
	 *
	 * @return void
	 */
	public function edd_get_license_data() {

		$this->edd_check_license();

		$data = get_option( 'edd_prad_license_data', array() );

		wp_send_json_success( array( 'license_data' => $this->format_license_data_response( $data ) ) );
	}

	/**
	 * Check the license status and update the license data.
	 *
	 * This method checks the current license status by sending a request to
	 * the license server. It caches the response using a transient to avoid
	 * frequent checks. If the check is forced or hasn't been done within
	 * the specified interval, it will revalidate the license and store
	 * the result in the options table.
	 *
	 * @param bool $force Optional. Whether to force the license check. Default is false.
	 * @return void
	 */
	public function edd_check_license( $force = false ) {
		$site_hash      = md5( home_url() );
		$transient_name = 'edd_prad_' . $site_hash;
		$check_interval = DAY_IN_SECONDS * 15;

		if ( ! get_transient( $transient_name ) || $force ) {
			$license    = get_option( 'edd_prad_license_key', '' );
			$api_params = array(
				'edd_action' => 'check_license',
				'license'    => $license,
				'item_id'    => $this->ITEM_ID,
				'url'        => home_url(),
			);

			$response = wp_remote_post(
				$this->SERVER_URL,
				array(
					'timeout'   => 50,
					'sslverify' => false,
					'body'      => $api_params,
				)
			);

			set_transient( $transient_name, true, $check_interval );
			if ( ! is_wp_error( $response ) &&
				200 === wp_remote_retrieve_response_code( $response )
			) {
				$license_data = json_decode( wp_remote_retrieve_body( $response ) );
				update_option( 'edd_prad_license_data', (array) $license_data );
			}
		}
	}

	/**
	 * Activate the provided license key.
	 *
	 * This method sends a request to the license server to activate the
	 * provided license key. If the activation is successful, the license
	 * key and related data are stored in the options table. If activation
	 * fails, an error message is returned.
	 *
	 * @return void|string JSON response containing the activation status and message.
	 */
	public function edd_activate_license() {
		$activation_return = array(
			'status' => false,
			'data'   => __( 'Invalid license.', 'product-addons-pro' ),
		);

		$license_key                  = isset( $_POST['license_key'] ) ? sanitize_text_field( $_POST['license_key'] ) : '';
		$activation_return['license'] = $license_key;

		if ( ! empty( $license_key ) ) {
			$api_params = array(
				'edd_action' => 'activate_license',
				'license'    => $license_key,
				'item_id'    => $this->ITEM_ID,
				'url'        => home_url(),
			);

			$response = wp_remote_post(
				$this->SERVER_URL,
				array(
					'timeout'   => 50,
					'sslverify' => false,
					'body'      => $api_params,
				)
			);

			if ( is_wp_error( $response ) ||
				200 !== wp_remote_retrieve_response_code( $response )
			) {
				$activation_return['data'] = is_wp_error( $response ) ? $response->get_error_message() : __( 'An error occurred while Activation, please try again.', 'product-addons-pro' );
			} else {
				update_option( 'edd_prad_license_key', $license_key );
				$license_data                      = json_decode( wp_remote_retrieve_body( $response ) );
				$activation_return                 = array(
					'status' => true,
					'data'   => __( 'License Activated!', 'product-addons-pro' ),
				);
				$activation_return['license_data'] = $this->format_license_data_response( $license_data );
				update_option( 'edd_prad_license_data', (array) $license_data );
			}
		}

		wp_send_json( $activation_return );
	}

	/**
	 * Activate the provided license key.
	 *
	 * This method sends a request to the license server to activate the
	 * provided license key. If the activation is successful, the license
	 * key and related data are stored in the options table. If activation
	 * fails, an error message is returned.
	 *
	 * @return void|string JSON response containing the activation status and message.
	 */
	public function edd_deactivate_license() {

		// Check if deactivation is requested
		if ( ! isset( $_POST['deactivate'] ) || 'yes' !== $_POST['deactivate'] ) {
			wp_send_json_error( array( 'data' => 'Deactivation not confirmed!' ) );
			return;
		}

		// Initialize response data
		$deactive_return = array(
			'status' => false,
			'data'   => 'Deactivation Failed!',
		);

		// Retrieve the stored license key
		$license = get_option( 'edd_prad_license_key', '' );

		// Prepare API request parameters
		$api_params = array(
			'edd_action' => 'deactivate_license',
			'license'    => $license,
			'item_id'    => $this->ITEM_ID,
			'url'        => home_url(),
		);

		// Send the request to the license server
		$response = wp_remote_post(
			$this->SERVER_URL,
			array(
				'timeout'   => 50,
				'sslverify' => false,
				'body'      => $api_params,
			)
		);

		// Check for errors in the response
		if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {
			$deactive_return['data'] = is_wp_error( $response ) ? $response->get_error_message() : __( 'An Deactivation error occurred, please try again.', 'product-addons-pro' );
		} else {
			$license_data    = json_decode( wp_remote_retrieve_body( $response ) );
			$deactive_return = array(
				'status' => true,
				'data'   => __( 'License Deactivated.', 'product-addons-pro' ),
			);
			$this->edd_check_license( true );

			$deactive_return['license_data'] = $this->format_license_data_response( $license_data );

			// Update the license data in the options
			update_option( 'edd_prad_license_data', (array) $license_data );
		}
		wp_send_json( $deactive_return );
	}

	/**
	 * Generates a license status message based on license data.
	 *
	 * This function takes license data and returns an appropriate status message
	 * based on the license status (expired, disabled, valid, etc.).
	 *
	 * @param array $license_data License data containing status information.
	 * @return array|void Currently builds $res_data but doesn't return it. Consider adding 'return $res_data;' at the end.
	 */
	public function get_license_message( $license_data ) {
		$message = '';
		switch ( $license_data['license'] ) {
			case 'expired':
				$message = sprintf(
					// translators: %s is the expiration date.
					__( 'Your license key expired on %s.', 'product-addons-pro' ),
					date_i18n( $this->DATE_FORMAT, strtotime( $license_data['expires'], current_time( 'timestamp' ) ) )
				);
				break;
			case 'disabled':
			case 'revoked':
				$message = __( 'Your license key has been disabled.', 'product-addons-pro' );
				break;
			case 'missing':
				$message = __( 'Invalid license.', 'product-addons-pro' );
				break;
			case 'invalid':
			case 'site_inactive':
				$message = __( 'Your license is not active for this URL.', 'product-addons-pro' );
				break;
			case 'key_mismatch':
			case 'item_name_mismatch':
				$message = __( 'This appears to be an invalid license key.', 'product-addons-pro' );
				break;
			case 'no_activations_left':
				$message = __( 'Your license key has reached its activation limit.', 'product-addons-pro' );
				break;
			case 'license_not_activable':
				$message = __( 'The key you entered belongs to a bundle, please use the product specific license key.', 'product-addons-pro' );
				break;
			case 'deactivated':
				$message = __( 'Your license key has been deactivated.', 'product-addons-pro' );
				break;
			case 'valid':
				$message_data = $this->get_valid_message( $license_data );
				$message      = $message_data;
				break;
			default:
				$message = __( 'An error occurred, please try again.', 'product-addons-pro' );
				break;
		}

		return $message;
	}

	/**
	 * Generates a message for valid license keys.
	 *
	 * Creates appropriate messaging based on the license expiration status:
	 * - For lifetime licenses
	 * - For licenses expiring soon (within 30 days)
	 * - For licenses with standard expiration dates
	 *
	 * @param array $license_data License data containing expiration information.
	 * @return string Message indicating the license expiration status.
	 */
	public function get_valid_message( $license_data ) {
		$message   = '';
		$toExpired = false;
		if ( ! empty( $license_data['expires'] ) && 'lifetime' === $license_data['expires'] ) {
			$message = __( 'Never', 'product-addons-pro' );
		} else {
			$current_time = current_time( 'timestamp' );
			$_expiration  = current_time( 'timestamp' );
			if ( ! empty( $license_data['expires'] ) && 'lifetime' !== $license_data['expires'] ) {
				if ( ! is_numeric( $license_data['expires'] ) ) {
					$_expiration = strtotime( $license_data['expires'], $current_time );
				} else {
					$_expiration = $license_data['expires'];
				}
			}

			if ( ( $_expiration > $current_time ) && ( $_expiration - $current_time < ( DAY_IN_SECONDS * 30 ) ) ) {
				$message = sprintf(
				/* translators: the license expiration date. */
					__( 'Your license key expires soon! It expires on %s.', 'product-addons-pro' ),
					date_i18n( $this->DATE_FORMAT, $_expiration )
				);
				$toExpired = true;
			}

			$message = sprintf(
			/* translators: the license expiration date. */
				__( 'Your license key expires on %s.', 'product-addons-pro' ),
				date_i18n( $this->DATE_FORMAT, $_expiration )
			);
		}

		return array(
			'message'   => $message,
			'toExpired' => $toExpired,
		);
	}

	/**
	 * Formats license data for API response.
	 *
	 * Processes raw license data into a standardized format for front-end display
	 * or API consumption. Handles various license types (yearly/lifetime) and
	 * determines license type based on price ID and activation limits.
	 *
	 * @param array $license_data Optional. License data to format. If empty, retrieves
	 *                            from options table. Default empty array.
	 * @return array Formatted license data containing:
	 *               - license: The license status
	 *               - success: Whether the license operation was successful
	 *               - expires: Expiration date or "Never" for lifetime licenses
	 *               - licenseType: Human-readable license type description
	 *               - licenseKeySuffix: Last 4 characters of the license key for reference
	 */
	public function format_license_data_response( $license_data = array() ) {
		$formatted_data = array();
		if ( empty( $license_data ) ) {
			$license_data = get_option( 'edd_prad_license_data', array() );
		}

		if ( ! is_array( $license_data ) ) {
			$license_data = (array) $license_data;
		}

		if ( isset( $license_data['license'] ) ) {
			$formatted_data['license'] = $license_data['license'];
		}
		if ( isset( $license_data['success'] ) ) {
			$formatted_data['success'] = $license_data['success'];
		}

		$license_message           = $this->get_license_message( $license_data );
		$formatted_data['expires'] = is_array( $license_message ) ? $license_message['message'] : $license_message;
		if ( isset( $license_message['toExpired'] ) ) {
			$formatted_data['toExpired'] = $license_message['toExpired'];
		}
		// It may varry depending on the license type.
		if ( isset( $license_data['price_id'] ) ) {
			switch ( $license_data['price_id'] ) {
				case '1':
					$formatted_data['licenseType'] = '1 Site License | Yearly';
					break;
				case '2':
					$formatted_data['licenseType'] = 'Unlimited License | Yearly';
					break;
				case '3':
					$formatted_data['licenseType'] = '1 Site License | Lifetime';
					break;
				case '4':
					$formatted_data['licenseType'] = 'Unlimited Sites License | Lifetime';
					break;
				default:
					break;
			}
		}

		return $formatted_data;
	}
}
