<?php
/**
 * Bunny Video Model
 *
 * @package PrestoPlayer\Pro\Models\Bunny
 */

namespace PrestoPlayer\Pro\Models\Bunny;

use PrestoPlayer\Models\Setting;
use PrestoPlayer\Pro\Services\Bunny\URL;
use PrestoPlayer\Pro\Services\Bunny\BunnyService;

/**
 * Video class for handling Bunny video data.
 */
class Video {
	/**
	 * The ID of the video.
	 *
	 * @var string
	 */
	public $id;

	/**
	 * The type of the video.
	 *
	 * @var string
	 */
	public $type;

	/**
	 * The ID of the video library.
	 *
	 * @var string
	 */
	public $videoLibraryId;

	/**
	 * The GUID of the video.
	 *
	 * @var string
	 */
	public $guid;

	/**
	 * The title of the video.
	 *
	 * @var string
	 */
	public $title;

	/**
	 * The date the video was uploaded.
	 *
	 * @var string
	 */
	public $dateUploaded;

	/**
	 * The number of views the video has.
	 *
	 * @var int
	 */
	public $views = 0;

	/**
	 * Whether the video is public.
	 *
	 * @var bool
	 */
	public $isPublic;

	/**
	 * The length of the video.
	 *
	 * @var int
	 */
	public $length;

	/**
	 * The status of the video.
	 *
	 * @var string
	 */
	public $status;

	/**
	 * The framerate of the video.
	 *
	 * @var int
	 */
	public $framerate;

	/**
	 * The width of the video.
	 *
	 * @var int
	 */
	public $width;

	/**
	 * The height of the video.
	 *
	 * @var int
	 */
	public $height;

	/**
	 * The available resolutions of the video.
	 *
	 * @var string
	 */
	public $availableResolutions = '';

	/**
	 * The number of thumbnails for the video.
	 *
	 * @var int
	 */
	public $thumbnailCount;

	/**
	 * The progress of the video encoding.
	 *
	 * @var int
	 */
	public $encodeProgress;

	/**
	 * The storage size of the video.
	 *
	 * @var int
	 */
	public $storageSize;

	/**
	 * The captions for the video.
	 *
	 * @var array
	 */
	public $captions = array();

	/**
	 * Whether the video has an MP4 fallback.
	 *
	 * @var bool
	 */
	public $hasMP4Fallback = false;

	/**
	 * The ID of the collection the video belongs to.
	 *
	 * @var string
	 */
	public $collectionId;

	/**
	 * The filename of the thumbnail for the video.
	 *
	 * @var string
	 */
	public $thumbnailFileName;

	/**
	 * The URL of the playlist for the video.
	 *
	 * @var string
	 */
	public $playlistURL = '';

	/**
	 * The URL of the webp preview for the video.
	 *
	 * @var string
	 */
	public $webPURL = '';

	/**
	 * The URLs of the MP4 files for the video.
	 *
	 * @var array
	 */
	public $MP4URLs = array();

	/**
	 * The URL of the pullzone for the video.
	 *
	 * @var string
	 */
	public $pullzone_url;

	/**
	 * The signed URL of the playlist for the video.
	 *
	 * @var string
	 */
	public $playlistURLSigned = '';

	/**
	 * The signed URL of the webp preview for the video.
	 *
	 * @var string
	 */
	public $webPURLSigned = '';

	/**
	 * The signed URL of the thumbnail for the video.
	 *
	 * @var string
	 */
	public $thumbnailURLSigned = '';

	/**
	 * The signed URLs of the MP4 files for the video.
	 *
	 * @var array
	 */
	public $MP4URLsSigned = array();

	/**
	 * The URL of the thumbnail for the video.
	 *
	 * @var string
	 */
	public $thumbnailURL;

	/**
	 * The size of the video.
	 *
	 * @var int
	 */
	public $size;

	/**
	 * The thumbnail of the video.
	 *
	 * @var string
	 */
	public $thumbnail;

	/**
	 * The updated date of the video.
	 *
	 * @var string
	 */
	public $updated_at;

	/**
	 * The created date of the video.
	 *
	 * @var string
	 */
	public $created_at;

	/**
	 * Constructor for the Video class.
	 *
	 * @param array  $args The arguments for the video.
	 * @param string $type The type of the video (public or private).
	 * @throws \Exception If no type is provided.
	 */
	public function __construct( $args, $type ) {
		if ( ! $type ) {
			throw new \Exception( 'You must provide a type for the video (public or private)' );
		}

		$args = (object) $args;

		$has = get_object_vars( $this );
		foreach ( $has as $name => $_ ) {
			$this->$name = isset( $args->$name ) ? $args->$name : null;
		}

		$this->id                   = $args->guid;
		$this->type                 = $type;
		$this->pullzone_url         = Setting::get( 'bunny_stream_' . $this->type, 'pull_zone_url' );
		$this->availableResolutions = explode( ',', $this->availableResolutions ?? '' );

		$this->setURLS();
	}

	/**
	 * Set the URLs for the video.
	 */
	public function setURLS() {
		$this->playlistURL  = "{$this->getRootURL()}/playlist.m3u8";
		$this->webPURL      = "{$this->getRootURL()}/preview.webp";
		$this->thumbnailURL = "{$this->getRootURL()}/{$this->thumbnailFileName}";
		$this->MP4URLs      = array();
		if ( ! empty( $this->availableResolutions ) ) {
			foreach ( $this->availableResolutions as $resolution ) {
				$this->MP4URLs[] = "{$this->getRootURL()}/play_{$resolution}p.mp4";
			}
		}

		// sign urls
		$key                      = Setting::get( 'bunny_stream_private', 'token_auth_key' );
		$this->playlistURLSigned  = BunnyService::signURL( $this->playlistURL, $key );
		$this->webPURLSigned      = BunnyService::signURL( $this->webPURL, $key );
		$this->thumbnailURLSigned = BunnyService::signURL( $this->thumbnailURL, $key );
		$this->MP4URLsSigned      = array();
		if ( ! empty( $this->MP4URLs ) ) {
			foreach ( $this->MP4URLs as $mp4_url ) {
				$this->MP4URLsSigned[] = BunnyService::signURL( $mp4_url, $key );
			}
		}
	}

	/**
	 * Get the root url
	 *
	 * @return string
	 */
	public function getRootURL() {
		return untrailingslashit( "https://{$this->pullzone_url}/{$this->id}" );
	}
}
