/*
jquery.percentageloader.js

Copyright (c) 2012, Better2Web
All rights reserved.

This jQuery plugin is licensed under the Simplified BSD License. Please
see the file license.txt that was included with the plugin bundle.

*/

/*global jQuery */

(function ($) {
    /* Strict mode for this plugin */
    "use strict";
    /*jslint browser: true */

    /* Our spiral gradient data */
    var imgdata = "data:image/png;base64,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",
        gradient = new Image();
    gradient.src = imgdata;

    /** Percentage loader
     * @param	params	Specify options in {}. May be on of width, height, progress or value.
     *
     * @example $("#myloader-container).percentageLoader({
		    width : 256,  // width in pixels
		    height : 256, // height in pixels
		    progress: 0,  // initialise progress bar position, within the range [0..1]
		    value: '0kb'  // initialise text label to this value
		});
     */
    $.fn.percentageLoader = function (params) {
        var settings, canvas, percentageText, valueText, items, i, item, selectors, s, ctx, progress,
            value, cX, cY, lingrad, innerGrad, tubeGrad, innerRadius, innerBarRadius, outerBarRadius,
            radius, startAngle, endAngle, counterClockwise, completeAngle, setProgress, setValue,
            applyAngle, drawLoader, clipValue, outerDiv;

        /* Specify default settings */
        settings = {
            width: 256,
            height: 256,
            progress: 0,
            value: '0kb',
            controllable: false
        };

        /* Override default settings with provided params, if any */
        if (params !== undefined) {
            $.extend(settings, params);
        } else {
            params = settings;
        }

        outerDiv = document.createElement('div');
        outerDiv.style.width = settings.width + 'px';
        outerDiv.style.height = settings.height + 'px';
        outerDiv.style.position = 'relative';

        $(this).append(outerDiv);

        /* Create our canvas object */
        canvas = document.createElement('canvas');
        canvas.setAttribute('width', settings.width);
        canvas.setAttribute('height', settings.height);
        outerDiv.appendChild(canvas);

        /* Create div elements we'll use for text. Drawing text is
         * possible with canvas but it is tricky working with custom
         * fonts as it is hard to guarantee when they become available
         * with differences between browsers. DOM is a safer bet here */
        percentageText = document.createElement('div');
        percentageText.style.width = (settings.width.toString() - 2) + 'px';
        percentageText.style.textAlign = 'center';
        percentageText.style.height = '50px';
        percentageText.style.left = 0;
        percentageText.style.position = 'absolute';

        valueText = document.createElement('div');
        valueText.style.width = (settings.width - 2).toString() + 'px';
        valueText.style.textAlign = 'center';
        valueText.style.height = '0px';
        valueText.style.overflow = 'hidden';
        valueText.style.left = 0;
        valueText.style.position = 'absolute';

        /* Force text items to not allow selection */
        items = [valueText, percentageText];
        for (i  = 0; i < items.length; i += 1) {
            item = items[i];
            selectors = [
                '-webkit-user-select',
                '-khtml-user-select',
                '-moz-user-select',
                '-o-user-select',
                'user-select'];

            for (s = 0; s < selectors.length; s += 1) {
                $(item).css(selectors[s], 'none');
            }
        }

        /* Add the new dom elements to the containing div */
        outerDiv.appendChild(percentageText);
        //outerDiv.appendChild(valueText);

        /* Get a reference to the context of our canvas object */
        ctx = canvas.getContext("2d");


        /* Set various initial values */

        /* Centre point */
        cX = (canvas.width / 2) - 1;
        cY = (canvas.height / 2) - 1;

        /* Create our linear gradient for the outer ring */
        lingrad = ctx.createLinearGradient(cX, 0, cX, canvas.height);
        lingrad.addColorStop(0, '#ffffff');
        lingrad.addColorStop(1, '#ededed');

        /* Create inner gradient for the outer ring */
        innerGrad = ctx.createLinearGradient(cX, cX * 0.133333, cX, canvas.height - cX * 0.133333);
        innerGrad.addColorStop(0, '#ffffff');
        innerGrad.addColorStop(1, '#ffffff');

        /* Tube gradient (background, not the spiral gradient) */
        tubeGrad = ctx.createLinearGradient(cX, 0, cX, canvas.height);
        tubeGrad.addColorStop(0, '#c1dff4');
        tubeGrad.addColorStop(1, '#aacee6');

        /* The inner circle is 2/3rds the size of the outer one */
        innerRadius = cX * 0.6666;
        /* Outer radius is the same as the width / 2, same as the centre x
        * (but we leave a little room so the borders aren't truncated) */
        radius = cX - 2;

        /* Calculate the radii of the inner tube */
        innerBarRadius = innerRadius + (cX * 0.06);
        outerBarRadius = radius - (cX * 0.06);

        /* Bottom left angle */
        startAngle = 2.1707963267949;
        /* Bottom right angle */
        endAngle = 0.9707963267949 + (Math.PI * 2.0);

        /* Nicer to pass counterClockwise / clockwise into canvas functions
        * than true / false */
        counterClockwise = false;

        /* Borders should be 1px */
        ctx.lineWidth = 1;

        /**
         * Little helper method for transforming points on a given
         * angle and distance for code clarity
         */
        applyAngle = function (point, angle, distance) {
            return {
                x : point.x + (Math.cos(angle) * distance),
                y : point.y + (Math.sin(angle) * distance)
            };
        };


        /**
         * render the widget in its entirety.
         */
        drawLoader = function () {
            /* Clear canvas entirely */
            ctx.clearRect(0, 0, canvas.width, canvas.height);

            /*** IMAGERY ***/

            /* draw outer circle */
            ctx.fillStyle = lingrad;
            ctx.beginPath();
            ctx.strokeStyle = '#cecece';
            ctx.arc(cX, cY, radius, 0, Math.PI * 2, counterClockwise);
            ctx.fill();
            ctx.stroke();

            /* draw inner circle */
            ctx.fillStyle = innerGrad;
            ctx.beginPath();
            ctx.arc(cX, cY, innerRadius, 0, Math.PI * 2, counterClockwise);
            ctx.fill();
            ctx.strokeStyle = '#f6f6f6';
            ctx.stroke();

            ctx.beginPath();

            /**
             * Helper function - adds a path (without calls to beginPath or closePath)
             * to the context which describes the inner tube. We use this for drawing
             * the background of the inner tube (which is always at 100%) and the
             * progress meter itself, which may vary from 0-100% */
            function makeInnerTubePath(startAngle, endAngle) {
                var centrePoint, startPoint, controlAngle, capLength, c1, c2, point1, point2;
                centrePoint = {
                    x : cX,
                    y : cY
                };

                startPoint = applyAngle(centrePoint, startAngle, innerBarRadius);

                ctx.moveTo(startPoint.x, startPoint.y);

                point1 = applyAngle(centrePoint, endAngle, innerBarRadius);
                point2 = applyAngle(centrePoint, endAngle, outerBarRadius);

                controlAngle = endAngle + (3.142 / 2.0);
                /* Cap length - a fifth of the canvas size minus 4 pixels */
                capLength = (cX * 0.20) - 4;

                c1 = applyAngle(point1, controlAngle, capLength);
                c2 = applyAngle(point2, controlAngle, capLength);

                ctx.arc(cX, cY, innerBarRadius, startAngle, endAngle, false);
                ctx.bezierCurveTo(c1.x, c1.y, c2.x, c2.y, point2.x, point2.y);
                ctx.arc(cX, cY, outerBarRadius, endAngle, startAngle, true);

                point1 = applyAngle(centrePoint, startAngle, innerBarRadius);
                point2 = applyAngle(centrePoint, startAngle, outerBarRadius);

                controlAngle = startAngle - (3.142 / 2.0);

                c1 = applyAngle(point2, controlAngle, capLength);
                c2 = applyAngle(point1, controlAngle, capLength);

                ctx.bezierCurveTo(c1.x, c1.y, c2.x, c2.y, point1.x, point1.y);
            }

            /* Background tube */
            ctx.beginPath();
            ctx.strokeStyle = '#fefefe';
            makeInnerTubePath(startAngle, endAngle);

            ctx.fillStyle = tubeGrad;
            ctx.fill();
            ctx.stroke();

            /* Calculate angles for the the progress metre */
            completeAngle = startAngle + (progress * (endAngle - startAngle));

            ctx.beginPath();
            makeInnerTubePath(startAngle, completeAngle);

            /* We're going to apply a clip so save the current state */
            ctx.save();
            /* Clip so we can apply the image gradient */
            ctx.clip();

            /* Draw the spiral gradient over the clipped area */
            ctx.drawImage(gradient, 0, 0, canvas.width, canvas.height);

            /* Undo the clip */
            ctx.restore();

            /* Draw the outline of the path */
            ctx.beginPath();
            makeInnerTubePath(startAngle, completeAngle);
            ctx.stroke();

            /*** TEXT ***/
            (function () {
                var fontSize, string, smallSize, heightRemaining;
                /* Calculate the size of the font based on the canvas size */
                fontSize = cX / 2;

                percentageText.style.top = ((settings.height / 2) - (fontSize / 2)).toString() + 'px';
                percentageText.style.color = '#4677a2';
                percentageText.style.font = fontSize.toString() + 'px Arial';
                percentageText.style.textShadow = '0 1px 1px #FFFFFF';

                /* Calculate the text for the given percentage */
                string = (progress * 100.0).toFixed(0) + '%';

                percentageText.innerHTML = string;

                /* Calculate font and placement of small 'value' text */
                smallSize = cX / 5.5;
                valueText.style.color = '#80a9c8';
                valueText.style.font = smallSize.toString() + 'px Arial';
                valueText.style.height = smallSize.toString() + 'px';
                valueText.style.textShadow = 'None';

                /* Ugly vertical align calculations - fit into bottom ring.
                 * The bottom ring occupes 1/6 of the diameter of the circle */
                heightRemaining = (settings.height * 0.16666666) - smallSize;
                valueText.style.top = ((settings.height * 0.8333333) + (heightRemaining / 4)).toString() + 'px';
            }());
        };

        /**
        * Check the progress value and ensure it is within the correct bounds [0..1]
        */
        clipValue = function () {
            if (progress < 0) {
                progress = 0;
            }

            if (progress > 1.0) {
                progress = 1.0;
            }
        };

        /* Sets the current progress level of the loader
         *
         * @param value the progress value, from 0 to 1. Values outside this range
         * will be clipped
         */
        setProgress = function (value) {
            /* Clip values to the range [0..1] */
            progress = value;
            clipValue();
            drawLoader();
        };

        this.setProgress = setProgress;

        setValue = function (val) {
            value = val;
            valueText.innerHTML = value;
        };

        this.setValue = setValue;
        this.setValue(settings.value);

        progress = settings.progress;
        clipValue();

        /* Do an initial draw */
        drawLoader();

        /* In controllable mode, add event handlers */
        if (params.controllable === true) {
            (function () {
                var mouseDown, getDistance, adjustProgressWithXY;
                getDistance = function (x, y) {
                    return Math.sqrt(Math.pow(x - cX, 2) + Math.pow(y - cY, 2));
                };

                mouseDown = false;

                adjustProgressWithXY = function (x, y) {
                    /* within the bar, calculate angle of touch point */
                    var pX, pY, angle, startTouchAngle, range, posValue;
                    pX = x - cX;
                    pY = y - cY;

                    angle = Math.atan2(pY, pX);
                    if (angle > Math.PI / 2.0) {
                        angle -= (Math.PI * 2.0);
                    }

                    startTouchAngle = startAngle - (Math.PI * 2.0);
                    range = endAngle - startAngle;
                    posValue = (angle - startTouchAngle) / range;
                    setProgress(posValue);

                    if (params.onProgressUpdate) {
                        /* use the progress value as this will have been clipped
                         * to the correct range [0..1] after the call to setProgress
                         */
                        params.onProgressUpdate(progress);
                    }
                };

                $(outerDiv).mousedown(function (e) {
                    var offset, x, y, distance;
                    offset = $(this).offset();
                    x = e.pageX - offset.left;
                    y = e.pageY - offset.top;

                    distance = getDistance(x, y);

                    if (distance > innerRadius && distance < radius) {
                        mouseDown = true;
                        adjustProgressWithXY(x, y);
                    }
                }).mouseup(function () {
                    mouseDown = false;
                }).mousemove(function (e) {
                    var offset, x, y;
                    if (mouseDown) {
                        offset = $(outerDiv).offset();
                        x = e.pageX - offset.left;
                        y = e.pageY - offset.top;
                        adjustProgressWithXY(x, y);
                    }
                }).mouseleave(function () {
                    mouseDown = false;
                });
            }());
        }
        return this;
    };
}(jQuery));
