<?php
/**
 * Main controller
 *
 * @package otfd
 * @subpackage _controllers
 * @since One Time File Download
 */

namespace Otfd\Controllers;

defined( 'ABSPATH' ) or die( 'No script kiddies please!' );

/**
 * Class One Time File Download Controller
 *
 * @category Controller
 * @package  otfd
 * @author  IV | PIXOLETTE
 * @license  http://wp.pixollete.com
 * @link     http://wp.pixollete.com
 */
class Db {

	public $otfd_db_version;
	protected $table_name;
	protected $charset_collate;

	/**
	 * Function to be called on class init
	 */
	function __construct() {
		global $wpdb;

		$this->otfd_db_version  = '2.1';
		$this->files_table_name = $wpdb->base_prefix . 'otfd_files';
		$this->links_table_name = $wpdb->base_prefix . 'otfd_links';
		$this->charset_collate  = $wpdb->get_charset_collate();
	}

	/**
	 * Function to create tables on plugin install
	 */
	function create_table_on_install() {
		require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );

		$sql = "CREATE TABLE $this->files_table_name (
			id int NOT NULL AUTO_INCREMENT,
			file_id int NULL,
			external_file_link text NULL,
			site_id int NULL,
			created_by int NOT NULL,
			timestamp int NOT NULL,
			valid_after_download int DEFAULT 0 NULL,
			valid_after_link_is_created int DEFAULT 0 NULL,
			send_link_via_email int DEFAULT 0 NULL,
			title text NOT NULL,
			link_title text NOT NULL,
			link_css_class text NOT NULL,
			filename text NOT NULL,
			form_data text NULL,
			form_data_custom_fields text NULL,
			email_data text NULL,
			content_is_cached int DEFAULT 1 NULL,
			display_download_page int DEFAULT 0 NULL,
			download_page_content text NULL,
			status int DEFAULT 1 NULL,
			days_to_keep_entries int NULL,
			PRIMARY KEY (id)
		) $this->charset_collate;";

		dbDelta( $sql );

		$links_sql = "CREATE TABLE $this->links_table_name (
			id int NOT NULL AUTO_INCREMENT,
			otfd_id int NOT NULL,
			created_at int NOT NULL,
			expire_at int NOT NULL,
			is_valid int DEFAULT 1 NOT NULL,
			is_downloaded int DEFAULT 0 NOT NULL,
			downloaded_timestamp int DEFAULT 0 NULL,
			url_key text NULL,
			user_data text NULL,
			PRIMARY KEY (id)
		) $this->charset_collate;";

		dbDelta( $links_sql );

		if ( get_option( 'otfd_db_version' ) ) {
			update_option( 'otfd_db_version', $this->otfd_db_version );
		} else {
			add_option( 'otfd_db_version', $this->otfd_db_version );
		}
	}

	/**
	 * Function to create entry
	 *
	 * @param mixed $data .
	 * @param mixed $format .
	 */
	function create_new_file_entry( $data ) {
		global $wpdb;
		$res = $wpdb->insert( $this->files_table_name, $data );

		return $wpdb->insert_id;
	}

	/**
	 * Function to update entry
	 *
	 * @param mixed $data .
	 * @param int $id .
	 * @param mixed $format .
	 */
	function update_file_entry( $data, $id ) {
		global $wpdb;
		$wpdb->update( $this->files_table_name, $data, $id );
	}

	/**
	 * Function to get row entry
	 *
	 * @param int $id .
	 */
	function get_file_entry( $id ) {
		global $wpdb;
		$sql   = $wpdb->prepare( "SELECT * FROM $this->files_table_name WHERE id = %d AND status = 1 LIMIT 1", intval( $id ) );
		$entry = $wpdb->get_row( $sql, ARRAY_A );

		return $entry;
	}

	/**
	 * Function to get all entries
	 *
	 * @param int $page_nr .
	 * @param int $per_page .
	 */
	function get_file_entries( $page_nr = 1, $per_page = 20 ) {
		global $wpdb;
		$site_id = get_current_blog_id();
		$offset  = ( $page_nr - 1 ) * $per_page;
		$sql     = $wpdb->prepare( "SELECT * FROM $this->files_table_name WHERE status = %d AND site_id = %d ORDER BY id DESC LIMIT %d, %d", 1, $site_id, $offset, $per_page );
		$entries = $wpdb->get_results( $sql, ARRAY_A );

		return $entries;
	}

	/**
	 * Function to get all entries
	 *
	 * @param int $page_nr .
	 * @param int $per_page .
	 */
	function get_total_file_entries() {
		global $wpdb;
		$site_id = get_current_blog_id();
		$sql     = $wpdb->prepare( "SELECT count(id) as total_entries FROM $this->files_table_name WHERE status = %d AND site_id = %d", 1, $site_id );
		$entries = $wpdb->get_row( $sql, ARRAY_A );

		return $entries;
	}

	/**
	 * Function to get all entries
	 *
	 * @param int $page_nr .
	 * @param int $per_page .
	 */
	function get_file_entries_for_cleanup() {
		global $wpdb;
		$site_id = get_current_blog_id();
		$sql     = $wpdb->prepare( "SELECT * FROM $this->files_table_name WHERE status = %d AND site_id = %d AND send_link_via_email = %d AND days_to_keep_entries > %d", 1, $site_id, 1, 0 );
		$entries = $wpdb->get_results( $sql, ARRAY_A );

		return $entries;
	}

	// Link functionality.

	/**
	* Function to create entry
	*
	* @param mixed $data .
	* @param mixed $format .
	*/
	function create_new_link_entry( $data, $format ) {
		global $wpdb;
		$res = $wpdb->insert( $this->links_table_name, $data, $format );

		return $wpdb->insert_id;
	}

	/**
	 * Function to get row entry
	 *
	 * @param int $id .
	 */
	function get_link_entry_by_url_key( $url_key ) {
		global $wpdb;
		$sql   = $wpdb->prepare( "SELECT l.*, l.id as linkid, f.* FROM $this->links_table_name l LEFT JOIN $this->files_table_name as f ON l.otfd_id = f.id WHERE l.url_key = %s AND l.is_valid = 1 AND f.status = 1 LIMIT 1", $url_key );
		$entry = $wpdb->get_row( $sql, ARRAY_A );

		return $entry;
	}

	/**
	 * Function to update entry
	 *
	 * @param mixed $data .
	 * @param int $id .
	 * @param mixed $format .
	 */
	function update_link_entry( $data, $id ) {
		global $wpdb;
		$status = $wpdb->update( $this->links_table_name, $data, $id );
	}

	/**
	 * Function to get all entries
	 *
	 * @param int $page_nr .
	 * @param int $per_page .
	 */
	function get_form_entries_count( $page_nr = 1, $per_page = 20 ) {
		global $wpdb;
		$site_id = get_current_blog_id();
		$offset  = ( $page_nr - 1 ) * $per_page;
		$sql     = $wpdb->prepare( "SELECT f.*, l.is_valid, l.is_downloaded, l.user_data, count(*) as total FROM $this->files_table_name AS f LEFT JOIN $this->links_table_name AS l ON f.id = l.otfd_id WHERE f.status = %d AND f.site_id = %d AND f.send_link_via_email = %d GROUP BY f.id ORDER BY f.id DESC", 1, $site_id, 1 );
		$entries = $wpdb->get_results( $sql, ARRAY_A );

		return $entries;
	}

	/**
	 * Function to get all entries
	 *
	 * @param int $page_nr .
	 * @param int $per_page .
	 */
	function get_form_entries_for_form( $otfd_id, $page_nr = 1, $per_page = 20 ) {
		global $wpdb;
		$offset  = ( $page_nr - 1 ) * $per_page;
		$sql     = $wpdb->prepare( "SELECT * FROM $this->links_table_name WHERE otfd_id = %d ORDER BY id DESC LIMIT %d, %d", $otfd_id, $offset, $per_page );
		$entries = $wpdb->get_results( $sql, ARRAY_A );

		return $entries;
	}

	/**
	 * Function to get all entries
	 *
	 * @param int $page_nr .
	 * @param int $per_page .
	 */
	function get_count_form_entries_for_form( $otfd_id ) {
		global $wpdb;
		$sql     = $wpdb->prepare( "SELECT count( otfd_id ) as total_entries FROM $this->links_table_name WHERE otfd_id = %d", $otfd_id );
		$entries = $wpdb->get_row( $sql, ARRAY_A );

		return $entries;
	}

	/**
	 * Function to delete entry
	 *
	 * @param mixed $data .
	 * @param int $id .
	 * @param mixed $format .
	 */
	function delete_link_entry( $id ) {
		global $wpdb;
		$status = $wpdb->delete( $this->links_table_name, array( 'id' => $id ), array( '%d' ) );
	}

	/**
	 * Function to delete entry
	 *
	 * @param mixed $data .
	 * @param int $id .
	 * @param mixed $format .
	 */
	function cleanup_link_entries_older_than( $otfd_id, $older_than ) {
		global $wpdb;
		$db_query = $wpdb->prepare(
			"
			DELETE FROM $this->links_table_name
			WHERE otfd_id = %d
			AND is_downloaded = 1
			AND downloaded_timestamp > 0
			AND downloaded_timestamp < %d
			",
			$otfd_id, $older_than
		);
		$wpdb->query( $db_query );
	}

}
