<?php if ( ! defined( 'ABSPATH' ) ) exit;

/*
 * Plugin Name: Ninja Forms - Vimeo Uploader
 * Plugin URI: https://ninjaforms.com/extensions/vimeo-uploader/
 * Description: Adds the ability to upload videos directly to Vimeo from Ninja Forms
 * Version: 3.0.0
 * Author: Dave Kiss
 * Author URI: http://davekiss.com
 * Text Domain: ninja-forms-vimeo-uploader
 *
 * Copyright 2016 Dave Kiss.
 */

// define('NF_DEV', true);

// Put this in a readme.
// An example on how to get a field class from Ninja Forms
// $field_class = Ninja_Forms()->fields[$field_type];

if ( version_compare( get_option( 'ninja_forms_version', '0.0.0' ), '3.0.0', '<' ) || get_option( 'ninja_forms_load_deprecated', FALSE ) ) {

    if( ! defined( 'NF_VIMEO_UPLOADER_VERSION' ) ){
        define( 'NF_VIMEO_UPLOADER_VERSION', '3.0.0' );
    }
    include 'deprecated/ninja-forms-vimeo-uploader.php';
    include 'includes/Database/Upgrade.php';
    new NF_VimeoUploader_Database_Upgrade();

} else {

  /**
   * Class NF_VimeoUploader
   */
  final class NF_VimeoUploader {
    const VERSION = '3.0.0';
    const SLUG    = 'vimeo-uploader';
    const NAME    = 'Vimeo Uploader';
    const AUTHOR  = 'Dave Kiss';
    const PREFIX  = 'NF_VimeoUploader';

    /**
     * @var NF_VimeoUploader
     * @since 3.0
     */
    private static $instance;

    /**
     * Plugin Directory
     *
     * @since 3.0
     * @var string $dir
     */
    public static $dir = '';

    /**
     * Plugin URL
     *
     * @since 3.0
     * @var string $url
     */
    public static $url = '';

    /**
     * Main Plugin Instance
     *
     * Insures that only one instance of a plugin class exists in memory at any one
     * time. Also prevents needing to define globals all over the place.
     *
     * @since 3.0
     * @static
     * @static var array $instance
     * @return NF_VimeoUploader Highlander Instance
     */
    public static function instance() {
      if (!isset(self::$instance) && !(self::$instance instanceof NF_VimeoUploader)) {
        self::$instance = new NF_VimeoUploader();

        self::$dir = plugin_dir_path(__FILE__);

        self::$url = plugin_dir_url(__FILE__);

        if ( ! class_exists('Vimeography_Vimeo') ) {
          require_once self::$dir . 'includes/vendor/davekiss/vimeo-php/vimeo.php';
        }

        /*
         * Register our autoloader
         */
        spl_autoload_register(array(self::$instance, 'autoloader'));
      }

      return self::$instance;
    }

    public function __construct() {
      add_action( 'admin_init', array( $this, 'setup_license') );
      add_action( 'admin_init', array( $this, 'load_upgrade_routine') );
      add_action( 'ninja_forms_loaded', array($this, 'ninja_forms_loaded'));
      add_filter( 'ninja_forms_register_fields', array($this, 'register_fields'));
      add_filter( 'ninja_forms_register_actions', array($this, 'register_actions'));
      add_filter( 'ninja_forms_field_type_sections', array( 'NF_VimeoUploader_Config_FieldTypeSections', 'add_vimeo_field_type_section' ) );
      add_filter( 'ninja_forms_field_template_file_paths', array($this, 'register_templates_path'));
    }

    /**
     * [load_upgrade_routine description]
     * @return [type] [description]
     */
    public function load_upgrade_routine() {
      new NF_VimeoUploader_Database_Upgrade();
    }


    /**
     * Force load the remaining classes
     *
     * @return [type] [description]
     */
    public function ninja_forms_loaded() {
      new NF_VimeoUploader_Admin_Metaboxes_Submission();
      new NF_VimeoUploader_Admin_CPT_Submission();
      new NF_VimeoUploader_Admin_Settings();

      Ninja_Forms()->merge_tags[ 'vimeo_uploader' ] = new NF_VimeoUploader_Config_MergeTags();
    }


    /**
     * Adds the setting group to the Ninja Forms Settings menu page
     *
     * @param [type] $sections [description]
     */
    public function add_vimeo_field_type_section($sections) {
      $sections['vimeo_uploader'] = array(
        'id' => 'vimeo_uploader',
        'nicename' => __( 'Vimeo Uploader', 'ninja-forms-vimeo-uploader' ),
        'fieldTypes' => array(),
      );

      return $sections;
    }


    /**
     * [setup_license description]
     * @return [type] [description]
     */
    public function setup_license() {
      if ( ! class_exists( 'NF_Extension_Updater' ) ) return;

      new NF_Extension_Updater( self::NAME, self::VERSION, self::AUTHOR, __FILE__, self::SLUG );
    }


    /**
     * Add our templates to Ninja Form's registered template paths.
     *
     * @param  array $file_paths pre-registered template paths
     * @return array
     */
    public function register_templates_path($file_paths) {
      $file_paths[] = plugin_dir_path(__FILE__) . 'includes/Templates/';
      return $file_paths;
    }


    /**
     * Load a Template based on the filename and make the
     * passed data available to it.
     *
     * Currently used by the custom Vimeo Uploader submission metabox.
     * /includes/admin/metabox/submission.php
     *
     * @param string $file_name
     * @param array $data
     */
    public static function template( $file_name = '', array $data = array() ) {
      if ( ! $file_name ) return;
      extract( $data );
      include self::$dir . 'includes/Templates/' . $file_name;
    }


    /**
     * [register_fields description]
     * @param  [type] $fields [description]
     * @return [type]         [description]
     */
    public function register_fields($fields) {
      $fields[ 'vimeo_uploader_title' ]       = new NF_VimeoUploader_Fields_Title();
      $fields[ 'vimeo_uploader_description' ] = new NF_VimeoUploader_Fields_Description();
      $fields[ 'vimeo_uploader_dropzone' ]    = new NF_VimeoUploader_Fields_Dropzone();

      return $fields;
    }

    /**
     * [register_actions description]
     * @param  [type] $actions [description]
     * @return [type]          [description]
     */
    public function register_actions($actions) {
      $actions[ 'vimeo_uploader' ] = new NF_VimeoUploader_Actions_VimeoUploader();
      return $actions;
    }

    /**
     * [autoloader description]
     * @param  [type] $class_name [description]
     * @return [type]             [description]
     */
    public function autoloader($class_name) {
      if (class_exists($class_name)) return;

      if ( false === strpos( $class_name, self::PREFIX ) ) return;

      $class_name = str_replace( self::PREFIX, '', $class_name );
      $classes_dir = realpath(plugin_dir_path(__FILE__)) . DIRECTORY_SEPARATOR . 'includes' . DIRECTORY_SEPARATOR;
      $class_file = str_replace('_', DIRECTORY_SEPARATOR, $class_name) . '.php';

      if (file_exists($classes_dir . $class_file)) {
        require_once $classes_dir . $class_file;
      }
    }
  }

  /**
   * The main function responsible for returning The Vimeo Uploader Plugin
   * Instance to functions everywhere.
   *
   * Use this function like you would a global variable, except without needing
   * to declare the global.
   *
   * @since 3.0
   * @return {class} Vimeo Uploader Instance
   */
  function NF_VimeoUploader() {
    return NF_VimeoUploader::instance();
  }

  NF_VimeoUploader();
}
