// Create a new object for custom validation of a custom field.
var VimeoUploaderController = Marionette.Object.extend( {
  fieldType: 'vimeo_uploader_dropzone',
  resumable: '',
  progress: '',

  initialize: function() {

    // On the Form Submission's field validaiton...
    var submitChannel = Backbone.Radio.channel( 'submit' );
    this.listenTo( submitChannel, 'validate:field', this.validateRequired );

    // on the Field's model value change...
    var fieldsChannel = Backbone.Radio.channel( 'fields' );
    this.listenTo( fieldsChannel, 'change:modelValue', this.validateRequired );

    /* Call the setup method when fields are initiated */
    this.listenTo( fieldsChannel, 'init:model', this.setup );

    // Add additional values to the submitted information
    // We need this to get the final resource location for the new video
    Backbone.Radio.channel( this.fieldType ).reply( 'get:submitData', this.getSubmitData );
  },

  setup: function( fieldModel ) {
    var _self = this;

    // Only setup a specific fields type.
    if ( 'vimeo_uploader_dropzone' != fieldModel.get( 'type' ) ) return;

    var formID = fieldModel.get('formID');

    // Bind upload functionality
    // Start uploading when form is submitted.
    Backbone.Radio.channel( 'form-' + formID ).reply( 'maybe:submit', this.beforeSubmit, this, formID );

    this.listenTo( Backbone.Radio.channel( 'form' ), 'render:view', function(form) {

      // console.log( fieldModel.get('complete_uri') );
      // console.log(fieldModel, form);

      var scope = '#nf-field-' + fieldModel.get('id') + '-wrap',
          $dropzone = jQuery(form.el);

      _self.progress = new ProgressBar.Line(scope + ' .nf-vimeo-uploader-progress-bar', {
        color: "#27ae60",
        trailColor: "#999",
        strokeWidth: 0.5,
        easing: "easeIn",
        duration: 200
      });

      _self.resumable = new Resumable({
        target              : fieldModel.get('upload_link'), /* target URL for request */
        testChunks          : true,
        simultaneousUploads : 1,
        method              : 'octet',
        uploadMethod        : 'PUT',
        preprocess: function(chunk){
          if (chunk.startByte > 0) {
            chunk.opts.headers = {
              /* 'Content-Length': chunk.fileObjSize, */
              /* 'Content-Type'  : 'video/mp4', */
              'Content-Range' : 'bytes ' + chunk.startByte + '-' + chunk.endByte + '/' + chunk.fileObjSize
            };
          }
          /* console.log('Preprocessing chunk ' + chunk.offset); */
          /* console.log('Chunk #' + chunk.offset + ' state is ' + chunk.preprocessState); */
          chunk.preprocessFinished();
          /* console.log('Preprocess complete'); */
        }
      });

      /* Resumable.js isn't supported, fall back on a different method (form ticket) */
      /* if(! r.support) location.href = '/default-uploader'; */

      _self.resumable.assignBrowse( $dropzone.find('.ninja-forms-vimeo-uploader-file-browser') );
      _self.resumable.assignDrop( $dropzone.find('.ninja-forms-vimeo-uploader-file-dropzone') );

      /* Event listeners */
      _self.resumable.on('fileAdded', function(file){
        fieldModel.set('video_file_size', file.size);
        $dropzone.find('.ninja-forms-vimeo-uploader-file-dropzone').show();
        $dropzone.find('.ninja-forms-vimeo-uploader-file-dropzone h3').text(file.fileName);
        $dropzone.find('input[name="_nf_vimeo_uploader_filename"]').val(file.fileName).trigger( 'change' );
      });

      _self.resumable.on('fileSuccess', function(file){
        console.log(file);
      });

      _self.resumable.on('error', function(file, message){
        console.log(file);
        console.log(message);
      });

      _self.resumable.on('fileProgress', function(file) {
        var rounded = Math.round( file.progress()  * 1000 ) / 10;
        var percent = rounded.toFixed(1) + '%';
        $dropzone.find('.nf-vimeo-uploader-percentage').html(percent);

        _self.progress.animate(file.progress(), {
            duration: 800
        }, function() {
            /* console.log('Animation has finished'); */
        });

        /* $.each(file.chunks, function(i, chunk){
          if (chunk.status() == 'uploading') {
            console.log('Uploading chunk ' + i);
          }
        }); */
      });

      /* Upload complete */
      _self.resumable.on('complete', function(){

        $dropzone.find('.ninja-forms-vimeo-uploader-file').addClass('nf-vimeo-uploader-completed');

        // Set re-start flag when the upload is complete.
        Backbone.Radio.channel( 'form-' + form.model.get( 'id' ) ).request( 'add:extra', 'vimeo_upload_complete', true );

        // Manually re-start the submission.
        Backbone.Radio.channel( 'form-' + form.model.get( 'id' ) ).request( 'submit', form.model );
      });

    } );
  },

  validateRequired: function( model ) {
    // Only validate a specific fields type.
    if ( 'vimeo_uploader_dropzone' != model.get( 'type' ) ) return;

    // Only validate if the field is marked as required?
    if ( 0 == model.get( 'required' ) ) return;

    // Check if Model has a value
    if ( model.get( 'value' ) ) {
      // Remove Error from Model
      Backbone.Radio.channel( 'fields' ).request( 'remove:error', model.get( 'id' ), 'vimeo-uploader-dropzone-error' );
    } else {
      // Add Error to Model, passing model id, error id, and message
      Backbone.Radio.channel( 'fields' ).request( 'add:error', model.get( 'id' ), 'vimeo-uploader-dropzone-error', 'Don\'t forget to upload your video!' );
    }
  },

  getSubmitData: function( fieldData, field ) {
    fieldData.complete_uri = field.get('complete_uri');
    fieldData.video_file_size = field.get('video_file_size');
    return fieldData;
  },

  beforeSubmit: function( formID ) {

    var formModel = Backbone.Radio.channel( 'app' ).request( 'get:form', formID );

    if ( formModel.getExtra( 'vimeo_upload_complete' ) ) {
      return true;
    }

    this.upload( formModel );

    // Halt form submission.
    return false;
  },

  upload: function( formModel ) {
    jQuery(formModel.el).find('.ninja-forms-vimeo-uploader-file').addClass('nf-vimeo-uploader-uploading');
    this.resumable.upload();
  }
});


// On Document Ready...
jQuery( document ).ready( function( $ ) {

  // Instantiate our custom field's controller, defined above.
  new VimeoUploaderController();
});
