<?php if ( ! defined( 'ABSPATH' ) || ! class_exists( 'NF_Abstracts_Action' )) exit;

/**
 * Class NF_Action_VimeoUploader
 */
final class NF_VimeoUploader_Actions_VimeoUploader extends NF_Abstracts_Action {
  /**
   * @var string
   */
  protected $_name  = 'vimeo_uploader';

  /**
   * @var array
   */
  protected $_tags = array();

  /**
   * @var string
   */
  protected $_timing = 'normal';

  /**
   * @var int
   */
  protected $_priority = 10;

  /**
   * Vimeo class instance
   *
   * @var instanceof Vimeography_Vimeo
   */
  protected $_vimeo;

  /**
   * Vimeo Access token
   *
   * @var string
   */
  protected $_auth;

  /**
   * Constructor
   */
  public function __construct() {
    parent::__construct();

    $this->_nicename = __( 'Upload to Vimeo', 'ninja-forms-vimeo-uploader' );
    $this->_settings = array(
      'vimeo_uploader_title_processed' => array(
        'name' => 'vimeo_uploader_title_processed',
        'type' => 'textbox',
        'label' => __('Final Video Title', 'ninja-forms-vimeo-uploader'),
        'group' => 'primary',
        'value' => '', // No defauly value. Cannot assume the field key.
        'help'  => __('If you wish to make any changes to the uploaded video title before it is stored on Vimeo, you can do so here. The default behavior is to use the video title provided when the form was submitted.', 'ninja-forms-vimeo-uploader'),
        'use_merge_tags' => true,
        'width' => 'full'
      ),
      'vimeo_uploader_description_processed' => array(
        'name' => 'vimeo_uploader_description_processed',
        'type' => 'textarea',
        'label' => __('Final Video Description', 'ninja-forms-vimeo-uploader'),
        'group' => 'primary',
        'value' => '', // No default value. Cannot assume the field key.
        'help'  => __('If you wish to make any changes to the uploaded video description before it is stored on Vimeo, you can do so here. The default behavior is to use the video description provided when the form was submitted.', 'ninja-forms-vimeo-uploader'),
        'use_merge_tags' => true,
        'width' => 'full'
      ),
      // Minimum User Role required for uploading
      'vimeo_uploader_min_user_role' => array(
        'name' => 'vimeo_uploader_min_user_role',
        'type' => 'select',
        'group' => 'primary',
        'label' => __( 'Minimum User Role for Uploading', 'ninja-forms-vimeo-uploader' ),
        'options' => array(
          array( 'label' => 'Admin+',  'value' => 'manage_options' ),
          array( 'label' => 'Editor+', 'value' => 'publish_pages' ),
          array( 'label' => 'Author+', 'value' => 'publish_posts' ),
          array( 'label' => 'Contributor+', 'value' => 'edit_posts' ),
          array( 'label' => 'Subscriber+',  'value' => 'read' ),
          array( 'label' => 'Guests+',  'value' => 'guest' ),
        ),
        'value' => 'guest',
        'width' => 'full',
        'help' => __( 'Select the minimum WordPress role that your site visitors must have in order to upload videos to your Vimeo account. If you would like for all visitors to have uploading permission, choose Guests+', 'ninja-forms-vimeo-uploader' ),
      ),
      // Uploaded Video Privacy Settings
      'vimeo_uploader_video_privacy' => array(
        'name' => 'vimeo_uploader_video_privacy',
        'type' => 'select',
        'group' => 'primary',
        'label' => __( 'Video Privacy', 'ninja-forms-vimeo-uploader' ),
        'options' => array(
          array( 'label' => 'Only Me',           'value' => 'nobody' ),
          array( 'label' => 'Hidden from Vimeo', 'value' => 'disable' ),
          array( 'label' => 'Only My Contacts',  'value' => 'contacts' ),
          array( 'label' => 'Public',            'value' => 'anybody' ),
        ),
        'value' => 'nobody',
        'width' => 'full',
      ),
      'vimeo_uploader_add_to_collection' => array(
        'name' => 'vimeo_uploader_add_to_collection',
        'type' => 'toggle',
        'group' => 'primary',
        'label' => __( 'Add uploaded videos to a Vimeo collection', 'ninja-forms-vimeo-uploader' ),
        'value' => 0,
        'help' => __( 'Check this if you want to add the uploaded video to a Vimeo collection (groups, channels, and albums are supported.)', 'ninja-forms-vimeo-uploader' ),
      ),
      'vimeo_uploader_collections'        => array(
        'name'  => 'vimeo_uploader_collections',
        'type'  => 'textbox',
        'group' => 'primary',
        'label' => __( 'Vimeo Collection URL(s)', 'ninja-forms-vimeo-uploader' ),
        'value' => '',
        'width' => 'full',
        'deps'  => array(
          'vimeo_uploader_add_to_collection' => 1,
        ),
        'help' => __( 'Enter the link of the Vimeo collection you want to add uploaded videos to. To add videos to multiple collections, enter them below, separated by a comma.', 'ninja-forms-vimeo-uploader' ),
      ),
    );
  }


  /*
  * PUBLIC METHODS
  */

  public function save( $action_settings ) { }

  public function process( $action_settings, $form_id, $data ) {

    # Loop through the fields and extract the relevant data
    foreach ( $data['fields'] as $field ) {
      if ( $field['settings']['type'] == 'vimeo_uploader_dropzone' ) {
        $complete_uri      = $field['complete_uri'];
        $filename          = $field['value'];
        $dropzone_field_id = $field['id'];
        $file_size         = intval( $field['video_file_size'] );
      }
    }

    $name = ! empty( $action_settings['vimeo_uploader_title_processed'] ) ? $action_settings['vimeo_uploader_title_processed'] : $filename;
    $description = ! empty( $action_settings['vimeo_uploader_description_processed'] ) ? $action_settings['vimeo_uploader_description_processed'] : $filename;

    // Ensure user has permission to upload
    $cap = $action_settings['vimeo_uploader_min_user_role'];
    $capability = apply_filters('nf-vimeo-uploader/upload-capability', $cap );

    if ( $capability !== 'guest' ) {
      if ( ! current_user_can( $capability ) ) {
        $data[ 'errors' ][ 'fields' ][ $dropzone_field_id ] = __('You do not have permission to upload a video.', 'ninja-forms-vimeo-uploader');
        return $data;
      }
    }

    // Go for it
    try {
      $resource = $this->_complete_upload( $complete_uri );

      $this->patch_video_data( $resource, array(
        'name'         => $name,
        'description'  => $description,
        'privacy.view' => $action_settings['vimeo_uploader_video_privacy'],
      ) );

      $video = $this->_get_video( $resource );

      // Add to collections if it is in the settings
      if ( $action_settings['vimeo_uploader_add_to_collection'] == 1 && ! empty( $action_settings['vimeo_uploader_collections'] ) ) {
        $this->add_video_to_collections( $resource, $action_settings['vimeo_uploader_collections'] );
      }

    } catch (Exception $e) {
      $data[ 'errors' ][ 'fields' ][ $dropzone_field_id ] = $e->getMessage();
      return $data;
    }

    // Update the meta for the submission with NF 3.0
    $data['extra']['vimeo_uploader'] = array(
      'vimeo_resource' => $resource,
      'vimeo_url'      => $video->link,
      'video_file_size' => $file_size
    );

    // Set merge tag in memory
    Ninja_Forms()->merge_tags[ 'vimeo_uploader' ]->set_vimeo_url( $video->link );

    // $data['debug'] = $video->link;

    return $data;
  }


  /**
   * Fetch a brand new streaming upload ticket from Vimeo.
   *
   * @return array Response body
   */
  public function get_streaming_upload_ticket() {
    $this->_before_request();
    $this->_params = array('type' => 'streaming');

    $response = $this->_vimeo->request('/me/videos', $this->_params, 'POST');

    switch( $response['status'] ) {
      case 201:
        $this->_ticket = $response['body'];
        return $this->_ticket;
      default:
        throw new Exception( sprintf( __('Unable to get an upload ticket (%1$s): %2$s ', 'ninja-forms-vimeo-uploader'), $response['status'], $response['body']->error ) );
    }
  }


  /**
   * Send the saved video data in a PATCH request to the video resource
   *
   * @return [type] [description]
   */
  public function patch_video_data($link, $params = array() ) {
    $this->_before_request();

    if ( ! empty($params) ) {
      $result = $this->_vimeo->request( $link, $params, 'PATCH');

      switch ( $result['status'] ) {
        case 204:
          // Success, video resource at $result['headers']['Location']
          return TRUE;
          break;
        case 403: // The user is not allowed to perform that action. [Your access token does not have the "edit" scope].
        default:
          // Unknown patch response]
          throw new Exception($result['body']->error);
          break;
      }
    }
  }


  /**
   * Set up the Vimeo request class and authentication.
   *
   * @return void
   */
  private function _before_request() {
    if ( ! $this->_auth || ! $this->_vimeo ) {
      $access_token = Ninja_Forms()->get_setting('vimeo_uploader_access_token');

      if (! $access_token || $access_token == '') {
        throw new Exception( __('Please visit the Vimeo Uploader settings page and enter an access token.', 'nf-vimeo-uploader') );
      }

      $this->_auth = $access_token;
      $this->_vimeo = new Vimeography_Vimeo(NULL, NULL, $this->_auth);
    }
  }


  /**
   * Close the upload on the Vimeo server.
   *
   * This actually doesn't publish the upload anymore as of 8/31/2016 because
   * Vimeo will automatically complete and publish the video when it detects that
   * it has finished uploading based on the bit range header submitted during the upload.
   *
   * However, it's currently the only way we can fetch the Vimeo URL of the newly uploaded
   * video based on the ticket complete URI, so we need to continue to make this request.
   *
   * @param  string  $ticket_complete_uri  URI to send a DELETE request to close the upload.
   * @return string  URL of the newly uploaded Vimeo video
   */
  private function _complete_upload($ticket_complete_uri) {
    $this->_before_request();
    $completion = $this->_vimeo->request( $ticket_complete_uri, array(), 'DELETE');

    //  Validate that we got back 201 Created
    $status = (int) $completion['status'];

    if ($status != 201) {
      throw new VimeoUploadException('Error completing the upload: ' . $completion['body']->error);
    }

    //  Furnish the location for the new clip in the API via the Location header.
    return $completion['headers']['Location'];
  }


  /**
   * Get details about a video from the given resource
   *
   * @return object Vimeo video object
   */
  private function _get_video($resource) {
    $this->_before_request();
    $result = $this->_vimeo->request( $resource, array(), 'GET');

    if ($result['status'] == 200) {
      return wp_remote_retrieve_body( $result );
    } else {
      // Failure, throw error
      throw new Exception('get_video failed.');
    }
  }


  /**
   * Determine the resource and add the Video to a Vimeo collection.
   *
   * @param [type] $collection_urls [description]
   */
  public function add_video_to_collections( $resource, $collections ) {
    $collections = explode( ',', $collections );

    foreach ( $collections as $collection ) {
      $collection_info = array();

      $match = preg_match('%(album|channels|groups)\/([a-zA-Z0-9]+)%', $collection, $collection_info);

      if ($match == 1) {
        switch( $collection_info[1] ) {
          case 'album':
            $target = sprintf('/me/albums/%1$s%2$s', $collection_info[2], $resource);
            break;
          case 'channels': case 'groups':
            $target = sprintf('/%1$s/%2$s%3$s', $collection_info[1], $collection_info[2], $resource);
            break;
          default:
            break;
        }

        if ( $target ) {
          $result = $this->_add_video_to_collection( $target );
        }
      }
    }
  }


  /**
   * Put the video in a Vimeo collection
   *
   * @param  string $target Resource to request
   * @return bool           Did it work?
   */
  private function _add_video_to_collection($target) {
    $this->_before_request();
    $result = $this->_vimeo->request($target, array(), 'PUT');

    switch ( $result['status'] ) {
      case 202: case 204:
        // Success
        return TRUE;
        default:
          // Denied
          throw new Exception($result['body']->error);
          break;
    }
  }
}
