<?php

class NF_Vimeo_Uploader_Upload {

  /**
   * The ticket response from the Vimeo API containing the Vimeo
   * upload target and the ticket complete URI.
   *
   * @var object
   */
  private $_ticket;

  /**
   * Vimeo class instance
   * @var instanceof
   */
  protected $_vimeo;

  /**
   * Vimeo Access token
   * @var string
   */
  protected $_auth;

  /**
   * [$_params description]
   * @var [type]
   */
  protected $_params;

  /**
   * [__construct description]
   */
  public function __construct() {

    $plugin_settings = nf_get_settings();

    if ( isset( $plugin_settings['nf_vimeo_uploader_access_token'] ) && ! empty( $plugin_settings['nf_vimeo_uploader_access_token'] ) ) {
      $this->_auth = $plugin_settings['nf_vimeo_uploader_access_token'];
      $this->_vimeo = new Vimeography_Vimeo(NULL, NULL, $this->_auth);
    } else {
      throw new Exception( __('Please visit the Vimeo Uploader settings page and enter an access token.', 'nf-vimeo-uploader') );
    }
  }


  /**
   * Sends a request to Vimeo to retrieve a streaming upload ticket.
   *
   * Streaming uploads are only possible using the Vimeography Upload javascript.
   *
   * @return array  body, status, headers
   */
  public function get_streaming_upload_ticket() {
    $this->_params = array('type' => 'streaming');

    $response = $this->_vimeo->request('/me/videos', $this->_params, 'POST');

    switch( $response['status'] ) {
      case 201:
        $this->_ticket = $response['body'];
        return $this->_ticket;
      default:
        throw new Exception( sprintf( __('Unable to get an upload ticket (%1$s): %2$s ', 'nf-vimeo-uploader'), $response['status'], $response['body']->error ) );
    }
  }

  /**
   * [get_user description]
   * @return [type] [description]
   */
  public function get_user() {
    $response = $this->_vimeo->request('/me');

    switch( $response['status'] ) {
      case 200:
        return $response['body'];
      default:
        throw new Exception( sprintf( __('Unable to get user (%1$s): %2$s ', 'nf-vimeo-uploader'), $response['status'], $response['body']->error ) );
    }
  }

  /**
   * [get_albums description]
   * @return [type] [description]
   */
  public function get_albums( $params = array() ) {
    $response = $this->_vimeo->request('/me/albums', $params );

    switch( $response['status'] ) {
      case 200:
        return $response;
      default:
        throw new Exception( sprintf( __('Unable to get albums (%1$s): %2$s ', 'nf-vimeo-uploader'), $response['status'], $response['body']->error ) );
    }
  }

  /**
   * Close the upload on the Vimeo server and PATCH any video data saved in
   * the site transient.
   *
   * @return bool
   */
  public function complete_upload($ticket_complete_uri) {
    $completion = $this->_vimeo->request( $ticket_complete_uri, array(), 'DELETE');

    //  Validate that we got back 201 Created
    $status = (int) $completion['status'];

    if ($status != 201) {
      throw new VimeoUploadException('Error completing the upload.');
    }

    //  Furnish the location for the new clip in the API via the Location header.
    return $completion['headers']['Location'];
  }


  /**
   * Send the saved video data in a PATCH request to the video resource
   *
   * @return [type] [description]
   */
  public function patch_video_data($link, $params = array() ) {

    if ( ! empty($params) ) {
      $result = $this->_vimeo->request( $link, $params, 'PATCH');

      switch ( $result['status'] ) {
        case 204:
          // Success, video resource at $result['headers']['Location']
          return TRUE;
          break;
        case 403: // The user is not allowed to perform that action. [Your access token does not have the "edit" scope].
        default:
          // Unknown patch response]
          throw new Exception('PATCH failed.');
          break;
      }
    }
  }

  /**
   * [put_video_in_album description]
   *
   * PUT https://api.vimeo.com/me/albums/{album_id}/videos/{video_id}
   * @param  [type] $video [description]
   * @param  [type] $album [description]
   * @return [type]        [description]
   */
  public function put_video_in_album($video_uri, $album_id) {
    $resource = sprintf('/me/albums/%1$s%2$s', $album_id, $video_uri);
    $result   = $this->_vimeo->request($resource, array(), 'PUT');
  }

  /**
   * [video_data description]
   * @return [type] [description]
   */
  public function get_video($resource) {
    $result = $this->_vimeo->request( $resource, array(), 'GET');

    if ($result['status'] == 200) {
      return wp_remote_retrieve_body( $result );
    } else {
      // Failure, throw error
      return FALSE;
    }
  }

  /**
   * [get_featured_embed description]
   * @param  [type] $link [description]
   * @return [type]       [description]
   */
  public function get_video_embed_code( $link ) {
    $params = array(
      'url'       => $link,
      'autoplay'  => 0,
      'title'     => 1,
      'portrait'  => 0,
      'byline'    => 0,
      'api'       => 1,
      'player_id' => 'nf_vimeo_uploader'. rand('1', '999999')
    );

    $query = http_build_query($params);

    $oembed = wp_remote_get( 'http://vimeo.com/api/oembed.json?' . $query);

    if ( is_wp_error( $oembed ) ) {
      throw new Exception( __('Could not retrieve the video: ', 'nf-vimeo-uploader') . $oembed->get_error_message() );
    } else {
      switch( $oembed['response']['code'] ) {
        case 200:
          $oembed = json_decode($oembed['body']);
          $oembed->html = str_replace('<iframe', '<iframe id="' . $params['player_id'] . '"', $oembed->html);
          return $oembed->html;
        case 403: case 404:
          throw new Exception( __('Your video privacy settings for this video must be adjusted to allow displaying this video on your site.', 'nf-vimeo-uploader') );
        default:
          break;
      }
    }
  }

  /* Template Variables */

  /**
   * The link used as the target for the video to be uploaded.
   *
   * @return string
   */
  public function ticket_upload_link_secure() {
    return $this->_ticket->upload_link_secure;
  }

  /**
   * The Vimeo user object associated with the current access token.
   *
   * @return [type] [description]
   */
  public function vimeo_user() {
    return $this->_ticket->user;
  }

  /**
   * [user_thumbnail description]
   * @return {[type]} [description]
   */
  public function user_thumbnail() {
    return end($this->_ticket->user->pictures);
  }

  /**
   * [available_free_space_raw description]
   *
   * @return [type] [description]
   */
  public function available_free_space_raw() {
    return $this->_ticket->user->upload_quota->space->free;
  }

  /**
   * [available_free_space_formatted description]
   * @return [type] [description]
   */
  public function available_free_space_formatted() {
    return parent::format_bytes($this->_ticket->user->upload_quota->space->free);
  }

  /**
   * [max_space_raw description]
   * @return [type] [description]
   */
  public function max_space_raw() {
    return $this->_ticket->user->upload_quota->space->max;
  }

  /**
   * [max_space_formatted description]
   * @return [type] [description]
   */
  public function max_space_formatted() {
    return parent::format_bytes($this->_ticket->user->upload_quota->space->max);
  }

  /**
   * [used_space_raw description]
   * @return [type] [description]
   */
  public function used_space_raw() {
    return $this->_ticket->user->upload_quota->space->used;
  }

  /**
   * Convert a given number of bytes into a human readable format,
   * using the largest unit the bytes will fit into.
   *
   * size_format is a WP function.
   *
   * @link https://codex.wordpress.org/Function_Reference/size_format
   * @return string
   */
  public function used_space_formatted() {
    $size = size_format($this->_ticket->user->upload_quota->space->used);
    return $size ? $size : '0MB';
  }

  /* Unused below */

  public function quota_resets() {
    // when the quota resets (number of uploads allowed)
    return $this->_ticket->user->upload_quota->resets;
  }

  /**
   * [available_hd_uploads description]
   * @return [type] [description]
   */
  public function available_hd_uploads() {
    return $this->_ticket->user->upload_quota->quota->hd;
  }

  /**
   * [available_sd_uploads description]
   * @return [type] [description]
   */
  public function available_sd_uploads() {
    return $this->_ticket->user->upload_quota->quota->sd;
  }
}
