<?php

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;

class NF_Vimeo_Uploader_Register_Metabox {

  /**
   * The message content of the admin_notice
   * @var string
   */
  protected $_notice;

  /**
   * The class to add to the admin_notice
   * @var string
   */
  protected $_status = 'error';

  public function __construct() {
    add_action( 'admin_init', array($this, 'register_vimeo_uploader_tab' ) );
    add_action( 'admin_init', array( $this, 'register_vimeo_uploader_metabox' ) );
    add_action( 'admin_menu', array( $this, 'add_submenu' ), 99 );
  }

  /**
   * Adds a submenu entry to the ninja forms menu
   */
  public function add_submenu() {
    $settings = add_submenu_page(
      "ninja-forms",
      __( 'Ninja Forms Vimeo Uploader', 'nf-vimeo-uploader' ),
      __( 'Vimeo Uploader', 'nf-vimeo-uploader' ),
      apply_filters( 'ninja_forms_vimeo_uploader_settings_capabilities', 'manage_options' ),
      "ninja-forms-extension-vimeo-uploader",
      'ninja_forms_admin'
    );
  }


  /**
   * Registers the Vimeo Uploader tab on the Vimeo Uploader page.
   * @return void
   */
  public function register_vimeo_uploader_tab() {
    $args = array(
     'name'             => __('Vimeo Uploader', 'nf-vimeo-uploader'),
     'page'             => 'ninja-forms-extension-vimeo-uploader',
     'display_function' => '',
     'save_function'    => array($this, 'save_vimeo_access_token'),
    );

    if ( function_exists( 'ninja_forms_register_tab' ) ) {
      ninja_forms_register_tab('vimeo_uploader', $args);
    }
  }

  /**
   * Save function for the Vimeo Uploader metabox,
   * this simply adds the `nf_vimeo_uploader_access_token` to the
   * NF plugin settings.
   *
   * @return string  Status message
   */
  public function save_vimeo_access_token($data) {
    if ($this->_status == 'updated' || $data['nf_vimeo_uploader_access_token'] == '') {
      $plugin_settings = nf_get_settings();
      foreach( $data as $key => $val ){
        $plugin_settings[$key] = $val;
      }
      update_option( 'ninja_forms_settings', $plugin_settings );
      $update_msg = __( 'Settings Saved', 'ninja-forms' );
      return $update_msg;
    } else {
      $update_msg = __( 'Settings Not Saved', 'ninja-forms' );
      return $update_msg;
    }
  }

  /**
   * Adds the Vimeo Uploader settings metabox to the NF Settings page.
   *
   * @return void
   */
  public function register_vimeo_uploader_metabox() {

    $vimeo_access_token = get_site_option( 'nf_vimeo_uploader_access_token' );
    $args = array(
      'page'  => 'ninja-forms-extension-vimeo-uploader',
      'tab'   => 'vimeo_uploader',
      'title' => __('Vimeo Uploader Settings', 'nf-vimeo-uploader'),
      'slug'  => 'nf_vimeo_uploader_access_token',
      'save_function' => array($this, 'validate_vimeo_access_token'),
      'settings' => array(
        array(
          'name'  => 'nf_vimeo_uploader_access_token',
          'type'  => 'text',
          'label' => __('Vimeo Access Token', 'nf-vimeo-uploader'),
          'desc'  => __('To connect your Vimeo account to your WordPress site, visit <a href="https://vimeography.com/upload/authorize" target="_blank">this page</a> and copy/paste your access token above.', 'nf-vimeo-uploader'),
          'default_value' => $vimeo_access_token,
        ),
      ),
    );

    if ( function_exists( 'ninja_forms_register_tab_metabox' ) ) {
      ninja_forms_register_tab_metabox( $args );
    }
  }

  /**
   * Checks the tokens provided by the user and saves them if they are valid.
   *
   * NOTE: This can't actually prevent the value from being saved because NF
   * doesn't allow us to reject an invalid setting, so do our best to provide a message to the user
   * based on the validity of their entry.
   *
   * @access public
   * @param array $data
   * @return void
   */
  public function validate_vimeo_access_token($data) {
    if ( isset( $data['nf_vimeo_uploader_access_token'] ) && ! empty( $data['nf_vimeo_uploader_access_token'] ) ) {
      try {
        $vimeo = new Vimeography_Vimeo(NULL, NULL, $data['nf_vimeo_uploader_access_token'] );
        $response = $vimeo->request('/me');

        if (! $response) {
          $this->_notice = __('Looks like the Vimeo API is having some issues right now. Try this again in a little bit.', 'nf-vimeo-uploader');
        }

        switch ( $response['status'] ) {
          case 200:
            $this->_status = 'updated';
            $this->_notice = sprintf( __('Success! Your Vimeo access token for %1$s has been added and saved.', 'nf-vimeo-uploader'), $response['body']->name );
            break;
          case 401:
            throw new Exception(__("Your Vimeo access token didn't validate. Try again, and double check that you are entering the correct token.", 'nf-vimeo-uploader'));
          case 404:
            throw new Exception(__('how the heck did you score a 404?', 'nf-vimeo-uploader'). $response['body']->error);
          default:
            throw new Exception(__('Unknown response status from the Vimeo API: ', 'nf-vimeo-uploader'). $response['body']->error);
        }

      } catch (Exception $e) {
        $this->_notice = sprintf( __('Dangit. %1$s', 'nf-vimeo-uploader'), $e->getMessage() );
      }

      add_action( 'admin_notices', array( $this, 'display_notice' ) );
    }
  }

  /**
   * Displays a success or failure message to the user on submit.
   *
   * @return [type] [description]
   */
  public function display_notice() {
    ?>
    <div class="<?php echo $this->_status; ?>">
      <p><?php echo $this->_notice; ?></p>
    </div>
    <?php
  }

}