<?php

/**
 * Generates OAuth tokens for immediate access and future refreshes
 *
 */
class CapsuleTokens {

    /**
     * URL for generating authorization code
     */
    const AUTHORISE_URL = 'https://api.capsulecrm.com/oauth/authorise';

    /**
     * URL for generating refresh and access tokens
     */
    const TOKEN_URL = 'https://api.capsulecrm.com/oauth/token';

    /**
     * Open Auth Client ID 
     */
    const CLIENT_ID = '19a5lwg2nms6x';

    /**
     * Generates and returns access token
     * 
     * Checks if refresh token is stored and if not attempts to generate that
     * before attempting to generate access token
     */
    public static function generate_access_token() {

        $tokens = self::retrieve_stored_tokens();

        if (0 == strlen($tokens['refresh_token'])) {

            self::generate_refresh_token();
        }

        $access_token = self::generate_access_token_from_refresh_token();

        return $access_token;
    }

    /**
     * Selects and calls a method to generate refresh token
     * 
     * If authorization code isn't available, attempts using V1 authtoken;
     * if neither is available, use empty string from authorization code.  This
     * last option is known to not succeed, but provides good support response
     */
    public static function generate_refresh_token($redirect = '') {

        $credentials = self::retrieve_stored_credentials();

        if (0 < strlen($credentials[NF_CapsuleCRM_Constants::AUTH_CODE]) ||
                0 == strlen($credentials[NF_CapsuleCRM_Constants::AUTHTOKEN])) {

            self::refresh_token_from_auth_code($credentials);
        } elseif (0 < strlen($credentials[NF_CapsuleCRM_Constants::AUTHTOKEN])) {

            self::refresh_token_from_authtoken($credentials);
        }

        if (0 < strlen($redirect)) {

            wp_redirect($redirect);
            exit;
        }
    }

    /**
     * Generates an access token from stored refresh token
     * 
     * On failure to generate access, returns empty string
     * @return string Access token value (empty string on failure)
     */
    protected static function generate_access_token_from_refresh_token() {

        $tokens = self::retrieve_stored_tokens();

        $parameter_array = array(
            'grant_type' => 'refresh_token',
            'client_id' => self::CLIENT_ID,
            'refresh_token' => $tokens['refresh_token']
        );

        $post_args = self::return_default_post_args();
        $post_args['headers'] = array('Content-Type' => 'application/json');

        $post_args['body'] = json_encode($parameter_array);

        $response = wp_remote_post(self::TOKEN_URL, $post_args);

        $validated = self::validate_token_response($response);

        self::update_tokens($validated);

        return $validated['access_token'];
    }

    /**
     * Generates a refresh token from an authorization code
     * @param array $credentials
     */
    protected static function refresh_token_from_auth_code($credentials) {

        $parameter_array = array(
            'grant_type' => 'authorization_code',
            'client_id' => self::CLIENT_ID,
            'code' => self::strip_auth_code($credentials[NF_CapsuleCRM_Constants::AUTH_CODE])
        );

        $post_args = self::return_default_post_args();
        $post_args['headers'] = array('Content-Type' => 'application/json');

        $post_args['body'] = json_encode($parameter_array);

        $response = wp_remote_post(self::TOKEN_URL, $post_args);

        $validated = self::validate_token_response($response);

        self::delete_used_auth_code();

        self::update_tokens($validated);
    }

    /**
     * Generates a refresh token from existing authtoken
     * 
     * Special migration process from V1.  This can be removed after confirming
     * that V1 is no longer available and migration is ended
     * 
     * @param array $credentials
     */
    static function refresh_token_from_authtoken($credentials) {

        $url = 'https://' . $credentials[NF_CapsuleCRM_Constants::SUBDOMAIN] . '.capsulecrm.com/oauth/token';

        $parameter_array = array(
            'grant_type' => 'password',
            'client_id' => self::CLIENT_ID,
            'username' => $credentials[NF_CapsuleCRM_Constants::AUTHTOKEN],
        );

        $post_args = self::return_default_post_args();
        $post_args['headers'] = array('Content-Type' => 'application/json');

        $post_args['body'] = json_encode($parameter_array);

        $response = wp_remote_post($url, $post_args);

        $validated = self::validate_token_response($response);

        self::update_tokens($validated);
    }

    /**
     * Removes the authorization code from settings after generating token
     */
    protected static function delete_used_auth_code() {

        if (!class_exists('Ninja_Forms')) {
            return;
        }

        Ninja_Forms()->update_setting(NF_CapsuleCRM_Constants::AUTH_CODE, '');
    }

    /**
     * Evaluates input for successful refresh token and calls update method
     * 
     * If error or refresh token not available, uses default values.
     * 
     * @param object|array $raw_response Full wp_remote_post response
     * @return array Keyed array of tokens, with default value backup
     */
    protected static function validate_token_response($raw_response) {

        $default_token_response = self::return_default_token_array();
        $module = 'RefreshToken';
        if (is_wp_error($raw_response)) {

            $wp_errors = $raw_response->get_error_messages();

            $user_friendly = array(NF_CapsuleCRM_Constants::ACTION_REQUIRED.'There was an internal WP error trying to communicate ' . date('y-m-d h:i:s'));
            $message_array = array_merge($user_friendly, $wp_errors);

            self::create_log_entry($module, $message_array);

            return $default_token_response;
        }

        $response_array = json_decode($raw_response['body'], TRUE);


        if (isset($response_array['error_description'])) {

            $user_friendly = array(NF_CapsuleCRM_Constants::ACTION_REQUIRED. 'The request was rejected by the CRM ' . date('Y-m-d h:i:s'));

            $message_array = array_merge($user_friendly, array($response_array['error_description']));

            self::create_log_entry($module, $message_array);

            return $default_token_response;
        }

        $token_array = array_intersect_key($response_array, $default_token_response) + $default_token_response;

        $message_array = array(NF_CapsuleCRM_Constants::SUCCESS. 'The refresh token was successfully generated and stored ' . date('Y-m-d h:i:s'));

        self::create_log_entry($module, $message_array);

        return $token_array;
    }

    /**
     * Creates log entries and updates the wp_option
     * 
     * @param string $module
     * @param array $message_array
     */
    protected static function create_log_entry($module, $message_array) {

        $obj_str = NF_CapsuleCRM_Constants::COMM_LOG_CLASS;
        // TODO: replace with obj_str after PHP 5.2 support is dropped
        $comm = new CapsuleCommLog;

        foreach ($message_array as $message) {

            $comm->log_entry(NF_CapsuleCRM_Constants::COMM_STATUS, $module, $message);
        }

        $comm->update(NF_CapsuleCRM_Constants::SUPPORT_DATA);

        return;
    }

    /**
     * Updates wp_options with given token array
     * 
     * @param array $token_array
     */
    protected static function update_tokens($token_array) {

        update_option(NF_CapsuleCRM_Constants::OAUTH_CREDENTIALS_KEY, $token_array);
    }

    /**
     * Returns default token array
     * @return array
     */
    protected static function return_default_token_array() {

        return array('refresh_token' => '', 'access_token' => '');
    }

    /**
     * Returns default post arguments
     * @return array
     */
    protected static function return_default_post_args() {

        return array(
            'timeout' => 45,
            'redirection' => 0,
            'httpversion' => '1.0',
            'sslverify' => TRUE,
        );
    }

    /**
     * Extracts the auth code from the full URL value
     * 
     * Enables users to copy and paste the entire URL string from the CRM
     * @param string $raw_auth_code
     * @return string
     */
    protected static function strip_auth_code( $raw_auth_code ) {

        $prefix = '?code=';

        $url_present = strpos( $raw_auth_code, $prefix );

        if ( false === $url_present ) {

            $auth_code = $raw_auth_code;
        } else {

            $wip = strstr( $raw_auth_code, $prefix );
            $auth_code = str_replace( $prefix, '', $wip );
        }

        return $auth_code;
    }

    /**
     * Returns URL for Authorization Code generation
     * @return string HTML markup
     */
    public static function generate_auth_code_url() {

        $url = self::AUTHORISE_URL;

        $url .= '?response_type=code&scope=read write&client_id=' . self::CLIENT_ID;

        return $url;
    }

    /**
     * Returns all stored required credentials from ninja_forms_settings 
     * 
     * Uses native WP methods only in case NF not loaded
     * 
     * @return array
     */
    public static function retrieve_stored_credentials() {

        $default_array = array(
            NF_CapsuleCRM_Constants::AUTH_CODE => '',
            NF_CapsuleCRM_Constants::PERSONAL_ACCESS_TOKEN => '',
            NF_CapsuleCRM_Constants::AUTHTOKEN => '',
            NF_CapsuleCRM_Constants::SUBDOMAIN => '',
        );

        $options = get_option('ninja_forms_settings');

        $credentials = array_intersect_key($options, $default_array) + $default_array;

        return $credentials;
    }

    /**
     * Returns stored access and refresh tokens
     * @return array
     */
    public static function retrieve_stored_tokens() {

        $default_tokens = self::return_default_token_array();

        $from_wp_options = get_option(NF_CapsuleCRM_Constants::OAUTH_CREDENTIALS_KEY);

        if (!is_array($from_wp_options)) {

            return $default_tokens;
        }

        $stored_tokens = $from_wp_options + $default_tokens;

        return $stored_tokens;
    }

}
