<?php

/**
 * Extracts authorization parameters and provides them for CRM access
 * 
 * Helps with troubleshooting and support messages; may determine auth level
 * based on extracted auth parameters
 *
 * @author stuartlb3
 */
class CapsuleAuth {

    /**
     * Open Auth Access Token used in OAuth V2 communication
     * @var string
     */
    protected $access_token;

    /**
     * Personal Access Token used in V2 communication
     * @var string
     */
    protected $personal_access_token;

    /**
     * Authorization Token used in V1 communication
     * @var string
     */
    protected $authtoken;

    /**
     * Subdomain used
     * @var string
     */
    protected $subdomain;

    /**
     * The authorization level determined by highest setting stored
     * @var string
     */
    protected $auth_level;

    /**
     * Error message
     * @var string
     */
    protected $error_message;

    public function __construct() {

        $this->get_stored_credentials();

        $this->get_access_token();

        $this->determine_auth_level();
    }

    /**
     * Sets the auth level based on highest authority stored in settings
     * 
     * If a Personal Access token is set, set the auth level for V2
     * 
     * If
     * 
     */
    protected function determine_auth_level() {

        if (0 < strlen($this->access_token)) {

            $this->auth_level = NF_CapsuleCRM_Constants::AUTHLEVEL_OPENAUTH;
            $this->set_authlevel_oauth_status();
            return;
        }

        if (0 < strlen($this->personal_access_token)) {

            $this->auth_level = NF_CapsuleCRM_Constants::AUTHLEVEL_PAT;
            $this->set_authlevel_pat_status();
            return;
        }

        if (0 < strlen($this->authtoken) && 0 < strlen($this->subdomain)) {

            $this->auth_level = NF_CapsuleCRM_Constants::AUTHLEVEL_1;
            $this->set_authlevel_1_status();
            return;
        }

        $this->auth_level = NF_CapsuleCRM_Constants::AUTHLEVEL_UNAUTH;
        $this->set_unauth_error_status();
    }

    /**
     * Requests access token from Token class
     */
    protected function get_access_token() {

        $obj_str = NF_CapsuleCRM_Constants::TOKEN_CLASS;
        //TODO: replace with obj_str after PHP 5.2 support is dropped
        $this->access_token = CapsuleTokens::generate_access_token();
    }

    /**
     * Returns all stored required credentials from ninja_forms_settings 
     * 
     * Uses native WP methods only in case NF not loaded
     * 
     * @return array
     */
    protected function get_stored_credentials() {

        $this->personal_access_token = Ninja_Forms()->get_setting(NF_CapsuleCRM_Constants::PERSONAL_ACCESS_TOKEN);

        $this->authtoken = Ninja_Forms()->get_setting(NF_CapsuleCRM_Constants::AUTHTOKEN);

        $temp = Ninja_Forms()->get_setting(NF_CapsuleCRM_Constants::SUBDOMAIN);

        $this->subdomain = str_replace('.capsulecrm.com', '', $temp);
    }

    /**
     * Set the AuthLevel 2 message
     */
    protected function set_authlevel_oauth_status() {

        $this->status = __('Using Capsule Open Auth for API V2', 'ninja-forms-capsule-crm');
    }

    /**
     * Set the AuthLevel 2 message
     */
    protected function set_authlevel_pat_status() {

        $this->status = __('Using Capsule Personal Auth Token for API V2', 'ninja-forms-capsule-crm');
    }

    /**
     * Set the AuthLevel 1 message
     */
    protected function set_authlevel_1_status() {

        $this->status = __('Using Capsule Auth Token and Subdomain for API V1', 'ninja-forms-capsule-crm');
    }

    /**
     * Set the error message for missing authorization
     */
    protected function set_unauth_error_status() {

        $this->status = NF_CapsuleCRM_Constants::ACTION_REQUIRED;
        $this->status .= __('The authorization keys are missing/incomplete.  Please check your Ninja Forms Capsule settings', 'ninja-forms-capsule-crm');
    }

    /**
     * Returns true if an auth level has been determined
     * 
     * True if OAuth, PAT, or suthtoken/subdomain are present
     * 
     * @return boolean
     */
    public function is_authorized() {

        if (NF_CapsuleCRM_Constants::AUTHLEVEL_UNAUTH != $this->auth_level) {

            return true;
        } else {

            return false;
        }
    }

    /**
     * Returns the error message of the Auth Object
     * @return string
     */
    public function status() {

        return $this->status;
    }

    /**
     * Returns the auth level determined by highest credentials extracted
     * @return string
     */
    public function auth_level() {

        return $this->auth_level;
    }

    /**
     * Returns the authorization parameter based on auth level
     * 
     * @return string
     */
    public function authorization_header() {

        $authorization = '';

        switch ($this->auth_level) {

            case NF_CapsuleCRM_Constants::AUTHLEVEL_1:

                $authorization = 'Basic ' . base64_encode($this->authtoken . ': X');

                break;

            case NF_CapsuleCRM_Constants::AUTHLEVEL_PAT:

                $authorization = 'Bearer ' . $this->personal_access_token;

                break;

            case NF_CapsuleCRM_Constants::AUTHLEVEL_OPENAUTH:

                $authorization = 'Bearer ' . $this->access_token;

                break;
        }

        return $authorization;
    }

    /**
     * Returns the URL for requests based on the auth level
     * 
     * @return string
     */
    public function post_url() {

        $url = '';
        switch ($this->auth_level) {

            case NF_CapsuleCRM_Constants::AUTHLEVEL_1:

                $url = 'https://' . $this->subdomain . '.capsulecrm.com';

                break;

            case NF_CapsuleCRM_Constants::AUTHLEVEL_PAT:
            case NF_CapsuleCRM_Constants::AUTHLEVEL_OPENAUTH:

                $url = 'https://api.capsulecrm.com';

                break;
        }

        return $url;
    }

    /**
     * Returns markup for intelligent help migrating to Oauth
     * @return string
     */
    public static function markup_authorization_help() {

        $html = ''; // initialize
        $context_array = self::determine_help_context();

        if (in_array('auth_code_available', $context_array)) {
            // has authtoken or authcode available to generate refresh
            $html .= __('You have an authorization code that says you approve your Ninja Forms to communicate.') . "<br />";
            $html .= NF_CapsuleCRM_Constants::HELPFUL_HINT;
            $html .= __('Please click the button to generate a refresh token that enables us to continue communicating with Capsule', 'ninja_forms_capsule_crm') . "<br /><br />";

            $html .= self::build_refresh_token_link().'<br />';
        } elseif (in_array('refresh_token_available', $context_array)) {

            // already has refresh token
            $html .= __('You already have refresh token so you should be able to connect to your Capsule account.', 'ninja_forms_capsule_crm') . "<br />";

            $html .= __('If you are having difficulties, you can always start over by clicking the button.', 'ninja_forms_capsule_crm') . "<br /><br />";

            $html .= self::build_auth_code_link('RE-start authorization');
        } else {

            // no refresh token and no authtoken - new installation
            $html .= NF_CapsuleCRM_Constants::HELPFUL_HINT;
            $html .= __('To connect to Capsule, generate an authorization code by clicking on this link', 'ninja_forms_capsule_crm') . "<br />";
            $html .= self::build_auth_code_link();
        }

        $html .= "<br />";

        $html .= "<br />";

        return $html;
    }

    /**
     *  Returns markup for Authorization Code link
     * 
     * @param string $button_text Text for button
     * @return string HTML markup
     */
    protected static function build_auth_code_link($button_text = 'Get authorization code') {

        $obj_str = NF_CapsuleCRM_Constants::TOKEN_CLASS;
        // TODO: replace with obj_str after support for PHP 5.2 is dropped
        $url = CapsuleTokens::generate_auth_code_url();

        $link = '<a class="button " target = "_blank" href="' . $url . '">' . $button_text . '</a>';

        return $link;
    }

    /**
     *  Returns markup for Generate Refresh Token link
     * 
     * @param string $button_text Text for button
     * @return string HTML markup
     */
    protected static function build_refresh_token_link($button_text = 'Get Refresh Token') {

        $query_var = NF_CapsuleCRM_Constants::LISTENER_URL;
        $query_param = NF_CapsuleCRM_Constants::REFRESH_TOKEN_LISTENER;

        $url = home_url().'?'.$query_var.'='.$query_param;

        $nonce_url = wp_nonce_url($url, $query_param);
        
        $link = '<a class="button " target = "_blank" href="' . $nonce_url . '">' . $button_text . '</a>';

        return $link;
    }

    /**
     * Determine recommended authorization method
     * 
     * Checks the stored credentials and determine what is the best way to get
     * fully authorized
     */
    protected static function determine_help_context() {

        $context_array = array();

        $obj_str = NF_CapsuleCRM_Constants::TOKEN_CLASS;
        //TODO: replace with obj_str after support for PHP 5.2 is dropped
        $tokens = CapsuleTokens::retrieve_stored_tokens();

        if (0 < strlen($tokens['refresh_token'])) {

            $context_array[] = 'refresh_token_available';
        }

        $credentials = $obj_str::retrieve_stored_credentials();

        if (0 < strlen($credentials[NF_CapsuleCRM_Constants::AUTH_CODE])) {

            $context_array[] = 'auth_code_available';
        }

        if (0 < strlen($credentials[NF_CapsuleCRM_Constants::AUTHTOKEN])) {

            $context_array[] = 'authtoken_available';
        }

        return $context_array;
    }


}
