<?php

/**
 * Builds the nested array per CRM's V1 API and ID array of existing entries
 *
 * When finalized, structured request is an indexed array consisting of
 * associative arrays keyed on module
 */
class CapsuleV1StructuredRequest {

    /**
     * Incoming indexed array of submission data
     * 
     * keys are module, field_map, and form_field
     * @var array
     */
    protected $request_array;

    /**
     * Module configuration
     * 
     * Sets order for requests;
     * 
     * @var array
     */
    protected $module_config;

    /**
     * Authorization level determining which API version is used
     * @var string
     */
    protected $authlevel;
    
    /**
     * Configured field map array, optionally filtered
     * @var array
     */
    protected $field_map_lookup;

    /**
     * Request restructured in format optimized for the CRM's API
     * 
     * @var array
     */
    protected $structured_request = array();

    /**
     * IDs used for updated or linking previously created entries
     * 
     * have their IDs inserted manually into this array.
     * 
     * @var array
     */
    protected $update_id_array;

    /**
     * 
     * @param array $request_array Request array ready for iterating 
     * @param array $module_config Module configuration
     */
    public function __construct($request_array, $module_config, $authlevel) {

        $this->request_array = $request_array;

        $this->module_config = $module_config;

        $this->authlevel = $authlevel;
        
        $this->field_map_lookup = NF_CapsuleCRM()->get_field_map_lookup();

        $this->iterate_request_array();

        $this->finalize_structured_request();
    }

    /**
     * Cycle through entries in the request array and build structured request
     * 
     * Overwrites map instructions with V1 values
     */
    protected function iterate_request_array() {

        foreach ($this->request_array as $nested_field) {

            $nested_field['map_instructions'] = $this->field_map_lookup[$nested_field['field_map']]['map_instructions_v1'];

            $this->add_field_to_request($nested_field);
        }
    }

    /**
     * Adds a given entry to the structured request
     * @param array $nested_field
     */
    public function add_field_to_request($nested_field) {

        if ('none' == $nested_field['field_map'] || '' == $nested_field['form_field']) {
            return;
        } // no mapping shall be done this day


        $map_args = $this->validate_field($nested_field);
        $user_value = $map_args['form_field'];

        $mapped_field_array = explode('.', $map_args['map_instructions']);


        $mapped_field = $mapped_field_array[0]; // set the mapped field
        $field_grouping = $mapped_field_array[1];


        if (isset($mapped_field_array[4])) { //  module is defined
            $module = $mapped_field_array[4];
        } else {
            $module = 'person'; //default to person
        }

        switch ($field_grouping) { // handle groupings
            case 'single-with-type': // entries like phone number and email can be assigned a type

                $wrap = $mapped_field_array[2]; // pull in the wrap
                $major_grouping = $mapped_field_array[3]; // pull in the major_grouping 


                $temp_array = array(
                    $mapped_field => $user_value
                );

                if ('email' == $wrap) {// determine which types are allowed for this mapped field
                    $allowed_types = array('Home', 'Work');
                } else {

                    $allowed_types = array('Home', 'Work', 'Mobile', 'Fax', 'Direct');
                    // default to phone options
                }

                if (in_array($map_args['entry_type'], $allowed_types)) {
                    // an acceptable "type" was selected so add it to the request array

                    $temp_array['type'] = $map_args['entry_type'];
                }

                $this->structured_request[$module][$major_grouping][$wrap][] = $temp_array;
                break;

            case 'website': //website entries include social media and repositories

                $wrap = $mapped_field_array[2]; // pull in the wrap
                $major_grouping = $mapped_field_array[3]; // pull in the major_grouping

                $temp_array = array(
                    'webService' => $mapped_field,
                    'webAddress' => $user_value
                );
                $allowed_types = array('Home', 'Work'); // the allowed types for this mapped field

                if (in_array($map_args['entry_type'], $allowed_types)) {
                    // an acceptable "type" was selected so add it to the array

                    $temp_array['type'] = $map_args['entry_type'];
                }

                $this->structured_request[$module][$major_grouping][$wrap][] = $temp_array;
                break;

            case 'address': // gets consolidated based on entry_type and then finalized after all fields have been processed

                $allowed_types = array('Home', 'Office', 'Postal', 'none'); // the allowed types for this mapped field

                if (in_array($map_args['entry_type'], $allowed_types)) { // an acceptable "type" was selected so add it
                    $this->structured_request[$module]['contacts']['address'][$map_args['entry_type']][$mapped_field] = $user_value;
                }

                break;

            case 'custom':
                /*
                 * Use entry type on custom fields when using non-text custom fields
                 */
                switch ($map_args['entry_type']) {

                    case 'date':
                        $key = 'date';
                        $user_value = $this->format_date_for_capsule($user_value);
                        break;

                    case 'boolean':
                        $key = 'boolean';
                        $user_value = $this->format_boolean_for_capsule($user_value);
                        break;

                    default:
                        $key = 'text';
                }

                $this->structured_request[$module]['customField'][] = array(
                    'label' => $map_args['custom_field'],
                    $key => $user_value
                );

                break;

            case 'majorgroup':
                $major_grouping = $mapped_field_array[3]; // pull in the major_grouping

                $this->structured_request[$module][$major_grouping][$mapped_field] = $user_value;

                break;

            case 'multiples': // several entries in an indexed array

                $this->structured_request[$module][$mapped_field][] = $user_value;

                break;

            case 'standalone':

            default:

                $this->structured_request[$module][$mapped_field] = $user_value;
        }
        // end switch mapped field grouping
    }

    /**
     * Cycles validation functions spec'd in FieldMapLookup plus legacy fx
     * 
     * Legacy (pre-3) functions are converting checked/unchecked to string
     * true/false and an array-to-string implode
     * 
     * @param array $single_field_map
     * @return array
     */
    protected function validate_field($single_field_map) {

        $value_in = $single_field_map['form_field'];

        $validation_object_name = NF_CapsuleCRM_Constants::VALIDATION_CLASS;

        if ($single_field_map['validation_functions']) {

            $temp = $value_in;

            foreach ($single_field_map['validation_functions'] as $function_call) {

                if (!method_exists($validation_object_name, $function_call)) {

                    continue;
                }

                $temp = call_user_func(array($validation_object_name, $function_call), $temp);
            }

            $validated = $temp;
        } else {
            $validated = $value_in;
        }

        $bool_check = call_user_func(array($validation_object_name, 'convert_checkbox_to_true_false_string'), $validated);


        // Array conversion removed b/c 3.0 doesn't send arrays for multi-select

        $escaped = esc_attr($bool_check);

        //TODO: evaluate if esc_attr is best filtering
        $single_field_map['form_field'] = $escaped;

        return $single_field_map;
    }

    /**
     * Returns the structured request
     * 
     * @return array
     */
    public function get_structured_request() {

        return $this->structured_request;
    }

    /**
     * Finishes structuring the data array
     * Items like addresses can only be structured after all the fields have been processed
     * 
     * 
     */
    public function finalize_structured_request() {

        $this->set_address_type();
        $this->validate_module_required_fields();
        $this->reorder_structured_request();
    }

    /**
     * Remove the temporary keys and convert associative array to indexed
     */
    protected function set_address_type() {

        // organisation
        if (isset($this->structured_request['organisation']['contacts']['address'])) {

            foreach ($this->structured_request['organisation']['contacts']['address'] as $address_type => $address_array) {

                if ('none' != $address_array) {
                    // add type of address if it isn't 'none'
                    $address_array['type'] = $address_type;
                }
                unset($this->structured_request['organisation']['contacts']['address'][$address_type]);

                $this->structured_request['organisation']['contacts']['address'][] = $address_array;
            }
        }

        // person
        if (isset($this->structured_request['person']['contacts']['address'])) {

            foreach ($this->structured_request['person']['contacts']['address'] as $address_type => $address_array) {

                $address_array['type'] = $address_type;

                unset($this->structured_request['person']['contacts']['address'][$address_type]);

                $this->structured_request['person']['contacts']['address'][] = $address_array;
            }
        }
    }

    /**
     * Add req'd missing fields added to modules NOT being updated
     */
    protected function validate_module_required_fields() {

        foreach ($this->module_config as $module => $configuration) {

            $defaults = $configuration['required_fields'];

            $merged = array_merge($defaults, $this->structured_request[$module]);

            $this->structured_request[$module] = $merged;
        }
    }

    /**
     * Reorders modules to config order and adds indexed key
     * 
     * Cycles through tags and makes each one its own request for API V1
     * If in V2, tags will have been already structured in the person module
     */
    protected function reorder_structured_request() {

        $temp_array = array();

        foreach (array_keys($this->module_config) as $module) {

            if (!isset($this->structured_request[$module])) {
                
                continue;
            }
            
            if('tags'!=$module){
                
                $temp_array[] = array($module => $this->structured_request[$module]);
                
                continue;
            }
                
            foreach($this->structured_request['tags']['name'] as $tag){
                
                $temp_array[]=array('tags'=>$tag);
            }
        }

        $this->structured_request = $temp_array;
    }

    /**
     * Given a form field and module, sets the ID  updating
     * 
     * @param string $form_field
     * @param string $module
     */
    public function add_update_id($form_field, $module) {

        $this->update_id_array[$module]['Id'] = $form_field;
    }

    /**
     * Returns the Update Record ID array
     * @return array
     */
    public function get_update_id_array() {

        if (empty($this->update_id_array)) {

            return array();
        } else {

            return $this->update_id_array;
        }
    }

    /**
     * Converts incoming string to Capsule's required date format
     * 
     * If the incoming string cannot be converted to a date, it defaults
     * to Unix start time 1970
     * 
     * @param string $date_string
     * @return string Formatted date string for Capsule
     */
    protected function format_date_for_capsule($date_string) {

        $old_date_timestamp = strtotime($date_string);

        $formatted_date = date('Y-m-d\TH:i:s\Z', $old_date_timestamp);

        return $formatted_date;
    }

    /**
     * Converts incoming value into either 'true' or 'false
     * 
     * @param string $incoming_user_value
     * @return string 'true' or 'false' for Capsule's boolean custom field
     */
    protected function format_boolean_for_capsule($incoming_user_value) {

        switch ($incoming_user_value) {

            case 'checked':
            case 'CHECKED':
            case 'true':
                $formatted_boolean = 'true';
                break;

            case 'unchecked':
            case 'UNCHECKED':
            case 'false':
                $formatted_boolean = 'false';
                break;

            default:
                $formatted_boolean = 'false';
        }

        return $formatted_boolean;
    }

}
