<?php

/**
 * Builds the nested array per CRM's V2 API and ID array of existing entries
 *
 * When finalized, structured request is an indexed array consisting of
 * associative arrays keyed on module
 */
class CapsuleStructuredRequest {

    /**
     * Incoming indexed array of submission data
     * 
     * keys are module, field_map, and form_field
     * @var array
     */
    protected $request_array;

    /**
     * Module configuration
     * 
     * Sets order for requests;
     * 
     * @var array
     */
    protected $module_config;

    /**
     * Array of entry types that are valid for a given field
     */
    protected $entry_type_validation;

    /**
     * Authorization level determining which API version is used
     * @var string
     */
    protected $authlevel;

    /**
     * Configured field map array, optionally filtered
     * @var array
     */
    protected $field_map_lookup;

    /**
     * Request restructured in format optimized for the CRM's API
     * 
     * @var array
     */
    protected $structured_request = array();

    /**
     * IDs used for updated or linking previously created entries
     * 
     * have their IDs inserted manually into this array.
     * 
     * @var array
     */
    protected $update_id_array;

    /**
     * 
     * @param array $request_array Request array ready for iterating 
     * @param array $module_config Module configuration
     */
    public function __construct($request_array, $module_config, $authlevel) {

        $this->request_array = $request_array;

        $this->module_config = $module_config;

        $this->authlevel = $authlevel;

        $this->field_map_lookup = NF_CapsuleCRM()->get_field_map_lookup();

        $this->entry_type_validation = NF_CapsuleCRM::config('EntryTypeValidation');

        $this->iterate_request_array();

        $this->finalize_structured_request();
    }

    /**
     * Cycle through entries in the request array to build structured request
     * 
     * Calls validation functions on each field first
     */
    protected function iterate_request_array() {

        foreach ($this->request_array as $nested_field) {

            if ('none' == $nested_field['field_map'] || '' == $nested_field['form_field']) {
                continue;
            } // no mapping shall be done this day

            $field_args = $this->validate_field($nested_field);

            $this->add_field_to_request($field_args);
        }
    }

    /**
     * Adds each field to structured request after validating entry type
     * 
     * Reconstructs field args into single keyed array for readability
     * 
     * Uses callable function for simpler structure
     * 
     * @param array $field_args
     */
    protected function add_field_to_request($field_args) {

        $exploded = explode('.', $field_args['map_instructions']);
        
        $call_args = array(
            'module' => $exploded[0],
            'crm_field' => $exploded[1],
            'construct' => $exploded[2],
            'user_value' => $field_args['form_field'],
            'custom_field' => $field_args['custom_field'],
        );
        
        if(isset($exploded[3])){
            
            $call_args['nested']= $exploded[3];
        }
        
        $call_args['entry_type'] = $this->validate_entry_type($field_args['entry_type'], $call_args['construct']);

        if (!method_exists($this, 'add_' . $call_args['construct'])) {

            return;
        }

        call_user_func(array($this, 'add_' . $call_args['construct']), $call_args);
    }

    /**
     * Add single keyed value to structured request
     * 
     * @param array $call_args
     * @see CapsuleStructuredRequest->add_field_to_request() for keys
     */
    protected function add_top_level($call_args) {

        extract($call_args);

        $this->structured_request[$module][$crm_field] = $user_value;
    }

    /**
     * Adds address field to structured request
     * 
     * @param array $call_args    
     * @see CapsuleStructuredRequest->add_field_to_request() for keys
     */
    protected function add_address($call_args) {

        extract($call_args);

        $this->structured_request[$module]['addresses'][$entry_type][$crm_field] = $user_value;
    }

    /**
     * Adds email to structured request
     * 
     * @param array $call_args
     * @see CapsuleStructuredRequest->add_field_to_request() for keys
     */
    protected function add_email($call_args) {

        extract($call_args);
        $array = array(
            'type' => $entry_type,
            'address' => $user_value
        );

        $this->structured_request[$module]['emailAddresses'][] = $array;
    }

    /**
     * Adds phone number to structured request
     * 
     * @param array $call_args
     * @see CapsuleStructuredRequest->add_field_to_request() for keys
     */
    protected function add_phone($call_args) {

        extract($call_args);

        $array = array(
            'type' => $entry_type,
            'number' => $user_value
        );

        $this->structured_request[$module]['phoneNumbers'][] = $array;
    }

    /**
     * Adds website to structured request
     * 
     * @param array $call_args
     * @see CapsuleStructuredRequest->add_field_to_request() for keys
     */
    protected function add_website($call_args) {

        extract($call_args);

        $array = array(
            'service' => $crm_field,
            'type' => $entry_type,
            'address' => $user_value
        );

        $this->structured_request[$module]['websites'][] = $array;
    }

    /**
     * Adds tag to structured request
     * 
     * @param array $call_args
     * @see CapsuleStructuredRequest->add_field_to_request() for keys
     */
    protected function add_tag($call_args) {

        extract($call_args);

        $array = array(
            'name' => $user_value
        );
        $this->structured_request[$module]['tags'][] = $array;
    }

    /**
     * Adds tag to structured request
     * 
     * @param array $call_args
     * @see CapsuleStructuredRequest->add_field_to_request() for keys
     */
    protected function add_nested_value($call_args) {

        extract($call_args);

        $this->structured_request[$module][$crm_field][$nested] = $user_value;
    }

    /**
     * Adds custom field to structured request
     * 
     * @param array $call_args
     * @see CapsuleStructuredRequest->add_field_to_request() for keys
     */
    protected function add_custom($call_args){
   
        extract($call_args);

        $array = array(
            'value' => $user_value,
            'definition'=>array(
                'name'=>$custom_field
            ),
        );
        
        $this->structured_request[$module]['fields'][] = $array;
    }
    
    /**
     * Given a crm_field and entry type, ensures type is valid for field
     * 
     * If type is not valid, replace it with first valid available
     */
    protected function validate_entry_type($entry_type, $construct) {

        if (!in_array($entry_type, $this->entry_type_validation[$construct])) {

            $entry_type = $this->entry_type_validation[$construct][0];
        }

        return $entry_type;
    }

    /**
     * Cycles validation functions spec'd in FieldMapLookup plus legacy fx
     * 
     * Legacy (pre-3) functions are converting checked/unchecked to string
     * true/false and an array-to-string implode
     * 
     * @param array $single_field_map
     * @return array
     */
    protected function validate_field($single_field_map) {

        $value_in = $single_field_map['form_field'];

        $validation_object_name = NF_CapsuleCRM_Constants::VALIDATION_CLASS;

        if ($single_field_map['validation_functions']) {

            $temp = $value_in;

            foreach ($single_field_map['validation_functions'] as $function_call) {

                if (!method_exists($validation_object_name, $function_call)) {

                    continue;
                }

                $temp = call_user_func(array($validation_object_name, $function_call), $temp);
            }

            $validated = $temp;
        } else {
            $validated = $value_in;
        }

        $bool_check = call_user_func(array($validation_object_name, 'convert_checkbox_to_true_false_string'), $validated);

        $single_field_map['form_field'] = $bool_check;

        // TODO: watch for rejections after removed esc_attr
        
        return $single_field_map;
    }

    /**
     * Finishes structuring the data array
     * Items like addresses can only be structured after all the fields have been processed
     */
    protected function finalize_structured_request() {

        $this->set_address_type();
        
//        $this->link_person_to_org();
        
        $this->validate_module_required_fields();
        
        $this->reorder_structured_request();
    }

    /**
     * Person links on organisation name instead of id
     * 
     * If duplicate org name detected, org is not created.  Link on org name
     * ensures person is linked to existing org in CRM
     */
    protected function link_person_to_org() {

        if (isset($this->structured_request['organisation']['name']) &&
                isset($this->structured_request['person'])) {

            $this->structured_request['person']['organisation']['name'] = $this->structured_request['organisation']['name'];
        }
    }

    /**
     * Remove the temporary keys and convert associative array to indexed
     */
    protected function set_address_type() {

        $modules = array( 'organisation', 'person' );

        foreach ($modules as $module) {
            
            if (!isset($this->structured_request[$module]['addresses'])) {

                continue;
            }
            
            foreach ($this->structured_request[$module]['addresses'] as $address_type => $address_array) {

                if ('none' != $address_type) {
                    // add type of address if it isn't 'none'
                    $address_array['type'] = $address_type;
                }
                
                unset($this->structured_request[$module]['addresses'][$address_type]);

                $this->structured_request[$module]['addresses'][] = $address_array;
            }
        }
    }

    /**
     * Add req'd missing fields added to modules NOT being updated
     */
    protected function validate_module_required_fields() {

        foreach ($this->module_config as $module => $configuration) {

            $defaults = $configuration['required_fields'];

            $merged = array_merge($defaults, $this->structured_request[$module]);

            $this->structured_request[$module] = $merged;
        }
    }

    /**
     * Reorders modules to config order and adds indexed key
     * 
     * Cycles through tags and makes each one its own request for API V1
     * If in V2, tags will have been already structured in the person module
     */
    protected function reorder_structured_request() {

        $temp_array = array();

        foreach (array_keys($this->module_config) as $module) {

            if (!isset($this->structured_request[$module])) {

                continue;
            }

            if ('tags' != $module) {

                $temp_array[] = array($module => $this->structured_request[$module]);

                continue;
            }
        }

        $this->structured_request = $temp_array;
    }

    /**
     * Given a form field and module, sets the ID  updating
     * 
     * @param string $form_field
     * @param string $module
     */
    public function add_update_id($form_field, $module) {

        $this->update_id_array[$module]['Id'] = $form_field;
    }

    /**
     * Returns the Update Record ID array
     * @return array
     */
    public function get_update_id_array() {

        if (empty($this->update_id_array)) {

            return array();
        } else {

            return $this->update_id_array;
        }
    }
    
    /**
     * Returns the structured request
     * 
     * @return array
     */
    public function get_structured_request() {

        return $this->structured_request;
    }

}
