<?php

/**
 * Builds the request array from the field map data 
 * 
 * The request_array puts the submission values in an indexed array keyed
 * with the module, CRM field, and submission value During this iteration, any
 * data formatting based on the CRM field is called and performed
 * 
 */
class CapsuleRequestArray {

    /**
     *  Field map data array
     * 
     * This is pulled in from the $action_settings[ {key} ] and be iterated
     * to extract the form submission and the field mapping instructions
     * 
     * @var array
     */
    protected $field_map_data;

    /**
     * Array of submission field names for which to extract data
     * 
     * Co-ordinated with the ActionFieldMap and the FieldsToExtract configs
     * 
     * @var array
     */
    protected $fields_to_extract;

    /**
     * Array of each field of submitted data with mapping instructions
     * 
     * @var array
     */
    protected $request_array;

    /**
     * Array of Email values mapped for person
     * 
     * Used when searching for duplicates by email
     * 
     * @var array
     */
    protected $email_array = array();
    
    /**
     * Array to exchange each field map with detailed mapping instructions
     * 
     * $key is a READABLE and unique value that is sent with the form submission
     * 
     * $label is the i10n descriptive version of the field map, used to help the
     * form designer select the desired field map location
     * 
     */
    protected $field_map_lookup;

    public function __construct($field_map_data) {

        $this->field_map_data = $field_map_data;

        $this->fields_to_extract = NF_CapsuleCRM::config('FieldsToExtract');

        $this->field_map_lookup = NF_CapsuleCRM()->get_field_map_lookup();

        $this->request_array = array();  // initialize

        $this->iterate_field_map_data();
    }

    /**
     * Iterates each individual entry in the field map array to build the
     * request array.  Request array is nested array; first level is indexed
     * and second level is associative with each field to extract
     */
    protected function iterate_field_map_data() {

        foreach ($this->field_map_data as $unprocessed_single_field_map) {

            $single_field_map = array();

            foreach ($this->fields_to_extract as $field) {

                if (isset($unprocessed_single_field_map[$field])) {

                    $single_field_map[$field] = $unprocessed_single_field_map[$field];
                }
            }

            // if no form submission value, continue to next field
            if (!(0 < strlen($single_field_map['form_field']))) {

                continue;
            }

            $this->request_array[] = $this->add_lookup_parameters($single_field_map);
            
            $this->extract_emails($single_field_map);
        }
    }

    /**
     * Adds parameters from field lookup
     * 
     * If the entry type is specifying a format, add its validation function
     * 
     * @param array $single_field_map
     * @return array
     */
    protected function add_lookup_parameters($single_field_map) {

        $lookup_key = $single_field_map['field_map'];
        
        $single_field_map['map_instructions'] = $this->field_map_lookup[$lookup_key]['map_instructions'];
        
        if (isset($this->field_map_lookup[$lookup_key]['validation_functions'])) {

            $single_field_map['validation_functions'] = $this->field_map_lookup[$lookup_key]['validation_functions'];
        }else{
            
            $single_field_map['validation_functions'] =array();
        }

        switch($single_field_map['entry_type']){
            
            case 'date':
                $single_field_map['validation_functions'][]='format_date';
                break;
            
            case 'boolean':
                $single_field_map['validation_functions'][]='force_boolean';
                break;
        }
        
        return $single_field_map;
    }

    /**
     * Builds array of emails in request
     * 
     * Used when checking for duplicates
     * 
     * @param array $single_field_map
     */
    protected function extract_emails($single_field_map) {

        if (isset($single_field_map['field_map']) &&
                'Email' == $single_field_map['field_map']) {

            $this->email_array[] = $single_field_map['form_field'];
            return;
        }
    }

    /**
     * Request array for building
     * @return array
     */
    public function get_request_array() {

        return $this->request_array;
    }

    /**
     * Email array for Person module
     * @return array
     */
    public function get_email_array(){
        
        return $this->email_array;
    }
}
