<?php

/**
 * Given action_settings for submission, processes the form for the CRM
 *
 * @author stuartlb3
 */
class ProcessAddToCapsuleAction {

    /**
     * Communication Log object to collect and organize comm log data
     * 
     * @var object
     */
    protected $comm_log;

    /**
     * Authorization object
     * 
     * @var object
     */
    protected $auth;

    /**
     * Configured module array
     * 
     * @var array
     */
    protected $module_config;

    /**
     * Setting for duplicate check
     * 
     * @var boolean
     */
    protected $duplicate_check;

    /**
     * Array of each field of submitted data with mapping instructions
     * 
     * @var array
     */
    protected $request_array;

    /**
     * Request restructured in format optimized for the CRM's API
     * 
     * @var array
     */
    protected $structured_request;

    /**
     * Create Data Object
     * 
     * @var object
     */
    protected $create_obj;

    /**
     * Array of Email values mapped for person
     * 
     * Used when searching for duplicates by email
     * 
     * @var array
     */
    protected $email_array;

    /**
     * IDs used for updating or linking previously created entries
     * 
     * IDs are inserted manually into this array from form submission
     * 
     * @var array
     */
    protected $update_id_array;

    /**
     * IDs created during request, used to link downstream requests
     * 
     * @var array
     */
    protected $new_record_ids;

    public function __construct($action_settings) {

        $this->authorize();

        if (!$this->auth->is_authorized()) {

            $this->comm_log->update(NF_CapsuleCRM_Constants::SUPPORT_DATA);
            return;
        }

        $this->load_config_vars();

        $this->extract_form_settings($action_settings);

        $this->build_request_arrays($action_settings);

        if (empty($this->request_array)) {

            $this->comm_log->update(NF_CapsuleCRM_Constants::SUPPORT_DATA);
            return;
        }

        $this->build_structured_request();

        $this->check_duplicate_emails();

        if (NF_CapsuleCRM_Constants::AUTHLEVEL_1 == $this->auth->auth_level()) {

            $obj_str = 'CapsuleV1Create';
        } else {

            $obj_str = NF_CapsuleCRM_Constants::CREATE_CLASS;
        }

        $this->create_obj = new $obj_str($this->auth);


        $this->iterate_structured_request();

        $this->comm_log->update(NF_CapsuleCRM_Constants::SUPPORT_DATA);

        return;
    }

    /**
     * Instantiate authorization object
     */
    protected function authorize() {

        $str_auth_obj = NF_CapsuleCRM_Constants::AUTHORIZATION_CLASS;
        // TODO: replace with str_auth_obj after PHP 5.2 support is dropped
        $this->auth = new CapsuleAuth;

        $str_comm_obj = NF_CapsuleCRM_Constants::COMM_LOG_CLASS;
        // TODO: replace with str_auth_obj after PHP 5.2 support is dropped
        $this->comm_log = new CapsuleCommLog;

        $msg = $this->auth->status();
        $this->comm_log->log_entry(NF_CapsuleCRM_Constants::COMM_STATUS, 'Authorization', $msg);
    }

    /**
     * Extracts form settings
     * 
     * @param array $action_settings
     */
    protected function extract_form_settings($action_settings) {

        $this->duplicate_check = $action_settings[NF_CapsuleCRM_Constants::CHECK_FOR_DUPLICATES_KEY];
    }

    /**
     * Iterates option repeater to build request array and email array
     *
     * The request_array puts the submission values in an indexed array keyed
     * with the module, CRM field, and submission value During this iteration, any
     * data formatting based on the CRM field is called and performed
     *
     * The email array is all the emails added to the Person module
     * 
     * @param array $action_settings
     */
    protected function build_request_arrays($action_settings) {

        // isolate the action_data for the submission
        $field_map_data = $action_settings[NF_CapsuleCRM_Constants::FIELD_MAP_REPEATER_KEY];

        $obj_str = NF_CapsuleCRM_Constants::REQUEST_ARRAY_CLASS;
        //TODO: replace with obj_str after PHP 5.2 is dropped
        
        $request_object = new CapsuleRequestArray($field_map_data);

        $this->request_array = $request_object->get_request_array();

        $this->email_array = $request_object->get_email_array();

        // Request array covers all modules, so re-use key as universal module
        $this->comm_log->log_entry(
                NF_CapsuleCRM_Constants::REQUEST_ARRAY, NF_CapsuleCRM_Constants::REQUEST_ARRAY, $this->request_array
        );
    }

    /**
     * Builds the nested array per CRM's API and ID array of existing entries
     *
     * Includes the update_id_array, which is keyed by module
     */
    protected function build_structured_request() {

        if (NF_CapsuleCRM_Constants::AUTHLEVEL_1 == $this->auth->auth_level()) {

            $obj_str = 'CapsuleV1StructuredRequest';
            //TODO: replace with obj_str after support for PHP 5.2 is dropped
            $structured_request_object = new CapsuleV1StructuredRequest($this->request_array, $this->module_config, $this->auth->auth_level());
        } else {

            $obj_str = NF_CapsuleCRM_Constants::STRUCTURED_REQUEST_CLASS;
            //TODO: replace with obj_str after support for PHP 5.2 is dropped
            $structured_request_object = new CapsuleStructuredRequest($this->request_array, $this->module_config, $this->auth->auth_level());
        }


        $structured_request_object = new $obj_str($this->request_array, $this->module_config, $this->auth->auth_level());

        $this->structured_request = $structured_request_object->get_structured_request();

        // Store in comm log using key as universal module
        $this->comm_log->log_entry(
                NF_CapsuleCRM_Constants::STRUCTURED_REQUEST, NF_CapsuleCRM_Constants::STRUCTURED_REQUEST, $this->structured_request
        );
    }

    /**
     * Checks for duplicates by email if form setting demands it
     */
    protected function check_duplicate_emails() {

        if (!$this->duplicate_check || empty($this->email_array)) {

            return;
        }

        $obj_str_retrieve = NF_CapsuleCRM_Constants::RETRIEVE_CLASS;
        $obj_str_response = NF_CapsuleCRM_Constants::RESPONSE_CLASS;

        $count = 0;

        $retrieve_obj = new $obj_str_retrieve($this->auth);

        foreach ($this->email_array as $email) {

            $raw_response = $retrieve_obj->search_by_email($email);

            $response_obj = new $obj_str_response($raw_response, 'RecordCount');

            $processed_response = $response_obj->get_processed_response();

            $count = $count + $processed_response['record_count'];

            if ($processed_response['error']) {
                $this->comm_log->log_entry(NF_CapsuleCRM_Constants::COMM_STATUS, 'Duplicate Check for: ' . $email, $processed_response['error_message']);
            }
        }

        if (0 < $count) {
            $this->add_duplicate_email_task();
        }
    }

    /**
     * Appends task to structured array to search and merge duplicates
     */
    protected function add_duplicate_email_task() {

        $task_array = NF_CapsuleCRM::config('DuplicateEmailTask');

        $dueDate = date(NF_CapsuleCRM_Constants::DATE_FORMAT, time());

        $task_array['dueOn'] = $dueDate;
        $task_array['dueDate'] = $dueDate;

        $this->structured_request[] = array('task' => $task_array);
    }

    /**
     * Instantiates Create object, iterates structured request to make requests
     */
    protected function iterate_structured_request() {


        $this->new_record_ids = array();

        foreach ($this->structured_request as $keyed_request) {

            $this->make_comm_request($keyed_request);
        }

        // Make the new IDs available for other uses     
        do_action('nfcapsulecrm_process_new_record_ids', $this->new_record_ids);
    }

    /**
     * Makes specific module request, calls response object, stores response
     * 
     * @param array $keyed_request
     */
    protected function make_comm_request($keyed_request) {

        $module = key($keyed_request);

        $raw_response = $this->create_obj->create_entry($keyed_request, $this->module_config, $this->new_record_ids);

        $obj_str_2 = NF_CapsuleCRM_Constants::RESPONSE_CLASS;

        $response_obj = new $obj_str_2($raw_response, 'Create', $module, $this->module_config);

        $processed_response = $response_obj->get_processed_response();

        $this->comm_log->log_entry(NF_CapsuleCRM_Constants::COMM_STATUS, $module, $processed_response['crm_message']);

        $this->comm_log->log_entry(NF_CapsuleCRM_Constants::FULL_RESPONSE, $module, $processed_response['response_body']);

        if (isset($processed_response['new_record_id'])) {

            $this->new_record_ids[$module] = $processed_response['new_record_id'];
        }
    }

    /**
     * Load configured variables
     */
    protected function load_config_vars() {

        $filters = NF_CapsuleCRM()->is_alt_enabled();

        if ($filters) {

            $this->module_config = apply_filters('nfcapsulecrm_filter_module_config', NF_CapsuleCRM::config('Modules'));
        } else {

            $this->module_config = NF_CapsuleCRM::config('Modules');
        }
    }

}
