<?php

/**
 * Handles the response from the CRM
 *
 */
class CapsuleHandleResponse {

    /**
     * Complete response from Zoho or WP (if wp_error)
     * @var mixed
     */
    protected $raw_response;

    /**
     * Module of the response
     * @var string
     */
    protected $module;

    /**
     * Context is the CRM-specific method call
     * 
     * @var string
     */
    protected $context;

    /**
     * Response after processing
     * @var array
     */
    protected $processed_response;

    /**
     * The response code from the CRM
     * @var integer
     */
    protected $response_code;

    /**
     * The response message from the CRM
     * @var integer
     */
    protected $response_message;

    /**
     * Extracted headers from the CRM
     * @var array
     */
    protected $headers;

    /**
     * Array of response body
     * @var array
     */
    protected $body;
    
    /**
     * Module configuration
     * @var array
     */
    protected $module_config;
    
    public function __construct($raw_response, $context, $module=array(), $module_config=array()) {

        $this->raw_response = $raw_response;
        
        $this->module = $module;

        $this->context = $context;
        
        $this->module_config = $module_config;

        // set default
        $this->processed_response = NF_CapsuleCRM::config('ResponseArray');

        $this->wp_error_check();

        if ($this->processed_response['wp_error']) {
            return;
        }

        $this->extract_headers();

        $this->response_code = $this->raw_response['response']['code'];
        $this->response_message = $this->raw_response['response']['message'];
        
        $this->body = json_decode($this->raw_response['body'], TRUE);

        $this->processed_response['response_body'] = serialize($this->raw_response);
        
        $this->crm_error_check();

        if ($this->processed_response['crm_error']) {
            return;
        }

        $this->process_records();
    }

    /**
     * Selects the processing method based on the context
     * 
     * The data structure could be different based on what request was made;
     * this switch ensures the data is handled appropriately per the request
     */
    protected function process_records() {

        switch ($this->context) {
            case 'Create':
                $this->extract_new_record_id();
                break;
            
            case 'RecordCount':
                $this->count_records_returned();
                break;
        }
    }

    /**
     * Extracts the ID from the response and sets CRM message
     */
    protected function extract_new_record_id() {

//        $strip = $this->module_config[$this->module]['strip_response'];
//        $id = str_replace($strip, '', strstr($this->headers['location'], $strip));

        $resource_base = 'api.capsulecrm.com/api/v2/';
        
        $resource_and_id = str_replace($resource_base, '', strstr($this->headers['location'],$resource_base)) ;
       
        $id = str_replace('/', '', strstr($resource_and_id,'/'));

        $message = NF_CapsuleCRM_Constants::SUCCESS;
        $message .= __('Sucessfully created at: ', 'ninja-forms-capsule-crm');
        $message .= $this->headers['date'];

        $this->processed_response['new_record_id'] = $id;
        $this->processed_response['crm_message'] = $message;
    }

    /**
     * Counts the number of records returned in a search
     * 
     * Stored in parameter 'record_count' 
     */
    protected function count_records_returned(){
        
        $body_obj = json_decode($this->raw_response['body']);
        
        $this->processed_response['record_count']= $body_obj->{'parties'}->{'@size'};
    }
            
    /**
     * Builds keyed array of header responses
     */
    protected function extract_headers(){

        if(isset($this->raw_response['headers']['location'])){
           $this->headers['location'] =  $this->raw_response['headers']['location']; 
        }
          if(isset($this->raw_response['headers']['date'])){
           $this->headers['date'] =  $this->raw_response['headers']['date']; 
        }      
        
        $this->processed_response['headers']=$this->headers;
    }
    
    /**
     * Checks for CRM error
     */
    protected function crm_error_check() {

        $non_error_codes =  array(200,201);
        if (!in_array( $this->response_code, $non_error_codes)) {

            $this->processed_response['error'] = true;
            $this->processed_response['crm_error'] = true;
            $this->processed_response['error_code'] = $this->response_code;

            $this->determine_error_message();
        }
    }

    /**
     * Checks for WordPress error
     */
    protected function wp_error_check() {

        if (is_wp_error($this->raw_response)) {

            $this->processed_response['error'] = true;
            $this->processed_response['wp_error'] = true;
            $this->processed_response['error_code'] = 'wp';
            $this->processed_response['error_message'] = $this->raw_response->get_error_message();
            $this->processed_response['crm_message'] = NF_CapsuleCRM_Constants::ACTION_REQUIRED.$this->raw_response->get_error_message();
        }
    }

    /**
     * If error, see if response body has more descriptive message for support
     */
    protected function determine_error_message() {

        $this->processed_response['error_message'] = $this->response_message;
        $this->processed_response['crm_message'] = NF_CapsuleCRM_Constants::ACTION_REQUIRED.$this->response_message;

        if (!isset($this->body)) {
            return;
        }

        if (!isset($this->body['errors']) ||
                !is_array($this->body['errors'])) {
            return;
        }

        $message = array($this->body['message']);

        foreach ($this->body['errors'] as $error) {
            
            $line_item_error = $error['message'];
            
            if(isset($error['field'])){
                
                $line_item_error .= ' - See field '.$error['field'];
            }
            
            $message[] = $line_item_error;
        }

        $this->processed_response['error_message'] = implode(', ', $message);
        $this->processed_response['crm_message'] = NF_CapsuleCRM_Constants::ACTION_REQUIRED.implode(', ', $message);
        
    }

    /**
     * The complete processed response array
     * @return array
     */
    public function get_processed_response() {

        return $this->processed_response;
    }

}
