<?php

/**
 * Communicates with Capsule to create an entry using API V1
 *
 */
class CapsuleV1Create extends CapsuleCommParent {

    /**
     * Complete endpoint for a Create request
     * @var string
     */
    protected $endpoint;

    /**
     * Module configuration
     * @var array
     */
    protected $module_config;

    /**
     * Module for which request is being made
     * @var string
     */
    protected $module;

    /**
     * Request being made
     * @var array
     */
    protected $request;

    /**
     * Keyed array of newly created records for linking
     * @var array
     */
    protected $new_record_ids = array();

    /**
     * 
     * @param array $keyed_request
     * @param array $module_config Module configuration
     * @param array $new_record_ids
     * @return array
     */
    public function create_entry($keyed_request, $module_config, $new_record_ids = array()) {

        $this->module_config = $module_config;

        $this->module = key($keyed_request);
        $this->request = $keyed_request;

        $this->new_record_ids = $new_record_ids;

        $this->set_v1_endpoints();

        $this->link_v1_parent();

        $this->modify_v1_request_structure();

        $this->make_request();

        return $this->raw_response;
    }

    protected function make_request() {

        $this->post_args['method'] = $this->module_config[$this->module]['method'];

        $this->raw_response = wp_remote_post(
                $this->post_url . $this->endpoint, $this->post_args
        );
    }

    /**
     * Adds linking fields into request
     * 
     * Checks config if parent is specified, then checks if parent has new id;
     * if so, add parent id to linking key in request
     */
    protected function link_v1_parent() {

        if (!isset($this->module_config[$this->module]['parent'])) {

            return;
        } else {

            $parent = $this->module_config[$this->module]['parent'];
        }

        if (!isset($this->new_record_ids[$parent])) {
            return;
        }

        $linking_key = $this->module_config[$this->module]['linking_key'];

        $this->request[$this->module][$linking_key] = (integer) $this->new_record_ids[$parent];
    }

    /**
     * Modify request structure per API requirements
     * 
     * Notes, part of the 'history' module, does not use the module name as
     * the key, so it must be stripped out
     * 
     * Custom Fields use key different than structured request key
     * 
     * Tags in API V1 are sent in the URL, not the post body
     */
    protected function modify_v1_request_structure() {

        switch ($this->module) {

            case 'history':

                $this->post_args['body'] = json_encode($this->request['history']);

                break;

            case 'custompersonfields':


                $this->post_args['body'] = json_encode(array('customFields' => $this->request['custompersonfields']));

                break;

            case 'customopportunityfields':

                $this->post_args['body'] = json_encode(array('customFields' => $this->request['customopportunityfields']));

                break;

            case 'tags':

                $this->endpoint .= '/' . urlencode($this->request['tags']);

                unset($this->post_args['body']);

                break;

            default:

                $this->post_args['body'] = json_encode($this->request);
        }
    }

    /**
     * Set the endpoints for V1 communication
     * 
     * V1 uses parent endpoint, parent id, then child endpoint to link
     * 
     * Exception is person to organisation, which uses keyed parent id field
     */
    protected function set_v1_endpoints() {

        $this->endpoint = $this->module_config[$this->module]['endpoint'];

        if (isset($this->module_config[$this->module]['second_endpoint'])) {
            // append the second endpoint, with party or opportunity in the middle
            $this->endpoint .= $this->new_record_ids [$this->module_config[$this->module]['link_to']];
            $this->endpoint .= $this->module_config[$this->module]['second_endpoint'];
        }
    }

}
