<?php

if (!defined('ABSPATH'))
    exit;

/**
 * Class NF_CapsuleCRM_Admin_Settings
 */
final class NF_CapsuleCRM_Admin_Settings {

    /**
     * The stored communication data for support messages and status
     * 
     * Gets initialized to default array to ensure keys are set
     * 
     * @var array
     */
    protected $comm_data;

    /**
     * Array of settings for display in NF Settings
     * @var array 
     */
    protected $settings_array;

    public function __construct() {

        $this->retrieve_comm_data();

        add_filter('ninja_forms_plugin_settings', array($this, 'plugin_settings'), 10, 1);
        add_filter('ninja_forms_plugin_settings_groups', array($this, 'plugin_settings_groups'), 10, 1);
    }

    /**
     * Modifies settings prior to display on settings page
     * 
     * HTML settings are not hard-coded; they are retrieved from the options
     * database and marked up
     * 
     * Sorts values to display the most pertinent information first
     * 
     * @param array $settings
     * @return array
     */
    public function plugin_settings($settings) {

        $this->settings_array = NF_CapsuleCRM()->config('PluginSettings');

        $this->markup_comm_status();

        $this->markup_support_messages();

        $this->markup_request_array();

        $this->markup_structured_request();

        $this->markup_full_response();

        $this->markup_tokens();

        $this->markup_authorization_help();

        $this->manage_v1_cred_display();

        $this->manage_pat_cred_display();

        $this->markup_data_dump();

        $this->manage_support_settings_display();
        
        $settings['capsulecrm'] = $this->settings_array;

        return $settings;
    }

    /**
     * Retrieve stored comm data, merge with default to ensure values are set
     */
    protected function retrieve_comm_data() {

        $default_comm_data = NF_CapsuleCRM::config('CommLogConfig');
        $stored_comm_data = get_option(NF_CapsuleCRM_Constants::SUPPORT_DATA);

        if (!is_array($stored_comm_data)) {

            $this->comm_data = $default_comm_data;
        } else {

            $this->comm_data = array_merge($default_comm_data, $stored_comm_data);
        }
    }

    /**
     * Selectively display or hide settings for debug or for simplicity
     */
    protected function manage_support_settings_display() {

        NF_CapsuleCRM_Functions::add_advanced_command_filters('support');

        $display_support = apply_filters('nfcapsulecrm_display_support', FALSE);

        if (FALSE == $display_support) {
            $support_mode_settings = array(
                NF_CapsuleCRM_Constants::SUPPORT_MESSAGES,
                NF_CapsuleCRM_Constants::REQUEST_ARRAY,
                NF_CapsuleCRM_Constants::STRUCTURED_REQUEST,
                NF_CapsuleCRM_Constants::FULL_RESPONSE,
                'data_dump',
            );

            foreach ($support_mode_settings as $setting) {

                unset($this->settings_array[$setting]);
            }
        }
    }

    /**
     * Display V1 credentials only if in filter or advanced command
     */
    protected function manage_v1_cred_display() {

        $show_v1cred = apply_filters('nfcapsulecrm_show_v1cred', FALSE);

        if (FALSE == $show_v1cred) {

            unset($this->settings_array[NF_CapsuleCRM_Constants::AUTHTOKEN]);
            unset($this->settings_array[NF_CapsuleCRM_Constants::SUBDOMAIN]);
        }
    }

    /**
     * Display Personal Access Token only if in filter or advanced command
     */
    protected function manage_pat_cred_display() {

        $show_patcred = apply_filters('nfcapsulecrm_show_patcred', FALSE);

        if (FALSE == $show_patcred) {

            unset($this->settings_array[NF_CapsuleCRM_Constants::PERSONAL_ACCESS_TOKEN]);
        }
    }

    /**
     * Iterates and applies markup for Comm Status
     */
    protected function markup_comm_status() {

        $display = '';

        if(!is_array($this->comm_data[NF_CapsuleCRM_Constants::COMM_STATUS])){
            
            $this->settings_array[NF_CapsuleCRM_Constants::COMM_STATUS]['html'] = 
                    $this->comm_data[NF_CapsuleCRM_Constants::COMM_STATUS];
            return;
        }
        
        foreach ($this->comm_data[NF_CapsuleCRM_Constants::COMM_STATUS] as $module => $entry_array) {

            $display .= $module . '<br />&nbsp;&nbsp;&nbsp;&nbsp;';
            $entries = implode('<br />&nbsp;&nbsp;&nbsp;&nbsp;', $entry_array);
            $markup_entries = self::support_tags_to_icons($entries);
            $display .= $markup_entries . '<br />';
        }

        $this->settings_array[NF_CapsuleCRM_Constants::COMM_STATUS]['html'] = $display;
    }

    /**
     * Iterates and applies markup for Request Array
     */
    protected function markup_request_array() {

        $display = '<table><tr>';
        $display .= '<th>Form Field</th><th>Field Map</th><th>Entry Type</th><th>Custom Field</th>';


        $request_array = $this->comm_data[NF_CapsuleCRM_Constants::REQUEST_ARRAY][NF_CapsuleCRM_Constants::REQUEST_ARRAY][0];

        foreach ($request_array as $field_map_request) {

            $display .= '<tr><td>';
            $display .= $field_map_request['form_field'] . '</td><td>';
            $display .= $field_map_request['field_map'] . '</td><td>';
            $display .= $field_map_request['entry_type'] . '</td><td>';
            $display .= $field_map_request['custom_field'] . '</td></tr> ';
        }

        $display .= '</table>';

        $this->settings_array[NF_CapsuleCRM_Constants::REQUEST_ARRAY]['html'] = $display;
    }

    /**
     * Iterates and applies markup for Structured Request
     */
    protected function markup_structured_request() {

        $display = '<table><tr>';
        $display .= '<th>Module</th><th>Request</th>';

        if (isset($this->comm_data[NF_CapsuleCRM_Constants::STRUCTURED_REQUEST][NF_CapsuleCRM_Constants::STRUCTURED_REQUEST][0])) {

            $structured_array = $this->comm_data[NF_CapsuleCRM_Constants::STRUCTURED_REQUEST][NF_CapsuleCRM_Constants::STRUCTURED_REQUEST][0];

            foreach ($structured_array as $structured_module_request) {

                $module = key($structured_module_request);
                $temp = serialize($structured_module_request[$module]);
                $display .= '<tr><td>' . $module . '</td><td>' . $temp . '</td></tr>';
            }
        }


        $display .= '</table>';

        $this->settings_array[NF_CapsuleCRM_Constants::STRUCTURED_REQUEST]['html'] = $display;
    }

    /**
     * Applies markup for stored open auth tokens
     * 
     * Retrieves data stored directly in wp_options. Only displays Refresh 
     * Token, which demonstrates ability to make next request
     */
    protected function markup_tokens() {

        $oauth_settings = get_option(NF_CapsuleCRM_Constants::OAUTH_CREDENTIALS_KEY);

        if (0 == strlen($oauth_settings['refresh_token'])) {

            unset($this->settings_array[NF_CapsuleCRM_Constants::OAUTH_CREDENTIALS_KEY . '_refresh_token']);

            return;
        }
        $this->settings_array[NF_CapsuleCRM_Constants::OAUTH_CREDENTIALS_KEY . '_refresh_token']['html'] = $oauth_settings['refresh_token'];
    }

    /**
     * Applies markup for support messages
     * 
     * Checks for (legacy) case of not being stored as array
     */
    protected function markup_support_messages() {

        if (!is_array($this->comm_data[NF_CapsuleCRM_Constants::SUPPORT_MESSAGES])) {

            $markup = $this->comm_data[NF_CapsuleCRM_Constants::SUPPORT_MESSAGES];
        } else {

            $markup = implode('<br />', $this->comm_data[NF_CapsuleCRM_Constants::SUPPORT_MESSAGES]);
        }

        if(0==strlen($markup)){
            
            unset($this->settings_array[NF_CapsuleCRM_Constants::SUPPORT_MESSAGES]);
            return;
        }
        
        $this->settings_array[NF_CapsuleCRM_Constants::SUPPORT_MESSAGES]['html'] = $markup;
    }

    /**
     * Applies markup for the full response from the CRM
     */
    protected function markup_full_response() {
        
        $display = '';
        
        if(!is_array($this->comm_data[NF_CapsuleCRM_Constants::FULL_RESPONSE])){
            
            $this->settings_array[NF_CapsuleCRM_Constants::FULL_RESPONSE]['html'] = serialize($this->comm_data[NF_CapsuleCRM_Constants::FULL_RESPONSE]);
            
            return;
        }
    
        foreach ($this->comm_data[NF_CapsuleCRM_Constants::FULL_RESPONSE] as $module => $entry) {
            $display .= $module.'&nbsp;&nbsp;'. serialize ($entry).'<br /><br />';
        }
        
        $this->settings_array[NF_CapsuleCRM_Constants::FULL_RESPONSE]['html']= $display;
    }

    /**
     * Applies markup for the data dump
     */
    protected function markup_data_dump() {



        $this->settings_array['data_dump']['html'] = $this->comm_data['data_dump'];
    }

    /**
     * Inline documentation to assist migration to Capsule CRM API v2
     */
    protected function markup_authorization_help() {

        $obj_str = NF_CapsuleCRM_Constants::AUTHORIZATION_CLASS;

        //TODO: replace with obj_str after PHP 5.2 support dropped
        $unmarked = CapsuleAuth::markup_authorization_help();
        
        $marked_up = self::support_tags_to_icons($unmarked);
        
        $this->settings_array[NF_CapsuleCRM_Constants::AUTH_HELP]['html'] = $marked_up;
    }

    /**
     * Replaces support code tags with icons
     * 
     * @param string $incoming
     * @return string
     */
    protected static function support_tags_to_icons($incoming){
        
        $wip = $incoming;
        
        $wip = str_replace(NF_CapsuleCRM_Constants::ACTION_REQUIRED, '<span style = "color: orange;" class="dashicons dashicons-warning"></span>', $wip);
        
        $wip = str_replace(NF_CapsuleCRM_Constants::HELPFUL_HINT, '<span style = "color: blue;" class="dashicons dashicons-arrow-right-alt"></span>', $wip);
        
        $wip = str_replace(NF_CapsuleCRM_Constants::SUCCESS, '<span style = "color: green;" class="dashicons dashicons-yes"></span>', $wip);
        
        $outgoing = $wip;
  
        return $outgoing;
    }
    
    /**
     * Adds the Settings Group to Ninja Forms settings
     */
    public function plugin_settings_groups($groups) {

        $groups = array_merge($groups, NF_CapsuleCRM()->config('PluginSettingsGroups'));

        return $groups;
    }

}
