<?php

if (!defined('ABSPATH'))
    exit;

/* --- NINJA FORMS CAPSULE CRM INTEGRATION --- */

/**
 * Retrieve db options as global variables to minimize db calls
 * 
 * @global array $nfcapsulecrm_site_options
 * @global array $nfcapsulecrm_comm_data
 */
function nfcapsule_load_globals() {

    global $nfcapsulecrm_site_options;
    global $nfcapsulecrm_comm_data;

    $keys_to_extract = array(
        'nfcapsulecrm_authtoken',
        'nfcapsulecrm_subdomain',
        'nfcapsulecrm_personal_access_token',
        'nfcapsulecrm_advanced_codes'
    );


    /**
     * the most recent communication messages in a separate variable array 
     * so it doesn't get overwritten when updating the options
     */
    $nfcapsulecrm_comm_data = get_option('nfcapsulecrm_comm_data');


    /**
     * The site-wide settings 
     * 
     * If a Post-NF3 mode isn't defined, grab the array of legacy options stored
     * in the wp-options database 
     */
    $legacy_settings = get_option('nfcapsulecrm_settings');

    if ('2.9x' == NFCAPSULECRM_MODE) {

        $temp_array = $legacy_settings;
    } else {

        // In a NF 3.0 setup, the settings are all stored in option ninja_forms_settings
        $nf_settings_array = get_option(' ninja_forms_settings');

        foreach ($keys_to_extract as $key) {

            if (isset($nf_settings_array[$key])) {

                // use the NF3 version if already set
                $temp_array[$key] = $nf_settings_array[$key];
            } elseif (isset($legacy_settings[$key])) {

                // If NF3 key isn't set, grab the NF2.9 version
                $temp_array[$key] = $legacy_settings[$key];
            } else {

                // ensure it is at least set
                $temp_array[$key] = '';
            }
        }
    }
    // set the global
    $nfcapsulecrm_site_options = $temp_array;
}

/**
 * Returns an array of the communication data or optionally a single setting
 * 
 *     $comm_data = array(
 *          'capsule_comm_status' => readable communication status update
 *          'most_recent_request' => JSON request sent to Capsule
 *          'most_recent_capsule_comm_result' => full response from Capsule
 *          'advanced_codes' => comma-delimited list
 *          'field_map_array' => after cycling through each option in the repeater, the array used to build the raw request
 *          'data_dump'=> available field for debug
 *      )
 * 
 * 
 * @return mixed
 */
function nfcapsulecrm_extract_comm_data($single_setting = '') {

    global $nfcapsulecrm_comm_data;

    $global_comm_data = $nfcapsulecrm_comm_data;

    // configure default data
    if ('2.9x' == NFCAPSULECRM_MODE) {


        $comm_data = include NF2CAPSULECRM_PLUGIN_DIR . 'includes/Config/CommData.php';
    } else {

        $comm_data = NF_CapsuleCRM::config('CommData'); // Array of comm data
    }

    foreach ($comm_data as $key => $value) {

        if (isset($global_comm_data[$key])) {
            $comm_data[$key] = $global_comm_data[$key];
        }
    }

    if (0 < strlen($single_setting)) {

        if (!isset($global_comm_data[$single_setting])) {

            $comm_data[$single_setting] = '';
        }

        return $comm_data[$single_setting];
    } else {

        return $comm_data;
    }
}

/**
 * Checks that the authtoken and subdomain are set correctly
 * 
 * If not, raise an error flag and build support messages
 * Extract subdomain from full domain if user sets value incorrectly
 * 'subdomain.capsulecrm.com'
 * 
 *      error_flag_v1      => (bool)
 *      error_flag_v2      => (bool)
 *      authtoken       => (string)
 *      subdomain       => (string)
 *      personal_access_token => (string)
 *      comm_status_v1     => (string)
 *      comm_status_v2     = (string)
 *      support_message_v1 => (array) of (string)s
 *      support_message_v2 => (array) of (string)s
 */
function nfcapsulecrm_validate_api_settings() {

    global $nfcapsulecrm_site_options;

    // initialize response
    $error_flag_v1 = false; // if API v1 has the required credentials
    $error_flag_v2 = false; // if API v2 has the required credentials

    $authtoken = ''; // used for v1
    $subdomain = ''; // used for v1
    $personal_access_token = ''; // used for v2
    $comm_status_v1 = '';   // 2.9 builds the message inline; this is maintained here in case function is called in 2.9 (currently is not used in 2.9)
    $comm_status_v2 = '';
    $support_message_v1 = array();
    $support_message_v2 = array();

    // temp variables
    $authtoken_message = '';
    $subdomain_message = '';

    /*
     * ensure there is a value for the authtoken
     */
    if (!isset($nfcapsulecrm_site_options['nfcapsulecrm_authtoken']) || ('' == $nfcapsulecrm_site_options['nfcapsulecrm_authtoken'] )) {

        $authtoken_message = __('No authorization token was entered before the last communication attempt.', 'ninja-forms-capsule-crm') . '<br />';

        $support_message_v1[] = __('Ensure that the Capsule CRM Auth Token value is set', 'ninja-forms-capsule-crm');

        $error_flag_v1 = TRUE;
    } else {

        $authtoken = $nfcapsulecrm_site_options['nfcapsulecrm_authtoken'];
    }

    /*
     * ensure that there is a value for the subdomain AND
     * remove the errant capsulecrm.com from subdomain if set
     */
    if (!isset($nfcapsulecrm_site_options['nfcapsulecrm_subdomain']) || ( '' == $nfcapsulecrm_site_options['nfcapsulecrm_subdomain'])) {

        $subdomain_message = __('No subdomain was entered before the last communication attempt.', 'ninja-forms-capsule-crm') . '<br />';

        $support_message_v1[] = __('Ensure that the Capsule CRM Subdomain is set', 'ninja-forms-capsule-crm');

        $error_flag_v1 = TRUE;
    } else {

        $subdomain = str_replace('.capsulecrm.com', '', $nfcapsulecrm_site_options['nfcapsulecrm_subdomain']);
    }

    // one of the two checks failed, append additional support text
    if ($error_flag_v1) {

        $comm_status_v1 = $authtoken_message . $subdomain_message;
        $comm_status_v1 .= '<br />' . __('Please enter the missing information and test your form again.', 'ninja-forms-capsule-crm');
    }

    /*
     * ensure there is a value for the personal access token
     */
    if (!isset($nfcapsulecrm_site_options['nfcapsulecrm_personal_access_token']) || ('' == $nfcapsulecrm_site_options['nfcapsulecrm_personal_access_token'] )) {

        $support_message_v2[] = __('Ensure that the Capsule CRM Personal Access Token value is set', 'ninja-forms-capsule-crm');

        $comm_status_v2 = __('Could not communicate with Capsule - Personal Access Token  not set.', 'ninja-forms-capsule-crm');
        $error_flag_v2 = TRUE;
    } else {

        $personal_access_token = $nfcapsulecrm_site_options['nfcapsulecrm_authtoken'];
    }


    $validated_settings = array(
        'error_flag' => array(
            'v1' => $error_flag_v1,
            'v2' => $error_flag_v2,
        ),
        'comm_status' => array(
            'v1' => $comm_status_v1,
            'v2' => $comm_status_v2,
        ),
        'support_messages' => array(
            'v1' => $support_message_v1, // array of string messages
            'v2' => $support_message_v2, // array of string messages
        ),
        'authtoken' => $authtoken,
        'subdomain' => $subdomain,
        'personal_access_token' => $personal_access_token,
    );

    return $validated_settings;
}

/**
 * Returns the advanced codes array
 * 
 * Uses get_option function in WP core so can be called whenever; it is not
 * dependent on any NF or NF_Capsule class
 * 
 * @return array Defaults to empty array
 */
function nfcapsulecrm_extract_advanced_codes() {

    $settings_key = 'nfcapsulecrm_advanced_codes';

    $advanced_codes_array = array(); //initialize
    $nf_settings_array = get_option(' ninja_forms_settings');

    if (isset($nf_settings_array[$settings_key])) {

        $advanced_codes_setting = $nf_settings_array[$settings_key];

        $advanced_codes_array = array_map('trim', explode(',', $advanced_codes_setting));
    }

    return $advanced_codes_array;
}

function nfcapsulecrm_data_dump($data) {

    global $nfcapsulecrm_comm_data;

    $nfcapsulecrm_comm_data['data_dump'] = $data;

    update_option('nfcapsulecrm_comm_data', $nfcapsulecrm_comm_data);
}
