<?php if ( ! defined( 'ABSPATH' ) ) exit;

/*
 * Plugin Name: Ninja Forms - AWeber
 * Plugin URI: https://ninjaforms.com/extensions/aweber/
 * Description: Sign users up for your AWeber newsletter when submitting Ninja Forms
 * Version: 3.1.1
 * Author: Kathy Darling
 * Author URI: http://kathyisawesome.com/
 * Text Domain: ninja-forms-aweber
 *
 * Copyright 2015 Kathy Darling
 * 
 */

if( version_compare( get_option( 'ninja_forms_version', '0.0.0' ), '3', '<' ) || get_option( 'ninja_forms_load_deprecated', FALSE ) ) {

	include 'deprecated/ninja-forms-aweber.php';

} else {

	include_once 'includes/Libraries/Aweber/aweber.php';

	/**
	 * Class NF_AWeber
	 */
	final class NF_AWeber
	{
		const VERSION = '3.1.1';
		const SLUG    = 'aweber';
		const NAME    = 'AWeber';
		const AUTHOR  = 'Kathy Darling';
		const PREFIX  = 'NF_AWeber';
		const OPTION  = 'ninja_forms_aweber_options';

		/**
		 * @var NF_AWeber
		 * 
		 * @since 1.4.0
		 */
		private static $instance;

		/**
		 * Plugin Directory
		 *
		 * @var string $dir
		 * @since 1.4.0
		 */
		public static $dir = '';

		/**
		 * Plugin URL
		 *
		 * @var string $url
		 * @since 1.4.0
		 */
		public static $url = '';

		/**
		 * @var AWeber
		 * @since 1.4.0
		 */
		private $_api;

		/**
		 * @var AWeber::getAccount
		 * @since 1.4.0
		 */
		private $_account;

		/**
		 * @var settings array
		 * @since 1.4.0
		 */
		private $_settings;

		/*
		 * Holds an array of all the custom fields for a list
		 *
		 * @var settings array
		 * @since 1.4.0
		 */
		private $_key_lookup;

		/*
		 * Holds an array of all the lists
		 *
		 * @var settings array
		 * @since 1.4.0
		 */
		private $_list_lookup;

		/**
		 * Main Plugin Instance
		 *
		 * Insures that only one instance of a plugin class exists in memory at any one
		 * time. Also prevents needing to define globals all over the place.
		 *
		 * @static
		 * @static var array $instance
		 * @return NF_AWeber Highlander Instance
		 *
		 * @since 1.4.0
		 */
		public static function instance()
		{
			if (!isset(self::$instance) && !(self::$instance instanceof NF_AWeber)) {
				self::$instance = new NF_AWeber();

				self::$dir = plugin_dir_path(__FILE__);

				self::$url = plugin_dir_url(__FILE__);

				spl_autoload_register(array(self::$instance, 'autoloader'));

				new NF_AWeber_Admin_Settings();
			}

			return self::$instance;
		}

		/**
		 * NF_AWeber constructor
		 */
		public function __construct()
		{
			add_action( 'admin_init', array( $this, 'setup_license' ) );
			add_filter( 'ninja_forms_register_fields', array( $this, 'register_fields' ) );
			add_filter( 'ninja_forms_register_actions', array( $this, 'register_actions' ) );

			add_action( 'ninja_forms_loaded', array( $this, 'ninja_forms_loaded' ) );
		}

		/**
		 * Add actions once we know that NF is fully loaded
		 */
		public function ninja_forms_loaded()
		{
			new NF_AWeber_Admin_Metaboxes_Submission();
		}


		/**
		 * Register Fields
		 *
		 * @param array $actions
		 * @return array $actions
		 *
		 * @since  3.0.0
		 */
		public function register_fields($actions)
		{
			$actions[ 'aweber-optin' ] = new NF_AWeber_Fields_OptIn();

			return $actions;
		}

		/**
		 * Register Actions
		 *
		 * @param array $actions
		 * @return array $actions
		 *
		 * @since  3.0.0
		 */
		public function register_actions($actions)
		{
			$actions[ 'aweber' ] = new NF_AWeber_Actions_AWeber();

			return $actions;
		}

		/**
		 * Autoloader
		 *
		 * @param $class_name
		 * @return  void
		 *
		 * @since  3.0.0
		 */
		
		public function autoloader( $class_name )
		{
			if ( class_exists( $class_name ) ) return;

			if ( false === strpos( $class_name, self::PREFIX ) ) return;

			$class_name = str_replace( self::PREFIX, '', $class_name );
			$classes_dir = realpath( plugin_dir_path( __FILE__ ) ) . DIRECTORY_SEPARATOR . 'includes' . DIRECTORY_SEPARATOR;
			$class_file = str_replace( '_', DIRECTORY_SEPARATOR, $class_name ) . '.php';

			if ( file_exists($classes_dir . $class_file ) ) {
				require_once $classes_dir . $class_file;
			}
		}

		/**
		 * Setup License
		 *
		 * @return  void
		 *
		 * @since  3.0.0
		 */
		public function setup_license()
		{  

			if ( ! class_exists( 'NF_Extension_Updater' ) ) return;

			new NF_Extension_Updater( self::NAME, self::VERSION, self::AUTHOR, __FILE__, self::SLUG );
		}


		/*
		 * Plugin Settings
		 *
		 * @return  void
		 *
		 * @since  3.0.0
		 */
		public function get_settings()
		{
			if( ! $this->_settings ) {

				$this->_settings = get_option( self::OPTION, array() );

				$defaults = array ( 'consumer_key' => false, 'consumer_secret' => false, 'access_key' => false, 'access_secret' => false );

				$this->_settings = wp_parse_args( $this->_settings, $defaults );

			}

			return $this->_settings;
		}

		/*
		 * Individual Plugin Setting
		 *
		 * @param string $key		 *
		 * @return  mixed
		 *
		 * @since  3.0.0
		 */
		public function get_setting( $key )
		{
			$settings = $this->get_settings();

			return isset( $settings[ $key ] ) ? $settings[ $key ] : false;

		}

		/**
		 * Test if authorized with AWeber
		 *
		 * @return bool
		 *
		 * @since 1.4.0
		 */
		public function is_authorized(){

			return $this->get_setting( 'access_secret' ) && $this->get_setting( 'access_key' ) ? true : false;

		}

		/**
		 * API "Account"
		 *
		 * @return object AWeberAPI
		 * 
		 * @since 1.4.0
		 */
		public function api()
		{
			if( ! $this->_api ) {

				$debug = defined('WP_DEBUG') && WP_DEBUG;

				try {
					$this->_api = new AWeberAPI( $this->get_setting( 'consumer_key' ), $this->get_setting( 'consumer_secret' ) );
				} catch ( AWeberException $e ) {
					if( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ){
						error_log( $e->getMessage() );
					}
				}

			}

			return $this->_api;
		}


		/**
		 * API "Account"
		 *
		 * @return object the user's account
		 *
		 * @since 1.4.0
		 */
		public function get_account()
		{
			if( ! $this->_account ) {

				try {
					$this->_account = $this->api()->getAccount( $this->get_setting( 'access_key' ), $this->get_setting( 'access_secret' ) );
				} catch (AWeberException $e) {
					if( defined('WP_DEBUG_LOG') && WP_DEBUG_LOG ){
						error_log( $e->getMessage() );
					}
				}

			}

			return $this->_account;
		}

		/**
		 * API "Lists"
		 *
		 * @return array AWeber lists formatted to NF
		 *
		 * @since 1.4.0
		 */
		public function get_lists()
		{

			$lists = $list_lookup = array();

			if ( is_object( $this->get_account() ) && count( $this->get_account()->lists ) > 0 ){

				foreach ( $this->get_account()->lists->data['entries'] as $list ) {

					$lists[] = array(
						'value' => $list[ 'id' ],
						'label' => $list[ 'name' ],
						'groups' => array(),
						'fields' => $this->get_list_merge_vars( $list[ 'id' ] )
					);

					$list_lookup[ $list[ 'id' ] ] = $list[ 'name' ];

				}

				// Save the list look up.
				update_option( 'ninja_forms_aweber_list_lookup', $list_lookup );

			}

			return $lists;

		}

		/**
		 * AWeber Default fields
		 * - name and email are required, subscription will fail without
		 *
		 * @param str $list_id
		 * @return array AWeber lists formatted to NF
		 *
		 * @since 1.4.0
		 */
		public function get_default_merge_vars( $list_id )
		{

			$required_text = ' <small style="color:red">(required)</small>';

			$merge_vars = array(
				array(
					'value' => $list_id . '_name',
					'label' => __( 'Name', 'ninja-forms-aweber' ) . $required_text
				),
				array(
					'value' => $list_id . '_email',
					'label' => __( 'Email Address', 'ninja-forms-aweber' ) . $required_text
				),
				array(
					'value' => $list_id . '_ip_address',
					'label' => __( 'IP Address', 'ninja-forms-aweber' )
				),
				array(
					'value' => $list_id . '_misc_notes',
					'label' => __( 'Miscellaneous notes', 'ninja-forms-aweber' )
				),
				array(
					'value' => $list_id . '_ad_tracking',
					'label' => __( 'Ad Tracking', 'ninja-forms-aweber' )
				),

			);

			return $merge_vars;
		}


		/**
		 * AWeber Custom fields
		 * get all custom fields set up for a specific AWeber list
		 * NB: AWeber is picky about it's custom fields. so we stash their AWeber names against their NF keys in the ninja_forms_aweber_key_lookup option
		 *
		 * @param str $list_id
		 * @return array AWeber custom formatted to merge vars
		 *
		 * @since 1.4.0
		 */
		public function get_custom_fields( $list_id ){

			try {

				$key_lookup = get_option( 'ninja_forms_aweber_key_lookup', array() );

				$account = $this->get_account();

				$listURL = untrailingslashit( $account->url ) . '/lists/' . $list_id . '/custom_fields';
				$fields = $account->loadFromUrl($listURL);

				$field_key_lookup = $custom_fields = array();

				foreach ( $fields->data['entries'] as $field ) {

					// Set up the look up array.
					$setting_label = $field['name'];
					$setting_key = sanitize_title( $field['name'] );
					$field_key_lookup[$setting_key] = $setting_label;

					// Set up the custom fields to send back to NF.
					$custom_fields[] = array( 
						'value' => $list_id . '_' . $setting_key,
						'label' => $setting_label
					);

				}

				// Save the field look up.
				$key_lookup[$list_id] = $field_key_lookup;
				update_option( 'ninja_forms_aweber_key_lookup', $key_lookup );

			} catch ( AWeberException $e ) {
				if( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ){
						error_log( $e->getMessage() );
				}
				$custom_fields = array();
			}

			return $custom_fields;

		}

		/**
		 * Combine default and custom fields to get merge vars
		 *
		 * @param str $list_id
		 * @return array merge vars
		 *
		 * @since 1.4.0
		 */
		public function get_list_merge_vars( $list_id )
		{

			return array_merge( $this->get_default_merge_vars( $list_id ), $this->get_custom_fields( $list_id ) );

		}

		/**
		 * Subscribe a user to a list via API
		 *
		 * @param str $list_id
		 * @param $array $merge_vars default + custom fields
		 * @return array success/failure messages
		 *
		 * @since 1.4.0
		 */
		public function subscribe( $list_id, $merge_vars )
		{

			$subscriber_id = 0;

			try {

				$aweber = $this->api();
                $account = $this->get_account();

                // Direct API URL.
				$account_id = $account->id;
                $url = "/accounts/{$account_id}/lists/{$list_id}/subscribers";

				// Get custom fields.
				$custom_fields = array();

				// Create a subscriber.
				$params = array(
					'ws.op' => 'create',
					'email' => isset( $merge_vars[ 'email' ] ) ? $merge_vars[ 'email' ] : '',
					'name' => isset( $merge_vars[ 'name' ] ) ? $merge_vars[ 'name' ] : '',
				    'ip_address' => isset( $merge_vars[ 'ip_address' ] ) ? $merge_vars[ 'ip_address' ] : '',
				    'misc_notes' => isset( $merge_vars[ 'misc_notes' ] ) ? $merge_vars[ 'misc_notes' ] : '',
					'ad_tracking' => isset( $merge_vars[ 'ad_tracking' ] ) ? $merge_vars[ 'ad_tracking' ] : ''
				);
				
				// Add tags to susbcriber vars.
				if( ! empty ( $merge_vars[ 'aweber_tags' ] ) ) {
					$params[ 'tags' ] = $merge_vars[ 'aweber_tags' ];
					unset( $merge_vars[ 'aweber_tags' ] );
				}

				// Convert the custom merge vars into AWeber custom fields.
				foreach( $merge_vars as $var => $value ){
					// AWeber is picky about the key being *exactly* the same the value in the List settings.
					$key = $this->key_lookup( $list_id, $var );

					if( $key ){
						$custom_fields[$key] = $value;
					}

				}
			
				if( ! empty( $custom_fields ) ){
					$params['custom_fields'] = $custom_fields;
				}

				// Single POST to API.
                $data = $aweber->adapter->request( 'POST', $url, $params, array('return' => 'headers') );

		        // If successful, grab the subscriber ID from the return subscriber URL.
		        if( is_array( $data ) && $data['Status'] == '201 Created' && strpos( $data['Location'], '/subscribers/' ) !== false ){
		          $array = explode ( '/subscribers/' , $data['Location'] );
		          $subscriber_id = $array[1];
		        }

			} catch( AWeberException $e ) {
                return array( 'error' => $e->getMessage() );
            }

			// If succesful send back some details about the subscriber.
			return array( 'id' => $subscriber_id, 'list' => $list_id );

		}

		/**
		 * Convert the merge var back to AWeber custom key
		 *
		 * @param str $list_id
		 * @param str $key
		 * @return str
		 *
		 * @since 3.0.0
		 */
		public function key_lookup( $list_id, $key ){
			if( ! isset( $this->_key_lookup ) ){
				$this->_key_lookup = get_option( 'ninja_forms_aweber_key_lookup', array() );
			}
			return isset( $this->_key_lookup[$list_id] ) && isset( $this->_key_lookup[$list_id][$key] ) ? $this->_key_lookup[$list_id][$key] : '';
		}

		/**
		 * Convert the list ID to AWeber List name
		 *
		 * @param $list_id
		 * @return str
		 *
		 * @since 3.0.0
		 */
		public function list_lookup( $list_id ){
			if( ! isset( $this->_list_lookup ) ){
				$this->_list_lookup = get_option( 'ninja_forms_aweber_list_lookup', array() );
			}
			return isset( $this->_list_lookup[$list_id] ) ? $this->_list_lookup[$list_id] : '';
		}

		/*
		 * STATIC METHODS
		 */

		/**
		 * Load Template File
		 *
		 * @param string $file_name
		 * @param array $data
		 *
		 * @since 3.1.0
		 */
		public static function template( $file_name = '', array $data = array() )
		{
			if( ! $file_name ) return;

			extract( $data );

			include self::$dir . 'includes/Templates/' . $file_name;
		}

		/**
		 * Load Config File
		 *
		 * @param $file_name
		 * @return array
		 *
		 * @since 3.0.0
		 */
		public static function config( $file_name )
		{
			return include self::$dir . 'includes/Config/' . $file_name . '.php';
		}

	}

	/**
	 * The main function responsible for returning The Highlander Plugin
	 * Instance to functions everywhere.
	 *
	 * Use this function like you would a global variable, except without needing
	 * to declare the global.
	 *
	 * @since 1.4.0
	 * @return NF_AWeber
	 */
	function NF_AWeber()
	{
		return NF_AWeber::instance();
	}

	NF_AWeber();
}

add_filter( 'ninja_forms_upgrade_settings', 'NF_Aweber_Upgrade' );
function NF_Aweber_Upgrade( $data ){

	$actions = array();

	$field_mapping = array(
		'first_name' => '',
		'last_name'  => '',
		'email'      => '',
		'ip_address' => ''
	);

	foreach( $data[ 'fields' ] as $key => $field ){

		if( '_text' == $field[ 'type' ] ){

			if( isset( $field[ 'data' ][ 'first_name' ] ) && 1 == $field[ 'data' ][ 'first_name' ] ){
				$field_mapping[ 'first_name' ] = $field[ 'id' ];
			}

			if( isset( $field[ 'data' ][ 'last_name' ] ) && 1 == $field[ 'data' ][ 'last_name' ] ){
				$field_mapping[ 'last_name' ] = $field[ 'id' ];
			}

			if( isset( $field[ 'data' ][ 'email' ] ) && 1 == $field[ 'data' ][ 'email' ] ){
				$field_mapping[ 'email' ] = $field[ 'id' ];
			}

		}

		if( 'ip_address' == $field[ 'type' ] ){

			$field_mapping[ 'ip_address' ] = $field[ 'id' ];

			$data[ 'fields' ][ $key ][ 'type' ] = 'hidden';
			$data[ 'fields' ][ $key ][ 'default' ] = '{system:ip}';
		}

		if( 'aweber_optin' == $field[ 'type' ] ){

			if( isset( $field[ 'data' ][ 'aweber_list' ] ) && $field[ 'data' ][ 'aweber_list' ] ){
				$actions[] = array(
					'type' => 'aweber',
					'label' => $field[ 'data' ][ 'label' ],
					'newsletter_list' => $field[ 'data' ][ 'aweber_list' ],
				);
			}

			$data[ 'fields' ][ $key ][ 'type' ] = 'aweber-optin';

			switch( $data[ 'fields' ][ $key ][ 'data' ][ 'display_mode' ] ){
				case 'checked':
					// Opt-in by default
					$data[ 'fields' ][ $key ][ 'data' ][ 'default' ] = 1;
					break;
				case 'unchecked':
					$data[ 'fields' ][ $key ][ 'data' ][ 'default' ] = 0;
					break;
				case 'hidden':
					// Move to action, remove field.
					unset( $data[ 'fields' ][ $key ] );
					break;
			}
		}
	}

	if( is_array( $actions ) ){
		foreach( $actions as $key => $action ){

			if( ! isset( $actions[ $key ][ 'newsletter_list' ] ) || ! $actions[ $key ][ 'newsletter_list' ] ) continue;

			$newsletter_list = $actions[ $key ][ 'newsletter_list' ];

			if( $field_mapping[ 'first_name' ] ){
				$actions[ $key ][ $newsletter_list . '_name' ] = '{field:' . $field_mapping[ 'first_name' ] . '}';
			}

			if( $field_mapping[ 'last_name' ] ){
				$actions[ $key ][ $newsletter_list . '_name' ] .= ' {field:' . $field_mapping[ 'last_name' ] . '}';
			}

			if( $field_mapping[ 'email' ] ){
				$actions[ $key ][ $newsletter_list . '_email' ] = '{field:' . $field_mapping[ 'email' ] . '}';
			}

			if( $field_mapping[ 'ip_address' ] ){
				$actions[ $key ][ $newsletter_list . '_ip_address' ] = '{field:' . $field_mapping[ 'ip_address' ] . '}';;
			}
		}
		$data[ 'actions' ] = array_merge( $data[ 'actions' ], $actions );
	}
	return $data;
}