<?php if ( ! defined( 'ABSPATH' ) ) exit;
/**
 * Class NF_Example_Admin_Settings
 *
 * This is an example implementation of a Settings Class for a Ninja Forms Add-on.
 * The Ninja Forms Settings Submenu page handles registering, rendering, and saving settings.
 * Settings handled by Ninja Forms can be access using the Ninja Forms API.
 * Multiple WordPress Hooks are available for interacting with settings processing.
 */
final class NF_AWeber_Admin_Settings
{
    /**
     * NF_Example_Admin_Settings constructor.
     *
     * The following WordPress hooks are listed in processing order.
     */
    public function __construct()
    {
        // On Settings Page Load.
        add_filter( 'ninja_forms_plugin_settings',                array( $this, 'plugin_settings'        ), 10, 1 );
        add_filter( 'ninja_forms_plugin_settings_groups',         array( $this, 'plugin_settings_groups' ), 10, 1 );
        add_filter( 'ninja_forms_check_setting_aweber_oauth_id',  array( $this, 'check_setting'  ), 10, 1 );
        
        // On Settings Page Save (Submit).
        add_filter( 'ninja_forms_update_setting_aweber_oauth_id', array( $this, 'update_setting' ), 10, 1 );
        add_action( 'ninja_forms_save_setting_aweber_oauth_id',   array( $this, 'save_setting'   ), 10, 1 );
        add_action( 'ninja_forms_save_setting_aweber_remove',     array( $this, 'remove_setting'   ), 10, 1 );
    }
    
    /**
     * Add Plugin Settings
     *
     * Add a new setting within a defined setting group.
     * The setting's configuration is similar to Action Settings and Fields Settings.
     *
     * @param array $settings
     * @return array $settings
     *
     * @since  3.0.0
     */
    public function plugin_settings( $settings )
    {   
        $settings[ 'aweber' ] = NF_AWeber()->config( 'PluginSettings' );
        return $settings;
    }
    
    /**
     * Add Plugin Settings Groups
     *
     * Add a new Settings Groups for this plugin's settings.
     * The grouped settings will be rendered as a metabox in the Ninja Forms Settings Submenu page.
     *
     * @param array $groups
     * @return array $groups
     *
     * @since 3.0.0
     */
    public function plugin_settings_groups( $groups )
    {
        $groups = array_merge( $groups, NF_AWeber()->config( 'PluginSettingsGroups' ) );
        return $groups;
    }

    /**
     * Check Authorization Setting for Errors
     *
     * Check for existence of secrets/keys, otherwise display errors
     * Note: This check is performed when the Ninja Forms Settings Submenu page is loaded.
     *
     * @param array $setting
     * @return array $setting
     *
     * @since  3.0.0
     */
    public function check_setting( $setting )
    {
        $debug = defined( 'WP_DEBUG' ) && WP_DEBUG;

        $error = Ninja_Forms()->get_setting( 'aweber_error' );
        $access_secret = Ninja_Forms()->get_setting( 'aweber_access_secret' );
        
        if( trim( $setting['value' ] ) == '' ){
            $setting[ 'errors' ][] = __( 'You need to enter an AWeber authorization code in order to use the AWeber plugin.', 'ninja-forms-aweber');
        } else if ( ! NF_AWeber()->is_authorized() ){
            $setting[ 'errors' ][] = __( 'Unable to connect to your AWeber Account. Please generate a new authorization code and try again.', 'ninja-forms-aweber');                 
        }

        return $setting;
    }

    /**
     * Sanitize the auth code
     *
     * @param $value
     * @return mixed
     *
     * @since  3.0.0
     */
    public function update_setting( $value )
    {
        return sanitize_text_field( trim( $value ) );
    }

    /**
     * Save Auth Settings
     *
     * When the auth code is entered, split it into AWeber keys/secrets
     *
     * @param $value
     * @return void
     * 
     * @since  3.0.0
     */
    public function save_setting( $value )
    { 

        $access_secret = false;

        // Let's attempt the AWeber API.
        try {
            list( $consumer_key, $consumer_secret, $access_key, $access_secret ) = AWeberAPI::getDataFromAweberID( $value );
        } catch ( AWeberAPIException $exc) {
            // Make error messages customer friendly.
            $descr = $exc->message;
            $descr = preg_replace( '/http.*$/i', '', $descr );     # strip labs.aweber.com documentation url from error message
            $descr = preg_replace( '/[\.\!:]+.*$/i', '', $descr ); # strip anything following a . : or ! character
            $error_code = "($descr)";

            // TODO: Log Error, $e->getMessage(), for System Status Report.
        } 

        // Hurray, we're clear with AWeber.
        if ( $access_secret ) {

            // @TODO: no way to show a success message.
            $message = __( 'Successfully connected to your AWeber Account', 'ninja-forms-aweber' );

            Ninja_Forms()->update_setting( 'aweber_oauth_id', false );

            update_option( NF_AWeber::OPTION, compact( 'consumer_key', 'consumer_secret', 'access_key', 'access_secret' ) );

        }

    }

    /**
     * De-Authorize
     *
     * If the "remove button" is clicked then delete the auth codes
     *
     * @param $value
     * @return void
     *
     * @since  3.0.0
     */
    public function remove_setting( $value )
    { 

        // Remove all the settings.
        if ( 1 === absint( $value ) ) {

            delete_option( NF_AWeber::OPTION );

        }

    }

} // End Class NF_Example_Admin_Settings
