<?php


if ( ! defined( 'NINJA_FORMS_EDD_SL_STORE_URL' ) )
	define( 'NINJA_FORMS_EDD_SL_STORE_URL', 'http://ninjaforms.com/' );
define( 'NINJA_FORMS_EDD_AWEBER_PRODUCT_NAME', 'AWeber' );



/**
 * The Main NF_AWeber class
 **/
if ( ! class_exists( 'NF_AWeber' ) ) :

class NF_AWeber {

	/**
	 * @var NF_AWeber - the single instance of the class
	 */
	protected static $_instance = null;

	/**
	 * variables
	 */
	public $version = '3.0.2';


	/**
	 * Main NF_AWeber instance.
	 *
	 * Ensures only one instance of NF_AWeber is loaded or can be loaded
	 *
	 * @static
	 * @return NF_AWeber - Main instance
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}


	/**
	 * Cloning is forbidden.
	 */
	public function __clone() {
		_doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'ninja-forms-aweber' ) );
	}


	/**
	 * Unserializing instances of this class is forbidden.
	 */
	public function __wakeup() {
		_doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'ninja-forms-aweber' ) );
	}


	/**
	 * NF_AWeber Constructor
	 *
	 * @access 	public
     * @return 	NF_AWeber
	 */
	public function __construct() {

		// Plugin Folder Path
		if ( ! defined( 'NF_AWEBER_PLUGIN_DIR' ) ){
			define( 'NF_AWEBER_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
		}

		// load files
		$this->includes();

		// Load translation files
		add_action( 'plugins_loaded', array( $this, 'load_plugin_textdomain' ) );

		// plugin updates
		add_action( 'admin_init', array( $this, 'setup_license' ) );

		// plugin settings
		add_action( 'admin_menu', array( $this, 'add_menu' ), 20 );
		add_action( 'admin_init', array( $this, 'register_plugin_settings' ) );

		// form settings
		add_action( 'admin_init', array( $this, 'add_form_settings' ), 100 );

		// Register Sidebar for fields
		add_action( 'admin_init', array( $this, 'register_fields_sidebar' ), 20 );

		// Scripts & AJAX
		add_action( 'admin_enqueue_scripts', array( $this, 'admin_js' ) );
		add_action( 'wp_ajax_nf_aweber_refresh', array( $this, 'ajax_callback' ) );

		// AWeber Custom field mapping on all field types
		add_action( 'ninja_forms_edit_field_after_registered', array( $this, 'aweber_custom_fields' ), 10 );

		// Connect our signup check to form processing
		add_action( 'ninja_forms_process', array( $this, 'check_for_email_signup' ) );

    }


	/*-----------------------------------------------------------------------------------*/
	/* Load Files */
	/*-----------------------------------------------------------------------------------*/

	/**
	 * Include required core files used in admin and on the frontend.
	 *
	 * @return void
	 */
	public function includes(){

		/* Require AWeber API files */
		if ( ! class_exists( 'AWeberAPI' ) ){
			require_once( NF_AWEBER_PLUGIN_DIR . 'aweber_api/aweber_api.php');
		}

		/* Require AWeber-Specfic Fields */
		require_once( NF_AWEBER_PLUGIN_DIR . 'includes/fields/ip-address.php' );
		require_once( NF_AWEBER_PLUGIN_DIR . 'includes/fields/aweber-optin.php' );

	}


	/*-----------------------------------------------------------------------------------*/
	/* Localization */
	/*-----------------------------------------------------------------------------------*/


	/**
	 * Make the plugin translation ready
	 *
	 * @return void
	 */
	public function load_plugin_textdomain() {
		// Set filter for plugin's languages directory
		$lang_dir = dirname( plugin_basename( __FILE__ ) ) . '/languages/';
		$lang_dir = apply_filters( 'ninja_forms_aweber_languages_directory', $lang_dir );

		// Load the translations
		load_plugin_textdomain( 'ninja-forms-aweber', false, $lang_dir );
	}


	/**************************************************************
	/* 3. License and Updates
	/**************************************************************/

	/**
	 * Plugin Updater
	 *
	 * @since       1.0
	 * @return      void
	 */
	public function setup_license() {
	    if ( class_exists( 'NF_Extension_Updater' ) ) {
			$NF_Extension_Updater = new NF_Extension_Updater( 'AWeber', $this->version, 'Kathy Darling', __FILE__, 'aweber' );
	    }
	}


	/**************************************************************
	/* 4. Plugin Settings
	/**************************************************************/

	/**
	 * Register plugin settings
	 *
	 * @since       1.0
	 * @return      void
	 */
	public function register_plugin_settings(){
	    register_setting( 'ninja_forms_aweber_settings', 'ninja_forms_aweber_options', array( $this, 'validate' ) );
	}


	/**
	 * Add AWeber settings as a sub page of Ninja Forms
	 *
	 * @since       1.0
	 * @return      void
	 */
	public function add_menu(){

		$capabilities = 'manage_options';
		$capabilities = apply_filters( 'ninja_forms_admin_menu_capabilities', $capabilities );

		add_submenu_page( 'ninja-forms', __( 'AWeber Authorization', 'ninja-forms-aweber' ), __( 'AWeber', 'ninja-forms-aweber' ), $capabilities, 'ninja-forms-aweber', array( $this, 'admin_display' ) );

	}


	/**
	 * Custom callback for AWeber options
	 *
	 * @since       1.0
	 * @return      void
	 */
	public function admin_display(){

		$defaults = array ( 'oauth_id' => false, 'access_key' => false );
		$options = get_option( 'ninja_forms_aweber_options', $defaults );
		$options = wp_parse_args( (array) $options, $defaults );

		?>

		<div class="wrap">

			<!-- Display Plugin Icon, Header, and Description -->
			<h2><?php _e('AWeber Authorization', 'ninja-forms-aweber' );?></h2>

			<?php settings_errors( 'ninja_forms_aweber_settings' ); ?>

			<!-- Beginning of the Plugin Options Form -->
			<form method="post" action="<?php echo admin_url( 'options.php' );?>">

				<?php settings_fields( 'ninja_forms_aweber_settings' ); ?>

				<table class="form-table">
				<tbody>

				<?php
					// If we have the access key successfully, only show the option to remove connection
					if ( $options['access_key'] ){ ?>
						<tr>
							<th><?php _e( 'Click button to deauthorize AWeber', 'ninja-forms-aweber' ); ?></th>
								<td>
									<input type="submit" name="ninja_forms_aweber_options[remove]" class="button-primary" value="<?php _e( 'Remove Connection', 'ninja-forms-aweber' ); ?>">
								</td>
						</tr>
					<?php
					// otherwise need to show input for making connection to aweber
					 } else { ?>
						<tr>
							<th><?php _e( 'AWeber Authorization Code', 'ninja-forms-aweber' );?></th>
								<td>
									<input type="text" class="code widefat " name="ninja_forms_aweber_options[oauth_id]" id="aweber_oauth_id" value="<?php echo ninja_forms_esc_html_deep( $options["oauth_id"] ); ?>" />
								</td>

								<td>
									<input type="submit" name="ninja_forms_aweber_options[authorize]" class="button-primary" value="<?php _e( 'Make Connection', 'ninja-forms-aweber' ); ?>">
							</td>
						</tr>

						<tr>
							<th></th>
							<td class="howto"><?php printf ( __( 'Enter your AWeber Authorization Code. %sClick here to get your authorization code%s', 'ninja-forms-aweber'), '<a target="_blank" href="https://auth.aweber.com/1.0/oauth/authorize_app/81dfde27">', '</a>' ); ?>
							</td>
						</tr>

					<?php } ?>

			    	</tbody>
				</table>
			</form>
		</div>

	<?php
	}


	/**
	 * Validate/Save AWeber settings
	 *
	 * @since       1.0
	 * @return      void
	 */
	 public function validate( $input ){

	    $clean = array(); 

	    $type = $message = '';

		// If Make Connection button is clicked
		if ( isset( $input['authorize'] ) && isset ( $input['oauth_id'] ) ){

			// save the authorization code
			$oauth_id = sanitize_text_field( $input['oauth_id'] );
			$clean['oauth_id'] = $oauth_id;

			// let's attempt the AWeber API
			try {
				list( $consumer_key, $consumer_secret, $access_key, $access_secret ) = AWeberAPI::getDataFromAweberID( $oauth_id );
			} catch ( AWeberAPIException $exc) {
				list( $consumer_key, $consumer_secret, $access_key, $access_secret ) = null;
				# make error messages customer friendly.
				$descr = $exc->message;
				$descr = preg_replace( '/http.*$/i', '', $descr );     # strip labs.aweber.com documentation url from error message
				$descr = preg_replace( '/[\.\!:]+.*$/i', '', $descr ); # strip anything following a . : or ! character
				$error_code = "($descr)";
			} 

			// if no access secret, then we didn't get authorized
			if ( ! $access_secret ) {

				$type = 'error';
				$message = __( 'Unable to connect to your AWeber Account.', 'ninja-forms-aweber' );
				$message .= '<br/>' . $error_code;
				$message .= '<br/>' . __( 'Please make sure you entered the complete authorization code or generate a new authorization code and try again.', 'ninja-forms-aweber');
			// hurray, we're clear with AWeber
			} else {

				$type = 'updated';
				$message = __( 'Successfully connected to your AWeber Account', 'ninja-forms-aweber' );
				$clean['consumer_key'] = $consumer_key;
				$clean['consumer_secret'] = $consumer_secret;
				$clean['access_key'] = $access_key;
				$clean['access_secret'] = $access_secret;

				// delete transient on authorize
				$this->delete_transient();
			}

		}

		// If Remove Connection button is clicked
		if ( isset( $input['remove'] ) ){
			$type = 'updated';
			$message = __( 'Your connection to your AWeber account has been closed.', 'ninja-forms-aweber' );
			
			// delete transient on de-authorize
			$this->delete_transient();
		}

		if( $type && $message ){
			add_settings_error(
				'ninja_forms_aweber_settings',
				esc_attr( 'ninja_forms_aweber_settings_updated' ),
				$message,
				$type
		    );
		}

	    return $clean;
	}


	/**************************************************************
	/* Form Settings
	/**************************************************************/

	/**
	 * Register the form-specific settings
	 *
	 * @since       1.0
	 * @deprecated  1.3 - should use a form field now
	 * @return      void
	 */
	public function add_form_settings() {

		$link = remove_query_arg( array( 'update_message','notification-action' ) );
		$form_id = isset($_REQUEST['form_id']) ? absint( $_REQUEST['form_id'] ) : '';
		if($form_id != ''){
			$link = esc_url( add_query_arg( array( 'tab' => 'builder', 'form_id' => $form_id ), $link ) );
		} else{
			$link = esc_url( add_query_arg( array( 'tab' => 'builder' ), $link ) );
		}

		$args = array(
			'page' => 'ninja-forms',
			'tab' => 'form_settings',
			'slug' => 'aweber_fields',
			'state' => 'closed',
			'title' => __( 'AWeber Settings', 'ninja-forms-aweber' ),
			'settings' => array(
				array(
					'name'      => 'aweber_deprecated_notice',
					'type'      => 'desc',
					'label'     => __( 'Warning!', 'ninja-forms-aweber' ),
					'desc'      => sprintf( __( 'AWeber form settings have been deprecated since Ninja Forms AWeber 1.2.0. Please use an "AWeber Signup" form field instead when %sbuilding your form%s.', 'ninja-forms-aweber' ), '<a href="'.$link.'">', '</a>' )
				),				
				array(
					'name'      => 'aweber_signup_form',
					'type'      => 'checkbox',
					'label'     => __( 'Enable AWeber signup for this form?', 'ninja-forms-aweber' ),
					'desc'      => __( 'This will use fields from this form to create a new AWeber subscriber.', 'ninja-forms-aweber' )
				),
				array(
					'name'    => 'aweber_list',
					'label'   => __( 'Choose a list', 'ninja-forms-aweber' ),
					'desc'    => sprintf( __( 'Select the AWeber list to which you wish to subscribe users when submitting the form. If you don\'t see any lists, then try %sauthorizing your AWeber account%s.', 'ninja-forms-aweber' ), '<a href="' . admin_url( 'admin.php?page=ninja-forms-aweber' ) . '">', '</a>' ),
					'type'    => 'select',
					'options' => $this->get_lists()
				)
			)
		);

		if( function_exists( 'ninja_forms_register_tab_metabox' ) ){
			ninja_forms_register_tab_metabox( $args );
		}

	}

	/**
	 * Register sidebar for fields
	 *
	 * @return void
	 */
	public function register_fields_sidebar() {

		// now register the new sidebar
		$args = array(
			'name' => __( 'AWeber Fields', 'ninja-forms-aweber' ),
			'page' => 'ninja-forms',	
			'tab' => 'builder',
			'display_function' => 'ninja_forms_sidebar_display_fields'
		);

		if( function_exists( 'ninja_forms_register_sidebar' ) ) {
			ninja_forms_register_sidebar( 'aweber_fields', $args );
		}
	}


	/**************************************************************
	/* Scripts & Ajax callback
	/**************************************************************/

	/**
	 * Load Admin scripts
	 *
	 * @since       1.3.0
	 * @return      void
	 */
	public function admin_js( $hook ){
		if( 'toplevel_page_ninja-forms' == $hook ){
			$suffix = defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ? '' : '.min';
			wp_enqueue_script( 'ninja-forms-aweber', plugins_url( 'assets/js/ninja-forms-aweber' . $suffix . '.js ', __FILE__ ), array( 'jquery' ), $this->version, true );
		}
	}

	/**
	 * Get the lists for the AJAX function
	 *
	 * @since       1.3.0
	 * @return      void
	 */	
	public function ajax_callback(){
		// Verify that the incoming request is coming with the security nonce
  		if( wp_verify_nonce( $_POST['nonce'], 'ajax_aweber_refresh_nonce' ) ) {
			$this->delete_transient();
			echo json_encode( $this->get_lists() );
			die();
		} else {
			die(-1);
		}
	}


	/**************************************************************
	/* Field Settings - Add AWeber Custom Field Name to all Fields
	/**************************************************************/


	/**
	 * Output extra admin field input
	 *
	 * @param $field_id
	 * @since       1.1
	 * @return      void
	 */
	public function aweber_custom_fields( $field_id ){
		global $ninja_forms_fields;

		// @todo: check against NF2.9
		$field_row = ninja_forms_get_field_by_id( $field_id );
		$field_type = $field_row['type'];

		// only show on fields to be processed
		if ( 'aweber_optin' != $field_type && $ninja_forms_fields[$field_type] ['process_field'] ) {

			$field_data = $field_row['data'];

			if ( isset ( $field_data['aweber_custom'] ) ) {
				$aweber_custom = $field_data['aweber_custom'];
			} else {
				$aweber_custom = '';
			}

			?>

			<div class="description description-wide">
				<hr/>
				<h5><?php _e( 'AWeber Settings', 'ninja-forms-aweber' );?></h5>
				<?php ninja_forms_edit_field_el_output( $field_id, 'text', __( 'AWeber Custom Field Name', 'ninja-forms-aweber' ), 'aweber_custom', $aweber_custom, 'wide', '', 'widefat' );?>
				<span><?php _e( 'Use the exact same name you\'ve defined in your AWeber account.', 'ninja-forms-aweber' );?></span>
			</div>
			<?php

		}

	}


	/**************************************************************
	/* Front-End Signup
	/**************************************************************/


	/**
	 * Check for newsletter signups on form submission
	 *
	 * @since       1.0
	 * @return      void
	 */
	public function check_for_email_signup() {

		global $ninja_forms_processing;

		// start with the assumption that this isn't an AWeber form
		$aweber_lists = array();

		$form = $ninja_forms_processing->get_all_form_settings();
		$form_id = $ninja_forms_processing->get_form_ID();

		$subscriber = $ninja_forms_processing->get_user_info();

		//Get all the user submitted values
		$all_fields = $ninja_forms_processing->get_all_fields();

		//Make sure $all_fields is an array.
		if ( is_array( $all_fields ) ) { 

			//Loop through each of our submitted values.
			foreach ( $all_fields as $field_id => $value ) {

				$field = $ninja_forms_processing->get_field_settings( $field_id );

				// the opt-in field is checked, so find the related list
				if ( $field['type'] == 'aweber_optin' && isset( $field['data']['aweber_list'] ) && $value == 'checked' ){
					$aweber_lists[] = $field['data']['aweber_list'];
				} 

				// add the IP address to the $subscriber
				if ( $field['type'] == 'ip_address' ){
					$subscriber['ip_address'] = $value;
				}

				// add the custom fields to the $subscriber
				if ( isset( $field['data']['aweber_custom'] ) && ! empty( $field['data']['aweber_custom'] ) ) {
					$custom_field = $field['data']['aweber_custom'];
					$subscriber['custom_fields'][$custom_field] = $value;
				}

			}

		}

		// add a little extra data for AWeber
		$subscriber['form_id'] = $form_id;

		// Check if AWeber is enabled via form settings
		if ( isset( $form['aweber_signup_form'] ) && $form['aweber_signup_form'] == true && isset( $form['aweber_list'] ) ){
			$aweber_lists[] = $form['aweber_list'];
		}
		
		// remove any duplicates in case the form settings list is the same as any optin field
		// otherwise you get an error about a subscriber already being on the list
		$aweber_lists = array_unique( $aweber_lists );

		// sign user up to all lists
		if ( ! empty ( $aweber_lists ) && isset( $subscriber['email'] ) && ! empty( $subscriber['email'] ) ) { 
			foreach( $aweber_lists as $aweber_list ){
				$this->subscribe_email( $subscriber, $aweber_list );
			}
		}

	}



	/**
	 * Subscribe an email address to a AWeber list
	 *
	 * @since       1.0
	 * @return      bool
	 */
	public function subscribe_email( $subscriber = array(), $list_id = '' ) { 

		$defaults = array ( 'consumer_key' => false, 'consumer_secret' => false, 'access_key' => false, 'access_secret' => false );
		$options = get_option( 'ninja_forms_aweber_options', $defaults );
		$options = wp_parse_args( (array) $options, $defaults );

		try {
			$aweber = new AWeberAPI( $options['consumer_key'], $options['consumer_secret'] );
			$account = $aweber->getAccount( $options['access_key'], $options['access_secret'] );

			// direct API URL
			$account_id = $account->id;
            $url = "/accounts/{$account_id}/lists/{$list_id}/subscribers";

			$name = isset( $subscriber['first_name'] ) ? $subscriber['first_name'] : '';

			// add space between first and last names if both exist
			if( isset( $subscriber['first_name'] ) && isset( $subscriber['last_name'] ) ){
				$name .= ' ' . $subscriber['last_name'];
			} else if ( isset( $subscriber['last_name'] ) ){
				$name .= $subscriber['last_name'];
			}

			// get custom fields
			$custom_fields = isset( $subscriber['custom_fields'] ) ? (array) $subscriber['custom_fields'] : array();

			// get IP address
			$ip_address = isset( $subscriber['ip_address'] ) ? $subscriber['ip_address'] : '';

			# create a subscriber
			$params = array(
				'ws.op' => 'create',
				'email' => $subscriber['email'],
				'name' => $name,
				'ip_address' => $ip_address,
				'ad_tracking' => sprintf( __( 'Ninja Forms Form #%s', 'ninja-forms-aweber' ), $subscriber['form_id'] )
		    );

		    if( ! empty( $custom_fields ) ){
		    	$params['custom_fields'] = $custom_fields;
		    }

			// single POST to API
            $data = $aweber->adapter->request( 'POST', $url, $params, array('return' => 'headers') );

		} catch( AWeberException $e ) {
			
			$error_message = $e->getMessage() ? $e->getMessage() : '';
			$error_code = null; // AWeber doesn't have codes yet, but this holds a place for when they do
			
			// process the AWeber message and return something i18n-ready and somewhat more friendly
			$new_message = $this->error_message( $error_message, $error_code, $subscriber, $list_id );
			
			// if we get an exception, generally the user can't be added to the list
			if( $new_message && apply_filters( 'ninja_forms_aweber_show_errors', true, $error_message, $error_code, $subscriber, $list_id ) ){
				global $ninja_forms_processing;
				$ninja_forms_processing->add_error( 'aweber_error', $new_message );
			}
			
		}

	} 


	/**************************************************************
	/* Helper Functions
	/**************************************************************/

	
	/**
	 * Retrieve an array of AWeber lists
	 *
	 * @since       1.0
	 * @return      array
	 */
	public function get_lists() {

		if ( false === ( $lists = get_transient( 'ninja_forms_aweber_lists' ) ) ) {

			// start off with empty lists
			$lists = array( array( 'name' => __( '--select a list--', 'ninja-forms-aweber' ), 'value' => -1 ) );

			$defaults = array ( 'consumer_key' => false, 'consumer_secret' => false, 'access_key' => false, 'access_secret' => false );
			$options = get_option( 'ninja_forms_aweber_options', $defaults );
			$options = wp_parse_args( (array) $options, $defaults );

			if ( $options['consumer_key'] && $options['consumer_secret'] && $options['access_key'] && $options['access_secret'] ) {

				try {
					$aweber = new AWeberAPI( $options['consumer_key'], $options['consumer_secret'] );
					$account = $aweber->getAccount( $options['access_key'], $options['access_secret'] );

					$api_lists = $account->lists;

					if ( ! empty( $api_lists ) ){
						foreach ( $api_lists->data['entries'] as $list ) {

							$lists[] = array(
								'value' => $list['id'],
								'name'  => $list['name']
							);
						}

						// saving the transient here because we want to keep looking
						// for lists until aweber kicks back at least one useful list
						set_transient( 'ninja_forms_aweber_lists', $lists );

					}

				} catch ( AWeberException $e ) {
					// nothing for now
				} 

			} 

		}

		return $lists;
	}

	/**
	 * Create a simple function to delete our transient
	 *
	 * @since       1.3.0
	 */
	public function delete_transient() {
	     delete_transient( 'ninja_forms_aweber_lists' );
	}
	
	
	/**
	 * Create better error messages for AWeber
	 *
	 * @param 		$error_message Message from AWeber
	 * @param 		$error_code placeholder, does not exist yet in AWeber
	 * @param 		$subscriber array of form data submission
	 * @param 		$list_id string AWeber list id
	 * @return 		string
	 * @since       1.3.1
	 */
	public function error_message( $error_message = '', $error_code = '', $subscriber = array(), $list_id = '' ) {
		
		// if in debug mode, return the error from AWeber without modification
		if( ! defined( 'NF_AWEBER_DEBUG' ) || ! NF_AWEBER_DEBUG ){

		    switch( $error_message ){
		    	case strpos( $error_message, 'Consumer key is invalid' ) !== false :
		    		$new_message = __( 'The form could not be submitted because this form is not connected to AWeber. Please enter your authorization code in the plugin settings.', 'ninja-forms-aweber' );
		    		break;
		    	case strpos( $error_message, 'Subscriber already subscribed' ) !== false :
		    		$new_message = __( 'The form could not be submitted because you have already subscribed to this mailing list.', 'ninja-forms-aweber' );
		    		break;
		    	case strpos( $error_message, 'Email address blocked' ) !== false :
		    		$new_message = __( 'The form could not be submitted because you are using an unsupported email address. Please try another.', 'ninja-forms-aweber' );
		    		break;
		    	default:
		    		$new_message = __( 'We\'re sorry, but there has been an error in submitting your form. Please try again later.', 'ninja-forms-aweber' );
		    	
		    } 
		    $error_message = apply_filters( 'ninja_forms_aweber_subscribe_error_message', $new_message, $error_code, $error_message, $subscriber, $list_id );
		}
		return $error_message;
	}	
	
	

} //end class: do not remove or there will be no more guacamole for you

endif; // end class_exists check


/**
 * Returns the main instance of NF_AWeber to prevent the need to use globals.
 *
 * @return WooCommerce
 */
function NF_AWeber() {
	return NF_AWeber::instance();
}

// Launch the whole plugin
NF_AWeber();