<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class NF_Trello_Admin_Menus_Settings
 */
class NF_Trello_Admin_Menus_Settings {

	/**
	 * @var string
	 */
	protected $plugin;

	/**
	 * @var string
	 */
	protected $slug;

	/**
	 * @var string
	 */
	protected $setting_slug;

	/**
	 * @var string|void
	 */
	protected $callback;

	/**
	 * NF_Trello_Settings constructor.
	 */
	public function __construct() {
		$this->plugin       = NF_Trello()->plugin_name;
		$this->slug         = sanitize_title( NF_Trello()->plugin_name );
		$this->setting_slug = sanitize_title( $this->plugin ) . '_settings';
		$this->callback     = admin_url( 'admin.php?page=ninja-forms-' . $this->slug );
	}

	public function init() {
		if ( ! session_id() ) {
			session_start();
		}

		add_action( 'admin_menu', array( $this, 'add_menu' ), 100 );
		add_action( 'admin_init', array( $this, 'trello_connect' ) );
		add_action( 'admin_init', array( $this, 'trello_disconnect' ) );
	}

	/**
	 * Add settings page under NF menu
	 */
	public function add_menu() {
		$capabilities = 'administrator';
		$capabilities = apply_filters( 'ninja_forms_admin_menu_capabilities', $capabilities );

		add_submenu_page( 'ninja-forms', $this->plugin, $this->plugin, $capabilities, 'ninja-forms-' . $this->slug, array( $this, 'render_settings_page' ) );
	}

	/**
	 * Render the settings page
	 */
	public function render_settings_page() {
		$username       = '';
		$connect_url    = '';
		$disconnect_url = '';

		if ( ! ( $connected = $this->connected() ) ) {
			$trello = new trello_oauth();
			$connect_url    = $trello->get_authorise_url( $this->callback );
		} else {
			$username = $this->get_connected_member()->username;
			$disconnect_url = $this->callback . '&action=disconnect';
		}

		NF_Trello()->template( 'settings', compact( 'connected', 'connect_url', 'disconnect_url', 'username' ) );
	}

	/**
	 * Get plugin settings
	 *
	 * @return array|bool
	 */
	protected function get_settings() {
		$options = get_option( 'ninja_forms_settings', array() );

		return isset( $options[ $this->setting_slug ] ) ? $options[ $this->setting_slug ] : false;
	}

	/**
	 * Update settings
	 *
	 * @param bool|false $data
	 */
	protected function update_settings( $data = false ) {
		$options = get_option( 'ninja_forms_settings', array() );

		if ( $data ) {
			$options[ $this->setting_slug ] = $data;
		} else {
			unset( $options[ $this->setting_slug ] );
		}

		update_option( 'ninja_forms_settings', $options );
	}

	/**
	 * Is connected to Trello
	 *
	 * @return bool
	 */
	public function connected() {
		return ( $this->get_settings() ) ? true : false;
	}

	/**
	 * Get connected account
	 *
	 * @return string
	 */
	public function get_connected_member() {
		$trello_data = $this->get_settings();

		return isset( $trello_data['member'] ) ? $trello_data['member'] : '';
	}

	/**
	 * Get access token
	 *
	 * @return bool
	 */
	public function get_access_token() {
		$trello_data = $this->get_settings();

		return isset( $trello_data['token'] ) ? $trello_data['token'] : false;
	}

	/**
	 * Connect to Trello
	 */
	public function trello_connect() {
		if ( ! isset( $_GET['page'] ) || 'ninja-forms-' . $this->slug !== $_GET['page'] ) {
			return;
		}

		if ( ! isset( $_GET['oauth_token'] ) || ! isset( $_GET['oauth_verifier'] ) ) {
			return;
		}

		$auth_token        = $_SESSION['trello_oauth_token'];
		$auth_token_secret = $_SESSION['trello_oauth_token_secret'];
		$request_code      = $_GET['oauth_verifier'];

		$trello                = new trello_oauth( $auth_token, $auth_token_secret );
		$token                 = $trello->getAccessToken( $request_code, $this->callback );
		$member                = $trello->getMember();
		$trello_data['token']  = $token;
		$trello_data['member'] = $member;

		$this->update_settings( $trello_data );

		wp_redirect( $this->callback );
	}

	/**
	 * Disconnect from Trello
	 */
	public function trello_disconnect() {
		if ( ! isset( $_GET['page'] ) || 'ninja-forms-' . $this->slug !== $_GET['page'] ) {
			return;
		}

		if ( ! isset( $_GET['action'] ) || 'disconnect' !== $_GET['action'] ) {
			return;
		}


		$this->update_settings();
		wp_redirect( $this->callback );
	}

}