<?php

class ninja_form_trello {

	private $plugin;
	private $slug;
	private $setting_slug;
	private $callback;

	private $trello;
	private $form;


	function __construct( $plugin_name ) {
		$this->plugin       = $plugin_name;
		$this->slug         = sanitize_title( $this->plugin );
		$this->setting_slug = sanitize_title( $this->plugin ) . '_settings';
		$this->callback     = admin_url( 'admin.php?page=ninja-forms-' . $this->slug );

		if ( ! session_id() ) {
			session_start();
		}

		add_action( 'admin_menu', array( $this, 'add_menu' ), 100 );
		add_action( 'admin_init', array( $this, 'trello_connect' ) );
		add_action( 'admin_init', array( $this, 'trello_disconnect' ) );

		add_action( 'admin_init', array( $this, 'register_form_settings' ), 11 );
		add_action( 'ninja_forms_edit_field_after_registered', array( $this, 'card_inputs' ), 12 );
		add_action( 'ninja_forms_edit_field_after_registered', array( $this, 'card_attachments' ), 12 );
		add_action( 'ninja_forms_post_process', array( $this, 'process_trello' ), 20 );

		add_action( 'admin_enqueue_scripts', array( $this, 'admin_assets' ) );

		add_action( 'wp_ajax_nf_trello_get_objects', array( $this, 'nf_trello_get_objects' ) );
	}

	function admin_assets() {

		if ( isset( $_GET['page'] ) && $_GET['page'] == 'ninja-forms' ) {

			wp_enqueue_script(
				'ninja-forms-trello',
				NINJA_FORMS_TRELLO_URL . '/assets/js/deprecated/ninja-forms-trello.js',
				array( 'jquery' )
			);

		}
	}

	public function process_trello() {
		global $ninja_forms_processing;
		$form_id = $ninja_forms_processing->get_form_ID();
		$form    = ninja_forms_get_form_by_id( $form_id );

		if ( ! ( $list = $this->form_connected( $form ) ) ) {
			return;
		}

		$fields = ninja_forms_get_fields_by_form_id( $form_id );

		$card_name    = array();
		$card_desc    = array();
		$card_pos     = $this->saved_data( 'card_position', $form );
		$card_labels  = $this->saved_data( 'card_labels', $form );
		$card_members = $this->saved_data( 'card_members', $form );
		$card_link    = array();
		$card_files   = array();

		foreach ( $fields as $field ) {
			if ( isset( $field['data']['trello_card_type'] ) ) {
				$card_type = $field['data']['trello_card_type'];
				if ( in_array( $field['type'], $this->allowed_input_types() ) ) {
					$value = $ninja_forms_processing->get_field_value( $field['id'] );
					if ( trim( $value ) != '' ) {
						switch ( $card_type ) {
							case 'title':
								$card_name[] = $value;
								break;
							case 'descp':
								$card_desc[] = $value;
								break;
							case 'link':
								$card_link[] = $value;
								break;
						}
					}
				}
			}
		}

		$card_files = $this->get_uploaded_files();

		$this->create_card( $list, $card_name, $card_desc, $card_pos, $card_labels, $card_members, $card_link, $card_files );
	}

	private function get_uploaded_files() {
		global $ninja_forms_processing;
		$card_files = array();
		if ( ! function_exists( 'ninja_forms_uploads_setup_license' ) ) {
			return $card_files;
		}
		if ( $ninja_forms_processing->get_form_setting( 'create_post' ) != 1 ) {
			if ( ( $uploads = $ninja_forms_processing->get_extra_value( 'uploads' ) ) ) {
				foreach ( $uploads as $field_id ) {
					$field_row = $ninja_forms_processing->get_field_settings( $field_id );
					if ( in_array( $field_row['type'], $this->allowed_file_types() ) ) {
						if ( isset( $field_row['data']['trello_card_attach'] ) && $field_row['data']['trello_card_attach'] == 1 ) {
							$user_value = $ninja_forms_processing->get_field_value( $field_id );
							if ( isset( $field_row['data']['upload_location'] ) ) {
								if ( is_array( $user_value ) ) {
									foreach ( $user_value as $key => $file ) {
										if ( ! isset( $file['file_path'] ) ) {
											continue;
										}
										$filename = $file['file_path'] . $file['file_name'];
										if ( file_exists( $filename ) ) {
											$card_files[] = $file;
										}
									}
								}
							}
						}
					}
				}
			}
		}

		return $card_files;
	}

	private function create_card( $list, $card_name, $card_desc, $pos, $labels = false, $members = false, $card_link = array(), $card_files = array() ) {
		$name = implode( apply_filters( 'nf_trello_name_sep', ' ' ), $card_name );
		$desc = implode( apply_filters( 'nf_trello_desc_sep', ' ' ), $card_desc );

		if ( $name == '' ) {
			return;
		}

		$access_token = $this->get_access_token();
		if ( ! $access_token ) {
			return;
		}
		$params = array(
			'name'   => $name,
			'desc'   => $desc,
			'pos'    => $pos,
			'idList' => $list
		);
		if ( is_array( $labels ) && count( $labels ) > 0 ) {
			$params['labels'] = implode( ',', $labels );
		}
		if ( is_array( $members ) && count( $members ) > 0 ) {
			$params['idMembers'] = implode( ',', $members );
		}

		$this->trello = new trello_oauth( $access_token['oauth_token'], $access_token['oauth_token_secret'] );
		// Create card
		$card = $this->trello->postCard( $params );

		// Attachment links
		if ( is_array( $card_link ) && count( $labels ) > 0 ) {
			foreach ( $card_link as $link ) {
				$this->trello->postCardAttachment( $card->id, array( 'url' => $link ) );
			}
		}

		// Attachment File uploads
		if ( is_array( $card_files ) && count( $card_files ) > 0 ) {
			foreach ( $card_files as $file ) {
				$file_path = $file['file_path'] . $file['file_name'];
				$this->trello->uploadAttachment( $card->id, $file_path );
			}
		}
	}

	public function register_form_settings() {
		if ( isset( $_GET['page'] ) && $_GET['page'] == 'ninja-forms' && isset( $_GET['form_id'] ) ) {

			if ( ! $this->connected() ) {
				return;
			}

			$access_token = $this->get_access_token();
			if ( ! $access_token ) {
				return;
			}

			$form_id    = $_GET['form_id'];
			$this->form = ninja_forms_get_form_by_id( $form_id );

			$form_trello_settings = array();

			$this->trello = new trello_oauth( $access_token['oauth_token'], $access_token['oauth_token_secret'] );
			$orgs         = $this->trello->getOrganizations();
			$choices      = $this->get_choices( $orgs, 'organization' );

			$form_trello_settings[] = array(
				'name'    => 'trello_orgs',
				'type'    => 'select',
				'class'   => 'trello-orgs',
				'label'   => __( 'Organization', 'ninja-forms-trello' ),
				'options' => $choices
			);

			$org_id = $this->saved_data( 'trello_orgs' );
			$boards = array( array( 'name' => 'Select Board', 'value' => 0 ) );
			if ( $org_id ) {
				$boards = $this->trello->getBoards( $org_id );
				$boards = $this->get_choices( $boards, 'board' );
			}

			$form_trello_settings[] = array(
				'name'    => 'trello_boards',
				'type'    => 'select',
				'class'   => 'trello-boards',
				'label'   => __( 'Board', 'ninja-forms-trello' ),
				'options' => $boards
			);

			$board_id = $this->saved_data( 'trello_boards' );
			$lists    = array( array( 'name' => 'Select List', 'value' => 0 ) );
			if ( $board_id ) {
				$lists = $this->trello->getLists( $board_id );
				$lists = $this->get_choices( $lists, 'list' );
			}

			$form_trello_settings[] = array(
				'name'    => 'trello_lists',
				'type'    => 'select',
				'class'   => 'trello-lists',
				'label'   => __( 'List', 'ninja-forms-trello' ),
				'options' => $lists
			);

			$positions              = array(
				'bottom' => 'Bottom',
				'top'    => 'Top'
			);
			$form_trello_settings[] = array(
				'name'          => 'card_position',
				'type'          => 'select',
				'class'         => '',
				'default_value' => 'bottom',
				'label'         => __( 'Card Position on List', 'ninja-forms-trello' ),
				'options'       => $this->convert_choices( $positions )
			);

			$labels                 = array(
				'green'  => 'Green',
				'yellow' => 'Yellow',
				'orange' => 'Orange',
				'red'    => 'Red',
				'purple' => 'Purple',
				'blue'   => 'Blue'

			);
			$form_trello_settings[] = array(
				'name'    => 'card_labels',
				'type'    => 'checkbox_list',
				'class'   => '',
				'label'   => __( 'Card Labels', 'ninja-forms-trello' ),
				'options' => $this->convert_choices( $labels )
			);

			if ( $board_id ) {
				$members                = $this->trello->getBoardMembers( $board_id );
				$members                = $this->get_choices( $members, 'member', false );
				$form_trello_settings[] = array(
					'name'    => 'card_members',
					'type'    => 'checkbox_list',
					'class'   => '',
					'label'   => __( 'Card Members', 'ninja-forms-trello' ),
					'options' => $members
				);

			}

			$args = array(
				'page'             => 'ninja-forms',
				'tab'              => 'form_settings',
				'slug'             => 'form_trello',
				'title'            => __( 'Trello', 'ninja-forms-trello' ),
				'display_function' => '',
				'settings'         => $form_trello_settings
			);

			ninja_forms_register_tab_metabox( $args );
		}
	}

	private function init_field_settings( $field_id ) {
		$field_row = ninja_forms_get_field_by_id( $field_id );
		$form      = ninja_forms_get_form_by_id( $field_row['form_id'] );

		if ( ! $this->form_connected( $form ) ) {
			return false;
		}

		return $field_row;
	}

	private function form_connected( $form ) {
		$list_id = $this->saved_data( 'trello_lists', $form );

		return ( ! $list_id ) ? false : $list_id;
	}

	private function allowed_input_types() {
		return apply_filters( 'nf_input_trello_types', array( '_text', '_textarea' ) );
	}

	private function allowed_file_types() {
		return apply_filters( 'nf_file_trello_types', array( '_upload' ) );
	}

	public function card_inputs( $field_id ) {
		$field_row = $this->init_field_settings( $field_id );
		if ( ! $field_row ) {
			return;
		}

		if ( ! in_array( $field_row['type'], $this->allowed_input_types() ) ) {
			return false;
		}

		$field_data = $field_row['data'];
		if ( isset ( $field_data['trello_card_type'] ) ) {
			$trello_card_type = $field_data['trello_card_type'];
		} else {
			$trello_card_type = '';
		}

		$desc = '<hr><strong>' . __( 'Trello', 'ninja-forms-trello' ) . '</strong>';
		ninja_forms_edit_field_el_output( $field_id, 'desc', $desc );
		$types = array(
			array(
				'name'  => __( 'Not used', 'ninja-forms-trello' ),
				'value' => ''
			),
			array(
				'name'  => __( 'Title', 'ninja-forms-trello' ),
				'value' => 'title'
			),
			array(
				'name'  => __( 'Description', 'ninja-forms-trello' ),
				'value' => 'descp'
			)
		);

		if ( $field_row['type'] == '_text' ) {
			$types[] = array(
				'name'  => __( 'Attachment Link', 'ninja-forms-trello' ),
				'value' => 'link'
			);
		}
		ninja_forms_edit_field_el_output(
			$field_id, 'select', __( 'Use as Card Data', 'ninja-forms-trello' ), 'trello_card_type', $trello_card_type, 'thin', $types, 'widefat'
		);
	}

	public function card_attachments( $field_id ) {
		// check file uploads installed and activated
		if ( ! class_exists( 'NF_FU_File_Uploads' ) && ! function_exists( 'ninja_forms_uploads_setup_license' ) ) {
			return;
		}

		$field_row = $this->init_field_settings( $field_id );
		if ( ! $field_row ) {
			return;
		}

		if ( ! in_array( $field_row['type'], $this->allowed_file_types() ) ) {
			return false;
		}

		$field_data = $field_row['data'];
		if ( isset ( $field_data['trello_card_attach'] ) ) {
			$trello_card_type = $field_data['trello_card_attach'];
		} else {
			$trello_card_type = '';
		}

		$desc = '<hr><strong>' . __( 'Trello', 'ninja-forms-trello' ) . '</strong>';
		ninja_forms_edit_field_el_output( $field_id, 'desc', $desc );
		ninja_forms_edit_field_el_output(
			$field_id, 'checkbox', __( 'Attach to Card', 'ninja-forms-trello' ), 'trello_card_attach', $trello_card_type
		);
	}

	public function add_menu() {
		$capabilities = 'administrator';
		$capabilities = apply_filters( 'ninja_forms_admin_menu_capabilities', $capabilities );

		add_submenu_page(
			'ninja-forms', $this->plugin, $this->plugin, $capabilities, 'ninja-forms-' . $this->slug, array(
				$this,
				'settings_page'
			)
		);
	}

	public function settings_page() {
		if ( ! ( $connected = $this->connected() ) ) {
			$trello = new trello_oauth();
			$url    = $trello->get_authorise_url( $this->callback );
		}
		?>
		<div class="wrap">
			<h2><?php _e( 'Trello Settings', 'ninja-forms-trello' ) ?></h2>

			<p>
				<?php if (! $connected) : ?>
					<a class="button button-primary" href=<?php echo $url; ?>>Connect with Trello</a>
				<?php else : ?>

			<p>
				Connected as <code><?php echo $this->get_connected_member()->username; ?></code>
			</p>
			<a class="button" href=<?php echo $this->callback . '&action=disconnect'; ?>>Disconnect</a>
			<?php endif; ?>
			</p>
		</div>
	<?php
	}

	public function trello_connect() {
		if ( isset( $_GET['page'] ) && $_GET['page'] == 'ninja-forms-' . $this->slug && isset( $_GET['oauth_token'] ) && isset( $_GET['oauth_verifier'] ) ) {
			$auth_token        = $_SESSION['trello_oauth_token'];
			$auth_token_secret = $_SESSION['trello_oauth_token_secret'];
			$request_code      = $_GET['oauth_verifier'];

			$trello                = new trello_oauth( $auth_token, $auth_token_secret );
			$token                 = $trello->getAccessToken( $request_code, $this->callback );
			$member                = $trello->getMember();
			$trello_data['token']  = $token;
			$trello_data['member'] = $member;

			$this->update_option( $trello_data );
			header( 'Location: ' . $this->callback );
		}
	}

	public function trello_disconnect() {
		if ( isset( $_GET['page'] ) && $_GET['page'] == 'ninja-forms-' . $this->slug && isset( $_GET['action'] ) && $_GET['action'] == 'disconnect' ) {
			$this->update_option();
			header( 'Location: ' . $this->callback );
		}
	}

	public function nf_trello_get_objects() {
		if ( ! isset( $_POST['type'] ) || ! isset( $_POST['id'] ) ) {
			return 0;
		}
		$response['error']   = false;
		$response['message'] = '';
		$objects             = $this->get_dropdown_data( $_POST['type'], $_POST['id'] );
		if ( $objects ) {
			$response['objects'] = $objects;
			$response['message'] = 'success';
		} else {
			$response['error']   = true;
			$response['message'] = 'The was an error retrieving data from Trello';
		}
		echo json_encode( $response );
		die;
	}

	public function get_dropdown_data( $object, $id ) {
		$data         = false;
		$access_token = $this->get_access_token();
		if ( $access_token ) {
			$trello = new trello_oauth( $access_token['oauth_token'], $access_token['oauth_token_secret'] );
			$method = 'get' . ucfirst( $object );
			$data   = call_user_func( array( $trello, $method ), $id );
			$data   = $trello->getDropdown( $data, substr( $object, 0, - 1 ) );
		}

		return $data;
	}

	private function get_option() {
		$options = get_option( 'ninja_forms_settings' );

		return isset( $options[$this->setting_slug] ) ? $options[$this->setting_slug] : false;
	}

	private function update_option( $data = false ) {
		$options = get_option( 'ninja_forms_settings' );
		if ( $data ) {
			$options[$this->setting_slug] = $data;
		} else {
			unset( $options[$this->setting_slug] );
		}
		update_option( 'ninja_forms_settings', $options );
	}

	public function connected() {
		return ( $this->get_option() ) ? true : false;
	}

	public function get_connected_member() {
		$trello_data = $this->get_option();

		return isset( $trello_data['member'] ) ? $trello_data['member'] : '';
	}

	public function get_access_token() {
		$trello_data = $this->get_option();

		return isset( $trello_data['token'] ) ? $trello_data['token'] : false;
	}

	private function convert_choices( $choices ) {
		$new = array();
		foreach ( $choices as $key => $value ) {
			$new[] = array( 'name' => $value, 'value' => $key );
		}

		return $new;
	}

	private function get_choices( $choices, $type = 'organization', $select = true ) {
		$choices = $this->trello->getDropdown( $choices, $type, $select );

		return $this->convert_choices( $choices );
	}

	private function saved_data( $key, $form = false ) {
		if ( ! $form && $this->form ) {
			$form = $this->form;
		}
		$data = false;
		if ( isset( $form['data'][$key] ) ) {
			if ( $form['data'][$key] != '' && $form['data'][$key] != '0' ) {
				$data = $form['data'][$key];
			}
		}

		return $data;
	}
}