<?php if ( ! defined( 'ABSPATH' ) || ! class_exists( 'NF_Abstracts_Action' )) exit;

/**
 * Class NF_Action_RecurlyExample
 */
final class NF_Recurly_Actions_Subscribe extends NF_Abstracts_Action
{
    /**
     * @var string
     */
    protected $_name  = 'recurly_subscribe';

    /**
     * @var array
     */
    protected $_tags = array();

    /**
     * @var string
     */
    protected $_timing = 'normal';

    /**
     * @var int
     */
    protected $_priority = '10';

    /**
     * @var forms
     */
    protected $forms = array();

    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct();

        $this->_nicename = __( 'Recurly', 'ninja-forms-recurly' );

        $settings = NF_Recurly::config( 'ActionSubscribeSettings' );

        $this->_settings = array_merge( $this->_settings, $settings );

        add_action( 'ninja_forms_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
    }

    /*
    * PUBLIC METHODS
    */

    public function save( $action_settings )
    {

    }

    public function process( $action_settings, $form_id, $data )
    {
        // Check for Recurly Token.
        if( ! isset( $data[ 'extra' ][ 'recurly_token' ] ) || empty( $data[ 'extra' ][ 'recurly_token' ] ) ) {
            return $data;
        }

        /*
         * If we don't have a plan, return early.
         */
        if( ! isset( $action_settings[ 'recurly_plan' ] ) || empty( $action_settings[ 'recurly_plan' ] ) ) {
            return $data;
        }

        /*
         * If we have mapped data for first_name, last_name, and/or email, use $action_settings[ $setting ].
         * Otherwise, use $data[ 'extra' ][ $setting ]
         */
        $first_name = empty ( $action_settings[ 'recurly_first_name' ] ) ? $data[ 'extra' ][ 'recurly_first_name' ] : $action_settings[ 'recurly_first_name' ];
        $last_name = empty ( $action_settings[ 'recurly_last_name' ] ) ? $data[ 'extra' ][ 'recurly_last_name' ] : $action_settings[ 'recurly_last_name' ];
        $email = empty ( $action_settings[ 'recurly_email' ] ) ? $data[ 'extra' ][ 'recurly_email' ] : $action_settings[ 'recurly_email' ];

        /*
         * If we have mapped data for the account code, use $account_settings[ 'recurly_account_code' ].
         * If we don't, create a string using $first_name and $last_name.
         */
        if ( empty ( trim( $action_settings[ 'recurly_account_code' ] ) ) ) {
            $account_code = esc_html( strtolower( $first_name . '_' . $last_name ) );
        } else {
            $account_code = trim( $action_settings[ 'recurly_account_code' ] );
        }

        $subscription = new Recurly_Subscription();
        $subscription->plan_code = $action_settings[ 'recurly_plan' ];

        $subscription->account = new Recurly_Account();
        $subscription->currency = $this->get_currency( $data );

        /*
         * Required fields for creating a subscription.
         */
        $subscription->account->account_code = $account_code;
        $subscription->account->first_name = $first_name;
        $subscription->account->last_name = $last_name;
        $subscription->account->email = $email;

        /*
         * If we have address info, send that information.
         */

        if ( ! empty ( $action_settings[ 'recurly_address1' ] ) ) {
            $subscription->account->address->address1 = $action_settings[ 'recurly_address1' ];
        }

        if ( ! empty ( $action_settings[ 'recurly_address2' ] ) ) {
            $subscription->account->address->address2 = $action_settings[ 'recurly_address2' ];
        }

        if ( ! empty ( $action_settings[ 'recurly_city' ] ) ) {
            $subscription->account->address->city = $action_settings[ 'recurly_city' ];
        }

        if ( ! empty ( $action_settings[ 'recurly_state' ] ) ) {
            $subscription->account->address->state = $action_settings[ 'recurly_state' ];
        }

        if ( ! empty ( $action_settings[ 'recurly_zip' ] ) ) {
            $subscription->account->address->zip = $action_settings[ 'recurly_zip' ];
        }

        if ( ! empty ( $action_settings[ 'recurly_country' ] ) ) {
            $subscription->account->address->country = $action_settings[ 'recurly_country' ];
        }

        if ( ! empty ( $action_settings[ 'recurly_phone' ] ) ) {
            $subscription->account->address->phone = $action_settings[ 'recurly_phone' ];
        }

        /*
         * End of address information.
         */

        /*
         * Misc. information
         */
        $username = empty ( $action_settings[ 'recurly_username' ] ) ? $email : $account_settings[ 'recurly_username' ];
        $subscription->account->username = $username;

        if ( ! empty ( $action_settings[ 'recurly_company_name' ] ) ) {
            $subscription->account->company_name = $action_settings[ 'recurly_company_name' ];
        }

        if ( ! empty ( $action_settings[ 'recurly_vat_number' ] ) ) {
            $subscription->account->vat_number = $action_settings[ 'recurly_vat_number' ];
        }

        if ( ! empty ( $action_settings[ 'recurly_tax_exempt' ] ) ) {
            $subscription->account->tax_exempt = $action_settings[ 'recurly_tax_exempt' ];
        }

        /*
         * Create our subscription.
         */

        $subscription->account->billing_info = new Recurly_BillingInfo();
        $subscription->account->billing_info->token_id = $data[ 'extra' ][ 'recurly_token' ];

        try {
            $subscription->create();

            $data[ 'extra' ][ '_recurly_uuid' ] = $subscription->uuid;

            unset( $data[ 'extra' ][ 'recurly_token' ] );

        } catch ( Exception $e)  {
            /*
             * Check if this user is already subscribed.
             */
            if ( $e->errors[0] instanceof Recurly_FieldError && 'already_subscribed' == $e->errors[0]->symbol ) {
                $form_error = array( 'code' => $e->errors[0]->symbol, 'msg' => $e->errors[0]->description );
            } else {
                // Exception not specifically handled.
                if( is_user_logged_in() && current_user_can( 'manage_options' ) ) {
                    $form_error = array( 'code' => $e->getCode(), 'msg' => $e->getMessage() );
                }  else {
                    $form_error = array( 'code' => '', 'msg' => __( 'Could not create subscription. Please contact the site administrator.', 'ninja-forms-recurly' ) );
                }
            }

            $data[ 'errors' ][ 'form' ] = $form_error;
        }

        return $data;
    }

    public function enqueue_scripts( $data )
    {

        $recurly_actions = $this->get_active_recurly_actions( $data[ 'form_id' ] );
        if ( empty( $recurly_actions ) ) return;

        wp_enqueue_script( 'recurly', 'https://js.recurly.com/v4/recurly.js', array( 'jquery' ) );
        wp_enqueue_script( 'card', NF_Recurly::$url . 'assets/js/card.js', array( 'nf-front-end' ) );
        wp_enqueue_script( 'nf-recurly', NF_Recurly::$url . 'assets/js/recurly.js', array( 'recurly', 'nf-front-end' ) );

        wp_enqueue_style( 'jBox', Ninja_Forms::$url . 'assets/css/jBox.css' );
        wp_enqueue_style( 'payment-font', NF_Recurly::$url . 'assets/css/paymentfont.min.css' );
        wp_enqueue_script( 'jBox', Ninja_Forms::$url . 'assets/js/lib/jBox.min.js' );

        array_push( $this->forms, array( 'id' => $data[ 'form_id' ], 'actions' => $recurly_actions ) );

        wp_localize_script( 'nf-recurly', 'nfRecurly',
            array(
                'forms' => $this->forms, // Unique array of form IDs.
                'publicKey' => Ninja_Forms()->get_setting( 'recurly_public_key' )
            )
        );

        $this->output_form( $data[ 'form_id' ] );
    }

    public function output_form( $form_id ) {
        $recurly_actions = $this->get_active_recurly_actions( $form_id );
        if ( empty( $recurly_actions ) ) return;

        NF_Recurly::template( 'credit-card-form.html.php' );
    }

    private function get_active_recurly_actions( $form_id )
    {
        $form_actions = Ninja_Forms()->form( $form_id )->get_actions();
        $recurly_actions = array();

        foreach( $form_actions as $action ){
            if( 'recurly_subscribe' != $action->get_setting( 'type' ) ) continue;
            if ( ! $action->get_setting( 'active' ) ) continue;
            $recurly_actions[] = $action->get_id();
        }
        return $recurly_actions;
    }

    private function get_currency( $form_data )
    {
        /**
         * Currency Setting Priority
         * 2. Ninja Forms Currency Setting
         * 1. Form Currency Setting (default)
         */
        $plugin_currency = Ninja_Forms()->get_setting( 'currency', $stripe_currency );
        $form_currency   = ( isset( $form_data[ 'settings' ][ 'currency' ] ) && $form_data[ 'settings' ][ 'currency' ] ) ? $form_data[ 'settings' ][ 'currency' ] : $plugin_currency;
        return $form_currency;
    }
}
