<?php

if ( !defined( 'ABSPATH' ) || !class_exists( 'NF_Abstracts_Action' ) )
    exit;

/**
 * Class NF_Action_InsightlyCRMExample
 */
final class NF_InsightlyCRM_Actions_AddToInsightly extends NF_Abstracts_Action {

    /**
     * @var string
     */
    protected $_name = 'addtoinsightly'; // child CRM

    /**
     * @var array
     */
    protected $_tags = array();

    /**
     * @var string
     */
    protected $_timing = 'normal';

    /**
     * @var int
     */
    protected $_priority = '10';

    /**
     * The field data from the form submission needed for building the request
     * @var array
     */
    protected $fields_to_extract;

    /**
     * The lookup array that converts nice valule of the field map to the
     * period delimited array used for mapping
     * @var array
     */
    protected $field_map_lookup;

    /**
     * Lookup array for period-delimited mapping instructions for custom fields
     * @var array
     */
    protected $custom_field_entities;

    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct();

        $this->_nicename = __( 'Insightly', 'ninja-forms' );

        $this->field_map_lookup = NF_InsightlyCRM::config( 'InsightlyFields' );

        $this->custom_field_entities = NF_InsightlyCRM::config( 'CustomFieldEntities' );

        add_action( 'admin_init', array( $this, 'init_settings' ) );

        add_action( 'ninja_forms_builder_templates', array( $this, 'builder_templates' ) );

        add_action( 'nf_admin_enqueue_scripts', array( $this, 'enqueueAdminScripts' ) );
    }

    public function enqueueAdminScripts( $data ) {

        wp_enqueue_script( 'nf-insightly-metadata', NF_InsightlyCRM::$url . 'assets/js/metadata.js', array( 'nf-builder' ) );
    }

    public function save( $action_settings ) {

        return $action_settings;
    }

    public function process( $action_settings, $form_id, $data ) {

        global $nfinsightlycrm_settings;

        global $nfinsightlycrm_comm_data;

        $apikey = $nfinsightlycrm_settings[ 'nfinsightlycrm_apikey' ];

        $this->extract_field_data( $action_settings );

        if ( class_exists( 'InsightlyPlusRawRequest' ) ) {

            $request_object = new InsightlyPlusRawRequest();
        } else {

            $request_object = new InsightlyRawRequest();
        }

        if ( class_exists( 'InsightlyPlusCommunication' ) ) {

            $ico = new InsightlyPlusCommunication();
        } else {
            $ico = new InsightlyCommunication();
        }


        $ico->set_apikey( $apikey );

        foreach ( $this->request_array as $field_to_map ) {

            $form_value = $field_to_map[ 'form_field' ];

            unset( $field_to_map[ 'form_field' ] );

            $request_object->map_field_data( $form_value, $field_to_map );
        }


        // indicate end of form data to finish formatting request object
        $request_object->finalize_raw_request();

        $raw_request = $request_object->get_raw_request_array();

        $ico->process_form_request( $raw_request );

        $status_update = $ico->get_status_update();

        $raw_response = $ico->get_raw_response();

        $finalized_request = $ico->get_raw_request_array(); // the request array after additions made during communication

        $request_arrays = array(
            'before' => $raw_request,
            'finalized' => $finalized_request
        );

        $nfinsightlycrm_comm_data[ 'nfinsightlycrm_comm_status' ] = $status_update;

        $nfinsightlycrm_comm_data[ 'nfinsightlycrm_most_recent_raw_request' ] = serialize( $request_arrays );

        $nfinsightlycrm_comm_data[ 'nfinsightlycrm_most_recent_raw_response' ] = serialize( $raw_response );

        update_option( 'nfinsightlycrm_comm_data', $nfinsightlycrm_comm_data );

        $new_record_ids = $ico->get_newly_created_id();

        /*
         * Make the new IDs available for other uses
         */
        do_action( 'nfinsightlycrm_process_new_record_ids', $new_record_ids );

        return $data;
    }

    public function builder_templates() {

        NF_InsightlyCRM::template( 'custom-field-map-row.html' );
    }

    public function init_settings() {

        $settings = NF_InsightlyCRM::config( 'ActionFieldMapSettings' );

        $this->_settings = array_merge( $this->_settings, $settings );

        $standard_insightly_field_lookup = NF_InsightlyCRM::config( 'InsightlyFields' );

        $standard_insightly_fields = $this->buildFieldMapDropdown( $standard_insightly_field_lookup );

        $custom_insightly_fields = $this->build_custom_fields_map();

        $insightly_fields = array_merge( $standard_insightly_fields, $custom_insightly_fields );

        $this->_settings[ 'field_map' ][ 'columns' ][ 'field_map' ][ 'options' ] = $insightly_fields;

        $entry_type = NF_InsightlyCRM::config( 'EntryType' );

        $this->_settings[ 'field_map' ][ 'columns' ][ 'entry_type' ][ 'options' ] = $entry_type;

        $this->fields_to_extract = NF_InsightlyCRM::config( 'FieldsToExtract' );
    }

    protected function build_custom_fields_map() {

        global $nfinsightlycrm_crm_account_data;

        $custom_field_array = array();

        if ( isset( $nfinsightlycrm_crm_account_data[ 'nfinsightlycrm_custom_field_list' ] ) ) {

            $temp_array = unserialize( $nfinsightlycrm_crm_account_data[ 'nfinsightlycrm_custom_field_list' ] );

            if ( !empty( $temp_array ) ) {

                foreach ( $temp_array as $value => $label ) {

                    $structured_value = $this->buildFieldMapValue( $value );

                    $custom_field_array[] = array(
                        'label' => $label,
                        'value' => $structured_value,
                    );
                }
            }
        }

        return $custom_field_array;
    }

    /**
     * Given field map value, returns period-delimited value with object
     *
     * @param string $value
     * @return string
     */
    protected function buildFieldMapValue( $value ) {

        $outgoing_value = $value; // initialize

        $lookup_array = $this->custom_field_entities;

        // look for custom field value prefix
        foreach ( $lookup_array as $needle => $instructions ) {

            if ( 0 !== strpos( $value, $needle ) ) {
                continue;
            }

            // strip prefix if added solely for object identification
            if ( $instructions[ 'strip' ] ) {

                $wip = str_replace( $needle, '', $value );
            } else {

                $wip = $value;
            }

            $outgoing_value = $instructions[ 'prefix' ] . $wip;
        }

        return $outgoing_value;
    }

    protected function extract_field_data( $action_settings ) {

        $this->request_array = array();  // initialize

        $insightly_field_map_data = $action_settings[ 'insightly_field_map' ]; //grab each mapped field in the repeater

        if ( !is_array( $insightly_field_map_data ) ) {
            return; // stop if no array
        }

        foreach ( $insightly_field_map_data as $field_data ) {// cycle through each mapped field
            $map_args = array();

            foreach ( $this->fields_to_extract as $field_to_extract ) { // cycle through each column in the repeater
                if ( isset( $field_data[ $field_to_extract ] ) ) {

                    $value = $field_data[ $field_to_extract ];

                    // for the field map, replace the human readable version with the coded version
                    // custom fields are kept as is
                    if ( 'field_map' == $field_to_extract && isset( $this->field_map_lookup[ $value ] ) ) {

                        $value = $this->field_map_lookup[ $value ][ 'map_instructions' ];
                    }

                    $map_args[ $field_to_extract ] = $value;
                }
            }

            $this->request_array[] = $map_args;
        }
    }

    /**
     * Build the dropdown array of standard field mapping
     *
     * Given an indexed array of label-value arrays.  Label is human readable,
     * value is period-delimited map instructions
     *
     * @param type $field_map_array
     * @return array
     */
    protected function buildFieldMapDropdown( $field_map_array ) {

        $dropdown_array = array();

        foreach ( $field_map_array as $key => $label_map_array ) {

            $dropdown_array[] = array(
                'label' => $label_map_array[ 'label' ],
                'value' => $key,
            );
        }

        return $dropdown_array;
    }

}
