<?php

/* --- NINJA FORMS INSIGHTLY CRM INTEGRATION --- */


/* ----------
  Class InsightlyCommunication_v2_9
  ------------------------------------------------------------------------------------------------------------ */

class InsightlyCommunication_v2_9 {

    protected $apikey;
    protected $raw_request_array;
    protected $base_url;
    protected $raw_response;
    protected $processed_response;
    protected $status_update;
    protected $json_module_request_array;
    
    /**
     *
     * @var array IDs of the newly created entries
     */
    protected $newly_created_id;

    /**
     *
     * @var string Insightly response for attempt at API connection
     * 
     * @since 3.0
     */
    protected $api_connection_verification;

    function __construct() {

        $this->build_base_url();

        $this->apikey = '';

        $this->raw_response = array();  //initialize empty array
        $this->status_update = '';  //initialize empty value
        $this->processed_response = '';
    }

    /* Public Methods
      -------------------- */
    public function process_form_request( $raw_request_array ) {

        $this->raw_request_array = $raw_request_array;

        // ORGANIZATIONS
        if ( isset( $raw_request_array[ 'nforganizations' ] ) ) {

            $this->json_module_request_array[ 'create_organization' ] = json_encode( $this->raw_request_array[ 'nforganizations' ] );

            $this->create_new_module_entry( 'organization' );

            $this->link_organization_to_contact();
        }

        // OPPORTUNITIES
        if ( isset( $raw_request_array[ 'nfopportunities' ] ) ) {

            $this->json_module_request_array[ 'create_opportunity' ] = json_encode( $this->raw_request_array[ 'nfopportunities' ] );

            $this->create_new_module_entry( 'opportunity' );

            $this->link_opportunity_to_contact();
        }

        // PROJECTS
        if ( isset( $raw_request_array[ 'nfprojects' ] ) ) {

            $this->json_module_request_array[ 'create_project' ] = json_encode( $this->raw_request_array[ 'nfprojects' ] );

            $this->create_new_module_entry( 'project' );

            $this->link_project_to_contact();
        }

        // CONTACTS
        if ( isset( $raw_request_array[ 'nfcontacts' ] ) ) {
            $this->finalize_contact_links_array();

            $this->json_module_request_array[ 'create_contact' ] = json_encode( $this->raw_request_array[ 'nfcontacts' ] );

            $this->create_new_module_entry( 'contact' );
        }

        // LEADS

        if ( isset( $raw_request_array[ 'nfleads' ] ) ) {
            $this->json_module_request_array[ 'create_lead' ] = json_encode( $this->raw_request_array[ 'nfleads' ] );

            $this->create_new_module_entry( 'lead' );
        }
        // NOTES
        if ( isset( $raw_request_array[ 'nfnotes' ] ) ) {

            $this->link_person_to_note();

            $this->json_module_request_array[ 'create_note' ] = json_encode( $this->raw_request_array[ 'nfnotes' ] );

            $this->create_new_module_entry( 'note' );
        }


        //TASKS
        if ( isset( $raw_request_array[ 'nftasks' ] ) ) {

            // form designer must specify either owner user id AND responsible owner id OR there must be another non-contact module created

            if ( ( isset( $raw_request_array[ 'nftasks' ][ 'OWNER_USER_ID' ] ) && isset( $raw_request_array[ 'nftasks' ][ 'RESPONSIBLE_USER_ID' ] ) ) || isset( $this->newly_created_id[ 'owner_user_id' ] ) ) {

                $this->set_task_link_and_defaults();

                $this->json_module_request_array[ 'create_task' ] = json_encode( $this->raw_request_array[ 'nftasks' ] );

                $this->create_new_module_entry( 'task' );
            }
        }
    }

    public function get_resource( $resource = '' ) { // retrieves custom field list or country list from Insightly
        if ( '' == $resource ) {
            return false;
        }

        $query = $this->build_get_query();

        $url = $this->base_url . '/' . $resource . $query;

        $headers = array(
            'Authorization' => 'Basic ' . base64_encode( $this->apikey ),
            'Content-Type' => 'application/json'
        );

        $args = array(
            'timeout' => 45,
            'redirection' => 0,
            'httpversion' => '1.0',
            'sslverify' => FALSE,
            'headers' => $headers
        );


        $temp_response = wp_remote_get(
                $url, $args
        );

        switch ($resource) {

            case 'Countries':

                $this->processed_response = $this->process_countries_response( $temp_response );
                break;

            case 'CustomFields':
                $this->processed_response = $this->process_custom_fields_response( $temp_response );
                break;

            case 'Users':
                   $this->processed_response = $this->process_get_users_response( $temp_response );
                break;             
            
            default:
                $this->processed_response = $temp_response;
        }

        return $this->processed_response;
    }

    /*
      Internal Processing
      -------------------- */
    protected function create_new_module_entry( $resource ) {

        $resource_array = array(
            'organization' => array(
                'endpoint' => '/organisations',
                'request_array' => 'create_organization',
                'responses' => array(
                    'organization' => 'ORGANISATION_ID',
                    'owner_user_id' => 'OWNER_USER_ID',
                ),
                'status_update' => __( 'Attempt to create a new organization at: ', 'ninja-forms-insightly-crm' )
            ),
            'opportunity' => array(
                'endpoint' => '/opportunities',
                'request_array' => 'create_opportunity',
                'responses' => array(
                    'opportunity' => 'OPPORTUNITY_ID',
                    'owner_user_id' => 'OWNER_USER_ID',
                ),
                'status_update' => __( 'Attempt to create a new opportunity at: ', 'ninja-forms-insightly-crm' )
            ),
            'project' => array(
                'endpoint' => '/projects',
                'request_array' => 'create_project',
                'responses' => array(
                    'project' => 'PROJECT_ID',
                    'owner_user_id' => 'OWNER_USER_ID',
                ),
                'status_update' => __( 'Attempt to create a new project at: ', 'ninja-forms-insightly-crm' )
            ),
            'contact' => array(
                'endpoint' => '/contacts',
                'request_array' => 'create_contact',
                'responses' => array(
                    'contact' => 'CONTACT_ID'
                ),
                'status_update' => __( 'Attempt to create a new contact at: ', 'ninja-forms-insightly-crm' )
            ),
            'lead' => array(
                'endpoint' => '/leads',
                'request_array' => 'create_lead',
                'responses' => array(
                    'lead' => 'LEAD_ID'
                ),
                'status_update' => __( 'Attempt to create a new lead at: ', 'ninja-forms-insightly-crm' )
            ),
            'note' => array(
                'endpoint' => '/notes',
                'request_array' => 'create_note',
                'responses' => array(
                    'note' => 'NOTE_ID',
                    'owner_user_id' => 'OWNER_USER_ID',
                ),
                'status_update' => __( 'Attempt to create a new note at: ', 'ninja-forms-insightly-crm' )
            ),
            'task' => array(
                'endpoint' => '/tasks',
                'request_array' => 'create_task',
                'responses' => array(
                    'task' => 'TASK_ID'
                ),
                'status_update' => __( 'Attempt to create a new task at: ', 'ninja-forms-insightly-crm' )
            ),
        );

        $url = $this->base_url . $resource_array[ $resource ][ 'endpoint' ];

        $headers = array(
            'Authorization' => 'Basic ' . base64_encode( $this->apikey ),
            'Content-Type' => 'application/json'
        );


        $body = $this->json_module_request_array[ $resource_array[ $resource ][ 'request_array' ] ];

        $args = array(
            'timeout' => 45,
            'redirection' => 0,
            'httpversion' => '1.0',
            'sslverify' => FALSE,
            'headers' => $headers,
            'body' => $body
        );

        $create_new_module_response = wp_remote_post(
                $url, $args
        );

        $this->raw_response[] = $create_new_module_response;

        $response_evaluation = $this->get_response_codes( $create_new_module_response );


        if ( !$response_evaluation[ 'wp_error_flag' ] ) {

            if ( '201' == $response_evaluation[ 'response_code' ] ) {

                $quick_answer = __( 'Successful', 'ninja-forms-insightly-crm' );

                $temp_array = json_decode( $create_new_module_response[ 'body' ], true );

                if ( isset( $resource_array[ $resource ][ 'responses' ] ) ) {

                    foreach ( $resource_array[ $resource ][ 'responses' ] as $new_id_key => $response_key ) {

                        if ( isset( $temp_array[ $response_key ] ) ) {

                            $this->newly_created_id[ $new_id_key ] = $temp_array[ $response_key ];
                        }
                    }
                }
            } else {
                /*
                 * Get the response message, look up support values
                 */
                $response_message = $response_evaluation['response_message'];
                $support_response = $this->message_support_responses( $response_message, '<br />' );

                $quick_answer = 'Unsuccessful <br />' . $support_response . '<br />' . $response_message . '<br />';
            }
        }

        $this->status_update .= $resource_array[ $resource ][ 'status_update' ]
                . $response_evaluation[ 'datetime' ]
                . __( ' was: ', 'ninja-forms-insightly-crm' )
                . $quick_answer . '<br />';
    }

    protected function link_organization_to_contact() {

        if ( !isset( $this->newly_created_id[ 'organization' ] ) ) { // no organisation_id available so unset the organisation_id LINK from nfcontacts
            unset( $this->raw_request_array[ 'nfcontacts' ][ 'LINKS' ][ 'nforganisation' ] ); // remove organisation_id from links because organization link isn't available

            return false;
        } else {

            $this->raw_request_array[ 'nfcontacts' ][ 'LINKS' ][ 'nforganisation' ][ 'ORGANISATION_ID' ] = $this->newly_created_id[ 'organization' ];
        }
    }

    protected function link_opportunity_to_contact() {

        if ( !isset( $this->newly_created_id[ 'opportunity' ] ) ) { // no opportunity_id available so unset the organisation_id LINK from nfcontacts
            unset( $this->raw_request_array[ 'nfcontacts' ][ 'LINKS' ][ 'nfopportunities' ] ); // remove opportunity_id from links because opportunity link isn't available

            return false;
        } else {

            $this->raw_request_array[ 'nfcontacts' ][ 'LINKS' ][ 'nfopportunities' ][ 'OPPORTUNITY_ID' ] = $this->newly_created_id[ 'opportunity' ];
        }
    }

    protected function link_project_to_contact() {

        if ( !isset( $this->newly_created_id[ 'project' ] ) ) { // no project_id available so unset the project_id LINK from nfcontacts
            unset( $this->raw_request_array[ 'nfcontacts' ][ 'LINKS' ][ 'nfprojects' ] ); // remove project_id from links because project link isn't available

            return false;
        } else {

            $this->raw_request_array[ 'nfcontacts' ][ 'LINKS' ][ 'nfprojects' ][ 'PROJECT_ID' ] = $this->newly_created_id[ 'project' ];
        }
    }

    protected function finalize_contact_links_array() {

        if ( isset( $this->raw_request_array[ 'nfcontacts' ][ 'LINKS' ] ) ) {

            $this->raw_request_array[ 'nfcontacts' ][ 'LINKS' ] = array_values( $this->raw_request_array[ 'nfcontacts' ][ 'LINKS' ] );
        }
    }

    protected function link_person_to_note() { // add linking fields to json request
        if ( isset( $this->newly_created_id[ 'contact' ] ) ) {
            $link_subject_type = 'CONTACT';
            $this->raw_request_array[ 'nfnotes' ][ 'LINK_SUBJECT_TYPE' ] = $link_subject_type;
            $this->raw_request_array[ 'nfnotes' ][ 'LINK_SUBJECT_ID' ] = $this->newly_created_id[ 'contact' ];
        }
        if ( isset( $this->newly_created_id[ 'lead' ] ) ) {
            $link_subject_type = 'LEAD';
            $this->raw_request_array[ 'nfnotes' ][ 'LINK_SUBJECT_TYPE' ] = $link_subject_type;
            $this->raw_request_array[ 'nfnotes' ][ 'LINK_SUBJECT_ID' ] = $this->newly_created_id[ 'lead' ];
        }
    }

    protected function set_task_link_and_defaults() { // add linking field and default values to json request
        // only continue if either lead or contact has been created; default is false
        $continue = false;

        if ( isset( $this->newly_created_id[ 'contact' ] ) ) {

            $this->raw_request_array[ 'nftasks' ][ 'TASKLINKS' ][][ 'CONTACT_ID' ] = $this->newly_created_id[ 'contact' ];

            $continue = true;
        }

        if ( isset( $this->newly_created_id[ 'lead' ] ) ) {

            $this->raw_request_array[ 'nftasks' ][ 'TASKLINKS' ][][ 'LEAD_ID' ] = $this->newly_created_id[ 'lead' ];

            $continue = true;
        }

        if ( $continue ) {
            $this->raw_request_array[ 'nftasks' ][ 'PUBLICLY_VISIBLE' ] = apply_filters( 'nfinsightlycrm_set_task_publicly_visible', TRUE );

            $this->raw_request_array[ 'nftasks' ][ 'COMPLETED' ] = apply_filters( 'nfinsightlycrm_set_task_completed', FALSE );

            if ( !isset( $this->raw_request_array[ 'nftasks' ][ 'RESPONSIBLE_USER_ID' ] ) ) { //if responsible user id isn't set, default to owner id from previously created module
                $this->raw_request_array[ 'nftasks' ][ 'RESPONSIBLE_USER_ID' ] = $this->newly_created_id[ 'owner_user_id' ];
            }

            if ( !isset( $this->raw_request_array[ 'nftasks' ][ 'OWNER_USER_ID' ] ) ) {
                $this->raw_request_array[ 'nftasks' ][ 'OWNER_USER_ID' ] = $this->newly_created_id[ 'owner_user_id' ];
            }
        }
    }

    /**
     * Array of values extracted from raw response
     * returns array
     *      'wp_error_flag' => (bool)
      'datetime' => (datetime)
      'response_code' =>
      'response_message' => string or FALSE
     * 
     * @param type $raw_response
     * @return array
     */
    protected function get_response_codes( $raw_response ) {

        if ( is_wp_error( $raw_response ) ) {

            $response_codes = array(
                'wp_error_flag' => true,
                'datetime' => date( DATE_COOKIE ),
                'response_code' => $raw_response->get_error_code(), // create function to extract
                'response_message' => $raw_response->get_error_message(), // create function to extract
            );
        } else {


            $response_codes = array(
                'wp_error_flag' => false,
                'datetime' => $raw_response[ 'headers' ][ 'date' ],
                'response_code' => $raw_response[ 'response' ][ 'code' ],
            );

            /*
             * Begin NF3 Enhancements
             */
            
            $body = $raw_response[ 'body' ];
            $json_decode_body = json_decode( $raw_response[ 'body' ], true );

            /*
             * When there is an API connection error, there is no nested
             * array.  When there is an error in the request, it is a 
             * nested array.  This checks for both.
             * 
             * The response always has a message so it can't be used to detect
             * an error.
             */
            if ( $json_decode_body && isset( $json_decode_body[ 'Message' ] ) ) { // api not authorized
                
                $response_codes[ 'response_message' ] = $json_decode_body[ 'Message' ];
                
            } elseif ( is_string($body) ) { // error in the request
                
                $response_codes[ 'response_message' ] = $body;
            } else {

                $response_codes[ 'response_message' ] = serialize($body);
            }
            /*
             * End NF3 Enhancements
             */
        }

        return $response_codes;
    }

    /**
     * Builds and stores the custom field array and returns the array
     * 
     * Returns FALSE if unable to connect to Insightly or if there are no custom fields,
     * 
     * Updated the nfinsightlycrm_api_connection_verification data so that
     * anytime the settings page is visited and refreshed, updated status on 
     * the connection can be given without overwriting the comm status from
     * form submissions
     * 
     * @global array $nfinsightlycrm_comm_data
     * @param type $raw_response
     * @return mixed
     */
    protected function process_custom_fields_response( $raw_response ) {

        global $nfinsightlycrm_comm_data;

        $custom_field_array = false; // initialize fail-to-safe	
        $response_evaluation = $this->get_response_codes( $raw_response );

        /*
         * Start of NF3 Enhancements
         */
        $connection_time = $response_evaluation[ 'datetime' ];

        if ( !$response_evaluation[ 'response_message' ] ) {
            // false if no message returned
            // No message means there was no problem connecting to the API

            $api_response_message = __( 'Retrieved Custom Fields', 'ninja-forms-insightly-crm' ) . ' at ' . $connection_time;
        } else {

            $api_response_message = $response_evaluation[ 'response_message' ] . ' - Code ' . $response_evaluation[ 'response_code' ] . ' at ' . $connection_time;

            $message_support = $this->message_support_responses( $response_evaluation[ 'response_message' ] );

            if ( $message_support ) {

                $api_response_message.='<br />' . $message_support;
            }
        }

        $this->api_connection_verification = $api_response_message;
        /*
         * End NF3 Enhancements
         */

        if ( !$response_evaluation[ 'wp_error_flag' ] ) {

            $temp_array = json_decode( $raw_response[ 'body' ], true );

            if ( is_array( $temp_array ) ) {
                foreach ( $temp_array as $raw_custom_field_array ) {

                    if ( is_array( $raw_custom_field_array ) ) {
                        $custom_field_array[ $raw_custom_field_array[ "CUSTOM_FIELD_ID" ] ] = $raw_custom_field_array[ "FIELD_NAME" ];
                    }
                }
            }
        }

        return $custom_field_array;
    }

    /**
     * Extracts an associative array of users and ids
     * 
     * @param array $raw_response
     * @return array
     * @since 3.1.4
     */
     protected function process_get_users_response($raw_response) {

        $user_array = array(); // initialize fail-to-safe
        
        $response_evaluation = $this->get_response_codes($raw_response);

        $connection_time = $response_evaluation['datetime'];

        if (!$response_evaluation['response_message']) {
            // false if no message returned
            // No message means there was no problem connecting to the API

            $api_response_message = __('Retrieved Insightly Users', 'ninja-forms-insightly-crm') . ' at ' . $connection_time;
        } else {

            $api_response_message = $response_evaluation['response_message'] . ' - Code ' . $response_evaluation['response_code'] . ' at ' . $connection_time;

            $message_support = $this->message_support_responses($response_evaluation['response_message']);

            if ($message_support) {

                $api_response_message .= '<br />' . $message_support;
            }
        }

        $this->api_connection_verification = $api_response_message;

        if (!$response_evaluation['wp_error_flag']) {

            $data_string = $raw_response['body'];

            $full_user_array = json_decode($data_string, true);

            if (empty($full_user_array)) {

                return $user_array;
            }

            $fields_to_extract = array('USER_ID', 'FIRST_NAME', 'LAST_NAME', 'EMAIL_ADDRESS');

            foreach ($full_user_array as $user) {

                $temp = array();

                foreach ($fields_to_extract as $field) {

                    if (!isset($user[$field])) {

                        $temp[$field] = '';
                    } else {

                        $temp[$field] = $user[$field];
                    }
                }
                $user_array[] = $temp;
            }
        }

        return $user_array;
    }

    protected function process_countries_response( $raw_response ) {

        $response = array();

        $temp_array = json_decode( $raw_response[ 'body' ], true );

        foreach ( $temp_array as $value ) {
            $response[] = $value[ 'COUNTRY_NAME' ];
        }
        return implode( ',', $response );
    }

    /**
     * Returns array of troubleshooting responses for a message
     * 
     * @param string $message
     * @return array
     */
    protected function message_support_responses( $message = '', $delimiter = ", " ) {

        $support_response = array();  // initialize

        if ( class_exists( 'NF_InsightlyCRM' ) ) {

            $message_support_array = NF_InsightlyCRM()->config( 'MessageSupport' );
        } else {

            $message_support_array = include NF2INSIGHTLYCRM_PLUGIN_DIR . 'includes/Config/MessageSupport.php';
        }

        foreach ( $message_support_array as $lookup => $support ) {

            if ( FALSE !== strpos( $message, $lookup ) ) {

                $support_response[] = $support;
            }
        }

        $support_response = implode( $delimiter, $support_response );

        return $support_response;
    }

    protected function build_base_url() {

        $version = '2.2'; // set default Insightly API version

        if ( defined( 'INSIGHTLY_API_VERSION' ) ) {

            switch (INSIGHTLY_API_VERSION) {

                case '2.1':
                    $version = '2.1';
                    break;

                default:
                    $version = '2.2';
            }
        }

        $this->base_url = 'https://api.insight.ly/v' . $version . '/';
    }

    /*
     * Assembles any prebuilt get parameters
     * 
     * Returns empty string if no parameters are set
     * 
     */
    protected function build_get_query() {

        $get_query = ''; // default no query to append

        $top = $this->build_max_record_return_query();

        if ( 0 < strlen( $top ) ) {// append top if it is set
            $get_query .=$top;
        }


        if ( 0 < strlen( $get_query ) ) {// prepend ? if query has parameters
            $get_query = '?' . $get_query;
        }

        return $get_query;
    }

    protected function build_max_record_return_query() {

        $max_records_return_query = ''; //default no top query

        if ( defined( 'INSIGHTLY_RECORDS_TO_RETURN' ) && is_integer( INSIGHTLY_RECORDS_TO_RETURN ) ) {

            $qty = min( 500, INSIGHTLY_RECORDS_TO_RETURN );

            $max_records_return_query = 'top=' . $qty;
        }

        return $max_records_return_query;
    }

    /*
     * Gets and Sets
     */
    public function set_apikey( $apikey = "" ) {

        $this->apikey = $apikey;
    }

    public function get_raw_request_array() {

        if ( empty( $this->raw_request_array ) ) {
            return false;
        } else {
            return $this->raw_request_array;
        }
    }

    public function get_status_update() {

        if ( empty( $this->status_update ) ) {
            return false;
        } else {
            return $this->status_update;
        }
    }

    public function get_raw_response() {

        if ( empty( $this->raw_response ) ) {
            return false;
        } else {
            return $this->raw_response;
        }
    }

    /**
     * Returns the Insightly response when attempting to connect to the API
     * @return mixed
     */
    public function get_api_connection_verification() {

        if ( empty( $this->api_connection_verification ) ) {
            return false;
        } else {
            return $this->api_connection_verification;
        }
    }

    /**
     * 
     * @return boolean|array Returns array of the newly create IDs for further handling
     */
    public function get_newly_created_id(){
 
        if ( empty( $this->newly_created_id ) ) {
            return false;
        } else {
            return $this->newly_created_id;
        }
        
    }
}
