<?php if ( ! defined( 'ABSPATH' ) ) exit;

/*
 * Plugin Name: Ninja Forms - CleverReach
 * Plugin URI: http://etzelstorfer.com/
 * Description: Add subscribers to your Cleverreach campaigns from your Ninja Forms. 
You can create mutliple forms and connect to multiple subscriber lists. 
Supports auto subscription, single-opt-in via checkbox and douple-opt-in via confirmation mail.
 * Version: 3.1.3
 * Author: Hannes Etzelstorfer
 * Author URI: http://etzelstorfer.com/
 * Text Domain: nf_cleverreach
 *
 * Copyright 2017 Hannes Etzelstorfer.
 */

if( version_compare( get_option( 'ninja_forms_version', '0.0.0' ), '3.0', '<' ) || get_option( 'ninja_forms_load_deprecated', FALSE ) ) {
	include 'deprecated/ninja-forms-cleverreach.php';
} else {

	/**
	 * Class NF_CleverReach
	 */
	final class NF_CleverReach
	{
		const VERSION = '3.1.3';
		const SLUG    = 'cleverreach';
		const NAME    = 'CleverReach';
		const AUTHOR  = 'Hannes Etzelstorfer';
		const PREFIX  = 'NF_CleverReach';
		const NF_CLEVERREACH_API_URL = 'http://api.cleverreach.com/soap/interface_v5.1.php?wsdl';
		const NF_CLEVERREACH_REST_API_URL = 'https://rest.cleverreach.com/v2'; 

		/**
		 * @var NF_CleverReach
		 * @since 3.0
		 */
		private static $instance;

		/**
		 * Plugin Directory
		 *
		 * @since 3.0
		 * @var string $dir
		 */
		public static $dir = '';

		/**
		 * Plugin URL
		 *
		 * @since 3.0
		 * @var string $url
		 */
		public static $url = '';

		/**
		 * Main Plugin Instance
		 *
		 * Insures that only one instance of a plugin class exists in memory at any one
		 * time. Also prevents needing to define globals all over the place.
		 *
		 * @since 3.0
		 * @static
		 * @static var array $instance
		 * @return NF_CleverReach Highlander Instance
		 */
		public static function instance()
		{
			if (!isset(self::$instance) && !(self::$instance instanceof NF_CleverReach)) {
				self::$instance = new NF_CleverReach();

				self::$dir = plugin_dir_path(__FILE__);

				self::$url = plugin_dir_url(__FILE__);

				/*
				 * Register our autoloader
				 */
				spl_autoload_register(array(self::$instance, 'autoloader'));
			}

			return self::$instance;
		}

		public function __construct()
		{
			/*
			 * Required for all Extensions.
			 */
			add_action( 'admin_init', array( $this, 'setup_license') );

			/*
			 * Optional. If your extension creates a new field interaction or display template...
			 */
			add_filter( 'ninja_forms_register_fields', array($this, 'register_fields'));

			/*
			 * Optional. If your extension processes or alters form submission data on a per form basis...
			 */
			add_filter( 'ninja_forms_register_actions', array($this, 'register_actions'));

			add_filter( 'ninja_forms_plugin_settings', array($this, 'register_settings'));
			add_filter( 'ninja_forms_plugin_settings_groups', array($this, 'register_settings_group'));
			add_filter( 'ninja_forms_check_setting_nf_cleverreach_api_key', array( $this, 'test_api_key' ), 10, 1 );

			add_action( 'ninja_forms_enqueue_scripts', array( $this, 'enqueue_frontend_scripts' ), 10 );
		}


		/**
		 * Optional. If your extension creates a new field interaction or display template...
		 */
		public function register_fields($actions)
		{
			$actions[ 'cleverreachoptin' ] = new NF_CleverReach_Fields_CleverReachOptIn(); 

			return $actions;
		}

		/**
		 * Optional. If your extension processes or alters form submission data on a per form basis...
		 */
		public function register_actions($actions)
		{
			$actions[ 'cleverreach' ] = new NF_CleverReach_Actions_CleverReachSubscribe(); 

			return $actions;
		}


		/**
		 * Register a Group on the settings page
		 */
		public function register_settings_group($groups)
		{
			$api_key = Ninja_Forms()->get_setting('nf_cleverreach_api_key');
			if( $api_key ){ // only show SOAP api key for compatibility reasons
				$groups[ 'cleverreach_rest' ] = array(
					'id' => 'cleverreach_rest',
					'label' => __( 'CleverReach Settings', 'nf_cleverreach' ) . ' – new REST API (recommended)',
				);

				$groups[ 'cleverreach' ] = array(
					'id' => 'cleverreach',
					'label' => __( 'CleverReach Settings', 'nf_cleverreach' ) . ' – old SOAP API',
				);
			}else{
				$groups[ 'cleverreach_rest' ] = array(
					'id' => 'cleverreach_rest',
					'label' => __( 'CleverReach Settings', 'nf_cleverreach' ),
				);				
			}


			return $groups;
		}

		/**
		 * Register Settings fields
		 */
		public function register_settings($settings)
		{
			$api_key = Ninja_Forms()->get_setting('nf_cleverreach_api_key');
			if( $api_key ){ // only show SOAP api key for compatibility reasons
				$settings[ 'cleverreach' ] = NF_CleverReach()->config( 'PluginSettings' );
			}
			$settings[ 'cleverreach_rest' ] = NF_CleverReach()->config( 'PluginSettingsRest' );

			return $settings;
		}

		/*
		 * Optional methods for convenience.
		 */

		public function autoloader($class_name)
		{
			if (class_exists($class_name)) return;

			if ( false === strpos( $class_name, self::PREFIX ) ) return;

			$class_name = str_replace( self::PREFIX, '', $class_name );
			$classes_dir = realpath(plugin_dir_path(__FILE__)) . DIRECTORY_SEPARATOR . 'includes' . DIRECTORY_SEPARATOR;
			$class_file = str_replace('_', DIRECTORY_SEPARATOR, $class_name) . '.php';

			if (file_exists($classes_dir . $class_file)) {
				require_once $classes_dir . $class_file;
			}
		}


		/**
		 * Config
		 *
		 * @param $file_name
		 * @return mixed
		 */
		public static function config( $file_name )
		{
			return include self::$dir . 'includes/Config/' . $file_name . '.php';
		}

		
		/*
		 * Required methods for all extension.
		 */

		public function setup_license()
		{
			if ( ! class_exists( 'NF_Extension_Updater' ) ) return;

			new NF_Extension_Updater( self::NAME, self::VERSION, self::AUTHOR, __FILE__, self::SLUG );
		}


		/**
		 * Test API Key (valid / write-access)
		 *
		 * @return array
		 */
		public function test_api_key($setting){
			$api_key = trim( $setting[ 'value' ] );
			
			try{
				$api = new SoapClient(NF_CleverReach::NF_CLEVERREACH_API_URL);
				$result = $api->groupGetList($api_key);
				if($result->status!="SUCCESS"){
					$setting[ 'errors' ][] = __( 'Your API key is invalid.', 'nf_cleverreach' );
				}else{
					$test_group_name = 'Ninja-Forms-Cleverreach-Write-Access-Test-'.uniqid();
					$result = $api->groupAdd($api_key, $test_group_name);
					if($result->status == 'ERROR' && $result->statuscode == 6 ){
						$setting[ 'errors' ][] = __( 'Your API has read-only permissions, please change to read-write-access.', 'nf_cleverreach' );//.'<pre>'.print_r($result,true).'</pre>';
					}else{
						$api->groupDelete($api_key, $result->data->id);
					}
				}
			} catch(Exception $e){
				$setting[ 'errors' ][] = __( 'Your API key is invalid.', 'nf_cleverreach' );
			}
			return $setting;
		}



		public function enqueue_frontend_scripts( $settings ){
		    $form_id = $settings['form_id'];

		    $nf_ver     = Ninja_Forms::VERSION;
		    $nf_js_dir  = Ninja_Forms::$url . 'assets/js/min/';
		    $nf_css_dir = Ninja_Forms::$url . 'assets/css/';

		    $form_fields = Ninja_Forms()->form( $form_id )->get_fields();
		    $has_cleverreachoptin_field = false;
		    
		    if( $form_fields ){
		        foreach ($form_fields as $field) {
		            if( is_object( $field ) ) {
		                $field = array(
		                    'id' => $field->get_id(),
		                    'settings' => $field->get_settings()
		                );
		            }

		            if( 'cleverreachoptin' == $field[ 'settings' ][ 'type' ] ){
		                $has_cleverreachoptin_field = true;
		                break;
		            }
		        }
		    }

		    if( $has_cleverreachoptin_field ){
		        wp_enqueue_script('nf-cleverreachoptin', self::$url . 'assets/js/fieldCleverReachOptIn.js', array( 'jquery' ), self::VERSION );
		    }
		}
	}



	
	/**
	 * The main function responsible for returning The Highlander Plugin
	 * Instance to functions everywhere.
	 *
	 * Use this function like you would a global variable, except without needing
	 * to declare the global.
	 *
	 * @since 3.0
	 * @return {class} Highlander Instance
	 */
	function NF_CleverReach()
	{
		return NF_CleverReach::instance();
	}

	NF_CleverReach();
}


add_filter( 'ninja_forms_upgrade_settings', 'NF_CleverReach_Upgrade_Settings' );
function NF_CleverReach_Upgrade_Settings( $data ){

	$plugin_settings = get_option( 'nf_cleverreach_options' );
	
	Ninja_Forms()->update_settings(
		array(
			'nf_cleverreach_api_key'  => $plugin_settings[ 'api_key' ],
		)
	);

	$form_id = null;

	// convert the salutation field to a list field
	foreach ( $data['fields'] as $field_index => $field ) {
		if( $field['type'] == 'cleverreach_salutation' ){
			$data['fields'][ $field_index ]['type'] = '_list';
			$salutation_field_id = $field['id'];
			foreach ($field['data']['list']['options'] as $key => $option) {
				if( !$option['value'] )
					$data['fields'][ $field_index ]['data']['list']['options'][$key]['value'] = $option['label'];
			}
		}
		elseif( $field['type'] == '_text' && isset( $field['data']['user_email'] ) && $field['data']['user_email'] )
			$email_field_id = $field['id'];
		
		elseif( $field['type'] == '_text' && isset( $field['data']['last_name'] ) && $field['data']['last_name'] )
			$last_name_id = $field['id'];
		
		elseif( $field['type'] == '_text' && isset( $field['data']['first_name'] ) && $field['data']['first_name'] )
			$first_name_id = $field['id'];

		elseif( $field['type'] == 'cleverreach_auto' )
			$form_id = $field['data']['cleverreach_form'];

		//convert the optin-field
		elseif(  $field['type'] == 'cleverreach_checkbox' ){
			$form_id = $field['data']['cleverreach_form'];
			$data['fields'][ $field_index ]['type'] = 'cleverreachoptin';
		}
	}

	if( $form_id ){
		//create the cleverreach action
		$action = array(
				'label' => 'CleverReach',
				'type' => 'cleverreach',
				'newsletter_list' => $form_id,
				'double_opt_in' => ( !isset($plugin_settings[ 'opt_in' ]) || $plugin_settings[ 'opt_in' ]=='mail' ? 1 : 0 ),
			);

		if( isset( $salutation_field_id) )
			$action['NF_CleverReach_salutation'] = '{field:' . $salutation_field_id . '}';

		if( isset( $email_field_id) )
			$action['NF_CleverReach_email'] = '{field:' . $email_field_id . '}';

		if( isset( $last_name_id) )
			$action['NF_CleverReach_lastname'] = '{field:' . $last_name_id . '}';

		if( isset( $first_name_id) )
			$action['NF_CleverReach_firstname'] = '{field:' . $first_name_id . '}';

		$data[ 'actions' ][] = $action;
	}
	
	return $data;
}
