<?php

/*
 *
 * This class is called during form processing to setup our PayPal Express checkout using the NF_Paypal_Process class.
 *
 * @since 1.0
 *
 */

class NF_Paypal_Form_Process
{

	public $token = '';

	/*
	 *
	 * Function that handles putting together the request sent to PayPal initially during form processing.
	 *
	 * @since 1.0
	 * @return void
	 */

	public function __construct() {
		global $ninja_forms_processing;

		// Bail if this form isn't setup to use PayPal Express.
		if ( $ninja_forms_processing->get_form_setting( 'paypal_express' ) != 1 )
    		return;

    	// Bail if the form's total calculation field is empty.
    	$purchase_total = floatval ( $this->get_purchase_total() );

    	if ( empty (  $purchase_total ) )
    		return;
 
    	$this->do_checkout();
    	return;
	} // function __construct

	/*
	 *
	 * Function that runs our checkout process.
	 *
	 * @since 1.0
	 * @return void
	 */

	function do_checkout() {
		global $ninja_forms_processing;
		$plugin_settings = get_option( 'ninja_forms_paypal' );

    	// Get the product info for our query.
    	$item = $this->get_product_info();
    	$requestParams = $this->get_request_params();
    	$orderParams = $this->get_order_params();

    	if ( ( isset ( $plugin_settings['debug'] ) and $plugin_settings['debug'] == 1 ) or ( ! isset ( $plugin_settings['debug'] ) and NINJA_FORMS_PAYPAL_EXPRESS_DEBUG ) ) {
	    	echo "<pre>";
	    	var_dump( $item );
	    	echo "</pre>";

	    	echo "<pre>";
	    	var_dump( $orderParams );
	    	echo "</pre>";
    	}

    	$paypal = new NF_Process_Paypal();
		$response = $paypal -> request('SetExpressCheckout',$requestParams + $orderParams + $item);
		if ( ( isset ( $plugin_settings['debug'] ) and $plugin_settings['debug'] == 1 ) or ( ! isset ( $plugin_settings['debug'] ) and NINJA_FORMS_PAYPAL_EXPRESS_DEBUG ) ) {
	    	echo "<pre>";
			var_dump( $response );
			echo "</pre>";

			echo '<h3>';
	    	echo __( 'The debugging information above and any error output below has been generated by Ninja Forms - PayPal Express. To remove this information and any errors, please disable debugging in the Forms->PayPal Settings menu.', 'ninja-forms-paypal-express' );
	    	echo '</h3>';

		}
		if ( is_array ( $response ) AND ( $response['ACK'] == 'Success' OR $response['ACK'] == 'SuccessWithWarning' ) ) { //Request successfully
			$this->token = $response['TOKEN'];
			$this->checkout_success();
		} else {
			$this->checkout_fail();
		}
		return;
	}

	/*
	 *
	 * Function that gets the $purchase_total of our form.
	 *
	 * @since 1.0
	 * @return $purchase_total string
	 */

	public function get_purchase_total() {
		// Get our form total. This can be returned as an array or a string value.
		$total = ninja_forms_paypal_get_total();

		if ( is_array ( $total ) ) { // If this is an array, grab the string total.
			if ( isset ( $total['total'] ) ) {
			  $purchase_total = $total['total'];
			} else {
			  $purchase_total = '';
			}
		} else { // This isn't an array, so $purchase_total can just be set to the string value.
			$purchase_total = $total;
		}
		return $purchase_total;
	}

	/*
	 *
	 * Function that grabs the total, and items that make up the total, from our form.
	 * If no "Total" field is used, checks for the default form values.
	 *
	 * @since 1.0
	 * @return $item array
	 */

	public function get_product_info() {
		global $ninja_forms_processing;

		$purchase_total = $this->get_purchase_total();

	  	// Get our items.
	  	// Get our form total. This can be returned as an array or a string value.
		$total = ninja_forms_paypal_get_total();

	  	// Check to see if $total is an array. 
		if ( is_array ( $total ) ) { // If it is, then get the items from it.

			if ( is_array ( $total['fields'] ) ) {
				$item = array();
				$x = 0;
				foreach ( $total['fields'] as $field_id => $calc ) {
					$field = $ninja_forms_processing->get_field_settings( $field_id );
					$user_value = $ninja_forms_processing->get_field_value( $field_id );
					if ( isset ( $field['data']['paypal_item'] ) ) {
						$paypal_item = $field['data']['paypal_item'];
					} else {
						$paypal_item = 1;
					}

					if ( ( !isset ( $field['data']['payment_sub_total'] ) OR $field['data']['payment_sub_total'] != 1 ) AND $paypal_item == 1 ) {

						if ( isset ( $field['data']['label'] ) ) {
							$field_label = $field['data']['label'];
						} else {
							$field_label = '';
						}

						if ( isset ( $field['data']['desc_text'] ) ) {
							$field_desc = $field['data']['desc_text'];
						} else {
							$field_desc = '';
						}

						$field_label = strip_tags( $field_label );
						$field_desc = strip_tags( $field_desc );

						if ( strlen( $field_desc ) != 100 ) {
							$field_desc = substr( $field_desc, 0, 100 );
						}

						// Check to see if we're working with a checkbox. If we are, the calculation value will depend upon it's state.

						if ( $field['type'] == '_checkbox' ) {
							if ( $user_value == 'checked' ) { // This is a checkbox, so let's see if the value is checked or unchecked.
								$field_calc = $field['data']['calc_value']['checked'];
							} else {
								$field_calc = $field['data']['calc_value']['unchecked'];
							}
						} else { // We aren't working with a checkbox, so the value will be used as the calc value.
							$field_calc = $user_value;
						}

						$locale_info = localeconv();
						if ( isset ( $locale_info['thousands_sep'] ) ) {
							$thousands_sep = $locale_info['thousands_sep'];
						} else {
							$thousands_sep = ',';
						}
						
						$field_calc = str_replace( $thousands_sep, '', $field_calc );

						// Check to see if we're working with a list item
						if ( $field['type'] == '_list' ) {
							// Get our list of options.
							$options = $field['data']['list']['options'];
							if ( is_array ( $user_value ) ) {
							  	foreach ( $user_value as $val ) {

							  		
									foreach ( $options as $opt ) {
										// Check to see if we're using option values.
										if ( isset ( $field['data']['list_show_value'] ) AND $field['data']['list_show_value'] == 1 ) {
											// If the val is equal to our option value, then assign the label.
											if ( $opt['value'] == $val ) {
												// Normally, we use the label field setting for the product name. For a list, however, the user may want to use the selected list item's label instead.
												// Check to see if the user wants to use the list label for the selected item.
												if ( isset ( $field['data']['list_label_product_name'] ) AND $field['data']['list_label_product_name'] == 1 ) {
													$field_label = $opt['label'];
												}
												$field_calc = $opt['calc'];
												break;
											}
										} else {
										// If the val is equal to our option label, then assign the label.
											if ( $opt['label'] == $val ) {
												// Normally, we use the label field setting for the product name. For a list, however, the user may want to use the selected list item's label instead.
												// Check to see if the user wants to use the list label for the selected item.
												if ( isset ( $field['data']['list_label_product_name'] ) AND $field['data']['list_label_product_name'] == 1 ) {
													$field_label = $opt['label'];
												}
												$field_calc = $opt['calc'];
												break;                
											}
										}
									}


									
									if ( isset ( $field['data']['desc_text'] ) AND $field['data']['desc_text'] != '' ) {
										$field_desc = $field['data']['desc_text'];
									} else {
										$field_desc = '';
									}

									$field_label = strip_tags( $field_label );
									$field_desc = strip_tags( $field_desc );
									if ( strlen( $field_desc ) != 100 ) {
										$field_desc = substr( $field_desc, 0, 100 );
									}

									$field_calc = preg_replace('/([^\-0-9\.,])/i', '', $field_calc);

									$item['L_PAYMENTREQUEST_0_NAME'.$x] = apply_filters( 'ninja_forms_paypal_express_product_name', $field_label, $field_id );
									$item['L_PAYMENTREQUEST_0_DESC'.$x] = apply_filters( 'ninja_forms_paypal_express_product_desc', $field_desc, $field_id );
									$item['L_PAYMENTREQUEST_0_AMT'.$x] = $field_calc;
									$item['L_PAYMENTREQUEST_0_QTY'.$x] = '1';
									$x++;
							  	}
							} else {

								foreach ( $options as $opt ) {
									// Check to see if we're using option values.
									if ( isset ( $field['data']['list_show_value'] ) AND $field['data']['list_show_value'] == 1 ) {
										// If the val is equal to our option value, then assign the label.
										if ( $opt['value'] == $user_value ) {
											// Normally, we use the label field setting for the product name. For a list, however, the user may want to use the selected list item's label instead.
											// Check to see if the user wants to use the list label for the selected item.
											if ( isset ( $field['data']['list_label_product_name'] ) AND $field['data']['list_label_product_name'] == 1 ) {
												$field_label = $opt['label'];
											}
											$field_calc = $opt['calc'];
											break;
										}
									} else {
									// If the val is equal to our option label, then assign the label.
										if ( $opt['label'] == $user_value ) {
											// Normally, we use the label field setting for the product name. For a list, however, the user may want to use the selected list item's label instead.
											// Check to see if the user wants to use the list label for the selected item.
											if ( isset ( $field['data']['list_label_product_name'] ) AND $field['data']['list_label_product_name'] == 1 ) {
												$field_label = $opt['label'];
											}
											$field_calc = $opt['calc'];
											break;                
										}
									}
								}
							
								if ( isset ( $field['data']['desc_text'] ) AND $field['data']['desc_text'] != '' ) {
									$field_desc = $field['data']['desc_text'];
								} else {
									$field_desc = '';
								}

								$field_label = strip_tags( $field_label );
								$field_desc = strip_tags( $field_desc );
								if ( strlen( $field_desc ) != 100 ) {
									$field_desc = substr( $field_desc, 0, 100 );
								}
								$field_calc = preg_replace('/([^\-0-9\.,])/i', '', $field_calc);

								$item['L_PAYMENTREQUEST_0_NAME'.$x] = apply_filters( 'ninja_forms_paypal_express_product_name', $field_label, $field_id );
								$item['L_PAYMENTREQUEST_0_DESC'.$x] = apply_filters( 'ninja_forms_paypal_express_product_desc', $field_desc, $field_id );
								$item['L_PAYMENTREQUEST_0_AMT'.$x] = $field_calc;
								$item['L_PAYMENTREQUEST_0_QTY'.$x] = '1';
								$x++;
							}

						} else { // This isn't a list element
							$field_label = strip_tags( $field_label );
							$field_desc = strip_tags( $field_desc );
							if ( strlen( $field_desc ) != 100 ) {
								$field_desc = substr( $field_desc, 0, 100 );
							}

							$field_calc = preg_replace('/([^\-0-9\.,])/i', '', $field_calc);

							$item['L_PAYMENTREQUEST_0_NAME'.$x] = apply_filters( 'ninja_forms_paypal_express_product_name', $field_label, $field_id );
							$item['L_PAYMENTREQUEST_0_DESC'.$x] = apply_filters( 'ninja_forms_paypal_express_product_desc', $field_desc, $field_id );
							$item['L_PAYMENTREQUEST_0_AMT'.$x] = $field_calc;
							$item['L_PAYMENTREQUEST_0_QTY'.$x] = '1';
							$x++;
						}
					}
				}
			}

			if ( isset ( $total['tax_total'] ) ) {
				// Get the Tax field label
				foreach ( $ninja_forms_processing->get_all_fields() as $field_id => $user_value ) {
					$field = ninja_forms_get_field_by_id( $field_id );
					if ( $field['type'] == '_tax' ) {
						$field_label = $field['data']['label'];
						$field_desc = $field['data']['default_value'];
						break;
					}
				}

				$field_label = strip_tags( $field_label );
				$field_desc = strip_tags( $field_desc );
				if ( strlen( $field_desc ) != 100 ) {
					$field_desc = substr( $field_desc, 0, 100 );
				}
				$field_calc = preg_replace('/([^\-0-9\.,])/i', '', $field_calc);
				
				$item['L_PAYMENTREQUEST_0_NAME'.$x] = apply_filters( 'ninja_forms_paypal_express_product_name', $field_label, $field_id );
				$item['L_PAYMENTREQUEST_0_DESC'.$x] = apply_filters( 'ninja_forms_paypal_express_product_desc', $field_desc, $field_id );
				$item['L_PAYMENTREQUEST_0_AMT'.$x] = $total['tax_total'];
				$item['L_PAYMENTREQUEST_0_QTY'.$x] = '1';
				$x++;
			}

			if ( empty( $item ) ) {
				$product_name = __( 'Total1', 'ninja-forms-paypal-express' );
				$product_desc = '';
				$item = array(
				   'L_PAYMENTREQUEST_0_NAME0' => apply_filters( 'ninja_forms_paypal_express_product_name', $product_name, false ),
				   'L_PAYMENTREQUEST_0_DESC0' => apply_filters( 'ninja_forms_paypal_express_product_desc', $product_desc, false ),
				   'L_PAYMENTREQUEST_0_AMT0' => $purchase_total,
				   'L_PAYMENTREQUEST_0_QTY0' => '1',
				);
			}

		} else { // If it isn't, we are using the default product info.

			$product_name = $ninja_forms_processing->get_form_setting( 'paypal_product_name' );
			$product_desc = $ninja_forms_processing->get_form_setting( 'paypal_product_desc' );

			$item = array(
			   'L_PAYMENTREQUEST_0_NAME0' => apply_filters( 'ninja_forms_paypal_express_product_name', $product_name, false ),
			   'L_PAYMENTREQUEST_0_DESC0' => apply_filters( 'ninja_forms_paypal_express_product_desc', $product_desc, false ),
			   'L_PAYMENTREQUEST_0_AMT0' => $purchase_total,
			   'L_PAYMENTREQUEST_0_QTY0' => '1',
			);
		}

		return $item;
	} // function product_info

	/*
	 *
	 * Function that gets the Request Parameters needed to send an express checkout to PayPal.
	 *
	 * @since 1.0
	 * @return $requestParams array
	 */

	function get_request_params() {
		global $ninja_forms_processing;

		if ( $ninja_forms_processing->get_form_setting( 'form_url' ) ) {
			$url = $ninja_forms_processing->get_form_setting( 'form_url' );
		} else {
			$protocol = ((!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] != 'off') || $_SERVER['SERVER_PORT'] == 443) ? "https://" : "http://";
			$url = $protocol . $_SERVER["HTTP_HOST"] . $_SERVER["REQUEST_URI"];
		}

		$success_url = add_query_arg( array( 'nfpe_checkout' => 'success' ), $url );
		$cancel_url = add_query_arg( array( 'nfpe_checkout' => 'cancel' ), $url );

		//Our request parameters
		$requestParams = array(
		   'RETURNURL' => $success_url,
		   'CANCELURL' => $cancel_url
		);

		return $requestParams;
	} // function get_request_params

	/*
	 *
	 * Function that gets the Order Parameters needed to send an express checkout to PayPal.
	 *
	 * @since 1.0
	 * @return $orderParams array
	 */

	function get_order_params() {

		$purchase_total = $this->get_purchase_total();

		$currency = ninja_forms_paypal_get_currency();

		$orderParams = array(
			'PAYMENTREQUEST_0_PAYMENTACTION' => 'Sale',
			'PAYMENTREQUEST_0_AMT' => $purchase_total,
			'PAYMENTREQUEST_0_SHIPPINGAMT' => '0',
			'PAYMENTREQUEST_0_CURRENCYCODE' => $currency,
			'PAYMENTREQUEST_0_ITEMAMT' => $purchase_total
		);
		return $orderParams;
	} // function get_order_params

	/*
	 *
	 * Function that handles successfully sending an express checkout.
	 * 1) Set the landing_page form setting to the PayPal url.
	 * 2) Set the paypal_redirect form setting to 1, indicating that the form has been redirected.
	 * 3) Update our form submission, setting paypal_status to 'pending'
	 *
	 * @since 1.0
	 * @return void
	 */

	function checkout_success() {
		global $ninja_forms_processing;

		$purchase_total = $this->get_purchase_total();

		// Set checkout URL.
	    $plugin_settings = get_option( 'ninja_forms_paypal' );
		if ( $ninja_forms_processing->get_form_setting( 'paypal_test_mode' ) == 1 ) {
			$url = 'https://www.sandbox.paypal.com/webscr?cmd=_express-checkout&token=' . urlencode( $this->token ) . '&useraction=commit';
		} else {
			$url = 'https://www.paypal.com/webscr?cmd=_express-checkout&token=' . urlencode( $this->token ) . '&useraction=commit';
		}
		
	    // Update our 'landing_page' form setting with the new PayPal checkout url.
		//$ninja_forms_processing->update_form_setting( 'landing_page', $url );
		// Update our 'paypal_redirect' form setting to 1, indicating that this form has been redirected to PayPal.
		$ninja_forms_processing->update_form_setting( 'paypal_redirect', 1 );
		
		// Backwards compatibility code for Ninja Forms versions before 2.7.
		if ( nf_pe_pre_27() ) {
			// Call our function to save the submission thus far.
			ninja_forms_save_sub();
		} else {
			nf_save_sub();
		}

		// Update our submission, setting the paypal_status value to 'pending'.
		$sub_id = $ninja_forms_processing->get_form_setting( 'sub_id' );

		// Backwards compatibility code for Ninja Forms versions before 2.7.
		if ( nf_pe_pre_27() ) {
			$sub_row = ninja_forms_get_sub_by_id( $sub_id );

		    if ( $sub_row AND is_array ( $sub_row ) ) {
				$sub_row['paypal_status'] = 'pending';
				$sub_row['paypal_total'] = $purchase_total;
				unset( $sub_row['id'] );
				$sub_row['sub_id'] = $sub_id;
				ninja_forms_update_sub( $sub_row );
		    }
		} else {
			Ninja_Forms()->sub( $sub_id )->add_meta( '_paypal_status', 'pending' );
    		Ninja_Forms()->sub( $sub_id )->add_meta( '_paypal_total', $purchase_total );
		}

    	// Set our transient variables
	    ninja_forms_set_transient();

	    // Redirect the user to PayPal
	    wp_redirect( $url );

	    die();
	} // function checkout_success

	/*
	 *
	 * Function that handles a failed express checkout.
	 * 1) Add an error to the $ninja_forms_processing global.
	 *
	 * @since 1.0
	 * @return void
	 */

	function checkout_fail() {
		global $ninja_forms_processing;

		$ninja_forms_processing->add_error( 'paypal-fail', __( 'PayPal encountered an error in processing your transaction. Please try again.', 'ninja-forms-paypal-express' ) );
		return;
	} // function checkout_fail

} // Class

/*
 *
 * Function that initialized our PayPal Express processing
 *
 * @since 1.0
 * @return void
 */

function ninja_forms_paypal_express_process(){
	$NF_Paypal_Form_Process = new NF_Paypal_Form_Process();
}

add_action( 'ninja_forms_process', 'ninja_forms_paypal_express_process', 9999 );