<?php if ( ! defined( 'ABSPATH' ) ) exit;

/*
 * Plugin Name: Ninja Forms - Helpscout
 * Plugin URI: https//ninjaforms.com/extensions/helpscout/
 * Description: Integrate with Helpscout
 * Version: 3.0.0
 * Author: WP Ninjas
 * Author URI: http://ninjaforms.com/
 * Text Domain: ninja-forms-helpscout
 *
 * Copyright 2017 WP Ninjas.
 */

if( version_compare( get_option( 'ninja_forms_version', '0.0.0' ), '3', '<' ) || get_option( 'ninja_forms_load_deprecated', FALSE ) ) {

    include 'deprecated/ninja-forms-helpscout.php';

} else {

    /**
     * Class NF_Helpscout
     */
    final class NF_Helpscout
    {
        const VERSION = '3.0.0';
        const SLUG = 'helpscout';
        const NAME = 'helpscout';
        const AUTHOR = 'WP Ninjas';
        const PREFIX = 'NF_Helpscout';

        /**
         * @var NF_Helpscout
         * @since 3.0
         */
        private static $instance;

        /**
         * Plugin Directory
         * @since 3.0
         *
         * @var string $dir
         */
        public static $dir = '';

        /**
         * Plugin URL
         * @since 3.0
         *
         * @var string $url
         */
        public static $url = '';

        /**
         * The API key that is sent to Helpscout.
         * @since 3.0
         *
         * @var
         */
        private $_api;

        /**
         * Main Plugin Instance
         *
         * Insures that only one instance of a plugin class exists in memory at any one
         * time. Also prevents needing to define globals all over the place.
         *
         * @since 3.0
         * @static
         * @static var array $instance
         * @return NF_Helpscout Highlander Instance
         */
        public static function instance()
        {
            if ( !isset( self::$instance ) && !( self::$instance instanceof NF_Helpscout ) ) {
                self::$instance = new NF_Helpscout();

                self::$dir = plugin_dir_path( __FILE__ );

                self::$url = plugin_dir_url( __FILE__ );

                /*
                 * Register our autoloader
                 */
                spl_autoload_register( array( self::$instance, 'autoloader' ) );

                new NF_Helpscout_Admin_Settings();
            }

            return self::$instance;
        }

        /**
         * NF_Helpscout constructor.
         * @since 3.0
         *
         */
        public function __construct()
        {
            /*
             * Required for all Extensions.
             */
            add_action( 'admin_init', array( $this, 'setup_license' ) );

            // Get our TlS Transient
            $tls_transient = get_transient( 'nf_helpscout_tls_check' );
            // If the Transient is set.
            if( 1 == $tls_transient ) {
                // ...and if the query string check isset in the URL
                if( isset( $_GET[ 'tls-check' ] ) && 1 == $_GET[ 'tls-check' ] ){
                    // Then run our hooks.
                    add_filter( 'ninja_forms_register_actions', array( $this, 'register_actions' ) );
                    add_action( 'ninja_forms_builder_templates', array( $this, 'builder_templates' ) );
                    add_action( 'nf_admin_enqueue_scripts', array( $this, 'admin_scripts' ) );
                    add_action( 'wp_ajax_nf_helpscout_update_key', array( $this, 'update_key' ) );
                }
                // Throw our admin notice and return.
                add_action( 'admin_notices', array( $this, 'tls_notice' ) );
                return;
            }

                add_filter( 'ninja_forms_register_actions', array( $this, 'register_actions' ) );
                add_action( 'ninja_forms_builder_templates', array( $this, 'builder_templates' ) );
                add_action( 'nf_admin_enqueue_scripts', array( $this, 'admin_scripts' ) );
                add_action( 'wp_ajax_nf_helpscout_update_key', array( $this, 'update_key' ) );
        }

        /**************************************************
         * REGISTRATION METHODS
         *
         * These methods are used to register actions and
         * scripts.
         **************************************************/

        /**
         * Registers HelpScout action with Ninja Forms.
         * @since 3.0
         *
         * @param array $actions Gets array of actions from core and
         *                       appends the action we are building to this array.
         *
         *      $actions[ 'action_name' ] = action_data
         *
         * @return mixed(nested array) $actions list of actions with our action appended to the array.
         */
        public function register_actions( $actions )
        {
            $actions[ 'helpscout' ] = new NF_Helpscout_Actions_Helpscout();

            return $actions;
        }

        /**
         * Enqueues our builder scripts.
         * @since 3.0
         *
         * @return void
         */
        public function admin_scripts()
        {
            $ver = self::VERSION;
            wp_enqueue_style(  'nf-helpscout-builder',  plugin_dir_url( __FILE__ ) . 'assets/css/builder.css', array(), $ver );
            wp_enqueue_script( 'nf-helpscout-error',    plugin_dir_url( __FILE__ ) . 'assets/js/actionListener.js', array( 'nf-builder', 'jquery-effects-shake' ), $ver );
        }

        /**************************************************
         * API RELATED METHODS
         *
         * These methods are used either as helper to
         * communicate with the HelpScout API or to
         * Validate API keys.
         **************************************************/

        /**
         * Updates API key if a valid key is provide else sends an error message via AJAX.
         * @since 3.0
         *
         * @return false if user can't manage options, other wise @return void
         */
        public function update_key()
        {
            // Bail if user can't manage options.
            if( ! current_user_can( 'manage_options' ) ) return false;

            // Verify nonce.
            check_ajax_referer( 'ninja_forms_builder_nonce', 'security' );

            // Validate submitted API Key.
            $valid = NF_Helpscout()->validate_api_key( esc_html( $_REQUEST[ 'key' ] ) );

            // Setup our response array.
            $response = array(
                'valid_key'     => $valid,
                'message'       => '',
            );

            if( $valid ) {
                //Update settings key.
                Ninja_Forms()->update_setting( 'helpscout_api_key', esc_html( $_REQUEST[ 'key' ] ) );
            }else {
                // Set error message.
                $response[ 'message' ] = __( 'Your Helpscout API Key is invalid. ', 'ninja-forms-helpscout' );
            }

            // Send our json encoded response and die.
            echo json_encode( $response );
            die();
        }

        /**
         * Helper method that validates api keys.
         * @since 3.0
         *
         * @param string $key HelpScout API key
         *      $key = 'api_key'
         *
         * @return bool if response is 200 then @return TRUE else @return FALSE.
         */
        public function validate_api_key( $key )
        {
            // If NF_Helpscout doesn't exist bail.
            if( ! class_exists( 'NF_Helpscout' ) ) return false;

            // HelpScout endpoint
            $url = 'https://api.helpscout.net/v1/mailboxes.json';

            // Send request to HelpScout endpoint.
            $response= wp_remote_get( $url, array(
                'headers' => array(
                    "Authorization" => "Basic ". base64_encode( $key . ":X" ),
                    "content-type" => "application/json"
                )
            ) );

            // If the response is a WP Error...
            if( is_wp_error( $response ) ) {
                // ...then set a the transient value and return.
                set_transient( 'nf_helpscout_tls_check', 1, 86400);
                return array();
            }else {
                set_transient( 'nf_helpscout_tls_check', 0, 86400);
            }

            // Check for 200 in response and return bool accordingly.
            if( 200 == $response[ 'response' ][ 'code' ] ) {
                return true;
            } else {
                return false;
            }
        }

        /**
         * Build header for HelpScout API calls.
         * @since 3.0
         *
         * @return array $this->_api headers for calls to the HelpScout API.
         *      $this->_api = array
         *          'Authorization' => encoded API key.
         *          'content-type'  => 'application/json'
         */
        public function api_headers()
        {
            // Get key from settings.
            $api_key = trim( Ninja_Forms()->get_setting( 'helpscout_api_key' ) );

            // Build header.
            $this->_api = array(
                'Authorization' => 'Basic '. base64_encode( $api_key . ':X' ),
                'content-type' => 'application/json'
            );

            // Return header.
            return $this->_api;
        }


        /**
         * Pulls in the Mailboxes associated with the user's HelpScout account.
         * @since 3.0
         *
         * @return array(nested) $list_data builds settings used to populated HelpScout list data.
         */
        public function get_mailbox_list()
        {
            // HelpScout endpoint
            $url = 'https://api.helpscout.net/v1/mailboxes.json';

            // Send request to the mailbox endpoint.
            $response = wp_remote_get( $url, array(
                'headers' => $this->api_headers(),
            ) );

            // If the response is a WP Error...
            if( is_wp_error( $response ) ) {
                // ...then set a the transient value and return.
                set_transient( 'nf_helpscout_tls_check', 1, 86400);
                return array();
            } else {
                set_transient( 'nf_helpscout_tls_check', 0, 86400);
            }

            // If the response is 200 do stuff.
            if( 200 == $response[ 'response' ][ 'code' ] ) {
                // Decode the body of the response.
                $data = json_decode( $response[ 'body' ] );

                // This array is where we will build our list data.
                $list_data = array();

                // Loop over our decoded data to build our list
                foreach( $data->items as $item) {

                    // Create setting that we can use to grab list data later.
                    Ninja_Forms()->update_setting( 'helpscout_list_' . $item->id, $item->name );

                    // Build our list array.
                    $list_data[] = array(
                        'label'     => $item->name,
                        'value'     => $item->id,
                        'slug'      => $item->slug,
                        'email'     => $item->email,
                        'fields'    => array(
                            array(
                                'label' => __( 'First Name', 'ninja-forms-helpscout' ),
                                'value' => 'first_name',
                            ),
                            array(
                                'label' => __( 'Last Name', 'ninja-forms-helpscout' ),
                                'value' => 'last_name',
                            ),
                            array(
                                'label' => __( 'Email', 'ninja-forms-helpscout' ) . "<small style='color:red'>(required)</small>",
                                'value' => 'email',
                            ),
                            array(
                                'label' => __( 'Message Title', 'ninja-forms-helpscout' ),
                                'value' => 'title',
                            ),
                            array(
                                'label' => __( 'Message', 'ninja-forms-helpscout' ) . "<small style='color:red'>(required)</small>",
                                'value' => 'message',
                            ),
                        ),
                    );
                }
                return $list_data;
            }
        }

        /**
         * Send the users conversation off to the Helpscout API.
         * @since 3.0
         *
         * @param array $action_settings
         * @return array $response
         */
        public function send_conversation( $action_settings )
        {
            // Building the array to send to the API.
            $output = array(
                'type'      => 'email',
                'customer'  => (object)array(
                    'firstName' => $action_settings[ 'first_name' ],
                    'lastName'  => $action_settings[ 'last_name' ],
                    'email'     => $action_settings[ 'email' ],
                ),
                'subject'   => $action_settings[ 'title' ],
                'mailbox'   => (object)array(
                    'id'    => $action_settings[ 'newsletter_list']
                ),
                'threads' => array(
                    (object)array(
                        'type'      => 'customer',
                        'createdBy' => (object)array(
                            'firstName' => $action_settings[ 'first_name' ],
                            'lastName'  => $action_settings[ 'last_name' ],
                            'email'     => $action_settings[ 'email' ],
                            'type'      => 'customer'
                        ),
                        'body' => strip_tags( preg_replace( '/(<\/[^>]+?>)(<[^>\/][^>]*?>)/', '$1 $2'."\n\n", $action_settings[ 'message' ] ) )
                    )
                )
            );

            // JSON the body to be sent
            $output = json_encode( $output );

            // URL for creating a conversation
            $url = 'https://api.helpscout.net/v1/conversations.json';

            // Send data
            $response= wp_remote_post( $url, array(
                'headers'   => $this->api_headers(),
                'body'      => $output
            ) );

            return $response;
        }


        public function tls_notice()
        {
            ?>
            <div class="nf-admin-notice nf-admin-error error">
                <div class="nf-notice-logo"></div>
                <p class="nf-notice-title"><?php _e( 'Ninja Forms has detected an outdated TLS Version', 'ninja-forms-helpscout' ); ?></p>
                <div class="nf-notice-body"
                    <p>
                        <b><?php _e( 'Ninja Forms HelpScout and other add-ons will not function with outdated versions of TLS.', 'ninja-forms-helpscout' ); ?></b>
                        <?php _e( 'Please contact your host and have them update your environment to support TLS 1.2 and HTTP/1.1', 'ninja-forms-helpscout' ); ?>
                    </p>
                    <a href="<?php echo admin_url( 'admin.php?page=nf-settings&tls-check=1' ) ?>"><button style="float:left;" class="button-primary"><?php _e( 'Test TLS', 'ninja-forms-helpscout' );?></button></a>
                </div>
            </div>
            <?php
            wp_enqueue_style( 'nf-admin-notices', Ninja_Forms::$url .'assets/css/admin-notices.css?nf_ver=' . Ninja_Forms::VERSION );
        }

        /**************************************************
         * BOILER PLATE METHOD
         *
         * These methods were generated by kozo
         **************************************************/

        /**
         * Autoloads files that are passed in via their class name.
         * @since 3.0
         *
         * @param $class_name
         */
        public function autoloader( $class_name )
        {
            if ( class_exists( $class_name ) ) return;

            if ( false === strpos( $class_name, self::PREFIX ) ) return;

            $class_name = str_replace( self::PREFIX, '', $class_name );
            $classes_dir = realpath( plugin_dir_path(__FILE__) ) . DIRECTORY_SEPARATOR . 'includes' . DIRECTORY_SEPARATOR;
            $class_file = str_replace( '_', DIRECTORY_SEPARATOR, $class_name ) . '.php';

            if (file_exists($classes_dir . $class_file ) ) {
                require_once $classes_dir . $class_file;
            }
        }
        
        /**
         * Template
         *
         * @param string $file_name
         * @param array $data
         */
        public static function template( $file_name = '', array $data = array() )
        {
            if( ! $file_name ) return;

            extract( $data );

            include self::$dir . 'includes/Templates/' . $file_name;
        }
        
        /**
         * Config
         *
         * @param $file_name
         * @return mixed
         */
        public static function config( $file_name )
        {
            return include self::$dir . 'includes/Config/' . $file_name . '.php';
        }

        public function builder_templates()
        {
            NF_Helpscout::template( 'Modal.html.php' );
        }

        /*
         * Required methods for all extension.
         */

        public function setup_license()
        {
            if ( ! class_exists( 'NF_Extension_Updater' ) ) return;

            new NF_Extension_Updater( self::NAME, self::VERSION, self::AUTHOR, __FILE__, self::SLUG );
        }
    }

    /**
     * The main function responsible for returning The Highlander Plugin
     * Instance to functions everywhere.
     *
     * Use this function like you would a global variable, except without needing
     * to declare the global.
     *
     * @since 3.0
     * @return {class} Highlander Instance
     */
    function NF_Helpscout()
    {
        return NF_Helpscout::instance();
    }

    NF_Helpscout();
}

/**
 * NF Helpscout Upgrade Action
 *
 * All the logic for converting from 2.9.x to 3.0
 *
 * @param $action
 * @return array
 */
function NF_Helpscout_Upgrade_Action( $action ){
    if( 'helpscout' != $action[ 'type' ] ) return $action;

    //Build our Helpscout action.
    $data = array(
        'active'            => 1,
        'name'              => $action[ 'label' ],
        'type'              => 'helpscout',
        'email'             => '{field:email_' . $action[ 'helpscout_selector' ] . '}',
        'title'             => 'Submitted by Ninja Forms: Helpscout',
        'message'           => $action[ 'helpscout' ],
        'debug'             => $action[ 'helpscout_debug' ],
    );

    //Check to see if the mailbox id, has been set in the action.
    if( isset( $action[ 'settings-helpscout-mailbox' ] ) ) {
        $data[ 'newsletter_list' ] = $action[ 'settings-helpscout-mailbox' ];
    } else {
        $data[ 'newsletter_list' ] = get_option( 'nf_helpscout_mailbox' );
    }

    //Update the 3.0 API key.
    Ninja_Forms()->update_settings(
        array(
            'helpscout_api_key' => get_option( 'nf_helpscout_apikey' ),
        ) );

    //merge the data into the action array and return.
    $action = array_merge( $action, $data );
    return $action;
}

add_filter( 'ninja_forms_upgrade_action_helpscout', 'NF_Helpscout_Upgrade_Action' );