<?php

/**
 * Plugin Name: WP Rocket Integration
 * Plugin URI: https://mainwp.com
 * Description: WP Rocket Integration combines the power of your MainWP Dashboard with the popular WP Rocket Plugin. It allows you to mange WP Rocket settings and quickly Clear and Preload cache on your child sites.
 * Version: 5.1.5
 * Author: MainWP
 * Author URI: https://mainwp.com
 * Documentation URI: https://mainwp.com/kb/rocket-extension/
 * Icon URI:
 */

if ( ! defined( 'ABSPATH' ) ) {
    die( 'No direct access allowed' );
}

if ( ! defined( 'MAINWP_WP_ROCKET_PLUGIN_FILE' ) ) {
    define( 'MAINWP_WP_ROCKET_PLUGIN_FILE', __FILE__ );
}

if ( ! defined( 'MAINWP_WP_ROCKET_PATH' ) ) {
    define( 'MAINWP_WP_ROCKET_PATH', realpath( plugin_dir_path( MAINWP_WP_ROCKET_PLUGIN_FILE ) ) . '/' );
}

if ( ! defined( 'MAINWP_WP_ROCKET_URL' ) ) {
    define( 'MAINWP_WP_ROCKET_URL', plugins_url( '', __FILE__ ) );
}

if ( ! defined( 'MAINWP_WP_ROCKET_SLUG' ) ) {
    define( 'MAINWP_WP_ROCKET_SLUG', 'mainwp_wp_rocket_settings' );
}

if ( ! defined( 'MAINWP_ROCKET_GENERAL_SETTINGS' ) ) {
    define( 'MAINWP_ROCKET_GENERAL_SETTINGS', 'mainwp_rocket_general_settings' );
}

if ( ! defined( 'MAINWP_ROCKET_JAVASCRIPT_DELAY_EXCLUSIONS' ) ) {
    define( 'MAINWP_ROCKET_JAVASCRIPT_DELAY_EXCLUSIONS', 'mainwp_rocket_javaScript_delay_exclusions' );
}

if ( ! defined( 'MAINWP_ROCKET_FILE_PATH' ) ) {
    define( 'MAINWP_ROCKET_FILE_PATH', 'wp-rocket/wp-rocket.php' );
}

if ( ! defined( 'MAINWP_ROCKET_PREG_SANITIZE' ) ) {
    if ( defined( 'PREG_ROCKET_SANITIZE' ) ) {
        define( 'MAINWP_ROCKET_PREG_SANITIZE', PREG_ROCKET_SANITIZE );
    } else {
        define( 'MAINWP_ROCKET_PREG_SANITIZE', '#\?.*$#' );
    }
}


class MainWP_Rocket_Extension { //phpcs:ignore -- NOSONAR - register class.

    public $plugin_slug;
    public $wprocket_sites = null;
    public $version        = '3.2.1';

    const REQUIRES_MAINWP_VERSION       = '5.1';
    const REQUIRES_MAINWP_CHILD_VERSION = '5.2.1';

    public function __construct() {

        $this->plugin_slug = plugin_basename( __FILE__ );

        add_action( 'init', array( &$this, 'localization' ) );
        add_filter( 'plugin_row_meta', array( &$this, 'plugin_row_meta' ), 10, 2 );
        add_action( 'init', array( &$this, 'init' ) );
        add_action( 'admin_init', array( &$this, 'admin_init' ) );
        add_filter( 'mainwp_getsubpages_sites', array( &$this, 'managesites_subpage' ), 10, 1 );
        add_action( 'admin_notices', array( &$this, 'error_notice' ) );
        MainWP_Rocket_DB::get_instance()->install();
        MainWP_Rocket_Based::get_instance();
    }

    /**
     * MainWP Sucuri Extension localization.
     */
    public function localization() {
        load_plugin_textdomain( 'mainwp-rocket-extension', false, dirname( plugin_basename( __FILE__ ) ) . '/languages/' );
    }

    public function plugin_row_meta( $plugin_meta, $plugin_file ) {

        if ( $this->plugin_slug != $plugin_file ) {
            return $plugin_meta;
        }

        $slug     = basename( $plugin_file, '.php' );
        $api_data = get_option( $slug . '_APIManAdder' );
        if ( ! is_array( $api_data ) || ! isset( $api_data['activated_key'] ) || $api_data['activated_key'] != 'Activated' || ! isset( $api_data['api_key'] ) || empty( $api_data['api_key'] ) ) {
            return $plugin_meta;
        }

        $plugin_meta[] = '<a href="?do=checkUpgrade" title="Check for updates.">Check for updates now</a>';
        return $plugin_meta;
    }

    /**
     * init plugin
     *
     * @return void
     */
    public function init() {
    }

    public function managesites_subpage( $subPage ) {
        $subPage[] = array(
            'title'       => __( 'WP Rocket', 'mainwp-rocket-extension' ),
            'slug'        => 'WPRocket',
            'sitetab'     => true,
            'menu_hidden' => true,
            'callback'    => array( 'MainWP_Rocket', 'render' ),
        );
        return $subPage;
    }

    public function admin_init() {
        if ( isset( $_GET['page'] ) && ( 'Extensions-Mainwp-Rocket-Extension' == $_GET['page'] || 'managesites' == $_GET['page'] || 'ManageSitesWPRocket' == $_GET['page'] || 'mainwp_tab' == $_GET['page'] ) ) {
            wp_enqueue_style( 'mainwp-rocket-extension', MAINWP_WP_ROCKET_URL . '/css/mainwp-rocket.css', array(), $this->version );
            wp_enqueue_script( 'mainwp-rocket-extension', MAINWP_WP_ROCKET_URL . '/js/mainwp-rocket.js', array( 'jquery' ), $this->version );
        }

        wp_localize_script( 'mainwp-rocket-extension', 'mainwp_rocket_loc', array( 'nonce' => wp_create_nonce( 'mainwp_rocket_nonce' ) ) );

        MainWP_Rocket::get_instance()->admin_init();
        MainWP_Rocket_Plugin::get_instance()->admin_init();
    }

    public function error_notice() {
        if ( self::is_mainwp_pages() && version_compare( self::REQUIRES_MAINWP_VERSION, get_option( 'mainwp_plugin_version' ), '>' ) ) {
            echo '<div class="ui red message">' . sprintf( __( 'MainWP Rocket Add-on requires MainWP Dashboard plugin version %s to be installed on your dashboard site. Please update MainWP Dashboard plugin!', 'mainwp-rocket-extension' ), self::REQUIRES_MAINWP_VERSION ) . '</div>';
        }
    }

    public static function is_mainwp_pages() {
        $current_screen = get_current_screen();
        if ( $current_screen->parent_base == 'mainwp_tab' ) {
            return true;
        }
        return false;
    }
}

class MainWP_Rocket_Extension_Activator { //phpcs:ignore -- NOSONAR - multi methods.
    protected $mainwpMainActivated = false;
    protected $childEnabled        = false;
    protected $childKey            = false;
    protected $childFile;
    protected $plugin_handle    = 'mainwp-rocket-extension';
    protected $product_id       = 'MainWP Rocket Extension';
    protected $software_version = '5.1.5';

    public function __construct() {

        $this->childFile = __FILE__;
        $this->includes();

        spl_autoload_register( array( $this, 'autoload' ) );
        register_activation_hook( __FILE__, array( $this, 'activate' ) );
        register_deactivation_hook( __FILE__, array( $this, 'deactivate' ) );

        add_filter( 'mainwp_getextensions', array( &$this, 'get_this_extension' ) );
        $this->mainwpMainActivated = apply_filters( 'mainwp_activated_check', false );

        if ( $this->mainwpMainActivated !== false ) {
            $this->activate_this_plugin();
        } else {
            add_action( 'mainwp_activated', array( &$this, 'activate_this_plugin' ) );
        }

        add_action( 'admin_notices', array( &$this, 'mainwp_error_notice' ) );
    }

    public function autoload( $class_name ) {
        $allowedLoadingTypes = array( 'class' );
        $class_name          = str_replace( '_', '-', strtolower( $class_name ) );
        foreach ( $allowedLoadingTypes as $allowedLoadingType ) {
            $class_file = WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . str_replace( basename( __FILE__ ), '', plugin_basename( __FILE__ ) ) . $allowedLoadingType . DIRECTORY_SEPARATOR . $class_name . '.class.php';
            if ( file_exists( $class_file ) ) {
                include_once $class_file; //phpcs:ignore -- NOSONAR - include_once.
            }
        }
    }

    public function includes() {
      include_once 'includes/formatting.php'; //phpcs:ignore -- NOSONAR - include_once.
      include_once 'includes/functions.php'; //phpcs:ignore -- NOSONAR - include_once.
    }

    public function get_this_extension( $pArray ) {
        $pArray[] = array(
            'plugin'     => __FILE__,
            'api'        => $this->plugin_handle,
            'mainwp'     => true,
            'callback'   => array( &$this, 'settings' ),
            'apiManager' => true,
            'name'       => 'WP Rocket Integration',
        );
        return $pArray;
    }

    public function settings() {
        do_action( 'mainwp_pageheader_extensions', __FILE__ );
        MainWP_Rocket::render();
        do_action( 'mainwp_pagefooter_extensions', __FILE__ );
    }

    public function activate_this_plugin() {
        $this->mainwpMainActivated = apply_filters( 'mainwp_activated_check', $this->mainwpMainActivated );
        $this->childEnabled        = apply_filters( 'mainwp_extension_enabled_check', __FILE__ );
        $this->childKey            = $this->childEnabled['key'];
        if ( function_exists( 'mainwp_current_user_can' ) && ! mainwp_current_user_can( 'extension', 'mainwp-rocket-extension' ) ) {
            return;
        }

      new MainWP_Rocket_Extension(); //phpcs:ignore -- NOSONAR - register class.
    }

    public function get_child_key() {
        return $this->childKey;
    }

    public function get_child_file() {
        return $this->childFile;
    }

    public function mainwp_error_notice() {
        global $current_screen;
        if ( $current_screen->parent_base == 'plugins' && $this->mainwpMainActivated === false ) {
            echo '<div class="error"><p>WP Rocket Integration ' . __( 'requires <a href="https://mainwp.com/" target="_blank">MainWP Dashboard Plugin</a> to be activated in order to work. Please install and activate <a href="https://mainwp.com/" target="_blank">MainWP Dashboard Plugin</a> first.' ) . '</p></div>';
        }
    }

    public function activate() {
        $options = array(
            'product_id'       => $this->product_id,
            'software_version' => $this->software_version,
        );
        do_action( 'mainwp_activate_extention', $this->plugin_handle, $options );
    }

    public function deactivate() {
        do_action( 'mainwp_deactivate_extention', $this->plugin_handle );
    }
}

global $mainWPRocketExtensionActivator;
$mainWPRocketExtensionActivator = new MainWP_Rocket_Extension_Activator();
