<?php
/**
 * Credits
 *
 * Plugin-Name: WP Rocket
 * Plugin URI: https://wp-rocket.me
 * Description: The best WordPress performance plugin.
 * Version: 3.3.3.1
 * Code Name: Dagobah
 * Author: WP Media
 * Author URI: https://wp-media.me
 * Licence: GPLv2 or later
 */

class MainWP_Rocket_Based { //phpcs:ignore -- NOSONAR - multi methods.

  // phpcs:disable WordPress.Security.NonceVerification, WordPress.Security.SafeRedirect

    /**
     * Public static variable to hold the single instance of the class.
     *
     * @var mixed Default null
     */
    private static $instance = null;

    /**
     * Get the only instance of the class.
     *
     * @return self The only instance of a class
     */
    public static function get_instance() {
        if ( null == self::$instance ) {
            self::$instance = new MainWP_Rocket_Based();
        }
        return self::$instance;
    }

    /**
     * Initialize the class and add action.
     */
    public function __construct() {
        add_action( 'admin_init', array( $this, 'register_setting' ) );
        add_action( 'admin_post_mainwp_rocket_export', array( $this, 'do_options_export' ) );
        add_action( 'admin_post_mainwp_rocket_optimize_database', array( $this, 'do_optimize_database' ) );
        add_action( 'admin_post_mainwp_rocket_update_exclusion_list', array( $this, 'do_update_exclusion_list' ) );
    }

    /**
     * Register setting .
     */
    public function register_setting() {
        register_setting( 'mainwp_wp_rocket', MAINWP_WP_ROCKET_SLUG, array( $this, 'settings_callback' ) );
    }

    /**
     * Check security request
     */
    private function check_security_request() {
        return apply_filters( 'mainwp_check_security_request', false, 'mainwp_rocket_post_nonce', 'mainwp_rocket_post_nonce' );
    }

    /**
     * Handle import
     *
     * @return void
     */
    private function handle_import() {
        if ( isset( $_FILES['import'] ) && isset( $_POST['import'] ) ) {
            $this->handle_settings_import();
            return true;
        }
        return false;
    }

    /**
     * Process general minify css js options
     *
     * @param array $inputs
     */
    private function process_general_minify_css_js_options( $inputs ) {
        $inputs['minify_css'] = ! empty( $inputs['minify_css'] ) ? 1 : 0;
        $inputs['minify_js']  = ! empty( $inputs['minify_js'] ) ? 1 : 0;
        return $inputs;
    }

    /**
     * Process general purge delay option
     *
     * @param array $inputs
     */
    private function process_general_purge_delay_options( $inputs ) {
        $inputs['purge_cron_interval'] = isset( $inputs['purge_cron_interval'] ) ? (int) $inputs['purge_cron_interval'] : mainwp_get_rocket_option( 'purge_cron_interval' );
        $inputs['purge_cron_unit']     = isset( $inputs['purge_cron_unit'] ) ? $inputs['purge_cron_unit'] : mainwp_get_rocket_option( 'purge_cron_unit' );
        return $inputs;
    }
    /**
     * Process dns prefetch
     *
     * @param  array $inputs
     * @return void
     */
    private function process_dns_prefetch( $inputs ) {
        if ( ! empty( $inputs['dns_prefetch'] ) ) {
            if ( ! is_array( $inputs['dns_prefetch'] ) ) {
                $inputs['dns_prefetch'] = explode( "\n", $inputs['dns_prefetch'] );
            }
            $inputs['dns_prefetch'] = array_map( 'trim', $inputs['dns_prefetch'] );
            $inputs['dns_prefetch'] = array_map( 'esc_url', $inputs['dns_prefetch'] );
            $inputs['dns_prefetch'] = (array) array_filter( $inputs['dns_prefetch'] );
            $inputs['dns_prefetch'] = array_unique( $inputs['dns_prefetch'] );
        } else {
            $inputs['dns_prefetch'] = array();
        }

        return $inputs;
    }

    /**
     * Process cache options
     *
     * @param array $inputs
     */
    public function process_cache_options( $inputs ) {
        // Old Option : Empty the cache of the following pages when updating an article
        // $inputs['cache_purge_pages'] = ! empty( $input['cache_purge_pages'] ) ? mainwp_rocket_sanitize_textarea_field( 'cache_purge_pages', $inputs['cache_purge_pages'] ) : array();

        $inputs['cache_purge_pages'] = mainwp_rocket_sanitize_textarea_field( 'cache_purge_pages', $inputs['cache_purge_pages'] );

        /*
        * Option : Never cache the following pages
        */

        // Old option : Never cache the following pages
        // $inputs['cache_reject_uri'] = ! empty( $input['cache_reject_uri'] ) ? mainwp_rocket_sanitize_textarea_field( 'cache_reject_uri', $inputs['cache_reject_uri'] ) : array();

        $inputs['cache_reject_uri'] = mainwp_rocket_sanitize_textarea_field( 'cache_reject_uri', $inputs['cache_reject_uri'] );

        /*
        * Option : Don't cache pages that use the following cookies
        */
        // Old option : Don't cache pages that use the following cookies
        // $inputs['cache_reject_cookies'] = ! empty( $input['cache_reject_cookies'] ) ? mainwp_rocket_sanitize_textarea_field( 'cache_reject_cookies', $inputs['cache_reject_cookies'] ) : array();

        $inputs['cache_reject_cookies'] = mainwp_rocket_sanitize_textarea_field( 'cache_reject_cookies', $inputs['cache_reject_cookies'] );

        /*
        * Option : Cache pages that use the following query strings (GET parameters)
        */

        // Old option : Cache pages that use the following query strings (GET parameters)
        // $inputs['cache_query_strings'] = ! empty( $input['cache_query_strings'] ) ? mainwp_rocket_sanitize_textarea_field( 'cache_query_strings', $inputs['cache_query_strings'] ) : array();

        $inputs['cache_query_strings'] = mainwp_rocket_sanitize_textarea_field( 'cache_query_strings', $inputs['cache_query_strings'] );

        /*
        * Option : Never send cache pages for these user agents
        */
        // Old Option : Never send cache pages for these user agents
        // $inputs['cache_reject_ua'] = ! empty( $input['cache_reject_ua'] ) ? mainwp_rocket_sanitize_textarea_field( 'cache_reject_ua', $inputs['cache_reject_ua'] ) : array();

        $inputs['cache_reject_ua'] = mainwp_rocket_sanitize_textarea_field( 'cache_reject_ua', $inputs['cache_reject_ua'] );

        return $inputs;
    }

    /**
     *  Process preload options
     *
     * @param array $inputs
     */
    private function process_preload_options( $inputs ) {
        /**
         * Options: excluded uri preload
         */

        $inputs['preload_excluded_uri'] = mainwp_rocket_sanitize_textarea_field( 'preload_excluded_uri', $inputs['preload_excluded_uri'] );

        /**
         * Options: Activate bot preload
         */
        $inputs['manual_preload']    = ! empty( $inputs['manual_preload'] ) ? 1 : 0;
        $inputs['automatic_preload'] = ! empty( $inputs['automatic_preload'] ) ? 1 : 0;

        /*
        * Option: activate sitemap preload
        */
        $inputs['sitemap_preload'] = ! empty( $inputs['sitemap_preload'] ) ? 1 : 0;

        return $inputs;
    }

    /**
     * Process js options
     *
     * @param array $inputs
     */
    private function process_js_options( $inputs ) {
        /*
        * Option : css files to exclude of the minification
        */
        $inputs['exclude_css'] = mainwp_rocket_sanitize_textarea_field( 'exclude_css', $inputs['exclude_css'] );

        // Option : JS files to exclude from the minification.
        $inputs['exclude_js'] = ! empty( $inputs['exclude_js'] ) ? mainwp_rocket_sanitize_textarea_field( 'exclude_js', $inputs['exclude_js'] ) : array();

        // Option: inline JS patterns to exclude from combine JS.

        $inputs['exclude_inline_js'] = mainwp_rocket_sanitize_textarea_field( 'exclude_inline_js', $inputs['exclude_inline_js'] );

        // Option: Critical CSS.
        $input['critical_css'] = ! empty( $inputs['critical_css'] ) ? wp_strip_all_tags( str_replace( array( '<style>', '</style>' ), '', $inputs['critical_css'] ), array( "\'", '\"' ) ) : '';

        $inputs['defer_all_js'] = ! empty( $inputs['defer_all_js'] ) ? 1 : 0;

        /*
        * Option : JS files with deferred loading
        */
        $inputs['exclude_defer_js'] = mainwp_rocket_sanitize_textarea_field( 'exclude_defer_js', $inputs['exclude_defer_js'] );

        $inputs['delay_js'] = ! empty( $inputs['delay_js'] ) ? 1 : 0;

        /*
        * Option : JS files with deferred loading
        */
        $inputs['delay_js_scripts'] = ! empty( $input['delay_js_scripts'] ) ? mainwp_rocket_sanitize_textarea_field( 'delay_js_scripts', $inputs['delay_js_scripts'] ) : array();

        /*
        * Option : JS files with deferred loading
        */
        $inputs['exclude_lazyload'] = mainwp_rocket_sanitize_textarea_field( 'exclude_lazyload', $inputs['exclude_lazyload'] );

        return $inputs;
    }

    /**
     * Process database options
     *
     * @param array $inputs
     */
    private function process_database_options( $inputs ) {
        $inputs['database_revisions']          = ! empty( $inputs['database_revisions'] ) ? 1 : 0;
        $inputs['database_auto_drafts']        = ! empty( $inputs['database_auto_drafts'] ) ? 1 : 0;
        $inputs['database_trashed_posts']      = ! empty( $inputs['database_trashed_posts'] ) ? 1 : 0;
        $inputs['database_spam_comments']      = ! empty( $inputs['database_spam_comments'] ) ? 1 : 0;
        $inputs['database_trashed_comments']   = ! empty( $inputs['database_trashed_comments'] ) ? 1 : 0;
        $inputs['database_expired_transients'] = ! empty( $inputs['database_expired_transients'] ) ? 1 : 0;
        $inputs['database_all_transients']     = ! empty( $inputs['database_all_transients'] ) ? 1 : 0;
        $inputs['database_optimize_tables']    = ! empty( $inputs['database_optimize_tables'] ) ? 1 : 0;
        $inputs['schedule_automatic_cleanup']  = ! empty( $inputs['schedule_automatic_cleanup'] ) ? 1 : 0;

        if ( $inputs['schedule_automatic_cleanup'] != 1 && ( 'daily' != $inputs['automatic_cleanup_frequency'] || 'weekly' != $inputs['automatic_cleanup_frequency'] || 'monthly' != $inputs['automatic_cleanup_frequency'] ) ) {
            unset( $inputs['automatic_cleanup_frequency'] );
        }
        return $inputs;
    }

    /**
     * Process sitemaps options
     *
     * @param array $inputs input data.
     */
    private function process_sitemaps_options( $inputs ) {
        if ( ! empty( $inputs['sitemaps'] ) ) {
            if ( ! is_array( $inputs['sitemaps'] ) ) {
                $inputs['sitemaps'] = explode( "\n", $inputs['sitemaps'] );
            }
            $inputs['sitemaps'] = array_map( 'trim', $inputs['sitemaps'] );
            $inputs['sitemaps'] = (array) array_filter( $inputs['sitemaps'] );
            $inputs['sitemaps'] = array_unique( $inputs['sitemaps'] );
        } else {
            $inputs['sitemaps'] = array();
        }

        return $inputs;
    }

    /**
     * Process cdn options
     *
     * @param array $inputs input data.
     */
    public function process_cdn_options( $inputs ) {
        $inputs['cdn_cnames'] = ! empty( $inputs['cdn_cnames'] ) ? array_unique( array_filter( $inputs['cdn_cnames'] ) ) : array();

        if ( ! $inputs['cdn_cnames'] ) {
            $inputs['cdn_zone'] = array();
        } else {
            $max_key = max( array_keys( $inputs['cdn_cnames'] ) );
            for ( $i = 0; $i <= $max_key; $i++ ) {
                if ( ! isset( $inputs['cdn_cnames'][ $i ] ) ) {
                    unset( $inputs['cdn_zone'][ $i ] );
                } else {
                    $inputs['cdn_zone'][ $i ] = isset( $inputs['cdn_zone'][ $i ] ) ? $inputs['cdn_zone'][ $i ] : 'all';
                }
            }

            $inputs['cdn_cnames'] = array_values( $inputs['cdn_cnames'] );
            ksort( $inputs['cdn_zone'] );
            $inputs['cdn_zone'] = array_values( $inputs['cdn_zone'] );
        }

        // Option : Files to exclude from the CDN process.
        if ( ! empty( $inputs['cdn_reject_files'] ) ) {
            $inputs['cdn_reject_files'] = mainwp_rocket_sanitize_textarea_field( 'cdn_reject_files', $inputs['cdn_reject_files'] );
        } else {
            $inputs['cdn_reject_files'] = array();
        }

        return $inputs;
    }

    /**
     * Process support options
     *
     * @param array $inputs input data.
     */
    public function process_support_options( $inputs ) {

        $fake_options = array(
            'support_summary',
            'support_description',
            'support_documentation_validation',
        );

        foreach ( $fake_options as $option ) {
            if ( isset( $inputs[ $option ] ) ) {
                unset( $inputs[ $option ] );
            }
        }

        return $inputs;
    }

    /**
     * Update settings
     *
     * @param array $inputs input data.
     */
    private function update_settings( $inputs ) {
        $site_id = ! empty( $_POST['mainwp_rocket_current_site_id'] ) ? sanitize_text_field( wp_unslash( $_POST['mainwp_rocket_current_site_id'] ) ) : 0;
        if ( MainWP_Rocket::is_manage_sites_page() && $site_id ) {
            $update = array(
                'site_id'  => $site_id,
                'settings' => base64_encode( serialize( $inputs ) ), // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions -- base64_encode function is used for http encode compatible..
            );
            MainWP_Rocket_DB::get_instance()->update_wprocket( $update );
        } else {
            update_option( MAINWP_ROCKET_GENERAL_SETTINGS, $inputs );
        }
    }

    /**
     * Redirect after save
     */
    private function redirect_after_save() {

        wp_redirect(
            add_query_arg(
                array(
                    '_perform_action' => 'mainwp_rocket_save_opts_child_sites',
                    '_wpnonce'        => wp_create_nonce( 'mainwp_rocket_save_opts_child_sites' ),
                    'optimize_db'     => ! empty( $_POST['mainwp_wp_rocket_settings']['submit_optimize'] ) ? true : false,
                    'message'         => 1,
                ),
                remove_query_arg( 's', wp_get_referer() )
            )
        );
    }

    /**
     * Used to clean and sanitize the settings fields
     *
     * @param array $inputs data input.
     */
    public function settings_callback( $inputs ) {
        // mainwp check security request.
        if ( ! $this->check_security_request() ) {
            wp_die( 'Invalid request.' );
            return;
        }

        if ( $this->handle_import() ) {
            return; // all proccess in handle function.
        }

        /*
        * Option : Minification CSS & JS
        */
        $inputs = $this->process_general_minify_css_js_options( $inputs );

        /*
        * Option : Purge delay
        */
        $inputs = $this->process_general_purge_delay_options( $inputs );

        /*
        * Option : Prefetch DNS requests
        */
        $inputs = $this->process_dns_prefetch( $inputs );

        /*
        * Option : Empty the cache of the following pages when updating an article
        */
        $inputs = $this->process_cache_options( $inputs );

        /**
         * Option : Preload
         */
        $inputs = $this->process_preload_options( $inputs );

        /*
        * Option : CSS files to exclude of the minification
        */

        $inputs = $this->process_js_options( $inputs );

        /**
         * Database options
         */

        $inputs = $this->process_database_options( $inputs );

        /*
        * Option : XML sitemaps URLs
        */

        $inputs = $this->process_sitemaps_options( $inputs );

        /*
        * Option : CDN
        */

        $inputs = $this->process_cdn_options( $inputs );

        /*
        * Option: Support
        */

        $inputs = $this->process_support_options( $inputs );

        /*
        * update_settings
        */
        $this->update_settings( $inputs );
        $this->redirect_after_save();

        exit();
    }

    /**
     * Sanitize excluded files
     *
     * @param mixed $data file.
     * @param mixed $field file name.
     * @return array files
     */
    public function sanitize_excluded_files( $data, $field ) {
        $return = array();
        if ( ! empty( $data[ $field ] ) ) {
            if ( ! is_array( $data[ $field ] ) ) {
                $data[ $field ] = explode( "\n", $data[ $field ] );
            }
            $return = array_map( 'trim', $data[ $field ] );
            $return = array_map( 'mainwp_rocket_clean_exclude_file', $return );
            $return = (array) array_filter( $return );
            $return = array_unique( $return );
        }
        return $return;
    }

    /**
     * Do options export.
     */
    public function do_options_export() {

        if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'mainwp_rocket_export' ) ) {
            return;
        }

        $options = null;
        if ( isset( $_GET['id'] ) ) {
            $options = MainWP_Rocket_DB::get_instance()->get_wprocket_settings_by( 'site_id', sanitize_text_field( wp_unslash( $_GET['id'] ) ) );
        } else {
            $options = get_option( MAINWP_ROCKET_GENERAL_SETTINGS );
        }

        if ( empty( $options ) ) {
            return;
        }

        $filename = sprintf( 'wp-rocket-settings-%s-%s.json', gmdate( 'Y-m-d' ), uniqid() );
        $options  = wp_json_encode( $options ); // do not use get_rocket_option() here.
        nocache_headers();
        header( 'Content-Type: application/json' );
        header( 'Content-Disposition: attachment; filename="' . esc_attr( $filename ) . '"' );
        header( 'Content-Transfer-Encoding: binary' );
        header( 'Content-Length: ' . esc_attr( strlen( $options ) ) );
        header( 'Connection: close' );
        echo wp_kses_post( $options );
        exit();
    }

    /**
     * [Update inclusion and exclusion lists for do_update_exclusion_list]
     *
     * @return void
     */
    public function do_update_exclusion_list() {
        if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'mainwp_rocket_nonce_update_exclusion_list' ) ) {
            wp_nonce_ays( '' );
        }
        wp_redirect(
            add_query_arg(
                array(
                    '_perform_action' => 'mainwp_rocket_update_exclusion_list',
                    '_wpnonce'        => wp_create_nonce( 'mainwp_rocket_update_exclusion_list' ),
                ),
                remove_query_arg( 's', wp_get_referer() )
            )
        );
        die();
    }

    /**
     * Optimize database
     */
    public function do_optimize_database() {
        if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'mainwp_rocket_nonce_optimize_database' ) ) {
            wp_nonce_ays( '' );
        }

        wp_redirect(
            add_query_arg(
                array(
                    '_perform_action' => 'mainwp_rocket_optimize_database',
                    '_wpnonce'        => wp_create_nonce( 'mainwp_rocket_optimize_database' ),
                ),
                remove_query_arg( 's', wp_get_referer() )
            )
        );
        die();
    }

    /**
     * Validate uploaded file settings import
     *
     * @return bool
     */
    private function validate_uploaded_file_settings_import() {
        if (
        ! isset( $_FILES['import'] ) ||
        ! isset( $_FILES['import']['size'], $_FILES['import']['name'] ) ||
        0 === $_FILES['import']['size']
        ) {
            mainwp_rocket_settings_import_redirect( __( 'Settings import failed: no file uploaded.', 'rocket' ), 'error' );
            return false;
        }

        $file_name = sanitize_file_name( $_FILES['import']['name'] );

        if ( ! preg_match( '/wp-rocket-settings-20\d{2}-\d{2}-\d{2}-[a-f0-9]{13}\.(?:txt|json)/', $file_name ) ) {
            mainwp_rocket_settings_import_redirect( __( 'Settings import failed: incorrect filename.', 'rocket' ), 'error' );
            return false;
        }

        return true;
    }

    /**
     * Handle file upload settings import
     */
    private function handle_file_upload_settings_import() {
        add_filter( 'mime_types', 'mainwp_rocket_allow_json_mime_type' );
        add_filter( 'wp_check_filetype_and_ext', 'mainwp_rocket_check_json_filetype', 10, 4 );

        $file_data = array();
        if ( ! empty( $_FILES['import'] ) ) {
            $tmp_name  = ! empty( $_FILES['import']['tmp_name'] ) ? sanitize_file_name( $_FILES['import']['tmp_name'] ) : '';
            $name      = ! empty( $_FILES['import']['name'] ) ? sanitize_file_name( $_FILES['import']['name'] ) : '';
            $file_data = wp_check_filetype_and_ext( $tmp_name, $name );
        }

        if ( ! empty( $file_data ) && is_array( $file_data ) && ! in_array( $file_data['type'], array( 'text/plain', 'application/json' ), true ) ) {
            remove_filter( 'mime_types', 'mainwp_rocket_allow_json_mime_type' );
            remove_filter( 'wp_check_filetype_and_ext', 'mainwp_rocket_check_json_filetype', 10 );
            mainwp_rocket_settings_import_redirect( __( 'Settings import failed: incorrect filetype.', 'rocket' ), 'error' );
            return new WP_Error( 'incorrect_filetype', 'Incorrect file type' );
        }

        $_post_action    = ! empty( $_POST['action'] ) ? sanitize_text_field( wp_unslash( $_POST['action'] ) ) : '';
        $_POST['action'] = 'wp_handle_sideload';
        $file            = ! empty( $_FILES['import'] ) ? wp_handle_sideload( $_FILES['import'] ) : array();
        $_POST['action'] = $_post_action;

        remove_filter( 'mime_types', 'mainwp_rocket_allow_json_mime_type' );
        remove_filter( 'wp_check_filetype_and_ext', 'mainwp_rocket_check_json_filetype', 10 );

        if ( ! empty( $file['error'] ) ) {
            return new WP_Error( 'upload_error', $file['error'] );
        }

        return $file;
    }

    /**
     * Process file content settings import.
     *
     * @param mixed $file_data file data.
     */
    private function process_file_content_settings_import( $file_data ) {
        $settings  = mainwp_rocket_direct_filesystem()->get_contents( $file_data['file'] );
        $file_type = wp_check_filetype_and_ext( $file_data['file'], $file_data['name'] )['type'];

        if ( 'text/plain' === $file_type ) {
            $settings = 'gz' . strrev( 'etalfni' )( $settings );
            $settings = maybe_unserialize( $settings );
        } elseif ( 'application/json' === $file_type ) {
            $settings = json_decode( $settings, true );

            if ( null === $settings ) {
                mainwp_rocket_settings_import_redirect( __( 'Settings import failed: unexpected file content.', 'rocket' ), 'error' );
            }
        }

        mainwp_rocket_put_content( $file_data['file'], '' );
        mainwp_rocket_direct_filesystem()->delete( $file_data['file'] );

        return $settings;
    }

    /**
     * Update settings import
     *
     * @param mixed $settings settings data.
     */
    private function update_settings_import( $settings ) {
        if ( is_array( $settings ) ) {
            if ( MainWP_Rocket::is_manage_sites_page() && isset( $site_id ) && ! empty( $site_id ) ) {
                $update = array(
                    'site_id'  => $site_id,
                    'settings' => base64_encode( serialize( $settings ) ), // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions -- base64_encode function is used for http encode compatible..
                );
                MainWP_Rocket_DB::get_instance()->update_wprocket( $update );
            }
        } else {
            update_option( MAINWP_ROCKET_GENERAL_SETTINGS, $settings );
        }
    }

    /**
     * Redirect after import
     */
    private function redirect_after_import() {
        wp_safe_redirect(
            add_query_arg(
                array(
                    'message' => 2,
                ),
                remove_query_arg( 's', wp_get_referer() )
            )
        );
    }
    /**
     * [Settings import file for handle_settings_import]
     */
    public function handle_settings_import() {

        // Check uploaded file.
        if ( ! $this->validate_uploaded_file_settings_import() ) {
            return;
        }
        // File upload processing.
        $file_data = $this->handle_file_upload_settings_import();
        if ( is_wp_error( $file_data ) ) {
            mainwp_rocket_settings_import_redirect( __( 'Settings import failed: ', 'rocket' ) . $file_data->get_error_message(), 'error' );
            return;
        }
        // File content processing.
        $settings = $this->process_file_content_settings_import( $file_data );
        // update setting.

        $this->update_settings_import( $settings );
        $this->redirect_after_import();

        exit();
    }
}
