<?php
/**
 * SCREETS © 2018
 *
 * SCREETS, d.o.o. Sarajevo. All rights reserved.
 * This  is  commercial  software,  only  users  who have purchased a valid
 * license  and  accept  to the terms of the  License Agreement can install
 * and use this program.
 *
 * @package LiveChatX
 * @author Screets
 *
 */

if ( ! defined( 'ABSPATH' ) ) { exit; }

/**
 * Front-end class.
 *
 * @since Live Chat X (2.4.0)
 *
 */
class LiveChatX_Frontend extends LiveChatX_Abstract
{	

	private $display = false;

	/**
	 * Constructor.
	 */
	public function __construct() {
		
		if( ! LiveChatX()->is_request( 'ajax' ) ) {

			// Don't load application for older than IE11
			if( preg_match( '/(?i)msie [8\-10]/',$_SERVER['HTTP_USER_AGENT'] ) ) {
				return;
			}

			// Actions
			$this->addAction( 'wp_enqueue_scripts', 'isDisplay', 100 );
			$this->addAction( 'wp_enqueue_scripts', 'assets', 110 );
			$this->addAction( 'wp_footer', 'render', 120 );

		}
	}

	/**
	 * Check display options.
	 *
	 * @since LiveChatX (1.0)
	 * @return void
	 */
	public function isDisplay() {

		$db = lcx_get_option( 'realtime', 'firebase_apiKey' );
		$basics = lcx_get_option( 'general', 'display_basics' );
		$display_type = lcx_get_option( 'general', 'display_type' );
		$exceptPages = lcx_get_option( 'general', 'display_except_pages' );
		$blog = lcx_get_option( 'general', 'display_blog' );

		if( empty( $db ) ) {
			$this->display = false;
			return;
		}


		// Show except
		switch( $display_type ) {

			case 'show':
				if( !empty( $exceptPages ) ) {
					if( !is_page( $exceptPages ) ) {
						$this->display = true;
					}
				} else {
					$this->display = true;
				}

				// Force to hide blog-related pages
				if( $blog == 'hide' ) {
					if( fn_lcx_is_blog_page() )
						$this->display = false;
				}

				break;

			case 'hide':
				if( !empty( $exceptPages ) ) {
					if( is_page( $exceptPages ) ) {
						$this->display = true;
					}
				}

				// Force to show in blog-related pages
				if( $blog == 'show' ) {
					if( fn_lcx_is_blog_page() )
						$this->display = true;
				}
				break;
		}

		// 
		// Check basic display settings.
		// 
		if( !empty( $basics ) ) {

			if( $this->display === true ) {

				// Hide on mobile devices.
	 			if( in_array( 'hideMobile', $basics ) ) {
					if( fn_lcx_is_mobile() ) {
						$this->display = false;
					}
				}

				// Hide on SSL pages.
	 			if( in_array( 'hideSSL', $basics ) ) {
					if( is_ssl() ) {
						$this->display = false;
					}
				}

			}
		}

		// 
		// Check for homepage.
		// 
		if( is_home() || is_front_page() ) {
			$home = lcx_get_option( 'general', 'display_home' );

			switch( $home ) {
				case 'show':
					$this->display = true;
					break;
				case 'hide':
					$this->display = false;
					break;
			}
		}
	}

	/**
	 * Load css/js files.
	 *
	 * @access public
	 */
	public function assets() {

		if( !$this->display )
			return;

		$upload_uri = fn_lcx_get_upload_dir_var( 'baseurl', '/lcx' );

		wp_register_style(
			'lcx-app',
			$upload_uri . '/app.css', 
			null,
			LCX_VERSION
		);
		wp_enqueue_style( 'lcx-app' );

		// Is developing mode?
		$dev_mode = lcx_get_option( 'advanced', 'secure_dev_mode' );

		// 
		// Production mode
		// 
		if( empty( $dev_mode ) ) {
			
			wp_register_script(
				'lcx-app',
				LCX_URL . '/assets/js/screets.chat.min.js',
				null,
				LCX_VERSION
			);
			wp_enqueue_script( 'lcx-app' );

		// 
		// Developer mode
		// 
		} else {

			wp_register_script( 
				'firebase',
				'https://www.gstatic.com/firebasejs/' . LCX_FIREBASE_VERSION . '/firebase.js',
				 null,
				 LCX_VERSION
			);
			wp_enqueue_script( 'firebase' );
			
			wp_register_script(
				'lcx-events',
				LCX_URL . '/assets/js/src/events.js',
				null,
				LCX_VERSION
			);
			wp_enqueue_script( 'lcx-events' );

			wp_register_script(
				'lonewolf',
				LCX_URL . '/assets/js/src/app.js',
				array( 'firebase', 'lcx-events' ),
				LCX_VERSION
			);
			wp_enqueue_script( 'lonewolf' );

			wp_register_script(
				'lcx-db',
				LCX_URL . '/assets/js/src/db.firebase.js',
				array( 'lonewolf' ),
				LCX_VERSION
			);
			wp_enqueue_script( 'lcx-db' );

			wp_register_script(
				'lcx-frontend',
				LCX_URL . '/assets/js/src/frontend.js',
				array( 'lcx-db' ),
				LCX_VERSION
			);
			wp_enqueue_script( 'lcx-frontend' );
		}

	}

	/**
	 * Render widget.
	 *
	 * @access public
	 */
	public function render() {

		if( !$this->display )
			return;

		$user = wp_get_current_user();
		$msgs = lcx_get_option_group( 'msgs' );
		$siteName = lcx_get_option( 'site', 'info_name' );
		$siteLogo = lcx_get_option( 'site', 'info_logo' );

		echo fn_lcx_get_template( 'widget', array(
			'msg' => $msgs,
			'design' => lcx_get_option_group( 'design' ),
			'siteLogo' => $siteLogo,
			'appOpts' => apply_filters( 'lcx_frontend_app_opts',
				array(
					'db' => fn_lcx_get_realtime_config(),
					'ajax' => array(
						'nonce' => wp_create_nonce( LiveChatX_AJAX::NONCE ),
						'url' => LiveChatX()->ajax_url()
					),
					'user' => fn_lcx_get_user_data(),
					'autoinit' => true,
					'anonymousImage' => fn_lcx_get_anonymous_img(),
					'systemImage' => LCX_URL . '/assets/img/logo-120x.png',
					'companyName' => $siteName,
					'companyURL' => lcx_get_option( 'site', 'info_url' ),
					'companyLogo' => $siteLogo,

					'_pluginurl' => LCX_URL,
					'_optsurl' => admin_url( 'admin.php?page=options-general.php%3Fpage%3Dlcx_opts' )
				)
			),
			'strings' => apply_filters( 'lcx_frontend_app_strings', array(
				'conn' => $msgs['ntf_conn'],
				'reconn' => $msgs['ntf_reconn'],
				'noConn' => $msgs['ntf_no_conn'],
				'reqFields' => $msgs['ntf_reqFields'],
				'invEmail' => $msgs['ntf_invEmail'],
				'you' => $msgs['ntf_you'],
				'askContactInfo' => $msgs['popup_ask_contact'],
				'afterContactInfo_1' => fn_lcx_get_response_time( 'online', $msgs['popup_ask_contact_ntf_1'], false ),
				'afterContactInfo_2' => fn_lcx_get_response_time( 'online', $msgs['popup_ask_contact_ntf_2'], false ),
				'chatStatus_init' => $msgs['popup_chatStatus_init'],
				'chatStatus_open' => $msgs['popup_chatStatus_open'],
				'chatStatus_close' => $msgs['popup_chatStatus_close'],
				'welcNewVisitors' => str_replace( '{siteName}', '<span class="lcx--site-name">' . $siteName . '</span>', $msgs['popup_prechat_greeting_new'] ),
				'welcReturningVisitors' => str_replace( '{siteName}', '<span class="lcx--site-name">' . $siteName . '</span>', $msgs['popup_prechat_greeting_returning'] ),
				'newMsgTitle' => $msgs['ntf_newMsg_wtitle'],
				'onlineOpInfo_single' => $msgs['ops_online_welc_single'],
				'onlineOpInfo_multi' => $msgs['ops_online_welc_multi'],
				'time' => array(
					'prefix' => $msgs['date_prefix'],
			        'suffix' => $msgs['date_suffix'],
			        'seconds' => $msgs['date_seconds'],
			        'minute' => $msgs['date_minute'],
			        'minutes' => $msgs['date_minutes'],
			        'hour' => $msgs['date_hour'],
			        'hours' => $msgs['date_hours'],
			        'day' => $msgs['date_day'],
			        'days' => $msgs['date_days'],
			        'month' => $msgs['date_month'],
			        'months' => $msgs['date_months'],
			        'year' => $msgs['date_year'],
			        'years' => $msgs['date_years']
				)
			))
		));

	}

}

return new LiveChatX_Frontend();