/*!
 * Live Chat X, by Screets.
 *
 * SCREETS, d.o.o. Sarajevo. All rights reserved.
 * This  is  commercial  software,  only  users  who have purchased a valid
 * license  and  accept  to the terms of the  License Agreement can install
 * and use this program.
 */

'use strict';

class lcxFrontend {

	constructor( opts, strings ) {
		const defaults = {
			user: {},
			userType: 'webVisitor',
			db: {},
			ajax: {},
			showStarter: true,
			defaultWin: 'disconnect',
			anonymousImage: '',
			companyName: '',
			companyLogo: '',
			companyURL: '',
			dateFormat: 'd/m/Y H:i',
			hourFormat: 'H:i',
			shortDateFormat: 'd/m/Y',
			platform: 'frontend',
			askContactDelay: 3000, // milliseconds
			autoinit: true,
			pluginURL: '',
			allowedTags: [ 'STRONG', 'EM', 'A', 'IMG' ]
		};

		const defaultStrings = {
			you: 'You',
			today: 'Today',
			yesterday: 'Yesterday',
			connected: 'Connected',
			connecting: 'Connecting...',
			reconnecting: 'Reconnecting. Please wait...',
			loggedin: 'Logged in',
			loggedout: 'You\'re logged out.',
			noInternet: 'No connection!',
			duplicateWin: 'Your chat box is open in different window.',
			useHere: 'Use here',
			sessionExp: 'Your session has been expired!',
			changeUserType: 'You\'re logged in as %s.',
			smthWrong: 'Something went wrong! Please try again.',
			saved: 'Saved. Thank you!',
			errInvEmail: 'Invalid email address!',
			fromUser: 'from %s'
		};

		// To run application first ensure that browser supports localStroage.
		// Especially, Safari doesn't support it in private mode.
		if ( typeof localStorage === 'object' ) {
			try {
				localStorage.setItem( 'lcx-init', true );
				localStorage.removeItem( 'lcx-init', true );
			} catch(e) {
				console.warn( 'Live Chat doesn\'t work in Private Browser Mode.' );
				return;
			}
		}

		// Setup and establish options
		this.opts = defaults;
		if( typeof opts === 'object' ) {
			for( const k in opts ) { this.opts[k] = opts[k]; }
		}

		// Establish strings
		this.str = defaultStrings;
		if( typeof strings === 'object' ) {
			for( const k in strings ) { this.str[k] = strings[k]; }
		}

		const popupStatus = localStorage.getItem( 'lcx-popup-status' ) || 'close';

		// Useful data
		this._i = this.opts._iframe;
		this._d = this._i.contentWindow.document;
		this._w = this._i.contentWindow.window;
		this._widget = this._i.parentNode;
		this._currentWin = '';
		this._isOpen = popupStatus === 'open';
		this._mode = 'init'; // User mode ("init", disconnect", "online", "offline")
		this._chatid = ''; // Active chat id
		this._chat = ''; // Active chat data
		this._formData = {}; // Values of submit forms
		this._sending = false; // A reply box is currently sending?
		this._lastMsg = ''; // Currently sending message
		this._sessionExpired = false;
		this._userType = this.opts.userType; // Current user type
		this._customToken = '';
		this._timeouts = {};

		// Common objects
		this.$starter = this._d.getElementById( 'lcx-starter' );

		// Initialize the application
		if( this.opts.autoinit ) {
			this._init();
		}
	}

	_init() {

		// Open chat box
		if( this._isOpen ) {
			this.open();
		}

		// Get the real-time application
		this.db = new lcxDB( this.opts );

		// Setup UI
		this._ui();

		// Manage authentication
		lcxEvent.on( 'login', this.onOnline, this );
		lcxEvent.on( 'logout', this.onOffline, this );

		// Listen database events
		this._dbEvents();

		// Initialize the real-time database
		this.db.init( this._userType );

		// Never logged in before. 
		// Just forget real-time connections for now.
		if( !this.db._userid ) {
			this.db.disconnect( 'neverLoggedin' );
		}
	}

	/**
	 * Setup UI.
	 */
	_ui() {
		// Show connecting notification
		this.ntf( 'blink', this.str.connecting, 'auth' );

		// Show starter button
		if( this.opts.showStarter ) {
			this.$starter.classList.add( 'lcx-show' );
		}

		// 
		// Listen starter button clicks
		// 
		if( this.$starter ) {
			this.$starter.addEventListener( 'click', (e) => {
				if( !this._isOpen )
					this.open();
				else
					this.close();
			});
		}

		// Listen actions
		this.listenActions();

		// Listen reply boxes
		this.listenReplyBoxes();

		// Listen widget events (focus, blur)
		this._widget.addEventListener( 'mouseenter', this._onWidgetFocus.bind(this) );
		this._widget.addEventListener( 'mouseleave', this._onWidgetBlur.bind(this) );
	}

	/**
	 * Open chat popup.
	 */
	open() {
		const winName = this._currentWin || this.opts.defaultWin;

		// Update data
		this._isOpen = true;

		// Open window
		this.openWin( winName );

		// Remove all in-app messages
		this.hideAppMsgs();

		// Show popup
		this._widget.classList.add( 'lcx-is-open' );
		this._d.body.classList.add( 'lcx-is-open' ); // body tag

		// Update cookies
		localStorage.setItem( 'lcx-popup-status', 'open' );
	}

	/**
	 * Close chat popup.
	 */
	close() {

		// Close popup
		this._widget.classList.remove( 'lcx-is-open' );
		this._d.body.classList.remove( 'lcx-is-open' ); // body tag
		this._isOpen = false;

		// Close current window
		if( this._currentWin ) {
			this.closeWin( this._currentWin );
		}

		// Update cookies
		localStorage.setItem( 'lcx-popup-status', 'close' );
	}

	/**
	 * Open a window.
	 */
	openWin( name ) {

		// Show "disconnect" page on duplicate session
		if( this._sessionExpired && name !== 'disconnect' ) {
			name = 'disconnect';
		}

		const win = this._d.getElementById( 'lcx-window-' + name );
		if( win ) {
console.warn( 'WIN>>', name );

			// Prepare window before appearing
			lcxEvent.fire( 'prepareWin', this, win, name );

			// Update "online" page template tags
			if( name === 'online' && this._chat.operatorid ) {
				const winTitle = win.querySelector( '.lcx-window-header .lcx-title' );
				if( winTitle ) {
					winTitle.innerHTML = winTitle.innerHTML.replace( '{operator_name}', this._chat.operatorName );
				}
			}
			
			// Close the last one
			if( this._currentWin ) {
				this.closeWin( this._currentWin );
			}

			// Show up current window if 
			if( this._isOpen ) {
				win.classList.add( 'lcx-showin' );
			}

			// Update widget data-window
			this._widget.setAttribute( 'data-window', name );

			// Update current window data
			this._currentWin = name;

			// Scroll conversation down if any exists
			const cnv = win.querySelector( '.lcx-cnv' );
			if( cnv ) {
				this.scrollDown( cnv.parentNode );
			}

			// Invoke the event
			this._onOpenWin( win,name );
			
		}
	}

	/**
	 * Close a window.
	 */
	closeWin( name ) {
		const win = this._d.getElementById( 'lcx-window-' + name );

		if( win ) {
			win.classList.remove( 'lcx-showin' );

			// Clean data-window attribute from widget object
			this._widget.removeAttribute( 'data-window' );
		}
	}

	/**
	 * Play a sound.
	 */
	play( name ) {
		const audio = new Audio( this.opts.pluginURL + '/assets/sounds/' + name + '.mp3' );
		const p = audio.play();

		if (p && (typeof Promise !== 'undefined') && (p instanceof Promise)) {
			p.catch(() => {});
		}
	}

	/**
	 * Listen user interface actions.
	 */
	listenActions( wrapObj ) {

		let actions;

		if( wrapObj ) {
			actions = wrapObj.getElementsByClassName( 'lcx-action' );
		} else {
			actions = this._d.getElementsByClassName( 'lcx-action' );
		}

		const fn_runAction = ( obj ) => {
			const type = obj.getAttribute( 'data-type' );

			switch( type ) {
				// Open popup
				case 'open':
					this.open();
					break;

				// Close popup
				case 'close':
					this.close();
					break;

				// Open a window
				case 'openWin':
					this.openWin( obj.getAttribute( 'data-name' ) );
					break;
					
				// Start a chat properly
				case 'startChat':
					this.startChat();
					break;

				// Delete user and logout
				case 'die':
					this.die();
					break;

				// Submit a form
				case 'submitForm':
					this.submitForm( obj.getAttribute( 'data-bind-to' ) );
					break;

				// Use chat widget on this browser tab/window
				case 'useHere':
					if( !this.db.isConnected() ) {
						lcxEvent.off( 'connect', this._onUseHere, this );
						lcxEvent.on( 'connect', this._onUseHere, this );

						this.db.connect();
					}
					break;
			}
		};
		const fn_listenAction = function(e) {
			if( !this.hasAttribute( 'data-disable-click' ) ) {
				e.preventDefault();
			}

			fn_runAction( this );
		};

		if( actions ) {
			let listening, readLater;
			for( let i=0; i<actions.length; i++ ) {
				listening = actions[i].hasAttribute( 'data-listening' );
				readLater = actions[i].hasAttribute( 'data-read-later' );

				// Listen action at first try
				if( !listening && !readLater  ) {
					actions[i].addEventListener( 'click', fn_listenAction );
					actions[i].setAttribute( 'data-listening', true );
				
				// Listen action on the air
				} else if( readLater ) {
					actions[i].removeAttribute( 'data-read-later' );
				}
			}
		}
	}

	_onUseHere() {
		lcxEvent.off( 'connect', this._onUseHere, this );
		
		this.db.deleteSessions( this.db._userid, 'platform', this.opts.platform, this.refresh );
	}

	/**
	 * Listen reply boxes.
	 */
	listenReplyBoxes() {

		const replies = this._d.getElementsByClassName( 'lcx-reply' );
		const lastMsg = localStorage.getItem( 'lcx-reply-msg' );

		if( replies ) {
			let isListening;
			for( let i=0; i<replies.length; i++ ) {
				isListening = replies[i].getAttribute( 'data-listening' ) || false;

				if( !isListening ) {
					this.__listenReply( replies[i] );
				}

				if( lastMsg ) {
					replies[i].innerHTML = lastMsg;
				}

				replies[i].setAttribute( 'data-listening', true );
			}
		}
	}
	__listenReply( box ) {
		
		// Check if reply box is really exists! 
		if( !box ) { return; }

		// Listen reply box
		const fn_listen = (e) => {
			
			// When user type enter (and not shift+enter)       
			if( e && e.keyCode === 13 && !e.shiftKey ) {
				
				// Don't resend the message while already sending...
				if( this._sending ) {
					e.preventDefault();
					return;
				}

				// Get message
				this._lastMsg = box.innerHTML.trim();
				

				if( this._lastMsg.length > 0 ) {
					this._sending = true;

					// Sanitize the message
					this._lastMsg = this.sanitize( this._lastMsg );

					// Clear reply box
					box.innerHTML = '';
					
					// Once it is complete, call this.__chatReady function
					lcxEvent.off( 'readyToStartChat', this.__chatReady, this );
					lcxEvent.on( 'readyToStartChat', this.__chatReady, this );

					// SENT MESSAGE now.
					// But first, ensure that we've chat to send message
					// If no, it will create one :)
					this.prepareStartChat();
					
					// Prevent "enter" in text field
					e.preventDefault();
				}
			}
		};

		const fn_scrollTo = ( y ) => { this._w.scrollTo( this._w.scrollLeft, y ); };
		/*const fn_autosize = function() {
			this.style.height = this.style.height;
			this.style.height = this.scrollHeight + 'px';
			this.scrollTop = this.scrollHeight;
			fn_scrollTo( this.scrollTop + this.scrollHeight );
		};*/

		// Save message in cookies
		const fn_save = function() {
			const msg = this.value || '';

			localStorage.setItem( 'lcx-reply-msg', msg.trim() );
		};
		
		// Listen events
		box.addEventListener( 'keydown', fn_listen );
		// box.addEventListener( 'input', fn_autosize );
		box.addEventListener( 'blur', fn_save );

	}
	__chatReady() {
		this._sending = false;

		// Reset previous messages
		lcxEvent.off( 'readyToStartChat', this.__chatReady, this );

		// Push the message now
		this.db.pushMsg( this._chatid, this._lastMsg );

		// Clear data
		this._lastMsg = '';

	}
	/**
	 * Prepare visitor to start a chat properly.
	 */
	prepareStartChat() {

		// There is chat already, so directly send message.
		if( this._chatid ) {
			this.__readyToStartChat();
			return;
		}

		// Connect database and login first
		if( this._mode === 'disconnect' ) {
			lcxEvent.off( 'connect', this.__loginToStartChat, this );
			lcxEvent.on( 'connect', this.__loginToStartChat, this );
			this.db.connect();
		
		// Login first if its not logged in 
		// or visitor doesn't have a name (probably visitor's data deleted from DB)
		} else if( !this.db.isLogged() || !( name in this.db.$_user ) ) {
			this.__loginToStartChat();

		// Start chat directly!
		} else {
			this.__readyToStartChat();
		}
	}
	__loginToStartChat() {

		console.log( '__loginToStartChat' );

		this.post( 'getRandomName', {}, ( r ) => {

			if( !r.error ) {

				const userData = {
					name: r.data.name,
					animal: r.data.animal,
					hexColor: r.data.hex,
					caseNo: r.data.caseNo
				};

				// Reset previous events
				lcxEvent.off( 'connect', this.__loginToStartChat, this );

				// Setup events
				lcxEvent.off( 'login', this.__readyToStartChat, this );
				lcxEvent.on( 'login', this.__readyToStartChat, this );
				
				// Login first to start a chat
				const canWeLogin = this.db.login( userData );

				// It seems like we already logged in, but user data out-of-date
				// So first update user data to continue
				if( !canWeLogin ) {
					// Cancel login event
					lcxEvent.off( 'login', this.__readyToStartChat, this );

					// Setup events
					lcxEvent.off( 'updateMe', this.__readyToStartChat, this );
					lcxEvent.on( 'updateMe', this.__readyToStartChat, this );

					this.db.updateUser( userData );
					
				}

			} else {
				this.ntf( 'error', this.str.smthWrong, 'login' );
			}
		});

	}
	__readyToStartChat() {
		// Reset previous events
		lcxEvent.off( 'login', this.__readyToStartChat, this );

		// Invoke the event
		lcxEvent.fire( 'readyToStartChat' );
	}

	/**
	 * Submit a form.
	 */
	submitForm( formid ) {
		const form = this._d.getElementById( formid );

		if( form ) {
			const group = form.getAttribute( 'data-bound-group' );
			const elements = form.getElementsByClassName( 'lcx-field' );

			if( !this._formData[group] ) {
				this._formData[group] = {};
			}

			// Save user data
			let name, value;
			for( let i=0; i<elements.length; i++ ) {
				name = elements[i].getAttribute( 'data-name' );
				value = ( elements[i].classList.contains( 'lcx-editable' ) ) ? elements[i].innerHTML : elements[i].value;
				
				if( name && value ) {
					this._formData[group][name] = value;
				}
			}


console.warn( 'form.>', group, this._formData );


			switch( group ) {
				// Pre-chat form
				case 'prechat':

					lcxEvent.off( 'readyToStartChat', this.__prechatSend, this );
					lcxEvent.on( 'readyToStartChat', this.__prechatSend, this );

					// Prepare visitor first and then start chat
					this.prepareStartChat();

					break;

				// Save user information
				case 'saveUserInfo':
					this.db.updateUser( this._formData.saveUserInfo, () => {
						// Show notification
						this.ntf( 'success', this.str.saved, 'saveUserInfo', 3000 );
					}, ( error ) => {
						if( error.code === 'invalidEmail' )
							error.message = this.str.errInvEmail;

						// Show notification
						this.ntf( 'error', error.message, 'saveUserInfo', 5000 );
					});
					break;
			}

			// Invoke the event
			lcxEvent.fire( 'submitForm', this, this._formData[group] );
		}
	}
	__prechatSend() {

		// Clear previous events
		lcxEvent.off( 'readyToStartChat', this.__prechatSend, this );

		// Sanitize prechat message
		this._formData.prechat.msg = this.sanitize( this._formData.prechat.msg );

		this.db.startChatByVisitor( this._formData.prechat.msg, null, () => {

			// Update current window
			/*const mode = ( this.db.isAnyOpAcceptChats() ) ? 'online' : 'offline';

			console.log('modeee', this.db.isAnyOpAcceptChats(), mode );
			this.openWin( mode );*/

			// Push the message now
			this.db.pushMsg( this._chatid, this._formData.prechat.msg );


			if( !this.db.isAnyOpAcceptChats() ) {
				
				// Notify operators by email
				this.post( 'notifyOp', {
					caseNo: this.db.$_user.caseNo,
					visitorName: this.db.$_user.name,
					msg: this._formData.prechat.msg
				}, (r) => {
					console.log(r);
				});

				// Ask contact information to visitor.
				this._timeouts['ask-contact-msg'] = window.setTimeout( () => {

					// Prepare message data. Original ones will be
					// replaced with those below.
					const customMsgData = {
						name: this.opts.companyName,
						avatar: this.opts.companyLogo,
						type: 'ask-contact'
					};
					
					// Create custom message
					this.db.pushMsg( this._chatid, '{ask-contact}', customMsgData );


				}, this.opts.askContactDelay );
			}

			// Start a session
			// this.db.createSession();
		}, ( error ) => {
			console.error(error);
			this.ntf( 'error', error.message );
		});
	}

	/**
	 * Update widget forms with current user data.
	 */
	refreshForms( onFieldUpdate ) {

		const user = this.db.$_user;
		const userInfoForms = this._d.getElementsByClassName( 'lcx-form-user-info' );

		// 
		// Check user forms
		// 
		if( user && userInfoForms ) {
			let elements;
			for( let i=0; i<userInfoForms.length; i++ ) {
				elements = userInfoForms[i].elements;

				if( elements ) {
					for( let k=0; k<elements.length; k++ ) {
						onFieldUpdate( 'user', elements[i] );
					}
				}
			}
		}
	}

	/**
	 * Logout and delete visitor.
	 */
	die() {
		lcxEvent.off( 'deleteUser', this.__afterDeleteUser, this );
		lcxEvent.on( 'deleteUser', this.__afterDeleteUser, this );

		// Delete user now
		this.db.deleteUser();
	}
	__afterDeleteUser() {
		this.db.logout();
	}

	/**
	 * Render a message.
	 */
	renderMsg( msgid, data ) {

		// Include message id into data
		data.id = msgid;

		// Format time
		data._time = this.time( data.createdAt, this.opts.hourFormat );
		data._date = this.time( data.createdAt, this.opts.dateFormat );

		// Get message
		return this.render( 'msg', data );
	}
	/**
	 * Print a message into related conversations.
	 */
	printMsg( msgid, data, isUpdate ) {
		let cnv, msgObj, msgs, breakpointDate, authorDesc;
		const isMe = data.uid === this.db._userid;
		const cnvs = this._d.querySelectorAll( '.lcx-cnv .lcx-msgs' );

		if( !cnvs ) return;

		for( let i=0; i<cnvs.length; i++ ) {

			cnv = cnvs[i];

			// Create new message object
			if( !isUpdate ) {
				msgObj = this._d.createElement( 'li' );
				msgObj.id = 'lcx-msg-' + msgid;
				msgObj.className = 'lcx-msg-' + msgid;

			// Get existing message object
			} else {
				msgObj = cnv.querySelector( '.lcx-msg-' + msgid );
			}

			if( !msgObj ) return;

			// Update message id
			data.id = msgid;

			// Update author name
			data._name = ( isMe ) ? this.str.you : ( data.name || data.uid );

			// Include "by COMPANY_NAME" to the author name
			authorDesc = ( !isMe && data._name !== this.opts.companyName ) ? this.str.fromUser.replace( '%s', this.opts.companyName ) : '';
			data._name = this.render( 'author', { name: data._name, desc: authorDesc } );

			// Get text avatar if no avatar (first letter)
			if( isMe && !data.avatar && data.hex ) {
				data._avatar = this.render( 'textAvatar', { hex: data.hex, name: data.name.charAt(0) } );
			
			// Get image avatar
			} else {
				data._avatar = this.render( 'imgAvatar', ( data.avatar || this.opts.anonymousImage ));
			}

			// Format time
			data._time = this.time( data.createdAt, this.opts.hourFormat );
			data._date = this.time( data.createdAt, this.opts.dateFormat );

			// Parse shortcodes
			data.msg = this.parseShortcodes( data.msg );

			// Custom messages
			if( data.type ) {
				msgObj.classList.add( 'lcx-type-' + data.type );
				msgObj.classList.add( 'lcx-is-custom' );
				msgObj.setAttribute( 'data-type', data.type );	
			}

			// Set as new message
			if( data._isNew ) {
				msgObj.classList.add( 'lcx-is-new' );
			}

			// Include object data
			msgObj.setAttribute( 'data-id', data.id );
			msgObj.setAttribute( 'data-chatid', data.chatid );
			msgObj.setAttribute( 'data-status', data.status );

			// Get message
			const msg = this.render( 'msg', data );

			// Render breakpoint
			if( !isUpdate && data._breakpoint ) {
				if( data._isOlder ) {
					breakpointDate = data._isYesterday ? this.str.yesterday : this.time( data.createdAt, this.opts.shortDateFormat );
				} else {
					breakpointDate = this.str.today;
				}
				cnv.insertAdjacentHTML( 'beforeEnd', this.render( 'breakpoint', breakpointDate ) );
			}

			// Update the message
			msgObj.innerHTML = msg;

			// Add message into conversation
			if( !isUpdate ) {
				cnv.appendChild( msgObj );
			}

			// Scroll conversation list down (by focusing wrapper of .lcx-cnv)
			this.scrollDown( cnv.parentNode );

			// Invoke the event
			this._onPrintMsg( cnv, msgObj, data );
		}


		/*const cnv = this._d.querySelector( 'lcx-cnv' );
		let msgs, msgObj;
		let breakpointDate;

		if( cnv ) {

			// Create new message object
			if( !isUpdate ) {
				msgObj = this._d.createElement( 'li' );
				msgObj.className = 'lcx-msg-' + msg.id;

			// Get existing message object
			} else {
				msgObj = cnv.querySelector( '.lcx-msg-' + msg.id );
			}

			if( !msgObj ) return;

console.warn( isUpdate, msg );
			// Render breakpoint
			if( !isUpdate && msg._breakpoint ) {
				if( msg._isOlder ) {
					breakpointDate = msg._isYesterday ? this.str.yesterday : this.time( msg.createdAt, this.opts.shortDateFormat );
				} else {
					breakpointDate = this.str.today;
				}
				cnv.insertAdjacentHTML( 'beforeEnd', this.render( 'breakpoint', breakpointDate ) );
			}

			// Update message
			msgObj.innerHTML = html;
			
			// Add message into conversation
			if( !isUpdate ) {
				cnv.appendChild( msgObj );
			}

			// Scroll conversation list down (by focusing wrapper of .lcx-cnv)
			this.scrollDown( cnv.parentNode );
		}*/
	}

	/**
	 * Remove printed messages from related conversations.
	 */
	removeMsg( msgid ) {
		const msgs = this._d.getElementsByClassName( 'lcx-msg-' + msgid );

		if( msgs ) {
			for( let i=0; i<msgs.length; i++ ) {
				this._delObj( msgs[i] );
			}
		}
	}

	/**
	 * Create in-app message.
	 */
	appMsg( type, opts, onCreate ) {
		
		// Don't show up if chat box is already open
		if( this._isOpen ) 
			return;

		const fn_hide = () => {
			this.hideAppMsgs();
		};

		// First hide all in-app messages
		fn_hide();

		const obj = document.createElement( 'div' );
		obj.className = 'lcx-app-msg __lcx lcx-showin';

		obj.innerHTML = this._d.getElementById( 'lcx-app-msg' ).innerHTML;

		if( opts.thumb ) {
			obj.classList.add( 'lcx-has-thumb' );
			obj.insertAdjacentHTML( 'afterBegin', this.render( 'appMsgThumb', opts.thumb ) );
		}

		if( opts.featuredImg ) {
			obj.classList.add( 'lcx-has-featured' );
			obj.insertAdjacentHTML( 'afterBegin', this.render( 'appMsgFeatured', opts.featuredImg ) );
		}

		// Insert content
		obj.querySelector( '.lcx-content' ).innerHTML = opts.content;

		// Print message
		this._widget.appendChild( obj );
		
		// Listen close button
		obj.querySelector( '.lcx-clear' ).addEventListener( 'click', function(e) {
			e.preventDefault();
			fn_hide();
		});

		// Re-listen actions
		this.listenActions( obj );

		if( onCreate ) {
			onCreate( obj );
		}
	}

	/**
	 * Remove all in-app messages.
	 */
	hideAppMsgs() {
		const msgs = document.getElementsByClassName( 'lcx-app-msg' );

		if( msgs ) {
			for( let i=0; i<msgs.length; i++ ) {
				if( msgs[i] ) {
					this._delObj( msgs[i] );
				}
			}
		}
	}

	/**
	 * Update widget mode.
	 */
	updateMode( mode ) {
		if( this._mode === mode ) return;
console.warn( 'MODE', mode);
		this._mode = mode;

		// Invoke the event
		lcxEvent.fire( 'updateMode', this, mode );
	}

	/**
	 * Listen front-end events.
	 */
	onOnline() {

		// Update the mode
		this.updateMode( 'online' );

		// Invoke the event
		lcxEvent.fire( 'online' );
	}
	onOffline() {

		// If no user session,
		// disconnect from the real-time database server
		if( !this.db.isLogged() ) {
			this.db.disconnect( 'noUserSession' );
			return;
		}

		// Update the mode
		this.updateMode( 'offline' );

		// Invoke the event
		lcxEvent.fire( 'offline' );
		
	}
	onFirstTime() {
		// This is anonymous visitor probably.
		// Just let them to initiate a chat
		this.openWin( 'prechat' );

		// Invoke the event
		lcxEvent.fire( 'firstTime' );
	}

	/**
	 * Listen database events.
	 */
	_dbEvents() {
		
		// Authentication events
		lcxEvent.on( 'connect', this._onConnect.bind(this) );
		lcxEvent.on( 'disconnect', this._onDisconnect.bind(this) );
		lcxEvent.on( 'duplicateSession', this._onDuplicateSession.bind(this) );
		lcxEvent.on( 'sessionExpired', this._onSessionExpired.bind(this) );
		lcxEvent.on( 'login', this._onLogin.bind(this) );
		lcxEvent.on( 'logout', this._onLogout.bind(this) );
		lcxEvent.on( 'dbinit', this._onDbInit.bind(this) );
		lcxEvent.on( 'authError', this._onAuthError.bind(this) );

		// User updates
		lcxEvent.on( 'updateMe', this._onUpdateMe.bind(this) );
		lcxEvent.on( 'deleteMe', this._onDeleteMe.bind(this) );

		// Listen chats
		lcxEvent.on( 'newChat', this._onNewChat.bind(this) );
		lcxEvent.on( 'updateChat', this._onUpdateChat.bind(this) );
		lcxEvent.on( 'deleteChat', this._onDeleteChat.bind(this) );

		// Listen chat messages
		lcxEvent.on( 'newMsg', this._onNewMsg.bind(this) );
		lcxEvent.on( 'updateMsg', this._onUpdateMsg.bind(this) );
		lcxEvent.on( 'deleteMsg', this._onDeleteMsg.bind(this) );

		// Listen chat members
		lcxEvent.on( 'updateMember', this._onUpdateMember.bind(this) );
		
		// Operators
		lcxEvent.on( 'newOp', this._onNewOp.bind(this) );
		lcxEvent.on( 'deleteOp', this._onDeleteOp.bind(this) );
	}

	_onConnect() {
console.log( 'mode', this._mode );

		// Show "re-connected" notification
		this.ntf( 'success', this.str.connected, 'auth', 5000 );

		// If no user or chat, go to "prechat"
		/*if( !this.db.isLogged() || !this._chatid ) {
			this.openWin( 'prechat' );
		}*/

		console.log( 'connected' );
	}
	_onDisconnect( event, reason ) {

		this.hideNtf( 'auth' );

		switch( reason ) {

			// Invalid configs :D
			case 'invalidDBconfigs':

				// Hide chat widget
				this._widget.style.display = 'none';

				break;
			
			// On downgrade
			case 'downgraded':
				
				this.ntf( 'error', this.str.loggedout, 'auth', 7000 );

				this.openWin( 'prechat' );

				break;

			// Stay on default page when user never logged in
			case 'neverLoggedin':
			case 'noUserSession':
				this.onFirstTime();
				break;

			// Other reasons
			default:
				// Hide connecting message
				this.hideNtf( 'auth' );

				// Go to "disconnecting" window
				this.openWin( 'disconnect' );

				// Show "no connection" message 
				if( !this._sessionExpired ) {
					this.ntf( 'error', this.str.noInternet, 'auth' );
				} else {
					this.hideNtf( 'auth' );
				}
		}

		// Update mode
		this.updateMode( 'disconnect' );


		console.error( 'auth', reason );
	}

	_onDuplicateSession() {

		// No session
		this._sessionExpired = true;

		const msg = this.str.duplicateWin + '<br /><a href="#" class="lcx-action" data-type="useHere">' + this.str.useHere + '</a>';

		// Create new message
		this.ntf( 'error', msg, 'duplicateSession' );

		// Re-listen actions
		this.listenActions();

		// Now disconnect
		this.db.disconnect();
	}
	_onSessionExpired() {

		this._sessionExpired = true;

		// Show session expired error
		this.ntf( 'error', this.str.sessionExp, 'sessionExpired' );

		// Now disconnect
		this.db.disconnect();
	}
	_onLogin() {
		console.log( 'logged in', this.db.$_user );

		this.openWin( 'prechat' );

		/*// Hide "connecting" notification
		this.hideNtf( 'auth' );

		// Re-update current mode
		if( this.db.isLogged() ) {
			console.log('CHATID',this._chatid);
			const page = ( this._chatid ) ? 'offline' : 'prechat';

			// Show "You logged in as operator"
			if( this.db.$_user['type'] === 'operator' ) {
				this.ntf( 'success', this.str.changeUserType.replace( '%s', 'operator' ), 'auth', 7000 );
				
			// Show "logged in" notification
			} else {
				this.ntf( 'success', this.str.loggedin, 'auth', 3000 );
			}
			
			// Open window with new mode
			this.openWin( page );
		}*/
	}
	_onLogout() {
		console.log( 'logged out' );
	}
	_onDbInit() {}
	_onAuthError( event, error ) {

		// Show main error
		this.ntf( 'error', error.message, 'auth' );

		switch( error.code ) {
			case 'auth/requires-recent-login':

				// Show re-connecting notification
				this.ntf( 'blink', this.str.reconnecting, 'auth' );

				// Logout and refresh the window
				lcxEvent.on( 'login', this.__authRelogin, this );
				this.db.login();
				break;
		}
	}
	__authRelogin() {
		lcxEvent.on( 'logout', this.refresh, this );
	}

	_onNewOp( event, opid ) {
		// Update mode
		// this.updateMode( this.db.isAnyOpAvail() ? 'online' : 'offline' );
	}
	_onDeleteOp( event, opid ) {
		console.log('removed op');
		// Update mode
		// this.updateMode( this.db.isAnyOpAvail() ? 'online' : 'offline' );
	}
	_onUpdateMe( event, data ) {
		console.log('current user update', data );

		const fn_editField = ( type, obj ) => {

			// Only user info forms
			if( type !== 'user' ) return;

			const currentValue = this.db.$_user[obj.name] || null;

			// Replace the field with the data
			if( currentValue ) {
				obj.value = currentValue;
			}
		};

		// Re-check user info forms
		this.refreshForms( fn_editField );

		// If we don't have active chat, go "prechat"!
		// Probably we connected before and our chat is deleted or archived!
		if( data.chats ) {
			let hasChat = false;
			for( const id in data.chats ) {
				if( data.chats[id].startedOn !== 'console' ) {
					hasChat = true;
				}
			}

			if( !hasChat ) {
				this.openWin( 'prechat' );
			}
		} else {
			this.openWin( 'prechat' );
		}

		// Update case numbers
		const cases = this._d.getElementsByClassName( 'lcx-case-no' );
		const caseNo = this.db.$_user.caseNo;
		if( cases ) {
			for( let i=0; i<cases.length; i++ ) {

				if( caseNo ) {
					cases[i].innerHTML = this.replaceAll( 
						cases[i].innerHTML, {
							'{number}': '<span class="lcx-no">' + caseNo + '</span>' 
						});
				} else {
					cases[i].innerHTML = '';
				}
			}
		}
		
	}
	_onDeleteMe( data ) {
		// Current user deleted from database. 
		// So we can die now in peace :(
		this.die();
	}
	/**
	 * Chat events.
	 */
	_onNewChat( event, chatid, data ) {

		// Update data
		this._chatid = chatid;
		this._chat = data;

		console.log('new chat', chatid, data );
		
		// Hide authentication messages on startup
		this.hideNtf( 'auth' );

		// Refresh window
		this.openWin( ( data.operatorid ) ? 'online' : 'offline' );
		
	}
	_onUpdateChat( event, chatid, data ) {

		// Update data
		this._chatid = chatid;
		this._chat = data;

		console.log('update chat', chatid, data );

		// Refresh window
		this.openWin( ( data.operatorid ) ? 'online' : 'offline' );
	}
	_onDeleteChat( event, chatid, data ) {

		// Remove chat id
		if( this._chatid === chatid ) {
			this._chatid = '';
			this._chat = '';
		}

		console.warn('delete chat', chatid, data );
	}

	/**
	 * Chat message events.
	 */
	_onNewMsg( event, msgid, data ) {

		// Don't show or render hidden messages
		if( data.hidden ) return;

		const html = this.renderMsg( msgid, data );

		// Show app-in message if popup is closed
		if( !this._isOpen && data.status !== 'read' ) {
			this.appMsg( 'chat', {
				content: '<ul class="lcx-msgs">' + html + '</ul>',
				thumb: data.avatar
			});

			this.play( 'new-ntf' );
		}

		// Add message into related conversations
		this.printMsg( msgid, data, false );
	}
	_onUpdateMsg( event, msgid, data ) {

		// Don't show or render hidden messages
		if( data.hidden ) return;

		console.log('update msg', msgid, data );

		const html = this.renderMsg( msgid, data );

		// Update message
		this.printMsg( msgid, data, true );
	}
	_onDeleteMsg( event, msgid, data ) {
		console.warn('delete msg', msgid, data );
		this.removeMsg( msgid );
	}
	_onPrintMsg( cnv, msgObj, data ) {
		const actions = msgObj.getElementsByClassName( 'lcx-action' );
		const userDataInput = msgObj.querySelectorAll( '[data-input-type="user"]' );

		// Bind action buttons to objects
		if( actions ) {
			let actionName, boundId, bindTo, boundObj, randId;

			for( let k=0; k<actions.length; k++ ) {
				boundObj = boundId = null;
				bindTo = actions[k].getAttribute( 'data-bind-to' );
				actionName = actions[k].getAttribute( 'data-type' );
				randId = Math.floor( Math.random() * 99999 ) + 1; // between 1 and 99999

				if( bindTo ) {
					boundObj = msgObj.querySelector( bindTo );

					// Bind two object with unique ids
					if( boundObj ) {
						boundId = '__lcx-' + actionName + '-' + boundObj.getAttribute( 'data-bound-group' ) + '-' + randId;
						boundObj.id = boundId;
						actions[k].setAttribute( 'data-bind-to', boundId );
					}
				}
			}
		}

		// Update common user inputs
		if( userDataInput ) {
			let input;
			for( let k=0; k<userDataInput.length; k++ ) {
				input = userDataInput[k];

				if( this.db.$_user[input.type] ) {
					input.value = this.db.$_user[input.type];
				}
			}
		}

		// Listen actions for custom messages
		this.listenActions( msgObj );

		// Invoke the event
		lcxEvent.fire( 'printMsg', this, cnv, msgObj, data );
	}

	/**
	 * Chat member events.
	 */
	_onUpdateMember( event, chatid, members ) {
		console.log('update members', chatid, members, arguments );
	}

	/**
	 * Widget events.
	 */
	_onWidgetFocus() {
		// Read chat when popup is open
		// and user's focused on the widget
		if( this._isOpen ) {
			this.db.readChat( this._chatid );
		}

		// Invoke the event
		lcxEvent.fire( 'widgetFocus' );
	}
	_onWidgetBlur() {
		// Read chat when focus on widget
		this.db.readChat( this._chatid );

		// Invoke the event
		lcxEvent.fire( 'widgetBlur' );
	}


	/**
	 * Page events.
	 */
	_onOpenWin( win, name ) {

		// 
		// Resize pre-chat question box
		//
		if( name === 'prechat' ) {

			const prechatFields = this._d.querySelectorAll( '#lcx-window-prechat .lcx-window-content .lcx-field' );

			// If there is more than one field, don't resize question box
			if( prechatFields && prechatFields.length === 1 ) {
				const q = this._d.getElementById( 'lcx-prechat-ask-question' );
				const prechatContent = this._d.querySelector( '#lcx-window-prechat .lcx-window-content' );

				if( q ) {
					q.style.height = prechatContent.offsetHeight - 10 + 'px';
				}
			}
			
		}

		// Invoke the event
		lcxEvent.fire( 'openWin', this, win, name );
	}

	/**
	 * Scroll down a box.
	 */
	scrollDown( obj ) {
		obj.scrollTop = obj.scrollHeight;
	}

	/**
	 * Format time.
	 */
	time( ts, format = 'd/m/Y H:i:s' ) {
		
		const date = new Date( ts );
		const val = {
			Y: date.getFullYear(),
			m: date.getMonth()+1,
			d: date.getDate(),
			H: date.getHours(),
			i: date.getMinutes(),
			s: date.getSeconds()
		};

		// Convert one digit "0" to "00" for both hours and minutes
		if( val.H.toString().length === 1 ) { val.H = '0' + val.H; }
		if( val.i.toString().length === 1 ) { val.i = '0' + val.i; }

		/* Source: http://stackoverflow.com/a/15604206/272478 */
		format = format.replace(/Y|m|d|H|i|s/gi, function(matched){
			return val[matched];
		});

		return format;
	}

	/**
	 * Helper functions.
	 */
	refresh() {
		console.warn('reloading');
		
		// Refresh the browser window
		window.location.reload();
	}
	_delObj( obj ) {
		if( obj ) {
			obj.parentNode.removeChild( obj );
		}
	}
	clearTimeout( name ) {
		if( typeof this._timeouts[name] === 'number' ) {
			this._w.clearTimeout( this._timeouts[name] );
			this._timeouts[name] = undefined;
		}
	}
	replaceAll( str, searchObj ) {
		const re = new RegExp(Object.keys(searchObj).join("|"),"gi");

		return str.replace( re, function( matched ) {
			return searchObj[matched.toLowerCase()];
		});
	}
	idealTextColor( bgHexColor ) {
		const nThreshold = 105;
		const components = this.convertToRGB( bgHexColor );
		const bgDelta = ( components.R * 0.299 ) + ( components.G * 0.587 ) + ( components.B * 0.114 );

		return ( ( 255 - bgDelta ) < nThreshold ) ? '#333' : '#fff';   
	}
	convertToRGB( hex ) {       
		var r = hex.substring( 1, 3 );
		var g = hex.substring( 3, 5 );
		var b = hex.substring( 5, 7 );

		return {
			R: parseInt( r, 16 ),
			G: parseInt( g, 16 ),
			B: parseInt( b, 16 )
		};
	}
	parseShortcodes( msg ) {

		let shortcodes = {};
		const rx = /\{([^)]+)\}/;
		const matches = rx.exec( msg );

		if( matches ) {
			let shortcodeName = '';
			for( let i=0; i<matches.length; i++ ) {
				if( i % 2 == 0 )
					shortcodeName = matches[i];
				else
					shortcodes[ shortcodeName ] = this.renderShortcode( matches[i] );
			}

			return this.replaceAll( msg, shortcodes );
		}

		return msg;
	}
	renderShortcode( tag ) {
		const content = this._d.getElementById( 'lcx-shortcode-' + tag );

		if( !content ) return '';

		return content.innerHTML;
	}
	sanitize( msg ) {
		const div = this._d.createElement( 'div' );
	    div.innerHTML = msg;
	    this.removeTags( div );
	    
	    return div.innerHTML;
	}
	removeTags( el ) {
		const tags = Array.prototype.slice.apply(el.getElementsByTagName("*"), [0]);
	    for (let i = 0; i < tags.length; i++) {
	    	tags[i].removeAttribute( 'style' ); // Clean styles

	        if ( this.opts.allowedTags.indexOf(tags[i].nodeName) == -1 ) {
	            this.usurp( tags[i] );
	        }
	    }
	}
	usurp( parent ) {
		let last = parent, e;
	    for ( let i = parent.childNodes.length - 1; i >= 0; i-- ) {
	        e = parent.removeChild( parent.childNodes[i] );
	        parent.parentNode.insertBefore( e, last );
	        last = e;
	    }
	    parent.parentNode.removeChild( parent );
	}

	/**
	 * Remove spaces, tabs and new lines from HTML.
	 *
	 * @source: https://stackoverflow.com/a/40026669/272478
	 */
	/*clearHTML( html ) {
		return html.replace(/(<(pre|script|style|textarea)[^]+?<\/\2)|(^|>)\s+|\s+(?=<|$)/g, "$1$3");
	}*/

	/**
	 * Show a notification on current window.
	 */
	ntf( type, msg, group, duration ) {

		let ntf, data;
		const win = this._d.querySelectorAll( '.lcx-window' );

		if( win ) {

			// First hide similar group of notifications
			if( group ) {
				this.hideNtf( group );
			}

			for( let i=0; i<win.length; i++ ) {
				data = {};
				ntf = win[i].querySelector( '.lcx-ntf' );

				if( ntf ) {
					ntf.classList.add( 'lcx-showin' );

					data = {
						classes: [ 'lcx-' + type ],
						msg: msg
					};

					if( group ) {
						data.classes.push( 'lcx-group-' + group );
						data.group = group;
					}

					ntf.insertAdjacentHTML( 'beforeEnd', this.render( 'ntf', data ) );
				}
			}

			if( duration && group ) {
				// Cancel previous timeout of the group
				this.clearTimeout( group );

				// Set new timeout
				this._timeouts[group] = window.setTimeout( ( groupName ) => {
					this.hideNtf( groupName );
				}, duration, group );
			} else if( group ) {
				this.clearTimeout( group );
			}
		}
	}
	hideNtf( group ) {
		const ntfs = this._d.querySelectorAll( '.lcx-group-' + group );

		if( ntfs ) {
			for( let i=0; i < ntfs.length; i++ ) {
				this._delObj( ntfs[i] );
			}
		}
	}

	/**
	 * Send a post request to the server.
	 */
	post( mode, data, callback ) {

		data.mode = mode;
		data.action = data.action || 'lcx_action';
		data._ajax_nonce = this.opts.ajax.nonce;

		const xhr = new XMLHttpRequest();
		const fd = new FormData();

		xhr.open( 'POST', this.opts.ajax.url, true );

		// Handle response
		xhr.onreadystatechange = () => {

			if ( xhr.readyState == 4 ) {

				// Perfect!
				if( xhr.status == 200 ) {
					if( callback ) { callback( JSON.parse( xhr.responseText ) ); }

				// Something wrong!
				} else {
					if( callback ) { callback( null ); }
				}
			}

		};

		// Get data
		for( const k in data ) { fd.append( k, data[k] ) ; }

		// Initiate a multipart/form-data upload
		xhr.send( fd );
	}

	/**
	 * Render a template.
	 */
	render( name, p ) {
		let output = [];

		switch( name ) {
			/*case 'msg':
				output = [ '<li class="lcx-msg-',p.id,'"><span class="lcx-avatar"><img src="',p.avatar,'" alt="" /></span><div class="lcx-content"><span class="lcx-time" title="',p._date,'">',p._time,'</span>',p.name,'<span class="lcx-msg">',p.msg,'</span></div></li>' ];
				break;

			case 'author':
				output = [ '<span class="lcx-author"><span class="lcx-title">',p.name,'</span><span class="lcx-desc">',p.desc,'</span></span>' ];
				break;*/

			case 'msg':
				output = [ '<div class="lcx-msg-wrap">', p._avatar, '<div class="lcx-content"><div class="lcx-meta"><span class="lcx-time" title="',p._date,'">',p._time,'</span></div>',p._name,'<span class="lcx-msg">',p.msg,'</span></div></div>' ];
				break;

			case 'author':
				output = [ '<span class="lcx-author"><span class="lcx-title">',p.name,'</span><span class="lcx-desc">',p.desc,'</span></span>' ];
				break;

			case 'breakpoint':
				output = [ '<div class="lcx-breakpoint"><span class="lcx-legend">',p,'</span><span class="lcx-bg"></span></div>' ];
				break;

			case 'ntf':
				output = [ '<div class="lcx-wrap ', p.classes.join(' '), '" data-group="',p.group,'">',p.msg,'</div>' ];
				break;

			case 'textAvatar':
				output = [ '<span class="lcx-avatar lcx-is-text" style="color:', this.idealTextColor( p.hex ), ';background-color:', p.hex, ';">',p.name,'</span>' ];
				break;

			case 'imgAvatar':
				output = [ '<span class="lcx-avatar"><img src="',p,'" alt="" /></span>' ];
				break;

			case 'appMsgThumb':
				output = [ '<div class="lcx-thumb"><img src="',p,'" alt="" /></div>' ];
				break;

			case 'appMsgFeatured':
				output = [ '<div class="lcx-featured-img"><img src="',p,'" alt="" /></div>' ];
				break;

		}

		return output.join('');
	}

}