/*!
 * Live Chat X, by Screets.
 *
 * SCREETS, d.o.o. Sarajevo. All rights reserved.
 * This  is  commercial  software,  only  users  who have purchased a valid
 * license  and  accept  to the terms of the  License Agreement can install
 * and use this program.
 */

class nightBird {

	constructor( opts, strings ) {

		const defaults = {
			db: {},
			ajax: {},
			user: {},
			autoinit: true,
			authMethod: 'anonymous',
			initPopup: 'prechat',
			ntfDuration: 5000, // ms.
			platform: 'frontend',
			dateFormat: 'd/m/Y H:i',
			hourFormat: 'H:i',
			askContactInfo: true,

			// Company data
			companyName: '',
			companyURL: '',
			companyLogo: '',
			anonymousImage: '',
			systemImage: '',
		};

		const defaultStrs = {
			/*
			conn: 'Connecting...',
			reconn: 'Reconnecting. Please wait...',
			noConn: 'No internet connection!',
			reqFields: 'Please fill out all required fields!',
			invEmail: 'Email is invalid!',
			you: 'You',
			askContactInfo: 'Please enter your email, so we know how to reach you.',
			afterContactInfo_1: 'Awesome! We usually reply in a few minutes...',
			afterContactInfo_2: "p.s. Don't worry about missing our reply – if you're not here, we'll send it to your email.",
			chatStatus_init: 'Pending',
			chatStatus_open: 'In chat',
			chatStatus_close: 'Ended',
			welcNewVisitors: '',
			welcReturningVisitors: '',
			newMsgTitle: '%s replied your message...',
			onlineOpInfo_single: '%s was online recently.',
			onlineOpInfo_multi: '%s1 and %s2 were online recently.',
			time: {
				prefix: "",
		        suffix: " ago",
		        seconds: "less than a minute",
		        minute: "about a minute",
		        minutes: "%d minutes",
		        hour: "about an hour",
		        hours: "about %d hours",
		        day: "a day",
		        days: "%d days",
		        month: "about a month",
		        months: "%d months",
		        year: "about a year",
		        years: "%d years"
			}
			*/
		};

		// Setup and establish options
		this.opts = defaults;	
		for( const k in opts ) {
			this.opts[k] = opts[k]; 
		}

		// Establish strings
		this.str = defaultStrs;
		if( typeof strings === 'object' ) {
			for( const k in strings ) { this.str[k] = strings[k]; }
		}

		const popupstatus = localStorage.getItem( 'lcx-popup-status' );

		// Useful data
		this._wstate = true;
		this._wtitle = document.title;
		this._popup = localStorage.getItem( 'lcx-popup' ) || this.opts.initPopup;
		this._pstate = popupstatus === 'open'; // popup state.
		this._chatid = sessionStorage.getItem( 'lcx-chatid') || '';
		this._chatStatuses = {};
		this._initProfileUpdated = false;
		this._timer = null; // timer for offline form.
		this._unreadChats = {}; // unread chats

		// Useful objects
		this.$widget = document.getElementById( 'lcx' );
		this.$starter = document.getElementById( 'lcx-starter' );
		this.$popup = document.getElementById( 'lcx-popup-' + this.opts.initPopup );
		this.$popupsWrap = document.getElementById( 'lcx-popups-wrap' );
		this.$goBackCnvs = document.querySelector( '#lcx-popup-online .lcx-btn-go-chats' );
		this.$ntf = document.getElementById( 'lcx-ntfs' );
		this.$chats = document.getElementById( 'lcx-chats' );
		this.$msgs = document.getElementById( 'lcx-msgs' );
		this.$reply = document.getElementById( 'lcx-reply' );
		this.$replyWrap = document.getElementById( 'lcx-reply-wrap' );
		this.$msgsContainer = document.querySelector( '.lcx-msgs-container' );
		this.$welcMsg = document.getElementById( 'lcx-prechat-welcome-msg' );
		this.$inAppMsg = document.getElementById( 'lcx-inAppMsg' );

		if( this.opts.autoinit ) {
			this.init();
		}
	}

	init() {

		// Applications
		this.db = new wolfDB( this.opts );

		// Listen database events
		this._dbEvents();

		// Listen UI
		this._uiEvents();

		// Listen actions
		this.actions();

		// Initialize the database
		this.db.init();
	}

	/**
	 * Listen action buttons.
	 */
	actions( btn ) {

		const fn_run = function(e) {
			e.preventDefault();
			const type = this.getAttribute( 'data-action' );

			switch( type ) {
				case 'openPopup':
					fn_openPopup( this.getAttribute( 'data-name' ), this.getAttribute( 'data-chat-id') );
					break;
				case 'closePopup':
					fn_closePopup();
					break;
				case 'openChat':
					fn_openChat( this.getAttribute( 'data-id' ) );
					break;
				case 'closeInAppMsg':
					fn_closeInAppMsg();
					break;
			}
		};

		const fn_openPopup = ( name, chatid ) => {
			if( chatid ) {
				this._pstate = true; // force to show up
				this.openChat( chatid );
			} else 
				this.openPopup( name );
		};

		const fn_closePopup = () => {
			this.closePopup();
		};
		const fn_openChat = ( chatid ) => {
			if( chatid ) this.openChat( chatid );
		};
		const fn_closeInAppMsg = () => {
			this.closeInAppMsg();
		};

		// Use default button
		if( btn ) {
			btn.addEventListener( 'click', fn_run );

		// Search on entire UI
		} else {
			const btns = document.getElementsByClassName( 'lcx-action' );
			if( btns ) {
				for( var i=0; i<btns.length; i++ ) {
					btns[i].addEventListener( 'click', fn_run );
				}
			}
		}
	}

	/**
	 * 
	 * ======= USER INTERFACE =======
	 *
	 */
	_uiEvents() {

		// Set initial state of popup
		if( this._pstate ) {
			this.openPopup();
		}

		// Listen time ago field
		const fn_to = () => {
			NBird.timeago( null, this.str.time );
			setTimeout( fn_to, 30000 ); // refresh every 30 seconds.
		};
		fn_to();

		// Auto-fill the user fields
		if( this.opts.user ) {
			const __user = this.opts.user;
			const name = document.getElementsByClassName( 'lcx-input-name' );
			const email = document.getElementsByClassName( 'lcx-input-email' );

			if( name && __user.name ) {
				for( var i=0, l=name.length; i<l; i++ ) {
					name[i].value = __user.name
				}
			}

			if( email && __user.email ) {
				for( var i=0, l=email.length; i<l; i++ ) {
					email[i].value = __user.email
				}
			}
		}

		// Listen starter.
		if( this.$starter ) {
			this.$starter.addEventListener( 'click', (e) => {
				if( this._pstate )
					this.closePopup();
				else
					this.openPopup();
			});
		}

		//
		// Listen pre-chat message box and send button.
		// 
		let __sending = false;
		let chatData = {};
		const $btnStartChat = document.getElementById('lcx-btn-start-chat');
		const $msg = document.getElementById( 'lcx-field-prechat-msg' );

		const fn_startChat = () => {


			// User logged-in..
			if( Object.keys( this.db.$_user ).length > 0 ) {

				// Start chat directly
				if( this.db.$_user.identity ) {
					this.db.getNewCaseNo( ( caseNo ) => {
						fn_startChatDirectly( this.db.$_user.identity );
					});

				// Get new identity first.. then start.
				} else {
					fn_startChatAfterAuth();
				}

			// First authenticate, then start chat
			} else {
				// Sign-in as operator
				if( this.opts.user.__isOp ) {
					this.db.signin( 'custom', this.opts.db.token );

				// Sign-in as anonymous visitor
				} else {
					this.db.signin( 'anonymous' );
				}

				// Start chat on after authenticating user
				this.db.event.once( 'authState', fn_startChatAfterAuth );
			}
		};

		const fn_startChatDirectly = ( identity ) => {
			chatData.name = identity.nickname;
			chatData.caseNo = identity.lastCaseNo;

			this.db.updateProfile({
				identity: identity,
				currentPageUrl: this.opts.user.currentPageUrl
			}, () => {
				this.db.startChat( chatData, () => {
					fn_refreshPopupUI();
				}, ( error ) => {
					console.error( error );
				});
			});
		};

		const fn_startChatAfterAuth = () => {

			this.db.createIdentity( ( newIdentity ) => {
				let user = {
					name: this.opts.user.name || newIdentity.nickname,
					identity: newIdentity
				};

				user.email = chatData.email || this.opts.user.email || null;
				user.currentPageUrl = this.opts.user.currentPageUrl;

				chatData.name = user.name;
				chatData.caseNo = user.identity.lastCaseNo;

				this.db.updateProfile( user, () => {
					this.db.startChat( chatData, () => {
						fn_refreshPopupUI();
					}, ( error ) => {
						console.error( error );
					});
				});
			});
		};

		const fn_refreshPopupUI = function() {
			$msg.innerText = '';
			$btnStartChat.classList.remove( 'lcx--disabled' );
			__sending = false;
		};

		$msg.addEventListener( 'keydown', function(e) {
			if( e && e.keyCode === 13 && !e.shiftKey ) {
				e.preventDefault();

				if( __sending ) return;

				__sending = true;

				$btnStartChat.classList.add( 'lcx--disabled' );

				let msg = this.innerHTML;

				if( !msg ) {
					__sending = false;
					return;
				}

				chatData.msg = msg;

				// Start chat
				fn_startChat();
			}
		});

		$btnStartChat.addEventListener( 'click', (e) => {
			e.preventDefault();

			$btnStartChat.classList.add( 'lcx--disabled' );

			
			const message = $msg.innerText;

			// Validate message fields.
			if( message.length === 0 ) {
				this.newNtf( 'error', this.str.reqFields, 'prechat' );

				$msg.classList.add( 'lcx-error' )

				fn_refreshPopupUI();
				return;
			}

			$msg.classList.remove( 'lcx-error' );

			chatData.msg = message;

			/*for( var i=0, l=fields.length; i<l; i++ ) {
				field = fields[i]; 
				name = field.name;
				value = field.value;

				switch( name ) {
					case 'email':
						if( !NBird.isEmail( value ) ) {
							this.newNtf( 'error', this.str.invEmail, 'prechat' );

							field.classList.add( 'lcx-error' );

							fn_refreshPopupUI();
							return;
						}
						break;
					case 'msg':
						if( !value ) {
							this.newNtf( 'error', this.str.reqFields, 'prechat' );

							field.classList.add( 'lcx-error' );

							fn_refreshPopupUI();
							return;
						}
						break;
				}

				field.classList.remove( 'lcx-error' );

				chatData[name] = value;
			}*/

			// Clear form notifications
			this.hideNtfGroup( 'prechat' );

			// Start chat
			fn_startChat();
		});

		// 
		// Listen online reply box.
		//
		this.$reply.addEventListener( 'keydown', function(e) {
			if( e && e.keyCode === 13 && !e.shiftKey ) {
				e.preventDefault();

				if( __sending ) return;

				__sending = true;

				let msg = this.innerHTML;

				if( !msg ) {
					__sending = false;
					return;
				}
				
				// Clear reply box
				this.innerHTML = '';

				// Push message now
				fn_pushMsg( msg, function() {
					__sending = false;
				});
			}
		});
		this.$reply.addEventListener( 'paste', function(e) {
			// cancel paste
			e.preventDefault();

			// get text representation of clipboard
			var text = e.clipboardData.getData("text/plain");

			// insert text manually
			document.execCommand( 'insertHTML', false, text );
		});

		const fn_pushMsg = ( msg, cb ) => {
			this.db.pushMsg( this._chatid, msg, cb );
		};

		// 
		// Listen if visitor reads the new messages.
		// 
		document.getElementById( 'lcx-popup-online' ).addEventListener( 'mouseenter', (e) => {
			this.readCurrentChat();
		});

		// 
		// Listen window states.
		// 
		window.addEventListener( 'focus', () => { this._wstate = true; });
		window.addEventListener( 'blur', () => { this._wstate = false; });
	}

	refreshUI() {
		const prechatPopup = document.getElementById( 'lcx-popup-prechat' );

		// Calculate pre-chat popup body height
		if( prechatPopup ) {
			
			const prechatBody = document.getElementById( 'lcx-popup-prechat-body' );
			const prechatReply = document.getElementById( 'lcx-field-prechat-msg' );

			if( prechatBody.scrollHeight > 0 )
				prechatPopup.style.height = prechatBody.scrollHeight + prechatReply.offsetHeight + 'px';
		}

	}

	/**
	 * 
	 * ======= COMMON FUNCTIONS =======
	 *
	 */

	/**
	 * Manage popups.
	 */
	openPopup( name, autoPopup = true ) {
		name = name || this._popup || this.opts.initPopup;

		const popup = document.getElementById( 'lcx-popup-' + name );
		if( !popup ) return;

		// Close current popup if opens
		if( this._pstate ) {
			this.$popup.classList.add( 'lcx--hidden' );
			this.$popup.classList.remove( 'lcx--active' );
		}

		// Get current popup
		this._popup = name;
		this.$popup = popup;

		// Clear in-app messages
		this.closeInAppMsg();

		// Show up now.
		if( autoPopup || this._pstate ) {
			this.$popup.classList.add( 'lcx--active' );
			this.$popup.classList.remove( 'lcx--hidden' );
			
			localStorage.setItem( 'lcx-popup-status', 'open' );

			this._pstate = true;

			document.body.setAttribute( 'data-lcx-popup', 'open' );
			this.$widget.setAttribute( 'data-popup-status', 'open' );
		}

		// Update local storage
		localStorage.setItem( 'lcx-popup', name );

		// Trigger event.
		this._onOpenPopup( name );
	}
	closePopup() {

		if( !this._pstate )
			return;

		this.$popup.classList.add( 'lcx--hidden' );
		this.$popup.classList.remove( 'lcx--active' );

		// Reset data
		this.$popup = document.getElementById( 'lcx-popup-' + this.opts.initPopup );
		this._pstate = false;

		// Update widget data
		document.body.setAttribute( 'data-lcx-popup', 'close' );
		this.$widget.setAttribute( 'data-popup-status', 'close' );

		// Update local db
		localStorage.setItem( 'lcx-popup-status', 'close' );

		// Trigger event
		this._onClosePopup();
	}

	/**
	 * Manage notifications.
	 */
	newNtf( type, msg, group, autohide ) {
		const uniqid = 'lcx-ntf-' + Math.floor( Math.random() * 99999 ) + 1; // between 1 and 99999
		const ntf = document.createElement('div');

		ntf.id = uniqid;
		ntf.className = 'lcx-ntf--' + type;
		ntf.innerHTML = msg + ' <div class="lcx-close-btn">&times;</div>';

		if( group ) {
			ntf.className += ' lcx-ntf--group-' + group;
			this.hideNtfGroup( group );
		}

		// Show notification
		this.$ntf.appendChild( ntf );

		const close = ntf.querySelector( '.lcx-close-btn' );

		if( autohide ) {
			
			// Hide after a while
			setTimeout( this.hideNtf.bind( null, uniqid ), this.opts.ntfDuration );

			// Listen close button
			close.addEventListener( 'click', this.hideNtf.bind( null, uniqid ) );

		// Hide close button
		} else {
			close.classList.add( 'lcx--hidden' );
		}


	}
	hideNtf( id ) {
		NBird.delObj( id );
	}
	hideNtfGroup( group ) {
		const ntfs = document.getElementsByClassName( 'lcx-ntf--group-' + group );

		if( ntfs ) {
			for( var i=0; i<ntfs.length; i++ ) {
				NBird.delObj( ntfs[i] );
			}
		}
	}

	/**
	 * Throw a new in-app message.
	 *
	 * @param {String} msgid
	 * @param {String} msg
	 */
	throwInAppMsg( msgid, msg ) {
		// Play sound
		NBird.play( 'new-ntf', this.opts._pluginurl );

		if( this._pstate )
			return; // don't continue if popup is open

		// Render in-app elements
		this.$inAppMsg.querySelector( '.lcx-author' ).innerHTML = msg.name;
		this.$inAppMsg.querySelector( '.lcx-msg-content' ).innerHTML = NBird.renderMsg( msg.msg );
		this.$inAppMsg.querySelector( '.lcx-avatar-img' ).src = msg.photoURL;

		// Show up message now
		this.$inAppMsg.classList.remove( 'lcx--hidden' );

		// Set related chat id
		this.$inAppMsg.querySelector( '.lcx-inAppMsgWrap' ).setAttribute( 'data-chat-id', msg.chatid );
	}
	/**
	 * Close in-app messages.
	 */
	closeInAppMsg() {
		this.$inAppMsg.classList.add( 'lcx--hidden' );
	}

	/**
	 * Render message into a conversation.
	 *
	 * @param {String} msgid
	 * @param {String} msg
	 * @param {String} prevId - (optional) Previous messages ID of the current message.
	 * @param {Object} $wrap - (optional) DOM Object where messages loaded. By default, current conversation will be used.
	 *
	 */
	renderMsg( msgid, msg, prevId, $wrap ) {
		let classes = [];
		const isYou = msg.platform === 'frontend' && this.db._uid === msg.uid;
		const itemid = 'lcx-msg-' + msg.chatid + msgid;

		$wrap = $wrap || this.$msgs;

		if( !$wrap )
			return;

		const item = document.createElement( 'li' );
		item.id = itemid;
		classes.push( itemid );

		if( isYou ) {
			classes.push( 'lcx-msg--you' );
			msg.name = this.str.you;
		}

		// Set avatar
		let avatar;
		if( isYou )
			avatar = !msg.photoURL ? this.opts.anonymousImage : msg.photoURL;
		else
			avatar = msg.photoURL || this.opts.companyLogo || this.opts.anonymousImage;
		
		avatar = '<span class="lcx-avatar"><img src="' + avatar + '" alt="" /></span>';

		// Set date/time
		const time = NBird.time( msg.date, this.opts.hourFormat );
		const date = NBird.time( msg.date, this.opts.dateFormat );

		// Set author desc (i.e. by Screets)
		const authorDesc = '';

		// Join classes
		item.className = classes.join(' ');

		item.innerHTML = '<div class="lcx-msg-wrap">' + avatar + '<div class="lcx-content"><div class="lcx-meta"><span class="lcx-msg-status"></span><span class="lcx-time" title="'+ date +'">' + time + '</span></div><div class="lcx-author"><span class="lcx-title">' + msg.name + '</span><span class="lcx-desc">' + authorDesc + '</span></div><span class="lcx-msg">' + NBird.renderMsg( msg.msg ) + '</span></div></div>';

		$wrap.appendChild( item );

		NBird.scrollDown( $wrap.parentNode );
	}

	/**
	 * Load messages of a chat into a conversation.
	 *
	 * @param {String} chatid
	 * @param {Object} $wrap - (optional) DOM Object where messages loaded. By default, current conversation will be used.
	 *
	 */
	loadMsgs( chatid, $wrap ) {
		$wrap = $wrap || this.$msgs;
		const msgs = this.db.$_msgs[chatid];

		$wrap.innerHTML = '';

		if( msgs ) {
			let msg;
			let i=0;
			for( const msgid in msgs ) {
				msg = this.db.$_msgs[chatid][msgid];
				this.renderMsg( msgid, msg, msg.prevId, $wrap );

				// Ask contact info
				if( i === 0 ) {
					const chat = this.db.$_chats[msg.chatid];

					if( chat && chat.status === 'init' )
						this.askContact();
				}

				i++;
			}
		}
	}

	/**
	 * Ask contact info in the current chat.
	 */
	askContact() {

		if( !this.opts.askContactInfo ) return;

		const li = document.createElement( 'li' );
		const contactInfo = document.getElementById( '__lcx-tpl-ask-contact' );
		const user = this.db.$_user;

		if( user.emailForNtf || !contactInfo || !this._chatid ) 
			return;

		// Render form
		li.className = 'lcx-msg-askContact';
		li.innerHTML = contactInfo.innerHTML;
		this.$msgs.appendChild( li );

		// Listen save button
		let sending = false;
		const email = li.querySelector( '.lcx-field-email' );
		const savebtn = li.querySelector( '.lcx-save-btn' );

		if( user.email ) {
			email.value = user.email;
		}

		savebtn.addEventListener( 'click', (e) => {
			e.preventDefault();

			if( sending ) 
				return;

			sending = true;

			savebtn.classList.add( 'lcx--disabled' );

			if( !NBird.isEmail( email.value ) ) {
				this.newNtf( 'error', this.str.invEmail, 'askContact' );

				email.classList.add( 'lcx-error' );

				sending = false;
				savebtn.classList.remove( 'lcx--disabled' );
				return;
			}

			// Clear notifications
			email.classList.remove( 'lcx-error' );
			this.hideNtfGroup( 'askContact' );

			// Update user info
			this.db.updateUser( this.db._uid, {
				emailForNtf: email.value
			}, () => {
				savebtn.classList.remove( 'lcx--disabled' );
				li.classList.add( 'lcx--hidden' );

				let postMsgs; 
				if( this.str.afterContactInfo_1 )
					postMsgs = this.str.afterContactInfo_1;
				
				if( this.str.afterContactInfo_2 )
					postMsgs += '<br><br>' + this.str.afterContactInfo_2;

				if( postMsgs ) {
					this.renderMsg( 'postContactMsg', {
						name: this.opts.companyName,
						photoURL: this.opts.companyLogo,
						date: Date.now(),
						msg: postMsgs
					});
				}
			});

		});
	} 

	/**
	 * Handle chats and chats list.
	 */
	readCurrentChat( cb ) {

		const chatid = this._chatid || '';

		if( !chatid || !( chatid in this._unreadChats ) )
			return;

		delete this._unreadChats[chatid];

		this.db.readChat( this._chatid, cb );

		// No unread messages left. So clear "conversations" link
		if( Object.keys(this._unreadChats).length === 0 )
			this.$goBackCnvs.classList.remove( 'lcx--newNtf' );

		// Reset window title
		document.title = this._wtitle;

		// Unmark related chat in conversations list
		document.getElementById( 'lcx-chat-item-' + chatid ).classList.remove( 'lcx--newNtf' );


	}
	refreshChat( id, chat ) {

		if( this._chatid !== id ) // update current chat only...
			return;

		let status;

		// Show reply box by default
		this.$replyWrap.classList.remove( 'lcx--hidden' );

		// In chat...
		if( chat.opid && chat.status === 'open' ) {

			status = 'inChat';

			// Set template tags
			chat.operatorName = chat.opName;

		// Pending chat...
		} else if( chat.status === 'init' ) { 
			
			status = 'pending';

		// Closed chat
		} else {
			status = 'close';

			// Hide reply box
			this.$replyWrap.classList.add( 'lcx--hidden' );
		}

		// Update online popup header
		const headerTpl = document.querySelector( '#lcx-popup-online-header .__lcx-tpl-header' );
		const footerTpl = document.querySelector( '#lcx-popup-online-footer .__lcx-tpl-footer' );
		const header = document.querySelector( '#lcx-popup-online-header .__lcx-dynamic-header' );
		const footer = document.querySelector( '#lcx-popup-online-footer .__lcx-dynamic-footer' );

		header.innerHTML = NBird.replaceAll( 
			headerTpl.innerHTML,
			chat,
			'lcx-popup-online'
		);

		footer.innerHTML = NBird.replaceAll( 
			footerTpl.innerHTML,
			chat,
			'lcx-popup-online'
		);

		// Find template tags
		NBird.refreshTags( id, chat, 'chat-' );

		// Update online chat status
		this._chatStatuses[id] = status;
		
		// Update widget data
		this.$widget.setAttribute( 'data-chat-status', status );
	}

	addChatItem( id, chat ) {
		const li = this.getChatItem( id, chat );

		this.$chats.appendChild( li );

		// Apply time-ago functionality
		NBird.timeago( li.querySelector( '.lcx-time' ), this.str.time );

		// Listen new chat item
		this.actions( li.querySelector( 'a' ) );
	}

	updateChatItem( id, chat ) {
		// Clear last chat list item
		this.delChatItem( id );

		this.addChatItem( id, chat );
	}
	getChatItem( id, chat ) {
		const li = document.createElement( 'li' );
		const itemid = 'lcx-chat-item-' + id;
		let tags = [];

		const status = this.str['chatStatus_' + chat.status ];

		if( status )
			tags.push( '<span class="lcx-tag">', status, '</span>' );

		li.id = itemid;
		li.className = itemid + ' lcx-status-' + chat.status;
		li.setAttribute( 'data-status', chat.status );

		// Render content
		li.innerHTML = '<a href="#" data-action="openChat" data-id="' + id + '"><span class="lcx-chat-meta"><span class="lcx-timeago" datetime="' + Number( chat.date/1000 ) + '"></span><span class="lcx-tags">' + tags.join('') + '</span></span><span class="lcx-chat-title">' + chat.subject + '</span></a>';


		return li;
	}
	delChatItem( id ) {
		NBird.delObj( 'lcx-chat-item-' + id );
	}

	/**
	 * Open a chat conversation in popup.
	 *
	 * @param {String} chatid
	 * @param {Object} $wrap - (optional) DOM Object where messages loaded. By default, current conversation will be used.
	 *
	 */
	openChat( chatid, $wrap ) {

		if( chatid in this.db.$_chats ) {

			const chat = this.db.$_chats[chatid];

			this._chatid = chatid;
			sessionStorage.setItem( 'lcx-chatid', chatid );
			
			this.refreshChat( chatid, chat );

			// Load chat messages
			this.loadMsgs( chatid, $wrap );

			// Go online now
			this.openPopup( 'online', this._pstate );
		}
	}

	/**
	 * Handle other users.
	 */
	refreshUser( id, user ) {
		// Find template tags
		NBird.refreshTags( id, user, 'user-' );
	}


	/**
	 * 
	 * ======= EVENTS =======
	 *
	 */
	/**
	 * Listen database events.
	 */
	_dbEvents() {

		// Authentication events.
		this.db.event.on( 'connect', this._onConnect.bind(this) );
		this.db.event.on( 'disconnect', this._onDisconnect.bind(this) );
		this.db.event.on( 'authState', this._onAuthState.bind(this) );

		// Session events.
		this.db.event.on( 'endSession', this._onEndSession.bind(this) );
		this.db.event.on( 'duplicateSession', this._onDuplicateSess.bind(this) );

		// Online operator events.
		this.db.event.on( 'newOp', this._onNewOp.bind(this) );
		this.db.event.on( 'deleteOp', this._onDeleteOp.bind(this) );

		// Chat events.
		this.db.event.on( 'newSingleChat', this._onNewChat.bind(this) );
		this.db.event.on( 'updateSingleChat', this._onUpdateChat.bind(this) );
		this.db.event.on( 'deleteSingleChat', this._onDeleteChat.bind(this) );

		// Chat message events.
		this.db.event.on( 'newMsg', this._onNewMsg.bind(this) );
		this.db.event.on( 'updateMsg', this._onUpdateMsg.bind(this) );
		this.db.event.on( 'deleteMsg', this._onDeleteMsg.bind(this) );

		// Current user events.
		this.db.event.on( 'profile', this._onProfileUpdate.bind(this) );
	}

	/**
	 * Network is connected.
	 */
	_onConnect() {
		this.$widget.setAttribute( 'data-conn-status', 'connect' );

		// Keep chat console active 
		if( this.$popup )
			this.$popupsWrap.classList.remove( 'lcx--disabled' );

		// List online OPs if there is any...
		this.db.getOnlineOps( ( ops ) => {

			if( ops ) {
				this.$widget.setAttribute( 'data-onlineops', 'yes' );

				// List online ops
				const opLists = document.getElementsByClassName( '_lcx-list-onlineOps' );

				if( !opLists ) return;

				let tpl = [];
				let opNames = [];
				for( var id in ops ) {
					opNames.push( ops[id].name );
					tpl.push( '<div class="lcx-op-item"><span class="lcx-op-avatar"><img src="',ops[id].photoURL,'" alt="" /></span><span class="lcx-op-name">', ops[id].name, '</span></div>' );
				}

				let msg;
				if( Object.keys(ops).length === 1 ) {
					for( var id in ops )
						msg = this.str.onlineOpInfo_single.replace( '%s', ops[id].name );
				} else {
					msg = this.str.onlineOpInfo_multi.replace( '%s2', opNames.pop() );
					msg = msg.replace( '%s1', opNames.join( ', ' ) );
				}
				
				
				tpl.push( '<div class="lcx-op-msg">' + msg + '</div>' );
				tpl = tpl.join(' ');

				for( var i=0; i<opLists.length; i++ ) {
					opLists[i].innerHTML = tpl;
				}

			} else {
				this.$widget.setAttribute( 'data-onlineops', 'no' );
			}


			// Refresh UI
			window.setTimeout( () => {
				this.refreshUI();
			}, 200 );

		});

		// Throw notification
		this.hideNtfGroup( 'auth' );
	}

	/**
	 * Network is disconnected.
	 */
	_onDisconnect( reason ) {
		this.$widget.setAttribute( 'data-conn-status', 'disconnect' );

		// Disable chat console when no network connection
		this.$popupsWrap.classList.add( 'lcx--disabled' );
		
		// Throw notification
		if( !this.db._isFirstConn ) {
			this.newNtf( 'error', this.str.noConn, 'auth' );
		}
	}

	/**
	 * Handle authentication state changes.
	 */
	_onAuthState( user ) {
		let authState;

		// Signed in
		if ( user ) {

			authState = 'signedin';

			// Update basic info
			if( ! this._initProfileUpdated ) {
				this.db.updateProfile({
					currentPageUrl: this.opts.user.currentPageUrl
				});
			}

			this._initProfileUpdated = true;

		// Signed out
		} else {
			authState = 'signedout';

			// Update widget data
			this.$widget.removeAttribute( 'data-has-chats' );

			// Show welcome message for new visitors
			this.$welcMsg.innerHTML = this.str.welcNewVisitors;

			// Go to initial page not to stuck on other pages
			this.openPopup( this.opts.initPopup, this._pstate );

		}

		// Update body
		this.$widget.setAttribute( 'data-auth-state', authState );

	}

	/**
	 * Handle sessions.
	 */
	_onEndSession() {
		this._onDuplicateSess();
	}
	_onDuplicateSess( user ) {
	}

	/**
	 * Handle authentication errors.
	 */
	_onAuthErr( error ) {
		console.error( error );
	}

	/**
	 * Handle online operators.
	 */
	_onProfileUpdate() {

		const user = this.db.$_user;

		// Listen current chat on page refresh
		if( user.chatsAsVisitor ) {
			for( const chatid in user.chatsAsVisitor ) {
				this.db.listenChat( chatid, true );
			}
			// Let visitor to choose a chat
			if( !this._chatid ) {
				this.openPopup( 'chats', false );
			}

		// Go to initial popup page.. Because we've no chat!
		} else {
			this.openPopup( this.opts.initPopup, this._pstate );
		}

		// Show welcome message for returning visitors who has any chat
		this.$welcMsg.innerHTML = ( user.chatsAsVisitor ) ? this.str.welcReturningVisitors : this.str.welcNewVisitors;
	}

	/**
	 * Handle online operators.
	 */
	_onNewOp( userid, prevId ) {



		// Operators accept chats.
		
	}	
	_onDeleteOp( userid, prevId ) {
		
		// Operators NOT accepting chats.
	}

	/**
	 * Handle chats.
	 */
	_onNewChat( id, chat, prevId ) {

		// First chat ever.
		if( !this._chatid && chat.status === 'init' ) {
			this._chatid = id;
			sessionStorage.setItem( 'lcx-chatid', id );
			
			const lastNotifiedChatId = localStorage.getItem( 'lcx-lastNotifiedChat' ) || null;

			// Notify operators about new chat
			if( lastNotifiedChatId &&  lastNotifiedChatId !== id ) {

				// Update storage
				localStorage.setItem( 'lcx-newChatNtf', this._chatid );

				NBird.post( 'notifyOps', {
					caseNo: chat.caseNo,
					visitorName: chat.name,
					msg: chat.lastMsg
				}, this.opts.ajax, (r) => {
					console.log(r);					
				});
			}

			// Open chat popup
			this.openChat( id );
		}

		// Update online popup status
		if( !this._chatid ) {
			this._chatStatuses[id] = 'inActive';
			this.$widget.setAttribute( 'data-chat-status', 'inActive' );
		}

		// Refresh chat data
		this.refreshChat( id, chat );

		// Update widget data
		this.$widget.setAttribute( 'data-has-chats', true );

		// Add the chat in chats list
		this.addChatItem( id, chat );

	}
	_onUpdateChat( id, chat, prevId ) {
		this.refreshChat( id, chat );

		// Update chat item in the list
		this.updateChatItem( id, chat );

	}
	_onDeleteChat( id, prevId ) {
		if( Object.keys( this.db.$_chats ).length === 0 ) {
			this.$widget.removeAttribute( 'data-has-chats' );
		}

		// Go to pre-chat if we're on deleted chat page
		if( this._chatid === id ) {
			this.openPopup( 'prechat', this._pstate );
		}

		// Delete it from conversations
		this.delChatItem( id );
	}

	/**
	 * Handle chat messages.
	 */
	_onNewMsg( id, data, prevId ) {

		if( this._chatid === data.chatid ) {
			this.renderMsg( id, data, prevId );
		}

		// Ask contact info
		if( !prevId ) {
			const chat = this.db.$_chats[data.chatid];

			if( chat && chat.status === 'init' )
				this.askContact();
		}

		// Show up in-app messages
		if( data.__unread ) {
			this.throwInAppMsg( id, data );

			this._unreadChats[data.chatid] = true;

			// Mark "conversations" link if we're on different conversation
			if( this._popup !== 'online' || this._chatid !== data.chatid )
				this.$goBackCnvs.classList.add( 'lcx--newNtf' );

			// Mark related conversation in conversations list
			document.getElementById( 'lcx-chat-item-' + data.chatid ).classList.add( 'lcx--newNtf' );

			// Update window title
			document.title = this.str.newMsgTitle.replace( '%s', data.name ) + ' ' + this._wtitle;
		}
	}
	_onUpdateMsg( id, data, prevId ) {}
	_onDeleteMsg( id, data, prevId ) {
		NBird.delObj( 'lcx-msg-' + data.chatid + id );
	}

	/**
	 * Handle popups.
	 */
	_onOpenPopup( name ) {

		// Don't stuck on a page when there is no chat
		if( name !== this.opts.initPopup && Object.keys( this.db.$_chats ).length === 0 ) {
			this.openPopup( this.opts.initPopup );
			return;
		}
		
		switch( name ) {
			case 'prechat':
			case 'chats':
				

				// Reset current chat data on specific popup pages.
				this._chatid = '';

				break;

			case 'online':

				NBird.scrollDown( this.$msgs.parentNode );

				// Make chat read
				this.readCurrentChat();

				// Focus on reply box
				this.$reply.focus();

				break;
		}

		// Refresh UI
		this.refreshUI();

		// Trigger event
		this.db.event.emit( 'openPopup', name );
	}
	_onClosePopup() {
		// Trigger event
		this.db.event.emit( 'closePopup' );
	}
}