<?php
/**
 * Liker
 *
 * @encoding        UTF-8
 * @version         2.3.2
 * @copyright       (C) 2018-2025 Merkulove ( https://merkulov.design/ ). All rights reserved.
 * @license         Envato License https://1.envato.market/KYbje
 * @contributors    Nemirovskiy Vitaliy (nemirovskiyvitaliy@gmail.com), Dmitry Merkulov (dmitry@merkulov.design)
 * @support         help@merkulov.design
 * @license         Envato License https://1.envato.market/KYbje
 *
 * @wordpress-plugin
 * Plugin Name: Liker
 * Plugin URI: https://1.envato.market/liker
 * Description: Liker helps you rate and like articles on a website and keep track of results.
 * Version: 2.3.2
 * Requires at least: 5.0
 * Requires PHP: 5.6
 * Author: Merkulove
 * Author URI: https://1.envato.market/cc-merkulove
 * License: Envato License https://1.envato.market/KYbje
 * License URI: https://1.envato.market/KYbje
 * Text Domain: liker
 * Domain Path: /languages
 * Tested up to: 6.7
 * Elementor tested up to: 3.99
 * Elementor Pro tested up to: 3.99
 **/

namespace Merkulove;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
    header( 'Status: 403 Forbidden' );
    header( 'HTTP/1.1 403 Forbidden' );
    exit;
}

/** Include plugin autoloader for additional classes. */
require __DIR__ . '/src/autoload.php';

use Merkulove\Liker\Caster;
use Merkulove\Liker\Config;
use Merkulove\Liker\Unity\CheckCompatibility;
use Merkulove\Liker\Unity\Plugin;
use Merkulove\Liker\Unity\PluginHelper;
use Merkulove\Liker\Unity\Unity;

define( 'LIKER_SERVER_ID', 2 );
define( 'LIKER_ENVATO_ID', Plugin::get_envato_id() );

/**
 * The main Liker.
 *
 * This class is responsible for initializing the plugin, managing its lifecycle, and ensuring
 * compatibility with its environment. It follows a singleton pattern to maintain only one
 * instance of the class during execution.
 */
final class Liker {

    /**
     * The one true Liker.
     *
     * @var Liker
     * @since 1.0.0
     * @access private
     **/
    private static $instance;

    /**
     * Sets up a new plugin instance.
     *
     * @since 1.0.0
     * @access private
     *
     * @return void
     **/
    private function __construct() {
		Plugin::get_instance();
    }

	/**
	 * Setup the plugin.
	 *
     * @since 1.0.0
	 * @access public
     *
	 * @return void
	 **/
	public function on_loaded() {

		if (isset($_SERVER['REQUEST_URI']) && $_SERVER['REQUEST_URI'] === '/favicon.ico') {
			return; // Ignore favicon requests
		}

        /** Do critical compatibility checks and stop work if fails. */
		if ( ! Unity::get_instance()->initial_checks( ['php', 'curl'] ) ) { return; }

		// Starting WP 6.7 need to run all translatable functions in the init hook.
		add_action( 'init', function () {
			PluginHelper::load_textdomain();
			Config::get_instance()->prepare_settings();
			Unity::get_instance()->setup();
			Caster::get_instance()->setup();
		});

	}

    /**
     * Called when a plugin is activated.
     *
     * @static
     * @since 1.0.0
     * @access public
     *
     * @return void
     **/
	public static function on_activation() {

        if ( ! CheckCompatibility::do_activator_check() ) {
            deactivate_plugins( array( Plugin::get_basename() ) );
            return;
        }

        /** Call Unity on plugin activation.  */
        Unity::on_activation();

        /** Call Liker on plugin activation */
		Caster::get_instance()->activation_hook();

	}

    /**
     * Called when a plugin is deactivated.
     *
     * @static
     * @since 1.0.0
     * @access public
     *
     * @return void
     **/
    public static function on_deactivation() {

        /** MP on plugin deactivation.  */
        Unity::on_deactivation();

    }

	/**
	 * Main Instance.
	 *
	 * Insures that only one instance of plugin exists in memory at any one time.
	 *
	 * @static
	 * @since 1.0.0
     *
     * @return Liker
	 **/
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {

			self::$instance = new self;

		}

		return self::$instance;

	}

}

/** Run 'on_activation' when the plugin is activated. */
register_activation_hook( __FILE__, [ Liker::class, 'on_activation' ] );

/** Run 'on_deactivation' when the plugin is deactivated. */
register_deactivation_hook( __FILE__, [ Liker::class, 'on_deactivation' ] );

/** Run Plugin class once after activated plugins have loaded. */
add_action( 'plugins_loaded', [ Liker::get_instance(), 'on_loaded' ] );
