<?php
function lighthouse_is_fake_username( $username ) {
    // Check if the username is too short or too long
    //if ( strlen( $username ) < 5 || strlen( $username ) > 15 ) {
    //    return true; // Fake due to length
    //}

    // Check for a high percentage of uppercase letters
    $uppercase_count = preg_match_all( '/[A-Z]/', $username );
    $total_chars     = strlen( $username );

    if ( $total_chars === 0 ) {
        return true;
    } else {
        $uppercase_ratio = $uppercase_count / $total_chars;

        if ( $uppercase_ratio > 0.7 ) { // Over 70% uppercase
            return true;
        }
    }

    // Check for excessive alternation of case (like "AaAaAa")
    if ( preg_match( '/([a-z][A-Z]|[A-Z][a-z]){3,}/', $username ) ) {
        return true;
    }

    // Check for repetition of character patterns (like "ABABAB")
    if ( preg_match( '/(.{2,})\1/', $username ) ) {
        return true;
    }

    // Check for randomness based on character diversity
    $unique_chars   = count( array_unique( str_split( $username ) ) );
    $char_diversity = $unique_chars / $total_chars;

    if ( $char_diversity > 0.8 ) { // Over 80% unique characters
        return true; // High diversity often indicates randomness
    }

    // Check for lack of meaningful structure (vowel-consonant balance)
    $vowel_count = preg_match_all( '/[aeiouAEIOU]/', $username );
    $vowel_ratio = $vowel_count / $total_chars;

    if ( $vowel_ratio < 0.3 ) { // Less than 30% vowels
        return true;
    }

    // Check for mixed-case randomness (uppercase and lowercase mix)
    if ( preg_match( '/^[a-zA-Z]+$/', $username ) && $uppercase_ratio > 0.3 && $uppercase_ratio < 0.7 ) {
        return true; // Random mix of uppercase and lowercase letters
    }

    // Otherwise, assume the username is valid
    return false;
}

function lighthouse_run_blacklist_email( $sanitized_user_login, $user_email, $errors ) {
    global $wpdb;

    // Create the table name
    $table_name = $wpdb->prefix . 'lighthouse_blacklist';

    // Get the domain name and clean it
    [$username, $domain] = explode( '@', $user_email );
    $domain              = esc_sql( strtolower( trim( $domain ) ) );

    // Check the blacklist
    $blacklist_domain = $wpdb->get_results( "SELECT * FROM $table_name WHERE domain LIKE '%$domain%'" );

    if ( ! empty( $blacklist_domain ) || substr_count( $sanitized_user_login, '.' ) > 5 ) {
        apply_filters( 'lighthouse_spam_check_result', true, $sanitized_user_login, $user_email, 'local blacklist' );

        $errors->add( 'invalid_email', get_option( 'lighthouse_blacklist_email_message' ) );
    }

    if ( (int) get_option( 'lighthouse_use_external_blacklist' ) === 1 ) {
        // Check cache first
        $blacklist_cache_key = 'lhf_external_blacklist';
        $blacklist_domains   = get_transient( $blacklist_cache_key );

        if ( $blacklist_domains === false ) {
            // Initialize cURL with improved settings
            $ch = curl_init();
            curl_setopt_array(
                $ch,
                [
                    CURLOPT_URL            => 'https://raw.githubusercontent.com/wolffe/lighthouse-spam-emails/master/blacklist.txt',
                    CURLOPT_RETURNTRANSFER => true,
                    CURLOPT_TIMEOUT        => 10,
                    CURLOPT_CONNECTTIMEOUT => 5,
                    CURLOPT_SSL_VERIFYPEER => true,
                    CURLOPT_SSL_VERIFYHOST => 2,
                    CURLOPT_ENCODING       => 'gzip,deflate',
                    CURLOPT_USERAGENT      => 'Lighthouse/1.0',
                ]
            );

            $blacklist_data = curl_exec( $ch );
            $http_code      = curl_getinfo( $ch, CURLINFO_HTTP_CODE );
            $error          = curl_error( $ch );
            curl_close( $ch );

            if ( $error || $http_code !== 200 ) {
                error_log( 'Lighthouse: Failed to fetch external blacklist - ' . $error );
                return;
            }

            // Process and cache the blacklist
            $blacklist_domains = array_filter(
                array_map( 'trim', explode( "\n", $blacklist_data ) )
            );

            // Cache for 24 hours
            set_transient( $blacklist_cache_key, $blacklist_domains, DAY_IN_SECONDS );
        }

        // Check if the domain matches any of the blacklisted domains
        foreach ( $blacklist_domains as $blacklist_domain ) {
            if ( trim( $blacklist_domain ) === $domain ||
                strpos( $domain, trim( $blacklist_domain ) ) !== false ) {
                apply_filters( 'lighthouse_spam_check_result', true, $sanitized_user_login, $user_email, '4P blacklist' );
                $errors->add( 'invalid_email', get_option( 'lighthouse_blacklist_email_message' ) );
                break;
            }
        }
    }

    // Check for spam using Akismet if enabled
    if ( (int) get_option( 'lighthouse_use_akismet' ) === 1 && function_exists( 'akismet_check_submission' ) ) {
        $akismet_result = akismet_check_submission(
            [
                'comment_type'         => 'registration',
                'comment_author'       => $sanitized_user_login,
                'comment_author_email' => $user_email,
                'comment_content'      => '',
                'user_ip'              => $_SERVER['REMOTE_ADDR'],
                'user_agent'           => $_SERVER['HTTP_USER_AGENT'],
            ]
        );

        if ( $akismet_result ) {
            apply_filters( 'lighthouse_spam_check_result', true, $sanitized_user_login, $user_email, 'Akismet' );

            $errors->add( 'invalid_email', get_option( 'lighthouse_blacklist_email_message' ) );
        }
    }

    if ( (int) get_option( 'lighthouse_check_registration_periods' ) === 1 ) {
        $period_count        = substr_count( $user_email, '.' );
        $max_periods_allowed = (int) get_option( 'lighthouse_check_registration_periods_count' );
        $max_periods_allowed = $max_periods_allowed > 0 ? $max_periods_allowed : 2;

        if ( (int) $period_count > $max_periods_allowed ) {
            // The email address has more than max allowed number of periods
            apply_filters( 'lighthouse_spam_check_result', true, $sanitized_user_login, $user_email, 'Pattern #1: Periods' );

            $errors->add( 'invalid_email', get_option( 'lighthouse_blacklist_email_message' ) );

            // Increment counter
            $count = get_option( 'lighthouse_spam_registration_pattern1_count', 0 );
            ++$count;
            update_option( 'lighthouse_spam_registration_pattern1_count', $count );
        }
    }

    if ( (int) get_option( 'lighthouse_check_registration_fake_username' ) === 1 ) {
        if ( lighthouse_is_fake_username( $sanitized_user_login ) ) {
            // The username is fake
            apply_filters( 'lighthouse_spam_check_result', true, $sanitized_user_login, $user_email, 'Pattern #2: Fake username' );

            $errors->add( 'invalid_email', get_option( 'lighthouse_blacklist_email_message' ) );

            // Increment counter
            $count = get_option( 'lighthouse_spam_registration_pattern2_count', 0 );
            ++$count;
            update_option( 'lighthouse_spam_registration_pattern2_count', $count );
        }
    }

    // Increment counter
    $count = get_option( 'lighthouse_spam_registration_count', 0 );
    ++$count;
    update_option( 'lighthouse_spam_registration_count', $count );

    return $errors;
}

if ( (int) get_option( 'lighthouse_check_registration_spam' ) === 1 ) {
    add_filter( 'register_post', 'lighthouse_run_blacklist_email', 20, 3 );
}
