<?php
function lhf_curl_api( $url, $return ) {
    $lhf_speedfactor_audit_http = (string) get_option( 'lhf_speedfactor_audit_http' );
    $lhf_speedfactor_audit_tls  = (string) get_option( 'lhf_speedfactor_audit_tls' );
    $lhf_speedfactor_audit_ipv4 = (string) get_option( 'lhf_speedfactor_audit_ipv4' );

    $ch = curl_init();
    curl_setopt_array(
        $ch,
        [
            CURLOPT_URL            => $url,
            CURLOPT_TIMEOUT        => 30,
            CURLOPT_MAXREDIRS      => 10,
            CURLOPT_HEADER         => true,
            CURLOPT_NOBODY         => true,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_AUTOREFERER    => true,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2,
            CURLOPT_CERTINFO       => true,
            CURLOPT_VERBOSE        => 0,
            CURLOPT_ENCODING       => 'br,gzip,deflate',
            CURLOPT_USERAGENT      => 'Lighthouse: SpeedFactor',
            CURLOPT_HTTP_VERSION   => $lhf_speedfactor_audit_http === 'http1' ? CURL_HTTP_VERSION_1_1 :
                                    ( $lhf_speedfactor_audit_http === 'http2' ? CURL_HTTP_VERSION_2_0 : CURL_HTTP_VERSION_1_1 ),
            CURLOPT_SSLVERSION     => $lhf_speedfactor_audit_tls === 'tls12' ? CURL_SSLVERSION_TLSv1_2 :
                                ( $lhf_speedfactor_audit_tls === 'tls13' ? CURL_SSLVERSION_TLSv1_3 : CURL_SSLVERSION_TLSv1_2 ),
            CURLOPT_IPRESOLVE      => $lhf_speedfactor_audit_ipv4 === 'ipv4' ? CURL_IPRESOLVE_V4 : CURL_IPRESOLVE_WHATEVER,
        ]
    );

    $response  = curl_exec( $ch );
    $error     = curl_error( $ch );
    $http_code = curl_getinfo( $ch, CURLINFO_HTTP_CODE );
    curl_close( $ch );

    if ( $error || $http_code !== 200 ) {
        error_log( 'Lighthouse SpeedFactor: cURL API Error - ' . $error );
        return false;
    }

    if ( (string) $return === 'info' ) {
        $response = curl_getinfo( $ch );
    } elseif ( (string) $return === 'response' ) {
        $response = $response;
    }

    return $response;
}



/**
 * Get timing metrics with improved caching and data retention
 */
function get_timing_metric( $siteId, $metric, $days ) {
    global $wpdb;

    // Validate and sanitize input
    $days       = max( 1, min( 30, (int) $days ) ); // Limit days between 1 and 30
    $table_name = $wpdb->prefix . 'lhf_sf_curl';

    // Check cache first
    $cache_key     = 'lhf_timing_metric_' . md5( $siteId . $metric . $days );
    $cached_result = get_transient( $cache_key );
    if ( $cached_result !== false ) {
        return $cached_result;
    }

    // Clean up old data (keep only last 30 days)
    $wpdb->query(
        $wpdb->prepare(
            "DELETE FROM $table_name WHERE audit_timestamp < DATE_SUB(NOW(), INTERVAL 30 DAY)"
        )
    );

    // Get metrics with improved query
    $result = $wpdb->get_results(
        $wpdb->prepare(
            "SELECT 
            audit_namelookup_time,
            audit_connect_time,
            audit_pretransfer_time,
            audit_redirect_time,
            audit_starttransfer_time 
        FROM $table_name 
        ORDER BY audit_timestamp DESC 
        LIMIT %d",
            $days
        )
    );

    if ( empty( $result ) ) {
        return false;
    }

    $result = array_reverse( $result );

    $timingDNS      = [];
    $timingTCP      = [];
    $timingRequest  = [];
    $timingRedirect = [];
    $timingTTFB     = [];

    foreach ( $result as $page ) {
        $timingDNS[]      = $page->audit_namelookup_time;
        $timingTCP[]      = $page->audit_connect_time;
        $timingRequest[]  = $page->audit_pretransfer_time;
        $timingRedirect[] = $page->audit_redirect_time;
        $timingTTFB[]     = $page->audit_starttransfer_time;
    }

    // Get total time
    $timingDOM = end( $timingDNS ) + end( $timingTCP ) + end( $timingRequest ) +
                end( $timingRedirect ) + end( $timingTTFB );

    // Format timing values
    $timingValues = [
        'dns'      => format_timing_value( end( $timingDNS ) ),
        'tcp'      => format_timing_value( end( $timingTCP ) ),
        'request'  => format_timing_value( end( $timingRequest ) ),
        'redirect' => format_timing_value( end( $timingRedirect ) ),
        'ttfb'     => format_timing_value( end( $timingTTFB ) ),
    ];

    $response = ( (string) $metric === 'timingDOM' ) ?
                format_timing_value( $timingDOM ) :
                ( $timingValues['ttfb'] ?? false );

    // Cache the result for 5 minutes
    set_transient( $cache_key, $response, 5 * MINUTE_IN_SECONDS );

    return $response;
}

/**
 * Helper function to format timing values
 */
function format_timing_value( $value ) {
    if ( $value * 1000 < 1000 ) {
        return number_format( $value * 1000, 2 ) . 'ms';
    }
    return number_format( $value, 6 ) . 's';
}



/**
 * Metric Cards
 *
 */

// First Response Time
function get_metric_frt() {
    global $wpdb;

    $table_name = $wpdb->prefix . 'lhf_sf_curl';

    $value = $wpdb->get_var( "SELECT audit_starttransfer_time FROM $table_name ORDER BY audit_timestamp DESC LIMIT 1" );

    $timingTTFB = (float) $value;

    $timingTTFB = number_format( ( (float) $timingTTFB * 1000 ), 2 ) . 'ms';

    return $timingTTFB;
}

// Document Loaded (DOMLoaded) Time
function get_metric_dlt( $siteId ) {
    global $wpdb;

    $table_name = $wpdb->prefix . 'lhf_sf_curl';

    $siteId = (int) $siteId;

    $result = $wpdb->get_results( "SELECT audit_namelookup_time, audit_connect_time, audit_pretransfer_time, audit_redirect_time, audit_starttransfer_time FROM $table_name ORDER BY audit_timestamp DESC LIMIT 1" );

    foreach ( $result as $page ) {
        $timingDNS      = (float) $page->audit_namelookup_time;
        $timingTCP      = (float) $page->audit_connect_time;
        $timingRequest  = (float) $page->audit_pretransfer_time;
        $timingRedirect = (float) $page->audit_redirect_time;
        $timingTTFB     = (float) $page->audit_starttransfer_time;
    }

    // Get total time
    $timingDOM = $timingDNS + $timingTCP + $timingRequest + $timingRedirect + $timingTTFB;

    $timingDOM = number_format( ( (float) $timingDOM * 1000 ), 2 ) . 'ms';

    return $timingDOM;
}




function sf_get_ttfb( $column, $days ) {
    global $wpdb;

    $table_name = $wpdb->prefix . 'lhf_sf_curl';

    $score = $date = [];

    $result = $wpdb->get_results( "SELECT audit_starttransfer_time, audit_timestamp FROM $table_name ORDER BY audit_timestamp DESC LIMIT $days" );
    foreach ( $result as $value ) {
        $score[] = $value->audit_starttransfer_time;
        $date[]  = wp_date( 'M d, Y', strtotime( $value->audit_timestamp ) );
    }

    return ( (string) $column === 'score' ) ? $score : $date;
}

function sf_get_beacon_ttfb( $column, $days ) {
    global $wpdb;

    $table_name = $wpdb->prefix . 'lhf_sf_curl_beacon';

    $score = [];
    $date  = [];

    $result = $wpdb->get_results( "SELECT audit_starttransfer_time, audit_timestamp FROM $table_name ORDER BY audit_timestamp DESC LIMIT $days" );
    foreach ( $result as $value ) {
        $score[] = $value->audit_starttransfer_time;
        $date[]  = wp_date( 'M d, Y', strtotime( $value->audit_timestamp ) );
    }

    return ( (string) $column === 'score' ) ? $score : $date;
}



function sf_fix_uri_schema( $uri ) {
    if ( parse_url( $uri, PHP_URL_SCHEME ) === null ) {
        return 'https://' . ltrim( $uri, '/' );
    } else {
        return $uri;
    }
}

function get_total_asset_size( $asset_array ) {
    return array_reduce(
        $asset_array,
        function ( $total, $asset ) {
            return $total + (int) $asset[1];
        },
        0
    );
}

function get_color_grade( $count, $min, $max ) {
    $color = '';

    if ( $count <= $min ) {
        $color = 'sf-color--green';
    } elseif ( $count > $min && $count <= $max ) {
        $color = 'sf-color--orange';
    } else {
        $color = 'sf-color--red';
    }

    return $color;
}

/**
 * Generate a Sparkline chart container (SVG)

 * @param  string $element_id  ID of SVG element
 * @param  array  $data_labels Array of labels
 * @param  array  $data_values Array of values
 *
 * @return string              HTML Element
 */
function get_sparkline( $element_id, $data_labels, $data_values ) {
    $last_date  = end( $data_labels );
    $last_score = end( $data_values );

    $labels = implode( '|', $data_labels );
    $values = implode( ',', $data_values );

    $out = '<svg class="sparkline--svg" id="' . $element_id . '" width="200" height="40" stroke-width="3" stroke="var(--color-blue2)" fill="url(#sparkline--svg-gradient) rgba(5, 203, 99, 0.2)" data-labels="' . $labels . '" data-values="' . $values . '"></svg>
    <div class="sparkline--tooltip black-50" data-default="' . $last_date . ': ' . $last_score . '">' . $last_date . ': ' . $last_score . '</div>';

    return $out;
}

function get_percentage( $total, $number ) {
    if ( $total > 0 ) {
        return round( $number / ( $total / 100 ), 2 );
    } else {
        return 0;
    }
}

function lhf_ms_to_s( $value, $precision = 2 ) {
    $value = (int) $value / 1000;
    $value = round( $value, $precision );

    return $value;
}
