(function () {

	"use strict";

	const eventHub = new Vue();
	const { __, sprintf } = wp.i18n;
	const buildQuery = function( params ) {
		return Object.keys( params ).map( function( key ) {
			return key + '=' + params[ key ];
		} ).join( '&' );
	}
	const store = new Vuex.Store( {
		state: {
			...window.JetABAFConfig,
			perPage: 30,
			offset: 0,
			totalItems: 0,
			itemsList: [],
			isLoading: true,
			overlappingBookings: false,
		},
		mutations: {
			setValue( state, varObject ) {
				state[ varObject.key ] = varObject.value;
			},
		},
		actions: {
			getItems: function() {

				store.commit( 'setValue', {
					key: 'isLoading',
					value: true
				} );

				wp.apiFetch( {
					method: 'get',
					path: window.JetABAFConfig.api.bookings_list + '?' + buildQuery( {
						per_page: store.state.perPage,
						offset: store.state.offset,
					} ),
				} ).then( function( response ) {

					store.commit( 'setValue', {
						key: 'isLoading',
						value: false
					} );

					if ( response.success ) {
						store.commit( 'setValue', {
							key: 'itemsList',
							value: response.data
						} );

						if ( ! store.state.totalItems ) {
							store.commit( 'setValue', {
								key: 'totalItems',
								value: parseInt( response.total, 10 )
							} );
						}
					}

				} ).catch( function( e ) {

					store.commit( 'setValue', {
						key: 'isLoading',
						value: false
					} );

					eventHub.$CXNotice.add( {
						message: e.message,
						type: 'error',
						duration: 7000,
					} );

				} );

			}
		}
	} );

	Vue.component( 'jet-abaf-bookings-list', {
		template: '#jet-abaf-bookings-list',
		data: function() {
			return {
				deleteDialog: false,
				deleteItem: false,
				detailsDialog: false,
				currentItem: false,
				currentIndex: false,
				editDialog: false,
			};
		},
		computed: Vuex.mapState( {
			itemsList: 'itemsList',
			perPage: 'perPage',
			offset: 'offset',
			totalItems: 'totalItems',
			statuses: state => state.all_statuses,
			bookingInstances: state => state.bookings,
			overlappingBookings: 'overlappingBookings'
		} ),
		methods: {
			changePage: function( page ) {

				store.commit( 'setValue', {
					key: 'offset',
					value: this.perPage * ( page - 1 )
				} );

				store.dispatch( 'getItems' );

			},
			showEditDialog: function( item, index ) {
				this.editDialog = true;

				store.commit( 'setValue', {
					key: 'overlappingBookings',
					value: false
				} );

				this.currentItem = JSON.parse( JSON.stringify( item ) );
				this.currentIndex = index;
			},
			showDetailsDialog: function( item ) {
				this.detailsDialog = true;
				this.currentItem = item;
			},
			showDeleteDialog: function( itemID ) {
				this.deleteItem = itemID;
				this.deleteDialog = true;
			},
			beVisible: function ( key ) {
				switch ( key ) {
					case 'booking_id':
					case 'apartment_unit':
					case 'order_id':
						return false;
					default:
						return true;
				}
			},
			handleEdit: function() {

				var self = this;

				if ( ! self.currentItem ) {
					return;
				}

				store.commit( 'setValue', {
					key: 'overlappingBookings',
					value: false
				} );

				self.itemsList.splice( self.currentIndex, 1, self.currentItem );

				wp.apiFetch( {
					method: 'post',
					path: window.JetABAFConfig.api.update_booking + self.currentItem.booking_id + '/',
					data: { item: self.currentItem }
				} ).then( function( response ) {

					if ( ! response.success ) {

						if ( response.overlapping_bookings ) {

							self.$CXNotice.add( {
								message: response.data,
								type: 'error',
								duration: 7000,
							} );

							store.commit( 'setValue', {
								key: 'overlappingBookings',
								value: response.html
							} );

							self.editDialog          = true;

							return;

						} else {

							self.$CXNotice.add( {
								message: response.data,
								type: 'error',
								duration: 7000,
							} );

						}

					} else {
						self.$CXNotice.add( {
							message: 'Done!',
							type: 'success',
							duration: 7000,
						} );
					}

					self.currentItem = false;
					self.currentIndex = false;

				} ).catch( function( e ) {

					self.$CXNotice.add( {
						message: e.message,
						type: 'error',
						duration: 7000,
					} );

					self.currentItem = false;
					self.currentIndex = false;
				} );

			},
			handleDelete: function() {

				var self = this;

				if ( ! self.deleteItem ) {
					return;
				}

				wp.apiFetch( {
					method: 'delete',
					path: window.JetABAFConfig.api.delete_booking + self.deleteItem + '/',
				} ).then( function( response ) {
					if ( ! response.success ) {
						self.$CXNotice.add( {
							message: response.data,
							type: 'error',
							duration: 7000,
						} );
					}

					for ( var i = 0; i < self.itemsList.length; i++ ) {
						if ( self.itemsList[ i ].booking_id === self.deleteItem ) {
							self.itemsList.splice( i, 1 );
							break;
						}
					}

				} ).catch( function( e ) {
					self.$CXNotice.add( {
						message: e.message,
						type: 'error',
						duration: 7000,
					} );
				} );
			},
			getBookingLabel: function( id ) {

				if ( ! id ) {
					return '--';
				}

				return this.bookingInstances[ id ] || id;
			},
			getOrderLink: function( orderID ) {
				return window.JetABAFConfig.edit_link.replace( /\%id\%/, orderID );
			},
			isFinished: function( status ) {
				return ( 0 <= window.JetABAFConfig.statuses.finished.indexOf( status ) );
			},
			isInProgress: function( status ) {
				return ( 0 <= window.JetABAFConfig.statuses.in_progress.indexOf( status ) );
			},
			isInvalid: function( status ) {
				return ( 0 <= window.JetABAFConfig.statuses.invalid.indexOf( status ) );
			}
		},
	} );

	Vue.component( 'jet-abaf-add-new-booking', {
		template: '#jet-abaf-add-new-booking',
		data: function () {
			return {
				addDialog: false,
				newItem: {
					status: '',
					apartment_id: '',
					check_in_date: '',
					check_out_date: '',
				},
				datePickerFormat: 'dd-MM-yyyy',
				dateMomentFormat: 'DD-MM-YYYY',
			}
		},
		components: {
			vuejsDatepicker: window.vuejsDatepicker
		},
		computed: Vuex.mapState( {
			statuses: state => state.all_statuses,
			bookingInstances: state => state.bookings,
			overlappingBookings: 'overlappingBookings',
			fields: function ( state ) {
				return [ ...state.columns, ...state.additional_columns ];
			}
		} ),
		methods: {
			showAddDialog: function() {
				this.addDialog = true;

				store.commit( 'setValue', {
					key: 'overlappingBookings',
					value: false
				} );
			},

			selectedDate: function( date, type ) {
				let formattedDate = moment( date ).format( this.dateMomentFormat );

				this.$set( this.newItem, type, formattedDate );
			},

			beVisible: function ( key ) {
				switch ( key ) {
					case 'booking_id':
					case 'apartment_unit':
					case 'order_id':
						return false;
					default:
						return true;
				}
			},

			checkRequiredFields: function() {
				let requiredFields = [ 'status', 'apartment_id', 'check_in_date', 'check_out_date' ],
					emptyFields = [];

				for ( let field of requiredFields ){
					if ( ! this.newItem[ field ].length ) {
						emptyFields.push( field );
					}
				}

				if( ! emptyFields[0] ){
					return true;
				}

				emptyFields = emptyFields.join( ', ' ).toLowerCase();

				eventHub.$CXNotice.add( {
					message: wp.i18n.sprintf( __( 'Empty fields: %s.', 'jet-booking' ), emptyFields ),
					type: 'error',
					duration: 7000,
				} );

				return false;
			},

			handleAdd: function() {
				let self = this;

				if( ! self.checkRequiredFields() ) {
					this.addDialog = true;
					return;
				}

				store.commit( 'setValue', {
					key: 'overlappingBookings',
					value: false
				} );

				wp.apiFetch( {
					method: 'post',
					path: window.JetABAFConfig.api.add_booking,
					data: { item: self.newItem }
				} ).then( function( response ) {

					if ( ! response.success ) {
						if ( response.overlapping_bookings ) {
							eventHub.$CXNotice.add( {
								message: response.data,
								type: 'error',
								duration: 7000,
							} );

							store.commit( 'setValue', {
								key: 'overlappingBookings',
								value: response.html
							} );

							self.addDialog = true;

							return;
						} else {
							eventHub.$CXNotice.add( {
								message: response.data,
								type: 'error',
								duration: 7000,
							} );
						}
					} else {
						eventHub.$CXNotice.add( {
							message: 'Done!',
							type: 'success',
							duration: 7000,
						} );
					}

					store.dispatch( 'getItems' );

					self.newItem = {
						status: '',
						apartment_id: '',
						check_in_date: '',
						check_out_date: '',
					};

				} ).catch( function( e ) {
					eventHub.$CXNotice.add( {
						message: e.message,
						type: 'error',
						duration: 7000,
					} );
				} );
			}
		}
	} );

	new Vue( {
		el: '#jet-abaf-bookings-page',
		template: '#jet-abaf-bookings',
		store,
		computed: Vuex.mapState( {
			isSet: state => state.setup.is_set,
			isLoading: 'isLoading',
		} ),
		created: function () {
			store.dispatch('getItems');
		},
	} );

} )();
