<?php

namespace GutenCon;

use WP_REST_Request;
use WP_REST_Server;

defined( 'ABSPATH' ) OR exit;

require_once( 'microdata-parser-master/src/Microdata.php' );
require_once( 'microdata-parser-master/src/MicrodataDOMDocument.php' );
require_once( 'microdata-parser-master/src/MicrodataDOMElement.php' );
require_once( 'microdata-parser-master/src/MicrodataParser.php' );
require_once( 'microdata-parser-master/src/XpathParser.php' );

require_once( 'vendor/autoload.php' );

use YusufKandemirs\MicrodataParser\Microdata;
//use YusufKandemirs\MicrodataParser\MicrodataDOMDocument;

class REST {
	private $rest_namespace = 'gcrestapi/v2';


	private static $instance = null;

	/** @return Assets */
	public static function instance() {
		if ( is_null( static::$instance ) ) {
			static::$instance = new static();
		}

		return static::$instance;
	}

	private function __construct() {
		add_action( 'rest_api_init', array( $this, 'action_rest_api_init_trait' ) );
	}

	public function action_rest_api_init_trait() {
		//		if(!((is_user_logged_in() && is_admin()))) {
		//			return;
		//		}

		register_rest_route( $this->rest_namespace . '/posts',
			'/get',
			array(
				array(
					'methods'  => WP_REST_Server::CREATABLE,
					'permission_callback' => function ( WP_REST_Request $request ) {
						return current_user_can( 'editor' ) || current_user_can( 'administrator' );
					},
					'callback' => array( $this, 'rest_get_posts' ),
				)
			)
		);

		register_rest_route(
			$this->rest_namespace,
			"/parse-offer/",
			array(
				'methods'  => WP_REST_Server::CREATABLE,
				'permission_callback' => function ( WP_REST_Request $request ) {
					return current_user_can( 'editor' ) || current_user_can( 'administrator' );
				},
				'callback' => array( $this, 'rest_parse_offer_handler' ),
			)
		);

		register_rest_route(
			$this->rest_namespace,
			"/wc-listing/",
			array(
				'methods'  => WP_REST_Server::CREATABLE,
				'permission_callback' => function ( WP_REST_Request $request ) {
					return current_user_can( 'editor' ) || current_user_can( 'administrator' );
				},
				'callback' => array( $this, 'rest_wc_listing_handler' ),
			)
		);

		register_rest_route(
			$this->rest_namespace,
			"/product/(?P<id>\d+)",
			array(
				'methods'  => WP_REST_Server::READABLE,
				'permission_callback' => function ( WP_REST_Request $request ) {
					return current_user_can( 'editor' ) || current_user_can( 'administrator' );
				},
				'callback' => array( $this, 'rest_product_handler' ),
			)
		);
	}

	public function rest_get_posts( WP_REST_Request $request ) {
		$params    = array_merge(
			array(
				's'         => '',
				'include'   => '',
				'exclude'   => '',
				'page'      => 1,
				'post_type' => 'post',
			), $request->get_params()
		);
		$isSelect2 = ( $request->get_param( 'typeQuery' ) === 'select2' );

		$args = array(
			'post_status'    => 'publish',
			'posts_per_page' => 5,
			'post_type'      => $params['post_type'],
			'paged'          => $params['page'],
		);

		if ( ! empty( $params['s'] ) ) {
			$args['s'] = $params['s'];
		}
		if ( ! empty( $params['include'] ) ) {
			$args['post__in'] = is_array( $params['include'] ) ? $params['include'] : array( $params['include'] );
		}
		if ( ! empty( $params['exclude'] ) ) {
			$args['post__not_in'] = is_array( $params['exclude'] ) ? $params['exclude'] : array( $params['exclude'] );
		}

		$response_array = array();
		$keys           = $isSelect2 ?
			[ 'label' => 'text', 'value' => 'id' ] :
			[ 'label' => 'label', 'value' => 'value' ];

		$posts = new \WP_Query( $args );
		if ( $posts->post_count > 0 ) {
			/* @var \WP_Post $gallery */
			foreach ( $posts->posts as $_post ) {
				$response_array[] = array(
					$keys['label'] => ! empty( $_post->post_title ) ? $_post->post_title : esc_html__( 'No Title', 'gutencon' ),
					$keys['value'] => $_post->ID,
				);
			}
		}
		wp_reset_postdata();

		$return = array(
			'results'    => $response_array,
			'pagination' => array(
				'more' => $posts->max_num_pages >= ++ $params['page'],
			)
		);

		return rest_ensure_response( $return );
	}

	public function rest_product_handler( WP_REST_Request $request ) {
		$id   = $request->get_params()['id'];
		$data = array();

		if ( empty( $id ) ) {
			return new \WP_Error( 'empty_data', 'Pass empty data', array( 'status' => 404 ) );
		}

		$product              = wc_get_product( $id );
		$thumbID 			  = $product->get_image_id();
		$copy                 = $product->get_short_description();
		$product_on_sale      = $product->is_on_sale();
		$regular_price        = (float) $product->get_regular_price();
		$sale_price           = (float) $product->get_sale_price();
		$attributes           = $product->get_attributes();
		$gallery_ids          = $product->get_gallery_image_ids();
		$price_label          = '';
		$gallery_images       = array();
		$product_name         = $product->get_title();
		$add_to_cart_text     = $product->add_to_cart_text();
		$product_type         = $product->get_type();
		
		if ( empty( $thumbID ) ) {
			$thumbnail_url = plugin_dir_url( __FILE__ ) . 'assets/icons/noimage-placeholder.png';
		}else{
			$thumbnail_url = wp_get_attachment_image_url($thumbID);
		}
		
		if ( ! empty( $copy ) ) {
			$copy = wp_trim_words($copy, 15);
		}

		if ( $product_on_sale && $regular_price && $sale_price > 0 && $product_type !== 'variable' ) {
			$sale_proc   = 0 - ( 100 - ( $sale_price / $regular_price ) * 100 );
			$sale_proc   = round( $sale_proc );
			$price_label = $sale_proc . '%';
		}

		if ( ! empty( $attributes ) ) {
			ob_start();
			wc_display_product_attributes( $product );
			$attributes = ob_get_contents();
			ob_end_clean();
		}

		if ( ! empty( $gallery_ids ) ) {
			foreach ( $gallery_ids as $key => $value ) {
				$gallery_images[] = wp_get_attachment_url( $value );
			}
		}

		$data['thumbnail'] = array(
			'url' => $thumbnail_url,
			'id'  => $thumbID
		);
		$data['title']             = $product_name;
		$data['copy']              = $copy;
		$data['priceLabel']        = $price_label;
		$data['addToCartText']     = $add_to_cart_text;
		$data['productAttributes'] = $attributes;
		$data['galleryImages']     = $gallery_images;
		$data['userscore']		   = $product->get_average_rating();
		$data['priceHtml']         = $product->get_price_html();
		$data['id']                = $id;

		return json_encode( $data );
	}

	public function rest_wc_listing_handler( WP_REST_Request $request ) {
		$posts_id = $request['posts_id'];
		$data     = array();

		if ( empty( $posts_id ) || count( $posts_id ) === 0 ) {
			return new \WP_Error( 'empty_data', 'Pass empty data', array( 'status' => 404 ) );
		}

		foreach ( $posts_id as $index => $id ) {
			$product = wc_get_product( $id );

			$thumbID = $product->get_image_id();
			if ( empty( $thumbID ) ) {
				$thumbnail_url = plugin_dir_url( __FILE__ ) . 'assets/icons/noimage-placeholder.png';
			}else{
				$thumbnail_url = wp_get_attachment_image_url($thumbID);
			}
			$copy = $product->get_short_description();
			if ( ! empty( $copy ) ) {
				$copy = wp_trim_words($copy, 15);
			}
			
			$data[$index] = array(
				'userscore'          => $product->get_average_rating(),
				'thumbnail'      => array(
					'url' => $thumbnail_url,
					'id'  => $thumbID
				),
				'title'          => $product->get_title(),
				'copy'           => $copy,
				'currentPrice'   => $product->get_price(),
				'oldPrice'       => $product->get_regular_price(),
				'addToCartText'  => $product->add_to_cart_text(),
				'priceHtml'		 => $product->get_price_html(),
				'id'             => $id
			);
		}

		return json_encode( $data );
	}

	public function rest_parse_offer_handler( WP_REST_Request $request ) {
		$url = $request->get_params()['url'];

		if ( empty( $url ) || filter_var( $url, FILTER_VALIDATE_URL ) === false ) {
			return new \WP_Error( 'invalid_url', 'Not valid url', array( 'status' => 404 ) );
		}
		
		$hostName = $this->get_host_name( $url );
		
		$xpathArray = array();
		
		if( $hostName == 'amazon' ){
			$xpathArray = array(
				'name' => '//h1[@id="title"]',
				'image'=> '//img[@id="landingImage"]',
				'description' => '//div[@id="productDescription"]/p',
				'priceCurrency' => '//div[@id="cerberus-data-metrics"]',
				'price' => '//span[@id="priceblock_ourprice"]%DELIMITER%//span[@id="priceblock_dealprice"]%DELIMITER%//div[@id="cerberus-data-metrics"]',
			);
		}
		
		if( !empty( $xpathArray ) ){ //we check if we have xpath ready
			return Microdata::fromXpathFile( $url )->toJSON( $xpathArray );
		}else{
			$args = array( 
				'timeout' => 30,
				'httpversion' => '1.0',
				'user-agent'  => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/86.0.4240.75 Safari/537.36'
			);
			$request = wp_safe_remote_get($url, $args);
			$html = wp_remote_retrieve_body( $request );
			$reader = new \Brick\StructuredData\Reader\ReaderChain(
				new \Brick\StructuredData\Reader\MicrodataReader(),
				new \Brick\StructuredData\Reader\JsonLdReader()
			);
			$htmlReader = new \Brick\StructuredData\HTMLReader($reader);
			$items = $htmlReader->read($html, $url);
			$itemarray = array();
			foreach ($items as $index => $item) {
				$itemarray['items'][$index]['type'] = $item->getTypes();
				foreach ($item->getProperties() as $name => $values) {
					$name = str_replace(array('http://schema.org/', 'https://schema.org/'), '', $name);
					foreach ($values as $valueindex=>$value) {
						if ($value instanceof \Brick\StructuredData\Item) {
							$itemarray['items'][$index]['properties'][$name][$valueindex]['type'] = $value->getTypes();
							foreach ($value->getProperties() as $innername => $innervalues) {
								$innername = str_replace(array('http://schema.org/', 'https://schema.org/'), '', $innername);
								$itemarray['items'][$index]['properties'][$name][$valueindex]['properties'][$innername] = $innervalues;
							}
		
						}else{
							$itemarray['items'][$index]['properties'][$name][$valueindex] = $value;
						}
					}
				}
			}
			return json_encode($itemarray);
		}
	}
	
    public function get_host_name( $url ) {
		$domain = strtolower(str_ireplace('www.', '', parse_url($url, PHP_URL_HOST)));
		
		// remove subdomain
        if (preg_match('/(?P<domain>[a-z0-9][a-z0-9\-]{1,63}\.[a-z\.]{2,6})$/i', $domain, $regs)) {
            $domain = $regs['domain'];
        }
		
		$hostData = explode('.', $domain);
		
		return $hostData[0];
    }
}
\GutenCon\REST::instance();