<?php

//------------------------------------------

GFForms::include_addon_framework();

class GFCollapsibleSections extends GFAddOn {

	protected $_version = GFCS_VERSION;
	protected $_min_gravityforms_version = '2.6';

	protected $_slug = GFCS_SLUG;
	protected $_path = 'gf-collapsible-sections/gf-collapsible-sections.php';
	protected $_full_path = __FILE__;
	protected $_title = GFCS_NAME;
	protected $_short_title = 'Collapsible Sections';
	protected $_url = 'https://jetsloth.com/gravity-forms-collapsible-sections/';

	protected $_defaultTheme = "simple";
	protected $_defaultImageBackgroundColor = "#ffffff";
	protected $_defaultImageSize = "20";

	/**
	 * Members plugin integration
	 */
	protected $_capabilities = array( 'gravityforms_edit_forms', 'gravityforms_edit_settings' );

	/**
	 * Permissions
	 */
	protected $_capabilities_settings_page = 'gravityforms_edit_settings';
	protected $_capabilities_form_settings = 'gravityforms_edit_forms';
	protected $_capabilities_uninstall = 'gravityforms_uninstall';

	private static $_instance = null;

	/**
	 * Get an instance of this class.
	 *
	 * @return GFCollapsibleSections
	 */
	public static function get_instance() {
		if ( self::$_instance == null ) {
			self::$_instance = new GFCollapsibleSections();
		}

		return self::$_instance;
	}

	private function __clone() {
	} /* do nothing */

	public function use_new_features() {
		$use_legacy_setting = $this->get_plugin_setting('gf_collapsible_sections_use_legacy_styles');
		$use_legacy_value = ( !empty($use_legacy_setting) );
		return ( false === $use_legacy_value );
	}

	/**
	 * Handles anything which requires early initialization.
	 */
	public function pre_init() {
		parent::pre_init();
	}

	/**
	 * Handles hooks and loading of language files.
	 */
	public function init() {

		$this->init_license();

		// add aria attributes
		add_action( 'gform_field_container', array( $this, 'custom_field_container' ), 10, 6 );

		// add a special class to relevant fields so we can identify them later
		add_action( 'gform_field_css_class', array( $this, 'add_custom_field_class' ), 10, 3 );
		add_action( 'gform_pre_render', array( $this, 'add_custom_form_class' ), 10, 3 );

		add_action( 'gform_enqueue_scripts', array( $this, 'frontend_inline_scripts' ) );

		add_action( 'gform_noconflict_scripts', array( $this, 'register_our_noconflict_scripts' ), 10, 1 );

		// inline css overrides. Run as late as possible
		add_action( 'gform_enqueue_scripts', array( $this, 'frontend_inline_styles' ), PHP_INT_MAX, 1 );

		parent::init();

	}

	/**
	 * Initialize the admin specific hooks.
	 */
	public function init_admin() {

		if ( $this->use_new_features() ) {
			GFCommon::remove_dismissible_message("gf_collapsible_sections_legacy_mode_message");
		}
        else {
	        GFCommon::add_dismissible_message(
		        "Collapsible Sections is currently set to legacy mode. To take advantage of the new and improved styles and features, please turn legacy mode off <a href='" . $this->get_plugin_settings_url() . "#gform-settings-section-collapsible-sections-legacy-settings'>in settings</a> and <strong><i>test your forms</i></strong>, or <a href='" . "https://jetsloth.com/support/gravity-forms-collapsible-sections/new-styles-and-settings/" . "' target='_blank'>check out this article</a> for more info.",
		        "gf_collapsible_sections_legacy_mode_message",
		        'updated',
		        false,
		        true
	        );
        }

		// form editor
		add_filter( 'gform_field_settings_tabs', array( $this, 'custom_settings_tab' ), 10, 1 );
		add_action( 'gform_field_settings_tab_content_collapsible_sections', array( $this, 'custom_settings_markup' ), 20, 1 );

		// form editor
		add_filter( 'gform_tooltips', array( $this, 'add_collapsible_sections_field_tooltips' ) );

		$name = plugin_basename($this->_path);
		add_action( 'after_plugin_row_'.$name, array( $this, 'gf_plugin_row' ), 10, 2 );

		add_action( 'admin_enqueue_scripts', array( $this, 'admin_enqueue_scripts' ) );
		add_action( 'admin_footer', array($this, 'maybe_show_splash_page') );

		parent::init_admin();

	}


	public function admin_enqueue_scripts() {
		if ( $this->is_form_editor() ) {
			wp_enqueue_media();// For Media Library
		}
		if ( $this->is_plugin_settings( $this->_slug ) || $this->is_form_settings( $this->_slug ) ) {
			wp_enqueue_code_editor( array( 'type' => 'text/css' ) );// for custom CSS
		}
	}

	public function get_app_menu_icon() {
		return $this->get_base_url() . '/images/icons/icon-collapsible-sections.svg';
	}

	public function get_menu_icon() {
		return $this->get_base_url() . '/images/icons/icon-collapsible-sections.svg';
	}

	public function is_gravity_flow_workflow_detail_page() {
		return ( function_exists('gravity_flow') && gravity_flow()->is_workflow_detail_page() );
	}


	// # SCRIPTS & STYLES -----------------------------------------------------------------------------------------------

	public function register_our_noconflict_scripts( $scripts ) {
		$scripts[] = 'media-audiovideo';
		return $scripts;
	}

	/**
	 * Return the scripts which should be enqueued.
	 *
	 * @return array
	 */
	public function scripts() {

		$gf_collapsible_sections_js_deps = array( 'jquery' );
		$admin_js_deps = array( 'jquery' );

		if ( $this->is_plugin_settings( $this->_slug ) || $this->is_form_settings( $this->_slug ) ) {
			$admin_js_deps[] = 'code-editor';
		}

        $admin_script = array(
	        'handle'   => 'gf_collapsible_sections_admin',
	        'src'      => $this->get_base_url() . '/js/gf_collapsible_sections_admin.js',
	        'version'  => $this->_version,
	        'deps'     => $admin_js_deps,
	        'callback' => array( $this, 'localize_admin_scripts' ),
	        'enqueue'  => array(
		        array( 'admin_page' => array( 'form_editor', 'plugin_settings', 'form_settings' ) ),
	        ),
        );

        $collapsible_script = array(
	        'handle'  => 'gf_collapsible_sections',
	        'src'     => $this->get_base_url() . '/js/gf_collapsible_sections.js',
	        'version' => $this->_version,
	        'deps'    => $gf_collapsible_sections_js_deps,
	        'callback' => array( $this, 'localize_scripts' ),
	        'strings' => array(
		        'version' => $this->_version
	        ),
	        'enqueue' => array(
		        array( 'admin_page' => array( 'form_editor' ) ),
		        //array( 'field_types' => array( 'section' ) ),
		        array( $this, 'maybe_enqueue_main_scripts_styles' )
	        ),
        );


		if ( is_admin() && !$this->is_gravity_flow_workflow_detail_page() ) {
			$scripts = array(
				$admin_script
			);
		}
		else {
			$scripts = array(
				$collapsible_script,
			);
		}

		return array_merge( parent::scripts(), $scripts );
	}

	/**
	 * Return the stylesheets which should be enqueued.
	 *
	 * @return array
	 */
	public function styles() {

		$use_new_features = $this->use_new_features();

        $is_admin = is_admin();
		$admin_styles_deps = array('code-editor');

		$styles = array();

		$admin_styles = array(
			'handle'  => 'gf_collapsible_sections_admin',
			'src'     => $this->get_base_url() . '/css/gf_collapsible_sections_admin.css',
			'version' => $this->_version,
            'deps' => $admin_styles_deps,
			'enqueue' => array(
				array('admin_page' => array( 'form_editor', 'plugin_settings', 'form_settings', 'entry_view', 'entry_detail' )),
				array('query' => 'page=gf_entries'),
				array('query' => 'page=gf_edit_forms')
			),
		);

		$load_admin_css = apply_filters( 'collapsible_sections_load_form_editor_css', true );
		if ( $is_admin && !$this->is_gravity_flow_workflow_detail_page() && ( !empty($load_admin_css) || $use_new_features ) ) {
			$styles[] = $admin_styles;
		}


		$frontend_styles = array(
			'handle'  => $use_new_features ? 'gf_collapsible_sections' : 'gf_collapsible_sections_legacy',
			'src'     => $use_new_features ? $this->get_base_url() . '/css/gf_collapsible_sections.css' : $this->get_base_url() . '/css/gf_collapsible_sections_legacy.css',
			'version' => $this->_version,
			'media'   => 'screen',
			'enqueue' => array(
				array( 'admin_page' => array( 'form_editor' ) ),
				//array( 'field_types' => array( 'section' ) ),
				array( $this, 'maybe_enqueue_main_scripts_styles' )
			),
        );

		$load_main_css = apply_filters( 'collapsible_sections_load_main_css', true );
        if ( ($this->is_gravity_flow_workflow_detail_page() || !$is_admin) && ( !empty($load_main_css) || $use_new_features ) ) {
	        $styles[] = $frontend_styles;
        }

		return array_merge( parent::styles(), $styles );
	}


	public function maybe_enqueue_main_scripts_styles( $form ) {
		return( !empty($form) && $this->form_has_collapsible_sections($form) );
	}

	public function form_has_collapsible_sections($form) {

		$has_collapsible_sections = false;

		if ( !empty($form) && isset($form['fields']) ) {
			foreach ( $form['fields'] as &$field ) {
                if ( $this->get_field_settings_value("enableCollapsible", "none", $field) != "none" ) {
	                $has_collapsible_sections = true;
	                break;
                }
			}
		}

		return $has_collapsible_sections;

    }

	public function get_field_theme( $field, $fallback_to_form = true ) {
		$theme = false;

		$field_theme = $this->get_field_settings_value("theme", "form_setting", $field);
		if ( (empty($field_theme) || $field_theme == "form_setting") && $fallback_to_form ) {
			$form = GFAPI::get_form( $field['formId'] );
			$form_settings = $this->get_form_settings( $form );
			$form_theme = $this->get_form_settings_value("gf_collapsible_sections_theme", "global_setting", $form, $form_settings);
			if ( empty($form_theme) || $form_theme == "global_setting" ) {
				$plugin_settings = $this->get_plugin_settings();
				$theme = $this->get_plugin_settings_value("gf_collapsible_sections_global_theme", $this->_defaultTheme, $plugin_settings);
			}
			else {
				$theme = $form_theme;
			}
		}
		else {
			$theme = $field_theme;
		}

		return $theme;
	}

	public function frontend_inline_styles( $form ) {

		if ( !$this->form_has_collapsible_sections( $form ) ) {
			return;
		}

		$use_new_features = $this->use_new_features();

		$form_id = rgar( $form, 'id' );
		$form_settings = $this->get_form_settings( $form );
		$plugin_settings = $this->get_plugin_settings();

        $ignore_global_css_value = $this->get_form_settings_value('gf_collapsible_sections_ignore_global_css', 0, $form_id, $form_settings);

        if ( $use_new_features ) {
	        $global_css_value = $this->get_plugin_settings_value('gf_collapsible_sections_user_css_global', "", $plugin_settings);
	        $global_ref = "gf_collapsible_sections_user_css_global";
	        if ( empty($ignore_global_css_value) && !empty($global_css_value) && !wp_style_is($global_ref) ) {
		        wp_register_style( $global_ref, false );
		        wp_enqueue_style( $global_ref );
		        wp_add_inline_style( $global_ref, $global_css_value );
	        }

	        $form_css_value = $this->get_form_settings_value('gf_collapsible_sections_user_css_form', '', $form_id, $form_settings);
	        $ref = "gf_collapsible_sections_user_css_form_{$form_id}";
	        if ( !empty($form_css_value) && !wp_style_is($ref) ) {
		        wp_register_style( $ref, false );
		        wp_enqueue_style( $ref );
		        wp_add_inline_style( $ref, $form_css_value );
	        }
        }
        else {
	        $global_css_value = $this->get_plugin_settings_value('gf_collapsible_sections_custom_css_global', "", $plugin_settings);
	        $global_ref = "gf_collapsible_sections_custom_global";
	        if ( empty($ignore_global_css_value) && !empty($global_css_value) && !wp_style_is($global_ref) ) {
		        wp_register_style( $global_ref, false );
		        wp_enqueue_style( $global_ref );
		        wp_add_inline_style( $global_ref, $global_css_value );
	        }

	        $form_css_value = $this->get_form_settings_value('gf_collapsible_sections_custom_css', '', $form_id, $form_settings);
	        $ref = "gf_collapsible_sections_custom_{$form_id}";
	        if ( !empty($form_css_value) && !wp_style_is($ref) ) {
		        wp_register_style( $ref, false );
		        wp_enqueue_style( $ref );
		        wp_add_inline_style( $ref, $form_css_value );
	        }
        }


        if ( $use_new_features ) {

	        $form_image_styles_value = "";

            $image_background_color = "";
	        $global_image_background_color_setting = $this->get_plugin_settings_value("gf_collapsible_sections_global_image_background_color", "none", $plugin_settings);
            if ( $global_image_background_color_setting == "custom" ) {
	            $global_image_background_color_custom_setting = $this->get_plugin_settings_value("gf_collapsible_sections_global_image_background_color_custom", "", $plugin_settings);
                if ( !empty($global_image_background_color_custom_setting) ) {
	                $image_background_color = "{$global_image_background_color_custom_setting}";
                }
            }

	        $form_image_background_color_setting = $this->get_form_settings_value("gf_collapsible_sections_image_background_color", "global_setting", $form, $form_settings);
	        if ( $form_image_background_color_setting == "custom" ) {
		        $form_image_background_color_custom_setting = $this->get_form_settings_value("gf_collapsible_sections_image_background_color_custom", "", $form, $form_settings);
                if ( !empty($form_image_background_color_custom_setting) ) {
	                $image_background_color = "{$form_image_background_color_custom_setting}";
                }
	        }

            if ( !empty($image_background_color) ) {
	            //$form_image_styles_value .= " #gform_fields_{$form['id']} .collapsible-sections-field::before { background-color: {$image_background_color}; } ";
	            $form_image_styles_value .= " #gform_fields_{$form['id']} { --cs-image-background-color: {$image_background_color}; } ";
            }

            $image_size = "";
	        $global_image_size_setting = $this->get_plugin_settings_value("gf_collapsible_sections_global_image_size", "cover", $plugin_settings);
            if ( $global_image_size_setting == "custom" ) {
	            $global_image_size_custom_setting = $this->get_plugin_settings_value("gf_collapsible_sections_global_image_size_custom", $this->_defaultImageSize, $plugin_settings);
                if ( !empty($global_image_size_custom_setting) ) {
	                $image_size = "{$global_image_size_custom_setting}";
                }
            }

	        $form_image_size_setting = $this->get_form_settings_value("gf_collapsible_sections_image_size", "global_setting", $form, $form_settings);
	        if ( $form_image_size_setting == "custom" ) {
		        $form_image_size_custom_setting = $this->get_form_settings_value("gf_collapsible_sections_image_size_custom", $this->_defaultImageSize, $form, $form_settings);
                if ( !empty($form_image_size_custom_setting) ) {
	                $image_size = "{$form_image_size_custom_setting}";
                }
	        }

            if ( !empty($image_size) ) {
	            //$form_image_styles_value .= " #gform_fields_{$form['id']} .collapsible-sections-field::before { background-size: {$image_size}px; } ";
	            $form_image_styles_value .= " #gform_fields_{$form['id']} { --cs-image-background-size: {$image_size}px; } ";
            }

            $image_corners = "";
	        $global_image_corners_setting = $this->get_plugin_settings_value("gf_collapsible_sections_global_image_corners", "round", $plugin_settings);
            if ( $global_image_corners_setting == "custom" ) {
	            $global_image_corners_custom_setting = $this->get_plugin_settings_value("gf_collapsible_sections_global_image_corners_custom", "", $plugin_settings);
                if ( !empty($global_image_corners_custom_setting) ) {
	                $image_corners = "{$global_image_corners_custom_setting}";
                }
            }

	        $form_image_corners_setting = $this->get_form_settings_value("gf_collapsible_sections_image_corners", "global_setting", $form, $form_settings);
	        if ( $form_image_corners_setting == "custom" ) {
		        $form_image_corners_custom_setting = $this->get_form_settings_value("gf_collapsible_sections_image_corners_custom", "", $form, $form_settings);
                if ( !empty($form_image_corners_custom_setting) ) {
	                $image_corners = "{$form_image_corners_custom_setting}";
                }
	        }

            if ( !empty($image_corners) ) {
	            //$form_image_styles_value .= " #gform_fields_{$form['id']} .collapsible-sections-field::before { border-radius: {$image_corners}px; } ";
	            $form_image_styles_value .= " #gform_fields_{$form['id']} { --cs-image-radius: {$image_corners}px; } ";
            }


	        foreach( $form['fields'] as $field ) {
		        $collapsible_enabled = $this->get_field_settings_value("enableCollapsible", "none", $field);
		        $image_url = $this->get_field_settings_value("imageUrl", "", $field);
		        if ( empty($collapsible_enabled) || $collapsible_enabled == "none" || empty($image_url) ) {
			        continue;
		        }
		        $image_url = esc_html($image_url);
		        $form_image_styles_value .= " #field_{$form['id']}_{$field['id']}:before { background-image:url({$image_url}); } ";


		        $field_image_background_color_setting = $this->get_field_settings_value("imageBackgroundColor", "form_setting", $field);
                if ( $field_image_background_color_setting === "custom" ) {
	                $field_image_background_color_custom = $this->get_field_settings_value("imageBackgroundColorCustom", "", $field);
                    if ( !empty($field_image_background_color_custom) ) {
	                    //$form_image_styles_value .= " #field_{$form['id']}_{$field['id']}.collapsible-sections-field::before { background-color: {$field_image_background_color_custom}; } ";
	                    $form_image_styles_value .= " #field_{$form['id']}_{$field['id']} { --cs-image-background-color: {$field_image_background_color_custom}; } ";
                    }
                }

	        }

	        $form_image_styles_ref = "gf_collapsible_sections_images_{$form_id}";
	        if ( !empty($form_image_styles_value) && !wp_style_is($form_image_styles_ref) ) {
		        wp_register_style( $form_image_styles_ref, false );
		        wp_enqueue_style( $form_image_styles_ref );
		        wp_add_inline_style( $form_image_styles_ref, $form_image_styles_value );
	        }

        }

	}

	public function frontend_inline_scripts( $form ) {

        if ( !$this->form_has_collapsible_sections( $form ) ) {
            return;
        }

		$form_id = rgar( $form, 'id' );
		$form_settings = $this->get_form_settings( $form );

		$scroll_ref = "gf_collapsible_sections_scroll_".$form_id;
		if ( !wp_script_is($scroll_ref) ) {
			$scroll_to_val = $this->get_form_settings_value('gf_collapsible_sections_scroll_to', false, $form_id, $form_settings);
            $scroll_to = $scroll_to_val ? "true" : "false";
			$scroll_offset_value = (int) $this->get_form_settings_value('gf_collapsible_sections_scroll_to_offset', 0, $form_id, $form_settings);
			$scroll_to_duration = (int) $this->get_form_settings_value('gf_collapsible_sections_scroll_to_duration', 400, $form_id, $form_settings);
			wp_register_script( $scroll_ref, false );
            ob_start();
            ?>
(function(){
    if ( typeof window.gf_collapsible_sections_scroll_to_<?php echo $form_id; ?> === 'undefined' ) {
        window.gf_collapsible_sections_scroll_to_<?php echo $form_id; ?> = <?php echo $scroll_to; ?>;
    }
    if ( typeof window.gf_collapsible_sections_scroll_to_offset_<?php echo $form_id; ?> === 'undefined' ) {
        window.gf_collapsible_sections_scroll_to_offset_<?php echo $form_id; ?> = <?php echo $scroll_offset_value; ?>;
    }
    if ( typeof window.gf_collapsible_sections_scroll_to_duration_<?php echo $form_id; ?> === 'undefined' ) {
        window.gf_collapsible_sections_scroll_to_duration_<?php echo $form_id; ?> = <?php echo $scroll_to_duration; ?>;
    }
})();
            <?php
            $scroll_script = ob_get_clean();
			wp_add_inline_script( $scroll_ref, $scroll_script );
			wp_enqueue_script( $scroll_ref );
		}

	}


	/**
	 * Localize the strings used by the scripts.
	 */
	public function localize_admin_scripts() {

		$use_new_features = $this->use_new_features();
		$plugin_settings = $this->get_plugin_settings();

		$params = array(
			'version' => $this->_version,
		);
		wp_localize_script( 'gf_collapsible_sections_admin', 'collapsibleSectionsFieldVars', $params );

		//localize strings for the js file
		$strings = array(
			'toggle' => esc_html__( 'Make this section collapsible', 'gf_collapsible_sections' ),
			'toggleNone' => esc_html__( 'None (normal section)', 'gf_collapsible_sections' ),
			'toggleStart' => esc_html__( 'Start collapsible section', 'gf_collapsible_sections' ),
			'toggleEnd' => esc_html__( 'End previous collapsible section', 'gf_collapsible_sections' ),
			'toggleEndDescription' => esc_html__( 'If selected, the field before this will be the last within a collapsible section and any fields from this one will appear outside.', 'gf_collapsible_sections' ),

			'endSectionDisplay' => esc_html__( 'Display setting for this field', 'gf_collapsible_sections' ),
			'endSectionDisplayDefault' => esc_html__( 'Show (normal section)', 'gf_collapsible_sections' ),
			'endSectionDisplayHidden' => esc_html__( 'Hide (will purely be used to end previous collapsible section)', 'gf_collapsible_sections' ),

			'descriptionPlacement' => esc_html__( 'Collapsible section description placement', 'gf_collapsible_sections' ),
			'descriptionPlacementForm' => esc_html__( 'Use Form Setting', 'gf_collapsible_sections' ),
			'descriptionPlacementInside' => esc_html__( 'Inside section', 'gf_collapsible_sections' ),
			'descriptionPlacementTitle' => esc_html__( 'Below title', 'gf_collapsible_sections' ),
			'showDescriptionInside' => esc_html__( 'Show description inside', 'gf_collapsible_sections' ),
			'showDescriptionInsideText' => esc_html__( 'With this setting, the field description will be displayed inside the collapsible section and only visible when expanded.', 'gf_collapsible_sections' )
		);
		wp_localize_script( 'gf_collapsible_sections_admin', 'collapsibleSectionsFieldStrings', $strings );


		wp_localize_script( 'gf_collapsible_sections_admin', 'collapsibleSectionsVars', array(
			'gf_version' => GFCommon::$version,
            'form_settings' => admin_url( "admin.php?subview=settings&page=gf_edit_forms&view=settings&id=" ),
			'version' => $this->_version,
			'useNewFeatures' => $use_new_features ? 'true' : 'false',
			'defaults' => array(
				'theme' => $this->_defaultTheme,
			),
			'globals' => array(
				'theme' => $this->get_plugin_settings_value("gf_color_picker_global_display_style", $this->_defaultTheme, $plugin_settings),
			),
		) );


	}

	public function localize_scripts() {

		$use_new_features = $this->use_new_features();

		$params = array(
			'useNewFeatures' => $use_new_features ? 'true' : 'false',
			'version' => $this->_version,
		);
		wp_localize_script( 'gf_collapsible_sections', 'collapsibleSectionsVars', $params );


	}


	/**
	 * Creates a settings page for this add-on.
	 */
	public function plugin_settings_fields() {

		$use_new_features = $this->use_new_features();
		$plugin_settings = $this->get_plugin_settings();

		$const_key = $this->get_const_license_key();

		$license_field = array(
			'name' => 'gf_collapsible_sections_license_key',
			'tooltip' => esc_html__('Enter the license key you received after purchasing the plugin.', 'gf_collapsible_sections'),
			'label' => esc_html__('License Key', 'gf_collapsible_sections'),
			'type' => 'text',
			'input_type' => 'password',
			'class' => 'medium',
			'default_value' => $this->get_license_key(),
			'after_input' => ( !empty($const_key) ) ? '<div class="alert gforms_note_info">' . __( "Your license key is defined as a constant (likely in wp-config.php) and can't be edited here.", 'gf_collapsible_sections' ) . '</div>' : '',
			'disabled' => ( !empty($const_key) ),
			'validation_callback' => array($this, 'license_validation'),
			'feedback_callback' => array($this, 'license_feedback'),
			'error_message' => esc_html__( 'Invalid license', 'gf_collapsible_sections' ),
		);

        /*
		if (!empty($license) && !empty($status)) {
			$license_field['after_input'] = ($status == 'valid') ? ' License is valid' : ' Invalid or expired license';
		}
        */

		$license_section = array(
			'type' => 'section',
			'title'  => esc_html__('To unlock plugin updates, please enter your license key below', 'gf_collapsible_sections'),
			'fields' => array(
				$license_field
			)
		);


		$use_legacy_styles_value = $this->get_plugin_settings_value('gf_collapsible_sections_use_legacy_styles', true, $plugin_settings);
		$legacy_styles_toggle_field = array(
			'name' => 'gf_collapsible_sections_use_legacy_styles',
			'type' => 'toggle',
			'tooltip' => esc_html__('In legacy mode, collapsible sections will not make use of the new features and styles. It will mostly continue to run like 1.1.x versions', 'gf_collapsible_sections'),
			'label' => __( 'Switch to legacy mode', 'gf_collapsible_sections' ),
		);

		$new_styles_dependency = array(
			'live'   => true,
			'fields' => array(
				array(
					'field' => 'gf_collapsible_sections_use_legacy_styles',
					'values' => array( false, '0', null, 'false', '' ),
				),
			),
		);

		$style_switch_section = array(
			'type' => 'section',
			'title'  => esc_html__('Collapsible Sections Legacy Settings', 'gf_collapsible_sections'),
			'description' => esc_html("Version 1.2+ is a major update that includes new features and settings offering a better experience for Collapsible Sections. If you've updated from previous versions and are having issues in your forms with these new settings, switch to legacy mode which will run more like 1.1.x versions and without the new features. It's always best practice to test your forms after updating settings or versions.", 'gf_collapsible_sections'),
			'fields' => array(
				$legacy_styles_toggle_field,
			)
		);


		// THEME SETTING
		$theme_value = $this->get_plugin_settings_value( "gf_collapsible_sections_global_theme", $this->_defaultTheme, $plugin_settings );
		$theme_field = array(
			'name' => 'gf_collapsible_sections_global_theme',
			'label' => esc_html__( 'Default Theme', 'gf_collapsible_sections' ),
			'type' => 'select',
			'class' => 'medium',
			'choices' => $this->get_settings_select_choices_array( $this->get_settings_theme_choices() )
		);
		if ( !empty($theme_value) ) {
			$theme_field['default_value'] = $theme_value;
		}

		// THEME PREVIEW
		$theme_preview_field = array(
			'name' => 'gf_collapsible_sections_global_theme_preview',
			'label' => '',
			'type' => 'html',
			'class' => 'medium',
			'html' => $this->get_settings_theme_preview_html(),
		);

		$theme_section = array(
			'type' => 'section',
			'dependency' => $new_styles_dependency,
			'title' => esc_html__( 'Theme', 'gf_collapsible_sections' ),
			'class' => 'gform-settings-panel--half',
			'fields' => array(
				$theme_field,
				$theme_preview_field,
			)
		);



		// NEW: IMAGE BACKGROUND COLOR
		$image_background_color_value = $this->get_plugin_settings_value("gf_collapsible_sections_global_image_background_color", "none", $plugin_settings);
		$image_background_color_field = array(
			'name' => 'gf_collapsible_sections_global_image_background_color',
			'label' => esc_html__('Default Image Background Color', 'gf_collapsible_sections'),
			'type' => 'select',
			'class' => 'medium',
			'choices' => array(
				array(
					'value' => 'none',
					'label' => esc_html__('None', 'gf_collapsible_sections')
				),
				array(
					'value' => 'custom',
					'label' => esc_html__('Custom', 'gf_collapsible_sections')
				),
			)
		);
		if ( !empty($image_background_color_value) ) {
			$image_background_color_field['default_value'] = $image_background_color_value;
		}

		// NEW: IMAGE BACKGROUND - CUSTOM
		$image_background_color_custom_value = $this->get_plugin_settings_value("gf_collapsible_sections_global_image_background_color_custom", $this->_defaultImageBackgroundColor, $plugin_settings);
		$image_background_color_custom_field = array(
			'name' => 'gf_collapsible_sections_global_image_background_color_custom',
			'label' => esc_html__('Custom Image Background Color', 'gf_collapsible_sections'),
			'default_value' => $image_background_color_custom_value,
			'type' => 'text',
			'class' => 'medium',
			'dependency' => array(
				'live'   => true,
				'fields' => array(
					array(
						'field' => 'gf_collapsible_sections_global_image_background_color',
						'values' => array( 'custom' ),
					),
				),
			)
		);


		$image_size_value = $this->get_plugin_settings_value("gf_collapsible_sections_global_image_size", "cover", $plugin_settings);
		$image_size_field = array(
			'name' => 'gf_collapsible_sections_global_image_size',
			'label' => esc_html__('Default Image Size', 'gf_collapsible_sections'),
			'type' => 'select',
			'class' => 'medium',
			'choices' => array(
				array(
					'value' => 'cover',
					'label' => esc_html__('Cover', 'gf_collapsible_sections')
				),
				array(
					'value' => 'contain',
					'label' => esc_html__('Contain', 'gf_collapsible_sections')
				),
				array(
					'value' => 'custom',
					'label' => esc_html__('Custom', 'gf_collapsible_sections')
				),
			)
		);
		if ( !empty($image_size_value) ) {
			$image_size_field['default_value'] = $image_size_value;
		}

		$image_size_custom_value = $this->get_plugin_settings_value("gf_collapsible_sections_global_image_size_custom", $this->_defaultImageSize, $plugin_settings);
		$image_size_custom_field = array(
			'name' => 'gf_collapsible_sections_global_image_size_custom',
			'label' => esc_html__('Custom Image Size (px)', 'gf_collapsible_sections'),
			'default_value' => $image_size_custom_value,
			'type' => 'text',
			'class' => 'medium',
			'dependency' => array(
				'live'   => true,
				'fields' => array(
					array(
						'field' => 'gf_collapsible_sections_global_image_size',
						'values' => array( 'custom' ),
					),
				),
			)
		);


		$image_corners_value = $this->get_plugin_settings_value("gf_collapsible_sections_global_image_corners", "round", $plugin_settings);
		$image_corners_field = array(
			'name' => 'gf_collapsible_sections_global_image_corners',
			'label' => esc_html__('Default Image Corners', 'gf_collapsible_sections'),
			'type' => 'select',
			'class' => 'medium',
			'choices' => array(
				array(
					'value' => 'round',
					'label' => esc_html__('Round', 'gf_collapsible_sections')
				),
				array(
					'value' => 'squared',
					'label' => esc_html__('Squared', 'gf_collapsible_sections')
				),
				array(
					'value' => 'custom',
					'label' => esc_html__('Custom', 'gf_collapsible_sections')
				),
			)
		);
		if ( !empty($image_corners_value) ) {
			$image_corners_field['default_value'] = $image_corners_value;
		}

		$image_corners_custom_value = $this->get_plugin_settings_value("gf_collapsible_sections_global_image_corners_custom", "", $plugin_settings);
		$image_corners_custom_field = array(
			'name' => 'gf_collapsible_sections_global_image_corners_custom',
			'label' => esc_html__('Custom Image Corners (px)', 'gf_collapsible_sections'),
			'default_value' => $image_corners_custom_value,
			'type' => 'text',
			'class' => 'medium',
			'dependency' => array(
				'live'   => true,
				'fields' => array(
					array(
						'field' => 'gf_collapsible_sections_global_image_corners',
						'values' => array( 'custom' ),
					),
				),
			)
		);

		$image_section = array(
			'type' => 'section',
			'dependency' => $new_styles_dependency,
			'title' => esc_html__( 'Images', 'gf_collapsible_sections' ),
			'class' => 'gform-settings-panel--half',
			'fields' => array(
				$image_background_color_field,
				$image_background_color_custom_field,
				$image_size_field,
				$image_size_custom_field,
				$image_corners_field,
				$image_corners_custom_field,
			)
		);



		$legacy_custom_css_global_value = $this->get_plugin_settings_value('gf_collapsible_sections_custom_css_global', '', $plugin_settings);
		$legacy_custom_css_global_field = array(
			'name' => 'gf_collapsible_sections_custom_css_global',
			'tooltip' => $use_new_features ? null : esc_html__('These styles will be loaded for all forms.<br/>Find examples at <a href="https://jetsloth.com/support/gravity-forms-collapsible-sections/">https://jetsloth.com/support/gravity-forms-collapsible-sections/</a>', 'gf_collapsible_sections'),
			'label' => $use_new_features ? esc_html__("Note: This legacy CSS is ignored as it's most likely not needed with the new features. Use the Custom CSS box above for customising any of the new styling.", 'gf_collapsible_sections') : esc_html__('Custom Legacy CSS', 'gf_collapsible_sections'),
			'type' => 'textarea',
			'class' => 'large',
			'default_value' => $legacy_custom_css_global_value
		);

		$legacy_custom_css_section = array(
			'type' => 'section',
			'title' => $use_new_features ? esc_html__('Legacy CSS', 'gf_collapsible_sections') : esc_html__('Enter your own css to style legacy collapsible sections', 'gf_collapsible_sections'),
			'id' => 'gf_collapsible_sections_legacy_custom_css_global_section',
			'collapsible' => true,
			'is_collapsed' => $use_new_features,
			'fields' => array(
				$legacy_custom_css_global_field
			)
		);


		$custom_css_global_value = $this->get_plugin_settings_value('gf_collapsible_sections_user_css_global', '', $plugin_settings);
		$custom_css_global_field = array(
			'name' => 'gf_collapsible_sections_user_css_global',
			'tooltip' => esc_html__('These styles will be loaded for all forms.', 'gf_collapsible_sections'),
			'label' => esc_html__('Enter your own css to style collapsible sections or override any variables', 'gf_collapsible_sections'),
			'type' => 'textarea',
			'class' => 'large',
			'default_value' => $custom_css_global_value
		);

		$custom_css_section = array(
			'type' => 'section',
			'title'  => esc_html__('Custom CSS', 'gf_collapsible_sections'),
			'fields' => array(
				$custom_css_global_field
			)
		);

		return array(
			$license_section,
			$theme_section,
			$image_section,
			$custom_css_section,
            $style_switch_section,
			$legacy_custom_css_section
		);

	}

	/**
	 * Configures the settings which should be rendered on the Form Settings > Collapsible Sections tab.
	 *
	 * @return array
	 */
	public function form_settings_fields( $form ) {

        /*
		$legacy_markup_section = array();
        if ( $this->form_has_collapsible_sections( $form ) ) {

            ob_start();
?>
            <div id="collapsible_sections_settings_legacy_notice" class="jetbase-alert jetbase-alert--warning">
                <div class="jetbase-alert__inner">
                    <div class="jetbase-alert__body">
                        <div class="alert gforms_note_warning">
                            <p><?php _e("Collapsible Sections 1.2+ does not support legacy markup. Please update your forms settings and turn off legacy markup if you would like to continue using Collapsible Sections in this form.", "gf_collapsible_sections"); ?></p>
                            <p><a class="gform-button gform-button--white gform-button--size-xs" href="<?php echo esc_url( admin_url( "admin.php?subview=settings&page=gf_edit_forms&view=settings&id={$form['id']}" ) ); ?>#gform_setting_markupVersion" aria-label="">Form settings</a></p>
                        </div>
                    </div>
                </div>
            </div>
<?php
            $legacy_warning = ob_get_clean();
	        $legacy_markup_section = array(
		        'title' => esc_html__( 'Legacy Form Markup', 'gf_collapsible_sections' ),
		        'fields' => array(
                    array(
                        'name' => 'gf_collapsible_sections_legacy_html_warning',
                        'label' => '',
                        'type' => 'html',
                        'html' => $legacy_warning
                    )
		        )
	        );

        }
        */

		$settings = $this->get_form_settings( $form );
		$plugin_settings = $this->get_plugin_settings();
		$use_new_features = $this->use_new_features();

        if ( $use_new_features ) {

	        // THEME SETTING
	        $display_theme_choices = $this->get_settings_theme_choices();

	        $global_theme_value = $this->get_plugin_settings_value("gf_collapsible_sections_global_theme", $this->_defaultTheme, $plugin_settings);
	        $form_theme_value = $this->get_form_settings_value("gf_collapsible_sections_theme", "global_setting", $form, $settings);
	        $form_theme_field = array(
		        'name' => 'gf_collapsible_sections_theme',
		        'label' => esc_html__( 'Theme', 'gf_collapsible_sections' ),
		        'type' => 'select',
		        'class' => 'medium',
		        'choices' => array_merge(
			        array(
				        array(
					        'value' => 'global_setting',
					        'label' => sprintf( esc_html__("Use Global Setting (%s)", 'gf_collapsible_sections'), $display_theme_choices[$global_theme_value] )
				        )
			        ),
			        $this->get_settings_select_choices_array( $display_theme_choices )
		        )
	        );
	        if ( !empty($form_theme_value) ) {
		        //$form_theme_field['default_value'] = $form_theme_value;
	        }

	        // THEME PREVIEW
	        $form_theme_preview_field = array(
		        'name' => 'gf_collapsible_sections_theme_preview',
		        'label' => '',
		        'type' => 'html',
		        'class' => 'medium',
		        'html' => $this->get_settings_theme_preview_html(),
	        );

	        $theme_section = array(
		        'title' => esc_html__( 'Theme', 'gf_collapsible_sections' ),
		        'class' => 'gform-settings-panel--half',
		        'fields' => array(
			        $form_theme_field,
			        $form_theme_preview_field,
		        )
	        );


	        // BACKGROUND COLOR
	        $global_image_background_color_value = $this->get_plugin_settings_value("gf_collapsible_sections_global_image_background_color", "none", $plugin_settings);// none or custom
	        $global_image_background_color_custom_value = $this->get_plugin_settings_value("gf_collapsible_sections_global_image_background_color_custom", $this->_defaultImageBackgroundColor, $plugin_settings);

	        $form_image_background_color_value = $this->get_form_settings_value("gf_collapsible_sections_image_background_color", "global_setting", $form, $settings);
	        $form_image_background_color_field = array(
		        'name' => 'gf_collapsible_sections_image_background_color',
		        'label' => esc_html__('Image Background Color', 'gf_collapsible_sections'),
                'default_value' => $form_image_background_color_value,
		        'type' => 'select',
		        'class' => 'medium',
		        'choices' => array(
			        array(
				        'value' => 'global_setting',
				        'label' => ( $global_image_background_color_value == "custom" && !empty($global_image_background_color_custom_value) ) ? sprintf( esc_html__('Use Global Setting (%s)', 'gf_collapsible_sections'), $global_image_background_color_custom_value ) : sprintf( esc_html__('Use Global Setting (%s)', 'gf_collapsible_sections'), ucwords($global_image_background_color_value) )
			        ),
			        array(
				        'value' => 'custom',
				        'label' => esc_html__('Custom', 'gf_collapsible_sections')
			        ),
		        )
	        );

	        // BACKGROUND COLOR - CUSTOM
	        $form_image_background_color_custom_value = $this->get_form_settings_value("gf_collapsible_sections_image_background_color_custom", "", $form, $settings);
	        $form_image_background_color_custom_field = array(
		        'name' => 'gf_collapsible_sections_image_background_color_custom',
		        'label' => esc_html__('Custom Image Background Color', 'gf_collapsible_sections'),
		        'default_value' => $form_image_background_color_custom_value,
		        'type' => 'text',
		        'class' => 'medium',
		        'dependency' => array(
			        'live'   => true,
			        'fields' => array(
				        array(
					        'field' => 'gf_collapsible_sections_image_background_color',
					        'values' => array( 'custom' ),
				        ),
			        ),
		        )
	        );

	        $global_image_size_value = $this->get_plugin_settings_value("gf_collapsible_sections_global_image_size", "cover", $plugin_settings);// none or custom
	        $global_image_size_custom_value = $this->get_plugin_settings_value("gf_collapsible_sections_global_image_size_custom", $this->_defaultImageSize, $plugin_settings);

	        $form_image_size_value = $this->get_form_settings_value("gf_collapsible_sections_image_size", "global_setting", $form, $settings);
	        $form_image_size_field = array(
		        'name' => 'gf_collapsible_sections_image_size',
		        'label' => esc_html__('Image Size', 'gf_collapsible_sections'),
		        'type' => 'select',
		        'class' => 'medium',
		        'default_value' => $form_image_size_value,
		        'choices' => array(
			        array(
				        'value' => 'global_setting',
				        'label' => ( $global_image_size_value == "custom" && !empty($global_image_size_custom_value) ) ? sprintf( esc_html__('Use Global Setting (%s)', 'gf_collapsible_sections'), $global_image_size_custom_value ) : sprintf( esc_html__('Use Global Setting (%s)', 'gf_collapsible_sections'), ucwords($global_image_size_value) )
			        ),
			        array(
				        'value' => 'cover',
				        'label' => esc_html__('Cover', 'gf_collapsible_sections')
			        ),
			        array(
				        'value' => 'contain',
				        'label' => esc_html__('Contain', 'gf_collapsible_sections')
			        ),
			        array(
				        'value' => 'custom',
				        'label' => esc_html__('Custom', 'gf_collapsible_sections')
			        ),
		        )
	        );

	        $form_image_size_custom_value = $this->get_form_settings_value("gf_collapsible_sections_image_size_custom", $this->_defaultImageSize, $form, $settings);
	        $form_image_size_custom_field = array(
		        'name' => 'gf_collapsible_sections_image_size_custom',
		        'label' => esc_html__('Custom Image Size (px)', 'gf_collapsible_sections'),
		        'default_value' => $form_image_size_custom_value,
		        'type' => 'text',
		        'class' => 'medium',
		        'dependency' => array(
			        'live'   => true,
			        'fields' => array(
				        array(
					        'field' => 'gf_collapsible_sections_image_size',
					        'values' => array( 'custom' ),
				        ),
			        ),
		        )
	        );


	        $global_image_corners_value = $this->get_plugin_settings_value("gf_collapsible_sections_global_image_corners", "round", $plugin_settings);// none or custom
	        $global_image_corners_custom_value = $this->get_plugin_settings_value("gf_collapsible_sections_global_image_corners_custom", "", $plugin_settings);

            $form_image_corners_value = $this->get_form_settings_value("gf_collapsible_sections_image_corners", "global_setting", $form, $settings);
	        $form_image_corners_field = array(
		        'name' => 'gf_collapsible_sections_image_corners',
		        'label' => esc_html__('Image Corners', 'gf_collapsible_sections'),
                'default_value' => $form_image_corners_value,
		        'type' => 'select',
		        'class' => 'medium',
		        'choices' => array(
			        array(
				        'value' => 'global_setting',
				        'label' => ( $global_image_corners_value == "custom" && !empty($global_image_corners_custom_value) ) ? sprintf( esc_html__('Use Global Setting (%s)', 'gf_collapsible_sections'), $global_image_corners_custom_value ) : sprintf( esc_html__('Use Global Setting (%s)', 'gf_collapsible_sections'), ucwords($global_image_corners_value) )
			        ),
			        array(
				        'value' => 'round',
				        'label' => esc_html__('Round', 'gf_collapsible_sections')
			        ),
			        array(
				        'value' => 'squared',
				        'label' => esc_html__('Squared', 'gf_collapsible_sections')
			        ),
			        array(
				        'value' => 'custom',
				        'label' => esc_html__('Custom', 'gf_collapsible_sections')
			        ),
		        )
	        );

	        $form_image_corners_custom_value = $this->get_form_settings_value("gf_collapsible_sections_image_corners_custom", "", $form, $settings);
	        $form_image_corners_custom_field = array(
		        'name' => 'gf_collapsible_sections_image_corners_custom',
		        'label' => esc_html__('Custom Image Corners (px)', 'gf_collapsible_sections'),
		        'default_value' => $form_image_corners_custom_value,
		        'type' => 'text',
		        'class' => 'medium',
		        'dependency' => array(
			        'live'   => true,
			        'fields' => array(
				        array(
					        'field' => 'gf_collapsible_sections_image_corners',
					        'values' => array( 'custom' ),
				        ),
			        ),
		        )
	        );


	        $image_section = array(
		        'title' => esc_html__( 'Images', 'gf_collapsible_sections' ),
		        'class' => 'gform-settings-panel--half',
		        'fields' => array(
			        $form_image_background_color_field,
			        $form_image_background_color_custom_field,
			        $form_image_size_field,
			        $form_image_size_custom_field,
			        $form_image_corners_field,
			        $form_image_corners_custom_field,
		        )
	        );

        }

		$form_collapsible_description_placement_value = $this->get_form_settings_value("gf_collapsible_sections_description_placement", "", $form, $settings);
		$form_collapsible_description_placement_field = array(
			'name' => 'gf_collapsible_sections_description_placement',
			//'tooltip' => esc_html__('The selected collapsible section will be opened by default when the form loads.', 'gf_collapsible_sections'),
			'label' => esc_html__('Description placement', 'gf_collapsible_sections'),
			'type' => 'select',
			'class' => 'medium',
			'default_value' => 'title',
			'choices' => array(
				array(
					'value' => 'inside',
					'label' => esc_html__('Inside section', 'gf_collapsible_sections')
				),
				array(
					'value' => 'title',
					'label' => esc_html__('Below title', 'gf_collapsible_sections')
				)
			)
		);
		if (!empty($form_collapsible_description_placement_value)) {
			$form_collapsible_description_placement_field['default_value'] = $form_collapsible_description_placement_value;
		}

		$form_collapsible_footer_placement_value = $this->get_form_settings_value("gf_collapsible_sections_footer_placement", '', $form, $settings);
		$form_collapsible_footer_placement_field = array(
			'name' => 'gf_collapsible_sections_footer_placement',
			//'tooltip' => esc_html__('The selected collapsible section will be opened by default when the form loads.', 'gf_collapsible_sections'),
			'label' => esc_html__('Form footer placement', 'gf_collapsible_sections'),
			'type' => 'select',
			'class' => 'medium',
			'default_value' => 'after_last',
			'choices' => array(
				array(
					'value' => 'after_last',
					'label' => esc_html__('After last section', 'gf_collapsible_sections')
				),
				array(
					'value' => 'inside_last',
					'label' => esc_html__('Inside last section', 'gf_collapsible_sections')
				)
			)
		);
		if (!empty($form_collapsible_footer_placement_value)) {
			$form_collapsible_footer_placement_field['default_value'] = $form_collapsible_footer_placement_value;
		}

		$display_section = array(
			'title' => esc_html__( 'Display', 'gf_collapsible_sections' ),
			'class' => $use_new_features ? 'gform-settings-panel--half' : '',
			'fields' => array(
				$form_collapsible_description_placement_field,
				$form_collapsible_footer_placement_field,
			)
		);

		$form_disable_interaction_value = $this->get_form_settings_value("gf_collapsible_sections_disable_interaction", 0, $form, $settings);
		$form_disable_interaction_field = array(
			'name' => 'gf_collapsible_sections_disable_interaction',
			'label' => esc_html__('Disable interactions (visual-only mode)', 'gf_collapsible_sections'),
			'tooltip' => esc_html__('This setting puts Collapsible Sections into a visual-only mode to group fields into sections without any accordion expand/collapse interaction. All sections will always be in open state.', 'gf_collapsible_sections'),
			'type' => 'toggle',
		);

		$form_force_single_opens_value = $this->get_form_settings_value("gf_collapsible_sections_force_single_opens", 0, $form, $settings);
		$form_force_single_opens_field = array(
			'name' => 'gf_collapsible_sections_force_single_opens',
			'label' => esc_html__('Force only one collapsible section to be open at a time?', 'gf_collapsible_sections'),
			'tooltip' => esc_html__('If checked, when a user opens a collapsible section, the other collapsible sections will automatically close.', 'gf_collapsible_sections'),
			'type' => 'toggle',
			'dependency' => array(
				'live' => true,
				'fields' => array(
					array(
						'field' => 'gf_collapsible_sections_disable_interaction',
						'values' => array( '0', false, 'false', 0, '', null ),
					),
				),
			)
		);
        /*
		if ( !empty($form_force_single_opens_value) ) {
			$form_force_single_opens_field['choices'][0]['default_value'] = 1;
		}
        */

		$collapsible_sections_select_choices = array(array(
			'value' => '',
			'label' => 'None (all sections closed by default)'
		));
		$collapsible_sections_checkbox_choices = array();
		$collapsible_sections_checkbox_choices_lookup = array();
		$i = 0;
		foreach($form['fields'] as $field) {
            if ( $field->type != 'section' || !property_exists($field, 'collapsibleSections_enableCollapsible') ) {
                continue;
            }
			if ( $field->collapsibleSections_enableCollapsible == 'start' || ($field->collapsibleSections_enableCollapsible == 'end' && $field->collapsibleSections_endSectionDisplay != 'hidden') ) {
				array_push($collapsible_sections_select_choices, array(
					'value' => $field->id,
					'label' => $field->label
				));
				$checkbox_name = 'gf_collapsible_section_open_by_default_'.$field->id;
				array_push($collapsible_sections_checkbox_choices, array(
					'name' => $checkbox_name,
					'label' => $field->label
				));
				$collapsible_sections_checkbox_choices_lookup[$checkbox_name] = $i;
				$i++;
			}
		}

		$form_sections_single_open_by_default_value = $this->get_form_settings_value("gf_collapsible_sections_single_open_by_default", array(), $form, $settings);
		$form_sections_single_open_by_default_field = array(
			'name' => 'gf_collapsible_sections_single_open_by_default',
			'tooltip' => esc_html__('The selected collapsible section will be opened by default when the form loads.', 'gf_collapsible_sections'),
			'label' => esc_html__('Open by default', 'gf_collapsible_sections'),
			'type' => 'select',
			'class' => 'large',
			'choices' => $collapsible_sections_select_choices,
            'dependency' => array(
                'live' => true,
	            'operator' => 'ALL',
	            'fields' => array(
		            array(
			            'field' => 'gf_collapsible_sections_disable_interaction',
			            'values' => array( '0', false, 'false', 0, '', null ),
		            ),
		            array(
			            'field' => 'gf_collapsible_sections_force_single_opens',
			            'values' => array( '1', true, 'true' ),
		            ),
	            ),
            )
		);
		if (!empty($form_sections_single_open_by_default_value)) {
			$form_sections_single_open_by_default_field['default_value'] = $form_sections_single_open_by_default_value;
		}

		$form_sections_open_by_default_value = $this->get_form_settings_value("gf_collapsible_sections_open_by_default", array(), $form, $settings);
		$form_sections_open_by_default_field = array(
			'name' => 'gf_collapsible_sections_open_by_default',
			'tooltip' => esc_html__('The selected collapsible sections will be opened by default when the form loads.', 'gf_collapsible_sections'),
			'label' => esc_html__('Open by default', 'gf_collapsible_sections'),
			'type' => 'checkbox',
			'choices' => $collapsible_sections_checkbox_choices,
			'dependency' => array(
				'live' => true,
				'operator' => 'ALL',
				'fields' => array(
					array(
						'field' => 'gf_collapsible_sections_disable_interaction',
						'values' => array( '0', false, 'false', 0, '', null ),
					),
					array(
						'field' => 'gf_collapsible_sections_force_single_opens',
						'values' => array( false, '0', null, 'false', '' ),
					),
				),
			)
		);
		if (!empty($form_sections_open_by_default_value)) {
			foreach($form_sections_open_by_default_value as $value) {
				$choice_index = (isset($collapsible_sections_checkbox_choices_lookup[$value])) ? $collapsible_sections_checkbox_choices_lookup[$value] : -1;
				if ($choice_index != -1) {
					$form_sections_open_by_default_field['choices'][$choice_index]['default_value'] = 1;
				}
			}
		}

		$controls_section = array(
			'title' => esc_html__( 'Behavior', 'gf_collapsible_sections' ),
			'class' => 'gform-settings-panel--half',
			'fields' => array(
				$form_disable_interaction_field,
				$form_force_single_opens_field,
				$form_sections_single_open_by_default_field,
				$form_sections_open_by_default_field,
			)
		);


		$form_scroll_to_value = $this->get_form_settings_value("gf_collapsible_sections_scroll_to", 0, $form, $settings);
		$form_scroll_to_field = array(
			'name' => 'gf_collapsible_sections_scroll_to',
			'label' => esc_html__('Automatically scroll to top of opened sections?', 'gf_collapsible_sections'),
			'tooltip' => esc_html__('If checked, when a user clicks to open a collapsible section, the page will scroll to the top of that section.', 'gf_collapsible_sections'),
			'type' => 'toggle',
			'dependency' => array(
				'live' => true,
				'fields' => array(
					array(
						'field' => 'gf_collapsible_sections_disable_interaction',
						'values' => array( '0', false, 'false', 0, '', null ),
					),
				),
			)
		);
		//$form_scroll_to_field['choices'][0]['default_value'] = $form_scroll_to_value;

		$form_scroll_to_offset_value = $this->get_form_settings_value("gf_collapsible_sections_scroll_to_offset", '', $form, $settings);
		$form_scroll_to_offset_field = array(
			'name' => 'gf_collapsible_sections_scroll_to_offset',
			'tooltip' => esc_html__('If you want to offset the scroll top (eg if you have a fixed header) set the offset amount here.', 'gf_collapsible_sections'),
			'label' => esc_html__('Scroll top offset (px)', 'gf_collapsible_sections'),
			'type' => 'text',
			'class' => 'small',
            'dependency' => array(
                'live' => true,
	            'operator' => 'ALL',
                'fields' => array(
	                array(
		                'field' => 'gf_collapsible_sections_disable_interaction',
		                'values' => array( '0', false, 'false', 0, '', null ),
	                ),
	                array(
		                'field' => 'gf_collapsible_sections_scroll_to',
		                'values' => array( true, '1', 'true' ),
	                ),
                )
            )
		);
		if (!empty($form_scroll_to_offset_value)) {
			$form_scroll_to_offset_field['default_value'] = $form_scroll_to_offset_value;
		}

		$form_scroll_to_duration_value = $this->get_form_settings_value("gf_collapsible_sections_scroll_to_duration", 400, $form, $settings);
		$form_scroll_to_duration_field = array(
			'name' => 'gf_collapsible_sections_scroll_to_duration',
			'label' => esc_html__('Scroll duration (milliseconds)', 'gf_collapsible_sections'),
			'type' => 'text',
			'default_value' => 0,
			'class' => 'small',
			'dependency' => array(
				'live' => true,
				'fields' => array(
					array(
						'field' => 'gf_collapsible_sections_scroll_to',
						'values' => array( true, '1', 'true' ),
					),
				)
			)
		);
		if (!empty($form_scroll_to_duration_value)) {
			$form_scroll_to_duration_field['default_value'] = $form_scroll_to_duration_value;
		}

		$scroll_to_section = array(
			'title' => esc_html__( 'Scroll To', 'gf_collapsible_sections' ),
			//'class' => 'gform-settings-panel--half',
			'fields' => array(
				$form_scroll_to_field,
				$form_scroll_to_offset_field,
				$form_scroll_to_duration_field,
			)
		);


		// IGNORE GLOBAL CSS
        $form_ignore_global_css_field = array(
            'name' => 'gf_collapsible_sections_ignore_global_css',
            'type' => 'toggle',
            'label' => __( 'Ignore Global Custom CSS for this form?', 'gf_collapsible_sections' ),
            'tooltip' => __('If checked, the custom css entered in the global settings won\'t be loaded for this form. <br/><br/>IMPORTANT NOTE: multiple forms on a single page with conflicting settings to ignore global CSS may not work as expected.', 'gf_collapsible_sections'),
        );


		if ( $use_new_features ) {

			// CUSTOM CSS
			$form_custom_css_value = $this->get_plugin_settings_value('gf_collapsible_sections_user_css_form', '', $plugin_settings);
			$form_custom_css_field = array(
				'name' => 'gf_collapsible_sections_user_css_form',
				'tooltip' => esc_html__('These styles will be loaded for this form only.', 'gf_collapsible_sections'),
				'label' => esc_html__('Enter your own css to style collapsible sections or override any variables', 'gf_collapsible_sections'),
				'type' => 'textarea',
				'class' => 'large',
				'default_value' => $form_custom_css_value
			);
			$form_custom_css_section = array(
				'type' => 'section',
				'title'  => esc_html__('Custom CSS', 'gf_collapsible_sections'),
				'fields' => array(
					$form_custom_css_field,
					$form_ignore_global_css_field
				)
			);

		}

		$legacy_form_custom_css_value = $this->get_form_settings_value("gf_collapsible_sections_custom_css", "", $form, $settings);
		$legacy_form_custom_css_field = array(
			'name' => 'gf_collapsible_sections_custom_css',
			'tooltip' => $use_new_features ? null : esc_html__('These styles will be loaded for this form only.<br/>Find examples at <a href="https://jetsloth.com/support/gravity-forms-collapsible-sections/">https://jetsloth.com/support/gravity-forms-collapsible-sections/</a>', 'gf_collapsible_sections'),
			'label' => $use_new_features ? esc_html__("Note: With new styles enabled, legacy CSS is ignored as it's most likely not needed. Use the Custom CSS box above for any new styling.", 'gf_collapsible_sections') : esc_html__('Custom CSS', 'gf_collapsible_sections'),
			'type' => 'textarea',
			'class' => 'large',
			'default_value' => $legacy_form_custom_css_value
		);
		$legacy_form_custom_css_section = array(
			'type' => 'section',
			'title'  => $use_new_features ? esc_html__('Legacy CSS', 'gf_collapsible_sections') : esc_html__('Enter your own css to style collapsible sections in this form', 'gf_collapsible_sections'),
			'id' => 'gf_collapsible_sections_legacy_custom_form_css_section',
			'collapsible' => true,
			'is_collapsed' => true,
			'fields' => array(
				$legacy_form_custom_css_field
			)
		);


        if ( $use_new_features ) {
	        return array(
		        $theme_section,
		        $image_section,
		        $display_section,
		        $controls_section,
		        $scroll_to_section,
		        $form_custom_css_section,
		        $legacy_form_custom_css_section,
	        );
        }
        else {
	        return array(
		        $display_section,
		        $controls_section,
		        $scroll_to_section,
		        $legacy_form_custom_css_section,
	        );
        }

	}


	public function custom_field_container( $field_container, $field, $form, $css_class, $style, $field_content ) {

	    if ( GFCommon::is_form_editor() || $field->type != "section" ) {
	        return $field_container;
        }

        $collapsible_section_type = $this->get_field_settings_value("enableCollapsible", "none", $field);
        $collapsible_hidden_section = ( $collapsible_section_type == "end" && $this->get_field_settings_value("endSectionDisplay", "default", $field) == "hidden" );

	    if ( $collapsible_section_type != "none" && !$collapsible_hidden_section ) {

            $form_id = $form['id'];
		    $settings = $this->get_form_settings( $form );
		    $expanded = "false";
		    $collapsible_id = 'collapsible-section_' . $form_id . '_' . $field->id;
		    $field_id = 'field_' . $form_id . '_' . $field->id;

		    $disable_interaction_setting = (bool) $this->get_form_settings_value("gf_collapsible_sections_disable_interaction", 0, $form_id, $settings);
		    $disable_interaction = apply_filters('collapsible_sections_disable_interaction', $disable_interaction_setting, $form_id);
            if ( !empty($disable_interaction) ) {
	            $expanded = "true";
            }
            else {

	            $force_single_opens_setting = (bool) $this->get_form_settings_value("gf_collapsible_sections_force_single_opens", 0, $form_id, $settings);
	            $force_single_opens = apply_filters('collapsible_sections_force_single_opens', $force_single_opens_setting, $form_id);
	            if ( !empty($force_single_opens) ) {
		            // single opened by default only
		            $section_open_by_default_setting = (int) $this->get_form_settings_value("gf_collapsible_sections_single_open_by_default", 0, $form_id, $settings);
		            $section_open_by_default = apply_filters( 'collapsible_sections_single_open_by_default', $section_open_by_default_setting, $form_id );
		            if ( !empty($section_open_by_default) && $section_open_by_default == $field->id ) {
			            $expanded = "true";
		            }
	            }
	            else {
		            // multiples can be opened by default, see if this field is in the list
		            $sections_open_by_default_setting = [];
		            if ( !empty($settings) ) {
			            foreach($settings as $setting_key => $setting_value) {
				            if (strpos($setting_key, 'gf_collapsible_section_open_by_default_') !== FALSE) {
					            $setting_field_id = (int) str_replace('gf_collapsible_section_open_by_default_', '', $setting_key);
					            if ( !empty($setting_value) ) {
						            $sections_open_by_default_setting[] = $setting_field_id;
					            }
				            }
			            }
		            }
		            $sections_open_by_default = apply_filters( 'collapsible_sections_multiple_open_by_default', $sections_open_by_default_setting, $form['id'] );
		            if ( in_array($field->id, $sections_open_by_default) ) {
			            $expanded = "true";
		            }
	            }

            }

		    $collapsible_style = ( $expanded == "true" ) ? "" : "display:none;";
		    $element = ( !isset( $form['markupVersion'] ) || $form['markupVersion'] < 2 ) ? "li" : "div";

		    $use_tabindex = apply_filters('collapsible_sections_use_tabindex', true, $form['id'], $field->id);
		    $tabindex_value = apply_filters('collapsible_sections_tabindex_value', 0, $form['id'], $field->id);
		    $tabindex = $use_tabindex ? 'tabindex="'.$tabindex_value.'"' : '';

            $image_url = $this->get_field_settings_value("imageUrl", "", $field);
            $data_image_url = ( !empty($image_url) ) ? 'data-img' : "";

	        $field_container = str_replace( '<'.$element.' id', '<'.$element.' role="button" '.$tabindex.' aria-pressed="'.$expanded.'" aria-controls="'.$collapsible_id.'" aria-expanded="'.$expanded.'" data-target="#'.$collapsible_id.'" '.$data_image_url.' id', $field_container );
		    $field_container .= '<div id="'.$collapsible_id.'" class="collapsible-sections-collapsible-body gform-theme__no-reset--el" aria-labelledby="'.$field_id.'" style="'.$collapsible_style.'"></div>';

        }

	    return $field_container;
    }

	/**
	 * Add custom class(es) to the section collapsible section fields
	 *
	 * @param string $classes The CSS classes to be filtered, separated by empty spaces.
	 * @param GF_Field $field The field currently being processed.
	 * @param array $form The form currently being processed.
	 *
	 * @return string
	 */
	public function add_custom_field_class( $classes, $field, $form ) {

		$collapsible_section_type = $this->get_field_settings_value("enableCollapsible", "none", $field);

        if ( $field->type != 'section' || $collapsible_section_type == "none" ) {
            return $classes;
        }

        $use_new_features = $this->use_new_features();

		$collapsible_hidden_section = ( $collapsible_section_type == "end" && $this->get_field_settings_value("endSectionDisplay", "default", $field) == "hidden" );
		$settings = $this->get_form_settings( $form );
        $plugin_settings = $this->get_plugin_settings();

		if ( GFCommon::is_form_editor() ) {
			$classes .= ' collapsible-sections-admin-field ';
		}
		else if ( !$collapsible_hidden_section ) {
			$classes .= ' collapsible-sections-field ';
			if ( $collapsible_section_type == 'end' ) {
				$classes .= ' collapsible-sections-end-field ';
			}
		}
        else {
	        $classes .= ' collapsible-sections-end-field collapsible-sections-end-field-hidden ';
        }

		if ( !$collapsible_hidden_section ) {

            if ( $use_new_features ) {
	            $image_url = $this->get_field_settings_value("imageUrl", "", $field);
	            if ( !empty($image_url) ) {
		            $classes .= 'cs--image ';
	            }

            }

			$description_placement = $this->get_field_settings_value("descriptionPlacement", "form_setting", $field);
			if ($description_placement == 'form_setting') {
				$description_placement = (isset($settings['gf_collapsible_sections_description_placement'])) ? $settings['gf_collapsible_sections_description_placement'] : 'title';
			}
			$classes .= 'collapsible-sections-description-'.$description_placement.' ';

			$disable_interaction_setting = (isset($settings['gf_collapsible_sections_disable_interaction']) && $settings['gf_collapsible_sections_disable_interaction'] == 1) ? 1 : 0;;
			$disable_interaction = apply_filters('collapsible_sections_disable_interaction', $disable_interaction_setting, $form['id']);
            if ( !empty($disable_interaction) ) {
	            $classes .= 'collapsible-sections-open ';
            }
            else {

	            $force_single_opens_setting = (isset($settings['gf_collapsible_sections_force_single_opens']) && $settings['gf_collapsible_sections_force_single_opens'] == 1) ? 1 : 0;
	            $force_single_opens = apply_filters('collapsible_sections_force_single_opens', $force_single_opens_setting, $form['id']);
	            if (!empty($force_single_opens)) {
		            // single opened by default only
		            $section_open_by_default_setting = (isset($settings['gf_collapsible_sections_single_open_by_default']) && !empty($settings['gf_collapsible_sections_single_open_by_default']) ) ? $settings['gf_collapsible_sections_single_open_by_default'] : 0;
		            $section_open_by_default = apply_filters( 'collapsible_sections_single_open_by_default', $section_open_by_default_setting, $form['id'] );
		            if (!empty($section_open_by_default) && $section_open_by_default == $field->id) {
			            $classes .= 'collapsible-sections-open ';
		            }
	            }
	            else if (!empty($settings)) {
		            // multiples can be opened by default, see if this field is in the list
		            $sections_open_by_default_setting = [];
		            foreach($settings as $setting_key => $setting_value) {
			            if (strpos($setting_key, 'gf_collapsible_section_open_by_default_') !== FALSE) {
				            $setting_field_id = (int) str_replace('gf_collapsible_section_open_by_default_', '', $setting_key);
				            if ( !empty($setting_value) ) {
					            $sections_open_by_default_setting[] = $setting_field_id;
				            }
			            }
		            }
		            $sections_open_by_default = apply_filters( 'collapsible_sections_multiple_open_by_default', $sections_open_by_default_setting, $form['id'] );
		            if ( in_array($field->id, $sections_open_by_default) ) {
			            $classes .= 'collapsible-sections-open collapsible-sections-open-by-default ';
		            }

	            }

            }

		}

		return $classes;
	}


	/**
	 * Add custom class(es) to the form
	 *
	 * @param array $form The form currently being processed.
	 *
	 * @return array
	 */

	public function add_custom_form_class( $form ) {

        if ( !$this->form_has_collapsible_sections( $form ) ) {
            return $form;
        }

		if ( empty($form['cssClass']) ) {
			$form['cssClass'] = '';
		}

		$form['cssClass'] .= ( empty($form['cssClass']) ) ? 'form-has-collapsible-sections ' : ' form-has-collapsible-sections ';

		$settings = $this->get_form_settings( $form );
        $use_new_features = $this->use_new_features();

        if ( $use_new_features ) {
	        $global_theme_setting = $this->get_plugin_settings_value("gf_collapsible_sections_global_theme", $this->_defaultTheme);
	        $form_theme_setting = $this->get_form_settings_value("gf_collapsible_sections_theme", "global_setting", $form, $settings);
	        $theme_setting = ( $form_theme_setting == "global_setting" ) ? $global_theme_setting : $form_theme_setting;
	        if ( $theme_setting !== "none" ) {
		        $form['cssClass'] .= "cs-theme--{$theme_setting} ";
	        }

	        $global_image_size_setting = $this->get_plugin_settings_value("gf_collapsible_sections_global_image_size", "cover");
	        $form_image_size_setting = $this->get_form_settings_value("gf_collapsible_sections_image_size", "global_setting", $form, $settings);
	        $image_size_setting = ( $form_image_size_setting == "global_setting" ) ? $global_image_size_setting : $form_image_size_setting;
	        $form['cssClass'] .= "cs-image-size--{$image_size_setting} ";

	        $global_image_corners_setting = $this->get_plugin_settings_value("gf_collapsible_sections_global_image_corners", "round");
	        $form_image_corners_setting = $this->get_form_settings_value("gf_collapsible_sections_image_corners", "global_setting", $form, $settings);
	        $image_corners_setting = ( $form_image_corners_setting == "global_setting" ) ? $global_image_corners_setting : $form_image_corners_setting;
	        $form['cssClass'] .= "cs-image-corners--{$image_corners_setting} ";
        }

		$disable_interaction_setting = (isset($settings['gf_collapsible_sections_disable_interaction']) && $settings['gf_collapsible_sections_disable_interaction'] == 1) ? 1 : 0;;
		$disable_interaction = apply_filters('collapsible_sections_disable_interaction', $disable_interaction_setting, $form['id']);
        if ( !empty($disable_interaction) ) {
            if ( strpos($form['cssClass'], "collapsible-sections-disabled") === FALSE ) {
	            $form['cssClass'] .= (empty($form['cssClass'])) ? 'collapsible-sections-disabled ' : ' collapsible-sections-disabled ';
            }
        }
        else {
	        $force_single_opens_setting = (isset($settings['gf_collapsible_sections_force_single_opens']) && $settings['gf_collapsible_sections_force_single_opens'] == 1) ? 1 : 0;
	        $force_single_opens = apply_filters('collapsible_sections_force_single_opens', $force_single_opens_setting, $form['id']);
	        if (!empty($force_single_opens)) {
		        $form['cssClass'] .= (empty($form['cssClass'])) ? 'collapsible-sections-single-opens ' : ' collapsible-sections-single-opens ';
	        }

	        $form_collapsible_scroll_value = (isset($settings['gf_collapsible_sections_scroll_to'])) ? $settings['gf_collapsible_sections_scroll_to'] : 0;
	        if (!empty($form_collapsible_scroll_value)) {
		        $form['cssClass'] .= (empty($form['cssClass'])) ? 'collapsible-sections-scrollto ' : ' collapsible-sections-scrollto ';
	        }
        }

		$form_collapsible_description_placement_value = (isset($settings['gf_collapsible_sections_description_placement'])) ? $settings['gf_collapsible_sections_description_placement'] : '';
		if (!empty($form_collapsible_description_placement_value)) {
			$form['cssClass'] .= (empty($form['cssClass'])) ? 'collapsible-sections-description-'.$form_collapsible_description_placement_value.' ' : ' collapsible-sections-description-'.$form_collapsible_description_placement_value.' ';
		}

		$form_collapsible_footer_placement_value = (isset($settings['gf_collapsible_sections_footer_placement'])) ? $settings['gf_collapsible_sections_footer_placement'] : 'after_last';
		$form['cssClass'] .= (empty($form['cssClass'])) ? 'collapsible-sections-footer-'.$form_collapsible_footer_placement_value.' ' : ' collapsible-sections-footer-'.$form_collapsible_footer_placement_value.' ';

		return $form;
	}

	/**
	 * Add the tooltips for the field.
	 *
	 * @param array $tooltips An associative array of tooltips where the key is the tooltip name and the value is the tooltip.
	 *
	 * @return array
	 */
	public function add_collapsible_sections_field_tooltips( $tooltips ) {
		$tooltips['collapsible_sections_end'] = esc_html__( 'Set your last collapsible section to this for correct grouping and closing styles.', 'gf_collapsible_sections' );
		return $tooltips;
	}


	public function custom_settings_tab( $tabs ) {
		$tabs[] = array(
			'id' => 'collapsible_sections', // Tab id is used later with the action hook that will output content for this specific tab.
			'title' => esc_html__('Collapsible Sections', 'gf_collapsible_sections'),
			'toggle_classes' => '', // Goes into the tab button class attribute.
			'body_classes'   => '', // Goes into the tab content ( ul tag ) class attribute.
		);

		return $tabs;
	}


	public function get_field_setting_fallback_value( $value, $form_value, $global_value ) {
		if ( $value == "form_setting" ) {
			$value = $form_value;
		}
		if ( $value == "global_setting" ) {
			$value = $global_value;
		}
		return $value;
	}

	public function get_field_settings_value( $setting_name, $default_value, $field ) {
		if ( empty($field) || empty($setting_name) ) {
			return null;
		}
		$full_setting_name = "collapsibleSections_{$setting_name}";
		$form_id = $field["formId"];
		$field_id = $field["id"];

		$filter_name = strtolower(preg_replace(
			'/(?<=[a-z])([A-Z]+)/',
			'_$1',
			str_replace("collapsibleSections_", "gfcs_", $full_setting_name)
		));

		$value = ( isset($field[$full_setting_name]) ) ? $field[$full_setting_name] : $default_value;
		$value = apply_filters( $filter_name, $value, $form_id, $field_id, $default_value );// Eg: "collapsibleSections_theme" will do ALL fields across ALL forms
		$value = apply_filters( "{$filter_name}_{$form_id}", $value, $form_id, $field_id, $default_value );// Eg: "collapsibleSections_theme_2" will do ALL fields in form with id 2
		return apply_filters( "{$filter_name}_{$form_id}_{$field_id}", $value, $form_id, $field_id, $default_value );// Eg: "collapsibleSections_theme_2_1" will do field with id 1 in form with id 2
	}

	public function get_form_settings_value( $setting_name, $default_value, $form_or_id, $form_settings = false ) {
		if ( empty($form_or_id) || empty($setting_name) ) {
			return null;
		}
		$form_id = ( is_int( $form_or_id ) ) ? $form_or_id : rgar( $form_or_id, "id" );
		if ( empty($form_settings) ) {
			$form = ( is_int( $form_or_id ) ) ? GFAPI::get_form($form_or_id) : $form_or_id;
			$form_settings = $this->get_form_settings( $form );
		}

		$filter_name = str_replace("gf_collapsible_sections_", "gfcs_", $setting_name);

		$value = ( isset($form_settings[$setting_name]) ) ? $form_settings[$setting_name] : $default_value;
		$value = apply_filters( $filter_name, $value, $form_id, $default_value );// Eg: "gfcs_theme" // will do all forms
		return apply_filters( "{$filter_name}_{$form_id}", $value, $form_id, $default_value );// Eg "gfcs_theme_2" // will do form with id 2
	}

	public function get_plugin_settings_value( $setting_name, $default_value, $plugin_settings = false ) {
		if ( empty($setting_name) ) {
			return null;
		}
		if ( empty($plugin_settings) ) {
			$value = $this->get_plugin_setting( $setting_name );
		}
		else {
			$value = ( isset($plugin_settings[$setting_name]) ) ? $plugin_settings[$setting_name] : null;
		}
		if ( is_null($value) ) {
			$value = $default_value;
		}

		$filter_name = str_replace("gf_collapsible_sections_", "gfcs_", $setting_name);

		return apply_filters( $filter_name, $value, $default_value );// Eg "gfic_global_feature_color_custom" (global)
	}

	protected function get_settings_select_options_html( $choices ) {
		ob_start();
		foreach( $choices as $choice_value => $choice_label ) {
			echo '<option value="'.$choice_value.'">'.$choice_label.'</option>';
		}
		return ob_get_clean();
	}

	protected function get_settings_select_choices_array( $choices ) {
		$options = array();
		foreach( $choices as $choice_value => $choice_label ) {
			$options[] = array(
				'value' => $choice_value,
				'label' => $choice_label,
			);
		}
		return $options;
	}

	public function get_settings_theme_choices() {
		$choices = array(
			'simple' => esc_html__("Simple", 'gf_collapsible_sections'),
			'accordion' => esc_html__("Accordion", 'gf_collapsible_sections'),
			'bordered' => esc_html__("Bordered", 'gf_collapsible_sections'),
			'retro' => esc_html__("Retro", 'gf_collapsible_sections'),
            'none' => esc_html__("None", 'gf_collapsible_sections'),
		);
		return $choices;
	}

	public function get_settings_theme_preview_html() {
		$choices = $this->get_settings_theme_choices();
		ob_start();
		echo '<div id="collapsible-sections-theme-preview" class="collapsible-sections-theme-preview">';
		foreach( $choices as $choice_value => $choice_label ) {
			if ( empty($choice_value) || $choice_value == 'none' ) {
				continue;
			}
			echo '<img src="'.plugins_url('', __FILE__) . '/images/themes/theme-' . $choice_value . '.png?v=' . $this->_version . '" alt="' . $choice_label . '" class="collapsible-sections-theme-preview-' . $choice_value . '" />';
		}
		echo '</div>';
		return ob_get_clean();
	}


	public function custom_settings_markup( $form ) {

		$plugin_settings = $this->get_plugin_settings();
		$form_settings = $this->get_form_settings( $form );
        $use_new_features = $this->use_new_features();

		$description_placement_form_setting = $this->get_form_settings_value( "gf_collapsible_sections_description_placement", "", $form, $form_settings );
		$description_placement_form_setting_label = ( !empty($description_placement_form_setting) && $description_placement_form_setting == "inside" ) ? esc_html__("(Inside section)", 'gf_collapsible_sections') : esc_html__("(Below title)", 'gf_collapsible_sections');
		?>
        <!-- Collapsible Section Toggle -->
        <li class="collapsible-sections-field-setting collapsible-sections-field-toggle-setting field_setting">
            <label class="section_label"><?php esc_html_e("Collapsible Sections", 'gf_collapsible_sections'); ?></label>
            <input type="radio" name="gf_collapsible_sections_toggle" id="collapsible_sections_none_toggle" class="collapsible_sections_toggle" value="none" onchange="collapsibleSectionsAdmin.toggleCollapsibleSections(this.value);"> <label for="collapsible_sections_none_toggle"><?php echo esc_html__("None (normal section)", 'gf_collapsible_sections'); ?></label>
            <input type="radio" name="gf_collapsible_sections_toggle" id="collapsible_sections_start_toggle" class="collapsible_sections_toggle" value="start" onchange="collapsibleSectionsAdmin.toggleCollapsibleSections(this.value);"> <label for="collapsible_sections_start_toggle"><?php echo esc_html__("Collapsible section", 'gf_collapsible_sections'); ?></label>
            <input type="radio" name="gf_collapsible_sections_toggle" id="collapsible_sections_end_toggle" class="collapsible_sections_toggle" value="end" onchange="collapsibleSectionsAdmin.toggleCollapsibleSections(this.value);"> <label for="collapsible_sections_end_toggle"><?php echo esc_html__("Last collapsible section", 'gf_collapsible_sections'); gform_tooltip('collapsible_sections_end'); ?></label>
        </li>
		<?php if ( $use_new_features ): ?>
            <!-- Theme Image -->
            <li class="collapsible-sections-field-setting collapsible-sections-field-image-url-setting field_setting">
                <label for="collapsible-sections-image-url" class="section_label"><?php echo esc_html__("Section Image"); ?></label>
                <div id="collapsible-sections-image-preview"></div>
                <button type="button" id="collapsible-sections-image-button" class="button collapsible-sections-image-button" onclick="collapsibleSectionsAdmin.OpenMediaLibrary(this);" title="<?php echo esc_html__("Open Media Library", 'gf_collapsible_sections'); ?>"><?php echo esc_html__("Select image", 'gf_collapsible_sections'); ?></button>
                <button type="button" id="collapsible-sections-image-remove-button" class="button collapsible-sections-image-remove-button" onclick="collapsibleSectionsAdmin.updateThemeImageUrl('');"><?php echo esc_html__("Remove image", 'gf_collapsible_sections'); ?></button>
            </li>
            <!-- Image Background Color -->
			<?php
			$form_setting_image_background_color = $this->get_form_settings_value( "gf_collapsible_sections_image_background_color", "global_setting", $form, $form_settings );
			$global_setting_image_background_color = $this->get_plugin_settings_value( "gf_collapsible_sections_global_image_background_color", "none", $plugin_settings );
			$form_setting_image_background_color_label = ( $form_setting_image_background_color == "global_setting" ) ? sprintf( esc_html__("Global: %s", 'gf_collapsible_sections'), ucwords($global_setting_image_background_color) ) : ucwords($form_setting_image_background_color);
			?>
            <li class="collapsible-sections-field-setting collapsible-sections-field-image-background-color field_setting">
                <label for="collapsible-sections-image-background-color" class="section_label"><?php esc_html_e("Image Background Color", 'gf_collapsible_sections'); ?></label>
                <select id="collapsible-sections-image-background-color" class="collapsible-sections-image-background-color" onchange="collapsibleSectionsAdmin.updateImageBackgroundColorSetting(this.value);">
                    <option value="form_setting"><?php echo sprintf( esc_html__("Use Form Setting (%s)", 'gf_collapsible_sections'), $form_setting_image_background_color_label ); ?></option>
                    <option value="none"><?php esc_html_e("None", 'gf_collapsible_sections'); ?></option>
                    <option value="custom"><?php esc_html_e("Custom", 'gf_collapsible_sections'); ?></option>
                </select>
                <label for="collapsible-sections-image-background-color-custom"><?php esc_html_e("Custom Image Background Color", 'gf_collapsible_sections'); ?></label>
                <input id="collapsible-sections-image-background-color-custom" class="collapsible-sections-image-background-color-custom" type="text" />
            </li>
        <?php endif; ?>
        <!-- Collapsible Section Description Placement -->
        <li class="collapsible-sections-field-setting collapsible-sections-field-description-setting field_setting">
            <label class="gfield_value_label" for="collapsible-sections-description-placement"><?php echo esc_html__("Description placement", 'gf_collapsible_sections'); ?></label>
            <select id="collapsible-sections-description-placement" class="collapsible-sections-description-placement" onchange="collapsibleSectionsAdmin.updateDescriptionSetting(this.value);">
                <option value="form_setting"><?php echo esc_html__("Use Form Setting", 'gf_collapsible_sections') . " {$description_placement_form_setting_label}"; ?></option>
                <option value="inside"><?php esc_html_e("Inside section", 'gf_collapsible_sections'); ?></option>
                <option value="title"><?php esc_html_e("Below title", 'gf_collapsible_sections'); ?></option>
            </select>
        </li>
        <!-- Collapsible Section End Display -->
        <li class="collapsible-sections-field-setting collapsible-sections-field-end-display-setting field_setting">
            <label class="gfield_value_label" for="collapsible-sections-field-end-display"><?php esc_html_e("Display setting for end section", 'gf_collapsible_sections') ?></label>
            <select id="collapsible-sections-field-end-display" class="collapsible-sections-field-end-display" onchange="collapsibleSectionsAdmin.updateEndSectionDisplaySetting(this.value);">
                <option value="default"><?php esc_html_e("Show (normal section)", 'gf_collapsible_sections'); ?></option>
                <option value="hidden"><?php esc_html_e("Hide (only used to end previous collapsible section)", 'gf_collapsible_sections'); ?></option>
            </select>
        </li>
		<?php
    }


	/**
	 * Add custom messages after plugin row based on license status
	 */

	public function gf_plugin_row($plugin_file='', $plugin_data=array(), $status='') {
		$row = "";
		$license_key = $this->get_license_key();
		$license_status = $this->get_license_status($license_key);
		if ( empty($license_key) || empty($license_status) ) {
			ob_start();
			?>
            <tr class="plugin-update-tr">
                <td colspan="3" class="plugin-update gf_collapsible_sections-plugin-update">
                    <div class="update-message">
                        <a href="<?php echo admin_url('admin.php?page=gf_settings&subview=' . $this->_slug); ?>">Activate</a> your license to receive plugin updates and support. Need a license key? <a href="<?php echo $this->_url; ?>" target="_blank">Purchase one now</a>.
                    </div>
                    <style>
						.plugin-update.gf_collapsible_sections-plugin-update .update-message:before {
							content: "\f348";
							margin-top: 0;
							font-family: dashicons;
							font-size: 20px;
							position: relative;
							top: 5px;
							color: orange;
							margin-right: 8px;
						}
						.plugin-update.gf_collapsible_sections-plugin-update {
							background-color: #fff6e5;
						}
						.plugin-update.gf_collapsible_sections-plugin-update .update-message {
							margin: 0 20px 6px 40px !important;
							line-height: 28px;
						}
                    </style>
                </td>
            </tr>
			<?php
			$row = ob_get_clean();
		}
		else if( !empty($license_key) && $license_status != 'valid' && $license_status != 'unknown' ) {
			ob_start();
			?>
            <tr class="plugin-update-tr">
                <td colspan="3" class="plugin-update gf_collapsible_sections-plugin-update">
                    <div class="update-message">
                        Your license is invalid or expired. <a href="<?php echo admin_url('admin.php?page=gf_settings&subview=' . $this->_slug); ?>">Enter a valid license key</a> or <a href="<?php echo $this->_url; ?>" target="_blank">purchase a new one</a>.
                    </div>
                    <style>
						.plugin-update.gf_collapsible_sections-plugin-update .update-message:before {
							content: "\f348";
							margin-top: 0;
							font-family: dashicons;
							font-size: 20px;
							position: relative;
							top: 5px;
							color: #d54e21;
							margin-right: 8px;
						}
						.plugin-update.gf_collapsible_sections-plugin-update {
							background-color: #fff6e5;
						}
						.plugin-update.gf_collapsible_sections-update .update-message {
							margin: 0 20px 6px 40px !important;
							line-height: 28px;
						}
                    </style>
                </td>
            </tr>
			<?php
			$row = ob_get_clean();
		}

		echo $row;
	}


	public function upgrade( $previous_version ) {
		if ( empty($previous_version) ) {
			return;
		}

		$version_num = floatval( substr($previous_version, 0, 3) );

		if ( $version_num < 1.2 ) {
			// upgraded from < 1.2

			// switch to use legacy styles by default and show message

			$plugin_settings = $this->get_plugin_settings();
			$plugin_settings["gf_collapsible_sections_use_legacy_styles"] = "1";
			$this->update_plugin_settings($plugin_settings);

		}

		if ( ( empty(GFCS_SPLASH_ID) || empty(GFCS_SPLASH_URL) || false !== get_option(GFCS_SPLASH_ID . "_seen") ) ) {
			// splash already shown
			return;
		}

		// set transient to show the splash page
		set_transient(GFCS_SPLASH_ID, '1', MONTH_IN_SECONDS);// expire in 30 days

	}

	public function maybe_show_splash_page() {

		if ( false === get_transient( GFCS_SPLASH_ID ) ) {
			return;
		}

		delete_transient( GFCS_SPLASH_ID );
		// set the option so this splash doesn't get shown on future updates
		update_option( GFCS_SPLASH_ID . "_seen", '1' );
		?>
        <style>
			.jetbase-splash-page-overlay {
				position: fixed;
				left: 0;
				top: 0;
				right: 0;
				bottom: 0;
				background-color: rgba(36, 39, 70, 0.75);
				padding: 30px;
				z-index: 999999;
			}
			.jetbase-splash-page-overlay:not(.active) {
				display: none;
			}
			.jetbase-splash-page-wrap {
				position: relative;
				width: calc(100vw - 60px);
				height: calc(100vh - 60px);
				background-color: white;
				border-radius: 10px;
				overflow: scroll;
			}
			.jetbase-splash-page-close {
				position: absolute;
				right: 40px;
				top: 40px;
				display: block;
				padding: 0;
				margin: 0;
				background: none;
				border: none;
				cursor: pointer;
				z-index: 999998;
			}
			.jetbase-splash-page-close i {
				display: block;
				width: 40px;
				height: 40px;
				border-radius: 50%;
				overflow: hidden;
				position: relative;
			}
			.jetbase-splash-page-close i:before,
			.jetbase-splash-page-close i:after {
				content: "";
				overflow: hidden;
				display: block;
				width: 50%;
				height: 2px;
				border-radius: 2px;
				background-color: #526982;
				position: absolute;
				left: 50%;
				top: 50%;
			}
			.jetbase-splash-page-close i:before {
				transform: translate(-50%, -50%) rotate(-45deg);
			}
			.jetbase-splash-page-close i:after {
				transform: translate(-50%, -50%) rotate(45deg);
			}
			.jetbase-splash-page-close span {
				border: 0 !important;
				clip: rect(0 0 0 0) !important;
				width: 1px !important;
				height: 1px !important;
				margin: -1px !important;
				overflow: hidden !important;
				padding: 0 !important;
				position: absolute !important;
			}
			.jetbase-splash-page {
				width: 100%;
				height: 100%;
			}
        </style>
        <div id="<?php echo GFCS_SPLASH_ID; ?>" class="jetbase-splash-page-overlay">
            <button type="button" class="jetbase-splash-page-close"><i></i><span>Close</span></button>
            <div class="jetbase-splash-page-wrap">
                <iframe class="jetbase-splash-page" src="<?php echo GFCS_SPLASH_URL; ?>"></iframe>
            </div>
        </div>
        <script>
			(() => {
				const __initSplash = () => {
					const overlay = document.querySelector('.jetbase-splash-page-overlay[id^="gfcs_"]');
					if ( !overlay ) {
						return;
					}
					overlay.querySelector('.jetbase-splash-page-close').addEventListener('click', (e) => {
						e.preventDefault();
						const overlay = e.currentTarget.closest('.jetbase-splash-page-overlay');
						overlay.remove();
					});
					overlay.classList.add('active');
				};
				if ( document && document.readyState === "complete" ) {
					__initSplash();
				}
				else {
					document.addEventListener('DOMContentLoaded', (e) => {
						__initSplash();
					});
				}
			})();
        </script>
		<?php
	}

	// # LICENSING -----------------------------------------------------------------------------------------------

	public function init_license() {
		// make sure a license key defined as constant always overrides
		$saved_key = $this->get_license_key();
		$const_key = $this->get_const_license_key();

		if ( !empty($const_key) && $saved_key != $const_key ) {
			// the constant key is different to what's saved in settings
			// force it into settings
			$settings = $this->get_plugin_settings();
			if ( false === $settings ) {
				$settings = array();
			}
			$settings['gf_collapsible_sections_license_key'] = $const_key;
			$this->update_plugin_settings($settings);

			if ( !empty($saved_key) ) {
				// maybe deactivate the old key
				$this->deactivate_license_key($saved_key);
			}
			$key = $const_key;
		}
		else {
			$key = $saved_key;
		}

		// activate the current key if need be
		$this->activate_license_key( $key );
	}

	public function get_const_license_key() {
		return ( defined('GF_COLLAPSIBLE_SECTIONS_LICENSE') ) ? GF_COLLAPSIBLE_SECTIONS_LICENSE : '';
	}

	public function get_license_key() {
		return $this->get_plugin_setting( 'gf_collapsible_sections_license_key' );
	}

	private function activate_license_key( $key, $force = false ) {
		if ( empty($key) ) {
			return;
		}

		$activated = get_option("gf_collapsible_sections_license_activated_{$key}");
		if ( !empty($activated) && $force !== true ) {
			// already activated, skip unless forcing the activation
			return;
		}

		$status = $this->get_license_status($key);
		if ( $status == "valid" ) {
			// already activated for this site, just update our record
			update_option("gf_collapsible_sections_license_activated_{$key}", '1', false);
			return;
		}

		$response = $this->perform_edd_license_request( 'activate_license', $key );
		$status = rgobj($response, 'license', '');
		if ( !empty($status) ) {
			$this->update_license_status($key, $status);
			if ( $status == 'valid' ) {
				update_option("gf_collapsible_sections_license_activated_{$key}", '1', false);
			}
		}
	}

	private function deactivate_license_key( $key, $force = false ) {
		if ( empty($key) ) {
			return;
		}

		// delete data if exists
		delete_transient("gf_collapsible_sections_license_status_{$key}");
		delete_option("gf_collapsible_sections_license_activated_{$key}");

		$status = $this->get_license_status( $key, true );
		if ( $status === "site_inactive" && $force !== true ) {
			// key is already not active for this site, skip unless forcing the deactivation
			return;
		}

		$response = $this->perform_edd_license_request( 'deactivate_license', $key );
	}

	/**
	 * Determine if the license key is valid so the appropriate icon can be displayed next to the field.
	 *
	 * @param string $value The current value of the license_key field.
	 * @param array $field The field properties.
	 *
	 * @return bool|null
	 */
	public function license_feedback( $value, $field = null ) {
		$status = $this->get_license_status( $value );
		return ( is_null($status) ) ? null : ( $status === 'valid' );
	}

	private function update_license_status( $key, $status ) {
		set_transient("gf_collapsible_sections_last_status", $status, MONTH_IN_SECONDS);
		set_transient("gf_collapsible_sections_license_status_{$key}", $status, WEEK_IN_SECONDS);
	}

	/**
	 * Handle license key validation (maybe activate)
	 * Note: This gets called AFTER settings are saved
	 *
	 * @param array $field The field properties.
	 * @param string $field_setting The submitted value of the license_key field.
	 */
	public function license_validation( $field, $field_setting ) {
		$previous_key = get_option("gf_collapsible_sections_prev_key");
		if ( !empty($previous_key) && $previous_key != $field_setting ) {
			// we've had a new key submitted, maybe deactivate the old one
			$this->deactivate_license_key($previous_key);
		}

		$status = $this->get_license_status( $field_setting, true );
		$activated = get_option("gf_collapsible_sections_license_activated_{$field_setting}");
		if ( $status == 'site_inactive' || empty($activated) ) {
			// new key isn't activated for this site yet, activate it
			$this->activate_license_key($field_setting, true);
		}

		// keep track of last saved key
		update_option("gf_collapsible_sections_prev_key", $field_setting, false);
	}

	public function get_license_status( $key, $force = false ) {
		if ( empty($key) ) {
			return null;
		}

		$status = get_transient("gf_collapsible_sections_license_status_{$key}");

		$checked_recently = get_transient("gf_collapsible_sections_license_checked");
		$last_status = get_transient("gf_collapsible_sections_last_status");
		if ( $checked_recently !== false && $last_status !== false && !$force ) {
			$this->log_debug("get_license_status() using last_status::{$last_status}");
			return $last_status;
		}

		if ( $force === true || $status === false ) {

			$license_data = $this->perform_edd_license_request( 'check_license', $key );
			if ( !is_object($license_data) ) {
				set_transient("gf_collapsible_sections_last_status", "unknown", MONTH_IN_SECONDS);
				return "unknown";
			}
			$status = rgobj($license_data, 'license', '');
			if ( empty($status) ) {
				set_transient("gf_collapsible_sections_last_status", "unknown", MONTH_IN_SECONDS);
				return "unknown";
			}
			// if site_inactive, activate?
			$this->update_license_status($key, $status);
		}

		return $status;
	}


	/**
	 * Send a request to the EDD store url.
	 *
	 * @param string $edd_action The action to perform (check_license, activate_license or deactivate_license).
	 * @param string $license The license key.
	 *
	 * @return object
	 */
	public function perform_edd_license_request( $edd_action, $license ) {

		if ( $edd_action == "check_license" ) {
			set_transient("gf_collapsible_sections_license_checked", '1', HOUR_IN_SECONDS);
		}

		// Prepare the request arguments
		$args = array(
			'timeout' => GFCS_TIMEOUT,
			'sslverify' => GFCS_SSL_VERIFY,
			'body' => array(
				'version' => GFCS_VERSION,
				'edd_action' => $edd_action,
				'license' => trim($license),
				'item_name' => urlencode(GFCS_NAME),
				'url' => home_url(),
			)
		);

		// Send the remote request
		$response = wp_remote_post(GFCS_HOME, $args);

		$result = json_decode( wp_remote_retrieve_body( $response ) );

		$this->log_debug("perform_edd_license_request() result:: " . print_r($result, true));

		return $result;
	}

	public function debug_output($data = '', $background='black', $color='white') {
		echo '<pre style="padding:20px; background:'.$background.'; color:'.$color.';">';
			print_r($data);
		echo '</pre>';
	}

} // end class
