<?php

/**
 * Rest API - Master Log Controller.
 * 
 * @since 12.0.0
 */
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly
}

if (!class_exists('FGF_Master_Log_Controller')) {

	/**
	 * Class.
	 * 
	 * @since 12.0.0
	 */
	class FGF_Master_Log_Controller extends FGF_REST_Posts_Controller {

		/**
		 * Post type.
		 *
		 * @since 12.0.0
		 * @var string
		 */
		protected $post_type = FGF_Register_Post_Types::MASTER_LOG_POSTTYPE;

		/**
		 * Route base.
		 *
		 * @since 12.0.0
		 * @var string
		 */
		protected $rest_base = 'master_log';

		/**
		 * Register the Rest API.
		 * 
		 * @since 12.0.0
		 */
		public function register_routes() {
			register_rest_route($this->namespace, '/' . $this->rest_base, array(
				array(
					'methods' => WP_REST_Server::READABLE,
					'callback' => array( $this, 'get_items' ),
					'permission_callback' => array( $this, 'get_items_permissions_check' ),
				),
				array(
					'methods' => WP_REST_Server::CREATABLE,
					'callback' => array( $this, 'create_item' ),
					'permission_callback' => array( $this, 'create_item_permissions_check' ),
					'args' => $this->get_endpoint_args_for_item_schema(WP_REST_Server::CREATABLE),
				),
				'schema' => array( $this, 'get_public_item_schema' ),
				)
			);

			register_rest_route($this->namespace, '/' . $this->rest_base . '/(?P<id>[\d]+)',
					array(
						array(
							'methods' => WP_REST_Server::READABLE,
							'callback' => array( $this, 'get_item' ),
							'permission_callback' => array( $this, 'get_item_permissions_check' ),
						),
						array(
							'methods' => WP_REST_Server::EDITABLE,
							'callback' => array( $this, 'update_item' ),
							'permission_callback' => array( $this, 'update_item_permissions_check' ),
							'args' => $this->get_endpoint_args_for_item_schema(WP_REST_Server::CREATABLE),
						),
						array(
							'methods' => WP_REST_Server::DELETABLE,
							'callback' => array( $this, 'delete_item' ),
							'permission_callback' => array( $this, 'delete_item_permissions_check' ),
							'args' => array(
									'force' => array(
									'default' => false,
									'type' => 'boolean',
									'description' => __('Whether to bypass trash and force deletion.', 'free-gifts-for-woocommerce'),
								),
							),
						),
						'schema' => array( $this, 'get_public_item_schema' ),
					)
			);
		}

		/**
		 * Get object.
		 *
		 * @since 12.0.0
		 * @param int $id Object ID.
		 * @return Object FGF_Master_Log
		 */
		protected function get_object( $id ) {
			return fgf_get_master_log($id);
		}

		/**
		 * Get the master log.
		 *
		 * @since 12.0.0
		 * @param WP_REST_Request $request Full details about the request.
		 * @return WP_Error|WP_REST_Response
		 */
		public function get_items( $request ) {
			$master_log_ids = get_posts(array(
				'post_type' => $this->post_type,
				'post_status' => fgf_get_master_log_statuses(),
				'posts_per_page' => '-1',
				'fields' => 'ids',
				'order' => 'ASC',
			));

			$master_logs = array();
			foreach ($master_log_ids as $master_log_id) {
				$master_log = $this->get_object($master_log_id);
				if (!$master_log->exists()) {
					continue;
				}

				$master_logs[] = $this->prepare_item_for_response($master_log, $request);
			}

			return rest_ensure_response($master_logs);
		}

		/**
		 * Get the master log schema, conforming to JSON Schema.
		 *
		 * @since 12.2.0
		 * @return array
		 */
		public function get_item_schema() {
			$schema= array(
				'$schema' => 'http://json-schema.org/draft-04/schema#',
				'title' => $this->post_type,
				'type' => 'object',
				'properties' => array(
					'id' => array(
						'description' => __('Unique identifier for the object.', 'free-gifts-for-woocommerce'),
						'type' => 'integer',
						'context' => array( 'view' ),
						'readonly' => true,
					),
					'order_id' => array(
						'description' => __('Order ID.', 'free-gifts-for-woocommerce'),
						'type' => 'integer',
						'context' => array( 'view', 'edit' ),
					),
					'user_id' => array(
						'description' => __('User ID.', 'free-gifts-for-woocommerce'),
						'type' => 'integer',
						'context' => array( 'view', 'edit' ),
					),
					'status' => array(
						'description' => __('Status.', 'free-gifts-for-woocommerce'),
						'type' => 'string',
						'context' => array( 'view', 'edit' ),
					),
					'date_created' => array(
						'description' => __("The date the coupon was created, in the site's timezone.", 'free-gifts-for-woocommerce'),
						'type' => 'date-time',
						'context' => array( 'view' ),
						'readonly' => true,
					),
					'date_modified' => array(
						'description' => __("The date the coupon was last modified, in the site's timezone.", 'free-gifts-for-woocommerce'),
						'type' => 'date-time',
						'context' => array( 'view' ),
						'readonly' => true,
					),
					'rule_ids' => array(
						'description' => __('Rule IDs.', 'free-gifts-for-woocommerce'),
						'type' => 'array',
						'items' => array(
							'type' => 'integer',
						),
						'context' => array( 'view', 'edit' ),
					),
					'product_details' => array(
						'description' => __('Product Details.', 'free-gifts-for-woocommerce'),
						'type' => 'array',
						'items' => array(
							'type' => 'array',
							'properties'=>array(
								'product_id'=>array(
									'description' => __('Product ID.', 'free-gifts-for-woocommerce'),
									'type' => 'integer',
									'context' => array( 'view', 'edit' ),
								),
								'product_name'=>array(
									'description' => __('Product Name.', 'free-gifts-for-woocommerce'),
									'type' => 'string',
									'context' => array( 'view', 'edit' ),
								),
								'product_price'=>array(
									'description' => __('Product Price.', 'free-gifts-for-woocommerce'),
									'type' => 'string',
									'context' => array( 'view', 'edit' ),
									'default'=> 0,
								),
								'quantity'=>array(
									'description' => __('Quantity', 'free-gifts-for-woocommerce'),
									'type' => 'integer',
									'context' => array( 'view', 'edit' ),
									'default'=> 1,
								),
								'rule_id'=>array(
									'description' => __('Rule ID.', 'free-gifts-for-woocommerce'),
									'type' => 'string',
									'context' => array( 'view', 'edit' ),
								),
								'mode'=>array(
									'description' => __('Mode.', 'free-gifts-for-woocommerce'),
									'type' => 'string',
									'context' => array( 'view', 'edit' ),
								),
							),
						),
						'context' => array( 'view', 'edit' ),
					),
					'user_name' => array(
						'description' => __('User Name.', 'free-gifts-for-woocommerce'),
						'type' => 'string',
						'context' => array( 'view', 'edit' ),
					),
					'user_email' => array(
						'description' => __('User Email.', 'free-gifts-for-woocommerce'),
						'type' => 'string',
						'format' => 'email',
						'context' => array( 'view', 'edit' ),
					),
				),
			);

			return $this->add_additional_fields_schema( $schema );
		}

		/**
		 * Get meta keys.
		 * 
		 * @since 12.2.0
		 * @return array
		 */
		public function get_meta_keys() {
			return array(
				'order_id' => 'fgf_order_id',
				'user_name' => 'fgf_user_name',
				'user_email' => 'fgf_user_email',
				'product_details' => 'fgf_product_details',
				'rule_ids' => 'fgf_rule_ids',
			);
		}

		/**
		 * Prepare a single item output for response.
		 *
		 * @since 12.0.0
		 * @param FGF_Master_Log $master_log object.
		 * @param  WP_REST_Request  $request Request object.
		 * @return array $response Response data.
		 */
		public function prepare_item_for_response( $master_log, $request ) {
			$data = array(
				'id' => intval($master_log->get_id()),
				'status' => $master_log->get_status(),
				'order_id' => intval($master_log->get_order_id()),
				'user_id' => intval($master_log->get_user_id()),
				'user_name' => $master_log->get_user_name(),
				'user_email' => $master_log->get_user_email(),
				'product_details' => $master_log->get_product_details(),
				'rule_ids' => $master_log->get_rule_ids(),
				'created_date' => $master_log->get_created_date(),
				'modified_date' => $master_log->get_modified_date(),
			);

			/**
			 * This hook is used to alter the item response.
			 * 
			 * @since 12.0.0
			 * @param WP_REST_Response   $response   The response object.
			 * @param FGF_Master_Log     $master_log Master Log object.
			 * @param WP_REST_Request    $request    Request object.
			 */
			$data = apply_filters('fgf_prepared_master_log_item_response', $data, $master_log, $request);
			$data = $this->filter_response_by_context( $data, 'view' );

			// Wrap the data in a response object.
			$response = rest_ensure_response( $data );

			$response->add_links( $this->prepare_links( $master_log, $request ) );

			/**
			 * Filter the data for a response.
			 *
			 * The dynamic portion of the hook name, $this->post_type, refers to post_type of the post being
			 * prepared for the response.
			 *
			 * @since 12.2.0
			 * @param WP_REST_Response   $response   The response object.
			 * @param FGF_Master_Log     $master_log Master Log object.
			 * @param WP_REST_Request    $request    Request object.
			 */
			return apply_filters( "woocommerce_rest_prepare_{$this->post_type}", $response, $master_log, $request );
		}

		/**
		 * Prepare links for the request.
		 *
		 * @since 12.2.0
		 * @param FGF_Master_Log $master_log Master Log object.
		 * @param WP_REST_Request $request Request object.
		 * @return array Links for the given order.
		 */
		protected function prepare_links( $master_log, $request ) {
			$links = array(
			'self' => array(
				'href' => rest_url( sprintf( '/%s/%s/%d', $this->namespace, $this->rest_base, $master_log->get_id() ) ),
			),
			'collection' => array(
				'href' => rest_url( sprintf( '/%s/%s', $this->namespace, $this->rest_base ) ),
			),
			);
			if ( 0 !== (int) $master_log->get_user_id() ) {
				$links['customer'] = array(
				'href' => rest_url( sprintf( '/%s/customers/%d', 'wc/v3', $master_log->get_user_id() ) ),
				);
			}
			if ( 0 !== (int) $master_log->get_order_id() ) {
				$links['order'] = array(
				'href' => rest_url( sprintf( '/%s/orders/%d', 'wc/v3', $master_log->get_order_id() ) ),
				);
			}
		
			return $links;
		}
	}

}
