<?php
if ( ! defined( 'ABSPATH' ) ) {
	die();
}

/**
 * API methods class
 *
 * @since  1.2
 * @access public
 */
class Forminator_API {

	public static function initialize() {
		// If models are not initialized, init the plugin
		if ( ! class_exists( 'Forminator_Custom_Form_Model' ) ) {
			$forminator = forminator();
		}
	}

	/**
	 * Returns all form objects.
	 *
	 * @since      1.2.0
	 * @access     public
	 *
	 * @param null|... $form_ids
	 *
	 * @return array
	 */
	public static function get_forms( $form_ids = null ) {
		// Initialize API
		self::initialize();

		$temp = array();

		if ( is_null( $form_ids ) ) {
			$temp = Forminator_Custom_Form_Model::model()->get_all();
		} else {
			if ( ! is_array( $form_ids ) ) {
				$form_ids = func_get_args();
				$form_ids = array_map( 'trim', $form_ids );
			}

			foreach ( $form_ids as $form_id ) {
				$temp[] = self::get_form( $form_id );
			}
		}

		return $temp;
	}

	/**
	 * Returns form object by given ID.
	 *
	 * @since  1.2
	 * @access public
	 *
	 * @param  int $form_id ID of the form
	 *
	 * @return Forminator_Custom_Form_Model|WP_Error Custom Form Model on success or WP_Error otherwise
	 */
	public static function get_form( $form_id ) {
		// Initialize API
		self::initialize();

		if ( empty( $form_id ) ) {
			return new WP_Error( 'missing_id', __( 'Form ID is required!', Forminator::DOMAIN ) );
		}

		$model = Forminator_Custom_Form_Model::model()->load( $form_id );
		if ( ! $model instanceof Forminator_Custom_Form_Model ) {
			return new WP_Error( 'custom_form_not_found', __( 'Form Not Found!', Forminator::DOMAIN ) );
		}

		return $model;
	}

	/**
	 * Delete form with given ID
	 *
	 * @since  1.2
	 * @access public
	 *
	 * @param int $form_id ID of the form
	 *
	 * @return bool|WP_Error True when successfully deleted | WP Error
	 */
	public static function delete_form( $form_id ) {
		// Initialize API
		self::initialize();

		$model = Forminator_Custom_Form_Model::model()->load( $form_id );
		if ( is_object( $model ) ) {
			// Delete form object
			wp_delete_post( $form_id );

			// Delete form entries
			Forminator_Form_Entry_Model::delete_by_form( $form_id );

			// Delete form views
			$form_view = Forminator_Form_Views_Model::get_instance();
			$form_view->delete_by_form( $form_id );

			return true;
		} else {
			return new WP_Error( 'not_found', sprintf( __( 'Can not find a form with id: %s', Forminator::DOMAIN ), $form_id ), $form_id );
		}
	}

	/**
	 * Delete forms with given IDs
	 *
	 * @since        1.2
	 * @access       public
	 *
	 * @param  array|... $form_ids array of Form IDs
	 *
	 * @return bool|WP_Error True when successfully deleted | WP Error
	 */
	public static function delete_forms( $form_ids ) {
		// Initialize API
		self::initialize();

		if ( ! is_array( $form_ids ) ) {
			$form_ids = func_get_args();
		}

		$form_ids = array_map( 'trim', $form_ids );

		if ( is_array( $form_ids ) && ! empty( $form_ids ) ) {
			foreach ( $form_ids as $id ) {
				self::delete_form( $id );
			}

			return true;
		} else {
			return new WP_Error( 'invalid', __( 'Invalid or empty array with IDs', Forminator::DOMAIN ) );
		}
	}

	/**
	 * Add form
	 *
	 * @since  1.2
	 * @access public
	 *
	 * @param string $name     Form name
	 * @param array  $wrappers Array with form fields
	 * @param array  $settings Array with form settings
	 *
	 * @return int|WP_Error ID of new form, or WP_Error on failure
	 */
	public static function add_form( $name, $wrappers = array(), $settings = array() ) {
		// Initialize API
		self::initialize();

		if ( empty( $name ) ) {
			return new WP_Error( 'missing_name', __( 'Form name is required!', Forminator::DOMAIN ) );
		}


		// Create new form model
		$model = new Forminator_Custom_Form_Model();

		// Set the post data
		$settings['formName'] = $name;
		$model->name          = $name;

		if ( is_array( $settings ) ) {
			// Bind form settings
			$model->settings = $settings;
		}

		// Add fields to the form
		if ( is_array( $wrappers ) ) {
			// Loop wrappers
			foreach ( $wrappers as $wrapper_data ) {
				// Loop fields
				foreach ( $wrapper_data['fields'] as $field_data ) {
					// Create new field model
					$field          = new Forminator_Form_Field_Model();
					$field->form_id = $wrapper_data['wrapper_id'];
					$field->slug    = $field_data['element_id'];

					// Unset field ID
					unset( $field_data['element_id'] );

					// Import field data to model
					$field->import( $field_data );

					// Add field to the form
					$model->add_field( $field );
				}
			}
		}


		// Save the form
		$id = $model->save();

		if ( false === $id ) {
			return new WP_Error( 'form_save_error', __( 'There was a problem saving the form', Forminator::DOMAIN ) );
		}

		return $id;
	}

	/**
	 * Update form
	 *
	 * @since  1.2
	 * @access public
	 *
	 * @param int   $id       Form ID
	 * @param array $wrappers Array with form fields
	 * @param array $settings Array with form settings
	 *
	 * @return int|WP_Error id of updated form, or WP_Error on failure
	 */
	public static function update_form( $id, $wrappers = array(), $settings = array() ) {
		// Initialize API
		self::initialize();

		if ( empty( $id ) ) {
			return new WP_Error( 'missing_id', __( 'Form ID is required!', Forminator::DOMAIN ) );
		}

		// Create new form model
		$model = Forminator_Custom_Form_Model::model()->load( $id );

		if ( ! is_object( $model ) ) {
			return new WP_Error( 'missing_object', __( "Form model doesn't exist", Forminator::DOMAIN ) );
		}


		// Set the post data
		$model->name = $settings['formName'];

		if ( is_array( $settings ) ) {
			// Bind form settings
			$model->settings = $settings;
		}

		// Add fields to the form
		if ( is_array( $wrappers ) ) {
			//we need to empty fields cause we will send new data
			$model->clear_fields();

			// Loop wrappers
			foreach ( $wrappers as $wrapper_data ) {
				// Loop fields
				foreach ( $wrapper_data['fields'] as $field_data ) {
					// Create new field model
					$field          = new Forminator_Form_Field_Model();
					$field->form_id = $wrapper_data['wrapper_id'];
					$field->slug    = $field_data['element_id'];

					// Unset field ID
					unset( $field_data['element_id'] );

					// Import field data to model
					$field->import( $field_data );

					// Add field to the form
					$model->add_field( $field );
				}
			}
		}


		// Save the form
		$id = $model->save();

		if ( false === $id ) {
			return new WP_Error( 'form_save_error', __( 'There was a problem updating the form', Forminator::DOMAIN ) );
		}

		return $id;
	}


	/**
	 * Returns all poll objects.
	 *
	 * @since            1.2.0
	 * @access           public
	 *
	 * @param array|null|... $poll_ids
	 *
	 * @return array
	 */
	public static function get_polls( $poll_ids ) {
		// Initialize API
		self::initialize();

		$temp = array();

		if ( is_null( $poll_ids ) ) {
			$temp = Forminator_Poll_Form_Model::model()->get_all();
		} else {
			if ( ! is_array( $poll_ids ) ) {
				$poll_ids = func_get_args();
				$poll_ids = array_map( 'trim', $poll_ids );
			}

			foreach ( $poll_ids as $poll_id ) {
				$temp[] = self::get_poll( $poll_id );
			}
		}

		return $temp;
	}

	/**
	 * Returns poll object by given ID.
	 *
	 * @since  1.2
	 * @access public
	 *
	 * @param int $poll_id ID of the poll
	 *
	 * @return Forminator_Poll_Form_Model|WP_Error Forminator_Poll_Form_Model on success, or WP_Error otherwise
	 */
	public static function get_poll( $poll_id ) {
		// Initialize API
		self::initialize();

		if ( empty( $poll_id ) ) {
			return new WP_Error( 'missing_id', __( 'Poll ID is required!', Forminator::DOMAIN ) );
		}

		$model = Forminator_Poll_Form_Model::model()->load( $poll_id );
		if ( ! $model instanceof Forminator_Poll_Form_Model ) {
			return new WP_Error( 'poll_not_found', __( 'Poll Not Found!', Forminator::DOMAIN ) );
		}

		return $model;
	}

	/**
	 * Delete poll with given ID
	 *
	 * @since  1.2
	 * @access public
	 *
	 * @param int $poll_id ID of the poll
	 *
	 * @return bool|WP_Error True when successfully deleted | WP Error
	 */
	public static function delete_poll( $poll_id ) {
		// Initialize API
		self::initialize();

		$model = Forminator_Poll_Form_Model::model()->load( $poll_id );
		if ( is_object( $model ) ) {
			// Delete form object
			wp_delete_post( $poll_id );

			// Delete form entries
			Forminator_Form_Entry_Model::delete_by_form( $poll_id );

			// Delete form views
			$form_view = Forminator_Form_Views_Model::get_instance();
			$form_view->delete_by_form( $poll_id );

			return true;
		} else {
			return new WP_Error( 'not_found', sprintf( __( 'Can not find a poll with id: %s', Forminator::DOMAIN ), $poll_id ), $poll_id );
		}
	}

	/**
	 * Delete polls with given IDs
	 *
	 * @since  1.2
	 * @access public
	 *
	 * @param int $poll_ids Array with IDs
	 *
	 * @return bool|WP_Error True when successfully deleted | WP Error
	 */
	public static function delete_polls( $poll_ids ) {
		// Initialize API
		self::initialize();

		if ( is_array( $poll_ids ) && ! empty( $poll_ids ) ) {
			foreach ( $poll_ids as $id ) {
				self::delete_poll( $id );
			}

			return true;
		} else {
			return new WP_Error( 'invalid', __( 'Invalid or empty array with IDs', Forminator::DOMAIN ) );
		}
	}

	/**
	 * Add a poll
	 *
	 * @since  1.2
	 * @access public
	 *
	 * @param string $name     Poll name
	 * @param array  $fields   Array with poll fields
	 * @param array  $settings Array with settings
	 *
	 * @return int|WP_Error ID of new Poll on success, or WP_Error otherwise
	 */
	public static function add_poll( $name, $fields = array(), $settings = array() ) {
		// Initialize API
		self::initialize();

		if ( empty( $name ) ) {
			return new WP_Error( 'missing_name', __( 'Poll name is required!', Forminator::DOMAIN ) );
		}


		// Create new form model
		$model = new Forminator_Poll_Form_Model();

		// Set the post data
		$settings['formName'] = $name;
		$model->name          = $name;

		if ( is_array( $settings ) ) {
			// Bind form settings
			$model->settings = $settings;
		}

		// Add fields to the poll
		if ( is_array( $fields ) ) {
			foreach ( $fields as $field_data ) {
				// Create new field model
				$field            = new Forminator_Form_Field_Model();
				$field_data['id'] = $field_data['element_id'];

				// Import field data to model
				$field->import( $field_data );
				$field->slug = $field_data['element_id'];

				// Add field to the form
				$model->add_field( $field );
			}
		}

		// Save the form
		$id = $model->save();

		if ( false === $id ) {
			return new WP_Error( 'form_save_error', __( 'There was a problem saving the form', Forminator::DOMAIN ) );
		}

		return $id;
	}

	/**
	 * Update a poll
	 *
	 * @since  1.2
	 * @access public
	 *
	 * @param int   $id       Poll ID
	 * @param array $fields   Array with poll fields
	 * @param array $settings Array with settings
	 *
	 * @return int|WP_Error ID of updated Poll on success, or WP_Error otherwise
	 */
	public static function update_poll( $id, $fields = array(), $settings = array() ) {
		// Initialize API
		self::initialize();

		if ( empty( $id ) ) {
			return new WP_Error( 'missing_id', __( 'Poll ID is required!', Forminator::DOMAIN ) );
		}

		// Create new form model
		$model = Forminator_Poll_Form_Model::model()->load( $id );
		if ( ! is_object( $model ) ) {
			return new WP_Error( 'missing_object', __( "Poll model doesn't exist", Forminator::DOMAIN ) );
		}

		// Set the post data
		$model->name = $settings['formName'];

		if ( is_array( $settings ) ) {
			// Bind form settings
			$model->settings = $settings;
		}

		// Add fields to the poll
		if ( is_array( $fields ) ) {
			// Clear existing fields
			$model->clear_fields();

			foreach ( $fields as $field_data ) {
				// Create new field model
				$field            = new Forminator_Form_Field_Model();
				$field_data['id'] = $field_data['element_id'];

				// Import field data to model
				$field->import( $field_data );
				$field->slug = $field_data['element_id'];

				// Add field to the form
				$model->add_field( $field );
			}
		}

		// Save the form
		$id = $model->save();

		if ( false === $id ) {
			return new WP_Error( 'form_save_error', __( 'There was a problem saving the form', Forminator::DOMAIN ) );
		}

		return $id;
	}


	/**
	 * Returns all quiz objects.
	 *
	 * @since  1.2.0
	 * @access public
	 *
	 * @param array|null $quiz_ids
	 *
	 * @return array
	 */
	public static function get_quizzes( $quiz_ids = null ) {
		// Initialize API
		self::initialize();

		$temp = array();

		if ( is_null( $quiz_ids ) ) {
			$temp = Forminator_Quiz_Form_Model::model()->get_all();
		} else {
			if ( ! is_array( $quiz_ids ) ) {
				$quiz_ids = func_get_args();
				$quiz_ids = array_map( 'trim', $quiz_ids );
			}

			foreach ( $quiz_ids as $quiz_id ) {
				$temp[] = self::get_quiz( $quiz_id );
			}
		}

		return $temp;
	}

	/**
	 * Returns quiz object by given ID.
	 *
	 * @since  1.2
	 * @access public
	 *
	 * @param int $quiz_id ID of the Quiz
	 *
	 * @return Forminator_Quiz_Form_Model|WP_Error, Quiz Object on success, or WP_Error otherwise
	 */
	public static function get_quiz( $quiz_id ) {
		// Initialize API
		self::initialize();

		if ( empty( $quiz_id ) ) {
			return new WP_Error( 'missing_id', __( 'Quiz ID is required!', Forminator::DOMAIN ) );
		}

		$model = Forminator_Quiz_Form_Model::model()->load( $quiz_id );

		if ( ! $model instanceof Forminator_Quiz_Form_Model ) {
			return new WP_Error( 'quiz_not_found', __( 'Quiz Not Found!', Forminator::DOMAIN ) );
		}

		return $model;
	}

	/**
	 * Delete quiz with given ID
	 *
	 * @since  1.2
	 * @access public
	 *
	 * @param int $quiz_id ID of the quiz
	 *
	 * @return bool|WP_Error True when successfully deleted | WP Error
	 */
	public static function delete_quiz( $quiz_id ) {
		// Initialize API
		self::initialize();

		$model = Forminator_Quiz_Form_Model::model()->load( $quiz_id );
		if ( is_object( $model ) ) {
			// Delete form object
			wp_delete_post( $quiz_id );

			// Delete form entries
			Forminator_Form_Entry_Model::delete_by_form( $quiz_id );

			// Delete form views
			$form_view = Forminator_Form_Views_Model::get_instance();
			$form_view->delete_by_form( $quiz_id );

			return true;
		} else {
			return new WP_Error( 'not_found', sprintf( __( 'Can not find a poll with id: %s', Forminator::DOMAIN ), $quiz_id ), $quiz_id );
		}
	}

	/**
	 * Delete forms with given IDs
	 *
	 * @since  1.2
	 * @access public
	 *
	 * @param array $quiz_ids Array with IDs
	 *
	 * @return bool|WP_Error True when successfully deleted | WP Error
	 */
	public static function delete_quizzes( $quiz_ids ) {
		// Initialize API
		self::initialize();

		if ( is_array( $quiz_ids ) && ! empty( $quiz_ids ) ) {
			foreach ( $quiz_ids as $id ) {
				self::delete_quiz( $id );
			}

			return true;
		} else {
			return new WP_Error( 'invalid', __( 'Invalid or empty array with IDs', Forminator::DOMAIN ) );
		}
	}

	/**
	 * Add quiz
	 *
	 * @since  1.2
	 * @access public
	 *
	 * @param string $name      Quiz name
	 * @param string $type      Quiz type (nowrong|knowledge)
	 * @param array  $questions Array with quiz questions
	 * @param array  $results   Array with quiz results
	 * @param array  $settings  Array with settings
	 *
	 * @return int|WP_Error ID of new Quiz, or WP_Error otherwise
	 */
	public static function add_quiz( $name, $type, $questions = array(), $results = array(), $settings = array() ) {
		// Initialize API
		self::initialize();

		if ( empty( $name ) ) {
			return new WP_Error( 'missing_name', __( 'Quiz name is required!', Forminator::DOMAIN ) );
		}

		if ( empty( $type ) ) {
			return new WP_Error( 'missing_type', __( 'Quiz type is required!', Forminator::DOMAIN ) );
		}


		// Create new form model
		$model = new Forminator_Quiz_Form_Model();

		// Set the post data
		$settings['formName'] = $name;
		$model->name          = $name;

		// Set the type
		$model->quiz_type = $type;

		if ( is_array( $settings ) ) {
			// Bind form settings
			$model->settings = $settings;
		}

		// Bind questions
		if ( ! empty( $questions ) ) {
			$model->questions = $questions;
		}

		// Bind results
		if ( ! empty( $results ) ) {
			$model->results = $results;
		}

		// Save the form
		$id = $model->save();

		if ( false === $id ) {
			return new WP_Error( 'quiz_save_error', __( 'There was a problem saving the quiz', Forminator::DOMAIN ) );
		}

		return $id;
	}

	/**
	 * Update quiz
	 *
	 * @since  1.2
	 * @access public
	 *
	 * @param int   $id        Quiz ID
	 * @param array $questions Array with quiz questions
	 * @param array $results   Array with quiz results
	 * @param array $settings  Array with settings
	 *
	 * @return int|WP_Error ID of updated Quiz, or WP_Errr otherwise
	 */
	public static function update_quiz( $id, $questions = array(), $results = array(), $settings = array() ) {
		// Initialize API
		self::initialize();

		if ( empty( $id ) ) {
			return new WP_Error( 'missing_id', __( 'Quiz ID is required!', Forminator::DOMAIN ) );
		}

		// Create new form model
		/** @var Forminator_Quiz_Form_Model $model */
		$model = Forminator_Quiz_Form_Model::model()->load( $id );
		if ( ! is_object( $model ) ) {
			return new WP_Error( 'missing_object', __( "Quiz model doesn't exist", Forminator::DOMAIN ) );
		}

		// Set the post data
		$model->name = $settings['formName'];


		if ( is_array( $settings ) ) {
			// Bind form settings
			$model->settings = $settings;
		}

		// Bind questions
		if ( ! empty( $questions ) ) {
			// Clear fields
			$model->clear_fields();

			$model->questions = $questions;
		}

		// Bind results
		if ( ! empty( $results ) ) {
			$model->results = $results;
		}

		// Save the form
		$id = $model->save();

		if ( false === $id ) {
			return new WP_Error( 'quiz_save_error', __( 'There was a problem saving the quiz', Forminator::DOMAIN ) );
		}

		return $id;
	}

	// ENTRIES

	/**
	 * Get entries objects
	 *
	 * @uses  Forminator_Form_Entry_Model::get_entries
	 *
	 * @since 1.2
	 *
	 * @param int $form_id Form ID
	 *
	 * @return Forminator_Form_Entry_Model[]|WP_Error
	 */
	public static function get_entries( $form_id ) {
		// Initialize API
		self::initialize();

		// Check if Form ID is set
		if ( empty( $form_id ) ) {
			return new WP_Error( 'missing_form_id', __( 'Form ID is required!', Forminator::DOMAIN ) );
		}

		return Forminator_Form_Entry_Model::get_entries( $form_id );
	}

	/**
	 * Get entry object
	 *
	 * @since 1.2
	 *
	 * @param $form_id
	 * @param $entry_id
	 *
	 * @return Forminator_Form_Entry_Model|WP_Error
	 */
	public static function get_entry( $form_id, $entry_id ) {
		// Initialize API
		self::initialize();

		// Check if Form ID is set
		if ( empty( $form_id ) ) {
			return new WP_Error( 'missing_form_id', __( 'Form ID is required!', Forminator::DOMAIN ) );
		}

		// Check if Entry ID is set
		if ( empty( $entry_id ) ) {
			return new WP_Error( 'missing_entry_id', __( 'Entry ID is required!', Forminator::DOMAIN ) );
		}

		return new Forminator_Form_Entry_Model( $entry_id );
	}

	/**
	 * Delete entry
	 *
	 * @uses  Forminator_Form_Entry_Model::delete_by_entry
	 *
	 * @since 1.2
	 *
	 * @param $form_id
	 * @param $entry_id
	 *
	 * @return bool|WP_Error
	 */
	public static function delete_entry( $form_id, $entry_id ) {
		// Initialize API
		self::initialize();

		// Check if Form ID is set
		if ( empty( $form_id ) ) {
			return new WP_Error( 'missing_form_id', __( 'Form ID is required!', Forminator::DOMAIN ) );
		}

		// Check if Entry ID is set
		if ( empty( $entry_id ) ) {
			return new WP_Error( 'missing_entry_id', __( 'Entry ID is required!', Forminator::DOMAIN ) );
		}

		// Delete entry
		Forminator_Form_Entry_Model::delete_by_entry( $form_id, $entry_id );

		return true;
	}

	/**
	 * Delete multiple entries
	 *
	 * @uses  Forminator_Form_Entry_Model::delete_by_entrys
	 *
	 * @since 1.2
	 *
	 * @param $form_id
	 * @param $entries_ids
	 *
	 * @return bool|WP_Error
	 */
	public static function delete_entries( $form_id, $entries_ids ) {
		// Initialize API
		self::initialize();

		// Check if Form ID is set
		if ( empty( $form_id ) ) {
			return new WP_Error( 'missing_form_id', __( 'Form ID is required!', Forminator::DOMAIN ) );
		}

		// Check if Entry ID is set
		if ( empty( $entries_ids ) ) {
			return new WP_Error( 'missing_entry_id', __( 'Entry IDs are required!', Forminator::DOMAIN ) );
		}

		// Check if entries ids are array and convert to string
		if ( is_array( $entries_ids ) ) {
			$entries_ids = implode( ",", $entries_ids );
		}

		// Delete entries
		Forminator_Form_Entry_Model::delete_by_entrys( $form_id, $entries_ids );

		return true;
	}

	/**
	 * Count entries
	 *
	 * @uses  Forminator_Form_Entry_Model::count_entries
	 *
	 * @since 1.2
	 *
	 * @param $form_id
	 *
	 * @return int|WP_Error
	 */
	public static function count_entries( $form_id ) {
		// Initialize API
		self::initialize();

		// Check if Form ID is set
		if ( empty( $form_id ) ) {
			return new WP_Error( 'missing_form_id', __( 'Form ID is required!', Forminator::DOMAIN ) );
		}

		return Forminator_Form_Entry_Model::count_entries( $form_id );
	}

	/**
	 * Get Form entries
	 *
	 * @access public
	 * @since  1.2
	 *
	 * @param int $form_id Form ID
	 *
	 * @return Forminator_Form_Entry_Model[]|WP_Error
	 */
	public static function get_form_entries( $form_id ) {
		return self::get_entries( $form_id );
	}

	/**
	 * Get Form entry by Form ID and Entry ID
	 *
	 * @access public
	 * @since  1.2
	 *
	 * @param int $form_id  Form ID
	 * @param int $entry_id Entry ID
	 *
	 * @return Forminator_Form_Entry_Model|WP_Error
	 */
	public static function get_form_entry( $form_id, $entry_id ) {
		return self::get_entry( $form_id, $entry_id );
	}

	/**
	 * Delete Form entry
	 *
	 * @access public
	 * @since  1.2
	 *
	 * @param int $form_id
	 * @param int $entry_id
	 *
	 * @return bool|WP_Error
	 */
	public static function delete_form_entry( $form_id, $entry_id ) {
		return self::delete_entry( $form_id, $entry_id );
	}

	/**
	 * Delete Form entries
	 *
	 * @access public
	 * @since  1.2
	 *
	 * @param int          $form_id   Form ID
	 * @param array|string $entry_ids entry IDs in an array, or string glued with `,`(comma)
	 *
	 * @return bool|WP_Error
	 */
	public static function delete_form_entries( $form_id, $entry_ids ) {
		return self::delete_entries( $form_id, $entry_ids );
	}

	/**
	 * Count form entries
	 *
	 * @access public
	 * @since  1.2
	 *
	 * @param  $form_id
	 *
	 * @return int|WP_Error
	 */
	public static function count_form_entries( $form_id ) {
		return self::count_entries( $form_id );
	}

	/**
	 * Get Poll entries
	 *
	 * @access public
	 * @since  1.2
	 *
	 * @param int $poll_id Poll ID
	 *
	 * @return Forminator_Form_Entry_Model[]|WP_Error
	 */
	public static function get_poll_entries( $poll_id ) {
		return self::get_entries( $poll_id );
	}

	/**
	 * Get Poll entry by ID
	 *
	 * @access public
	 * @since  1.2
	 *
	 * @param int $poll_id  Poll ID
	 * @param int $entry_id Entry ID
	 *
	 * @return Forminator_Form_Entry_Model|WP_Error
	 */
	public static function get_poll_entry( $poll_id, $entry_id ) {
		return self::get_entry( $poll_id, $entry_id );
	}

	/**
	 * Delete Poll entry
	 *
	 * @access public
	 * @since  1.2
	 *
	 * @param int $poll_id
	 * @param int $entry_id
	 *
	 * @return bool|WP_Error
	 */
	public static function delete_poll_entry( $poll_id, $entry_id ) {
		return self::delete_entry( $poll_id, $entry_id );
	}

	/**
	 * Delete Poll entries
	 *
	 * @access public
	 * @since  1.2
	 *
	 * @param int          $poll_id
	 * @param array|string $entry_ids entry IDs in an array, or string separated with `,`(comma)
	 *
	 * @return bool|WP_Error
	 */
	public static function delete_poll_entries( $poll_id, $entry_ids ) {
		return self::delete_entries( $poll_id, $entry_ids );
	}

	/**
	 * Count poll entries
	 *
	 * @access public
	 * @since  1.2
	 *
	 * @param int $poll_id
	 *
	 * @return int|WP_Error
	 */
	public static function count_poll_entries( $poll_id ) {
		return self::count_entries( $poll_id );
	}

	/**
	 * Get Quiz entries
	 *
	 * @access public
	 * @since  1.2
	 *
	 * @param int $quiz_id Quiz ID
	 *
	 * @return Forminator_Form_Entry_Model[]|WP_Error
	 */
	public static function get_quiz_entries( $quiz_id ) {
		return self::get_entries( $quiz_id );
	}

	/**
	 * Get Quiz entry by ID
	 *
	 * @access public
	 * @since  1.2
	 *
	 * @param int $quiz_id  Quiz ID
	 * @param int $entry_id Entry ID
	 *
	 * @return Forminator_Form_Entry_Model|WP_Error
	 */
	public static function get_quiz_entry( $quiz_id, $entry_id ) {
		return self::get_entry( $quiz_id, $entry_id );
	}

	/**
	 * Delete Quiz entry
	 *
	 * @access public
	 * @since  1.2
	 *
	 * @param int $quiz_id
	 * @param int $entry_id
	 *
	 * @return bool|WP_Error
	 */
	public static function delete_quiz_entry( $quiz_id, $entry_id ) {
		return self::delete_entry( $quiz_id, $entry_id );
	}

	/**
	 * Delete Quiz entries
	 *
	 * @access public
	 * @since  1.2
	 *
	 * @param int   $quiz_id
	 * @param array $entry_ids
	 *
	 * @return bool|WP_Error
	 */
	public static function delete_quiz_entries( $quiz_id, $entry_ids ) {
		return self::delete_entries( $quiz_id, $entry_ids );
	}

	/**
	 * Count quiz entries
	 *
	 * @access public
	 * @since  1.2
	 *
	 * @param int $quiz_id
	 *
	 * @return int|WP_Error
	 */
	public static function count_quiz_entries( $quiz_id ) {
		return self::count_entries( $quiz_id );
	}
}