<?php 
namespace FoodBook;
/**
 *
 * @package     Foodbook
 * @author      ThemeLooks
 * @copyright   2020 ThemeLooks
 * @license     GPL-2.0-or-later
 *
 *
 */

class Woo_Hooks {

  function __construct() {

    // Add custom field to order object action hook
    add_action( 'woocommerce_checkout_create_order_line_item', [ __CLASS__ ,'foodbook_add_custom_data_to_order' ], 10, 4 );

    // Woocommerce Order meta data after shipping address action hook
    add_action( 'woocommerce_admin_order_data_after_shipping_address', [ __CLASS__, 'foodbook_edit_woocommerce_checkout_page' ], 10, 1 );

    //Woocommerce Order meta data after shipping address action
    add_action( 'woocommerce_checkout_update_order_meta', [ __CLASS__, 'checkout_update_order_meta' ], 10, 2 );

    // Override WooCommerce Templates from plugin filter hook
    add_filter( 'woocommerce_locate_template', [ __CLASS__, 'foodbook_woo_template' ], 1, 3 );

    // WooCommerce order meta query filter hook
    add_filter( 'woocommerce_order_data_store_cpt_get_orders_query', [ __CLASS__, 'foodbook_order_meta_query_var' ], 10, 2 );

    // WooCommerce product meta query filter hook
    add_filter( 'woocommerce_product_data_store_cpt_get_products_query', [ __CLASS__, 'foodbook_product_meta_query_var'], 10, 2 );

    // Order status set ( order place  or pre order )
    add_action( 'woocommerce_order_status_processing', [ __CLASS__, 'wc_order_status_changed' ], 10, 1 );
    add_action( 'woocommerce_order_status_completed', [ __CLASS__, 'wc_order_status_changed' ], 10, 1 );
    add_action( 'woocommerce_order_status_on-hold', [ __CLASS__, 'wc_order_status_changed' ], 10, 1 );

    // Order cancelled hook
    add_action( 'woocommerce_cancelled_order', [ __CLASS__, 'wc_cancelled_order'], 10, 1 );

    // order status failed hook
    add_action('woocommerce_order_status_failed', [ __CLASS__, 'wc_failed_order'], 15, 2);

    // 
    add_action( 'woocommerce_cart_calculate_fees', [ $this, 'checkout_radio_choice_fee' ], 20, 1 );

    // 
    add_action( 'woocommerce_checkout_update_order_review', [ $this, 'checkout_radio_choice_set_session' ] );

    //
    add_filter( 'wc_order_statuses', [ __CLASS__, 'add_pre_order_statuses' ] );
      
    //
    add_action( 'init', [ __CLASS__, 'register_pre_order_status' ] );
    
    //
    add_filter( 'woocommerce_add_to_cart_validation', [ __CLASS__, 'add_to_cart_validation' ], 10, 4 );

    //
    add_filter( 'manage_edit-product_columns', [__CLASS__, 'add_product_column'], 10, 1 );

    //
    add_action( 'manage_product_posts_custom_column', [__CLASS__, 'add_product_column_content'], 10, 2 );

    //
    add_filter( 'woocommerce_package_rates', [__CLASS__, 'hide_shipping_when_free_is_available'], 100 );

    //
    add_action( 'woocommerce_after_checkout_validation', [__CLASS__, 'checkout_page_validate' ], 10, 2);

    //
    add_action( 'woocommerce_checkout_process', [ __CLASS__, 'wc_checkout_minimum_order_amount' ] );

    //
    add_action( 'woocommerce_before_cart' , [ __CLASS__, 'wc_cart_minimum_order_amount' ] );

    // cancel unpaid order
    add_filter( 'woocommerce_cancel_unpaid_order', [ __CLASS__, 'action_woocommerce_cancel_unpaid_orders'], 10, 2 );

    add_filter( 'woocommerce_add_cart_item', [ __CLASS__, 'add_cart_item'], 20, 1 );
    add_filter( 'woocommerce_get_cart_item_from_session', [__CLASS__, 'get_cart_item_from_session'], 100, 2 );

  }

  public static function getText() {
    return \FoodBook\Inc\Text::getText();
  } 

  /**
   * Add custom field to order object
   */
  public static function foodbook_add_custom_data_to_order( $item, $cart_item_key, $values, $order ) {

    foreach( $item as $cart_item_key => $values ) {

      // Item Instructions
      if( isset( $values['item_instructions'] ) ) {

        $item->add_meta_data( esc_html__( 'Item Instructions', 'foodbook' ), sanitize_text_field( $values['item_instructions'] ), true );

      }

      // Item extra features
      if( isset( $values['extra_options'] ) ) {

        $item->add_meta_data( esc_html__( 'Item Extra:-', 'foodbook' ), sanitize_text_field( $values['extra_options'] ), true );
       
      }

    }
  }
  
/**
 * Woocommerce Order meta data after shipping address
 *
 * 
 */
public static function foodbook_edit_woocommerce_checkout_page( $order ) {

  global $post_id;
  $order = new \WC_Order( $post_id );

  $orderID      = $order->get_id();
  $time         = $order->get_meta('_pickup_time');
  $deliveryType = $order->get_meta('_delivery_type');
  $branch       = $order->get_meta('_fb_pickup_branch');

  //
  if( !empty( $deliveryType ) ) {
    echo '<p><strong>'.esc_html__('Delivery Type', 'foodbook' ).':</strong> ' . esc_html( $deliveryType ) . '</p>';
  }
  //
  if( !empty( $time ) ) {
    echo '<p><strong>'.esc_html__('Time to Deliver/Pickup', 'foodbook' ).':</strong> ' . esc_html( $time ) . '</p>';
  }
  //
  if( !empty( $branch ) ) {
    echo '<p><strong>'.esc_html__('Pickup Branch Name', 'foodbook' ).':</strong> ' . esc_html( get_the_title( $branch ) ) . '</p>';
  }
  
}

/**
 * Woocommerce Add Order meta data 
 *
 */
public static function checkout_update_order_meta( $order_id, $posted ) {

    $order = wc_get_order( $order_id );

    //
    if( isset( $_POST['fb_delivery_time'] ) ) {

      $time = explode( ',', $_POST['fb_delivery_time']);

      $order->update_meta_data( '_pickup_time', sanitize_text_field( $time[0] ) );
    }
    //
    if( isset( $_POST['fb_delivery_options'] ) ) {
      $order->update_meta_data( '_delivery_type', sanitize_text_field( $_POST['fb_delivery_options'] ) );
    }
    //
    if( isset( $_POST['fb_pickup_branch'] ) ) {
      $order->update_meta_data( '_fb_pickup_branch', sanitize_text_field( $_POST['fb_pickup_branch'] ) );
    }
    //
    if( isset( $_POST['fb_delivery_date'] ) ) {
      $order->update_meta_data( '_delivery_date', sanitize_text_field( foodbook_db_format_date_time( $_POST['fb_delivery_date'] ) ) );
    }
    
    $order->save();

} 

/**
 *
 * Override WooCommerce Templates
 * 
 */
public static function foodbook_woo_template( $template, $template_name, $template_path ) {

     global $woocommerce;
     $_template = $template;
     if ( ! $template_path ) 
        $template_path = $woocommerce->template_url;
 
     $plugin_path  = untrailingslashit( FOODBOOK_DIR_PATH )  . '/template/woocommerce/';
 
    // Look within passed path within the theme - this is priority
    $template = locate_template(
    array(
      $template_path . $template_name,
      $template_name
    )
   );
 
   if( file_exists( $plugin_path . $template_name ) )
    $template = $plugin_path . $template_name;
 
   if ( ! $template )
    $template = $_template;

   return $template;

}

/**
 *
 * WooCommerce order meta query 
 * 
 */
public static function foodbook_order_meta_query_var( $query, $query_vars ) {

  if ( ! empty( $query_vars['tracking_status'] ) ) {
    $query['meta_query'][] = array(
      'key' => '_order_tracking_status',
      'value' => esc_attr( $query_vars['tracking_status'] ),
    );
  }

  //
  if ( ! empty( $query_vars['branch'] ) ) {
    $query['meta_query'][] = array(
      'key' => '_fb_pickup_branch',
      'value' => esc_attr( $query_vars['branch'] ),
    );
  }
  
  //
  if ( ! empty( $query_vars['delivery_boy'] ) ) {
    $query['meta_query'][] = array(
      'key' => '_order_delivery_boy',
      'value' => esc_attr( $query_vars['delivery_boy'] ),
    );
  }
  //
  if ( ! empty( $query_vars['delivery_date'] ) ) {
    $query['meta_query'][] = array(
      'key' => '_delivery_date',
      'value' => esc_attr( $query_vars['delivery_date'] ),
    );
  }
  //
  if ( ! empty( $query_vars['pre_order_status'] ) ) {
    $query['meta_query'][] = array(
      'key' => '_pre_order_status',
      'value' => esc_attr( $query_vars['pre_order_status'] ),
    );
  }
  //
  if ( ! empty( $query_vars['pickup_time'] ) ) {
    $query['meta_query'][] = array(
      'key' => '_pickup_time',
      'value' => esc_attr( $query_vars['pickup_time'] ),
    );
  }

  return $query;
}

/**
 *
 * WooCommerce product meta query 
 * 
 */

public static function foodbook_product_meta_query_var( $query, $query_vars ) {

  // low to high price

  if ( ! empty( $query_vars['low_to_high_price'] ) ) {
    $query['meta_query'][] = array(
      'relation' => 'OR',
      array(
          'key' => '_price',
          'value' => esc_attr( $query_vars['low_to_high_price'] ),
          'compare' => '>',
          'type' => 'NUMERIC'
      ),         
      array(
          'key' => '_sale_price',
          'value' => esc_attr( $query_vars['low_to_high_price'] ),
          'compare' => '>',
          'type' => 'NUMERIC'
      )
    );
  }

  // Average rating
  if ( ! empty( $query_vars['average_rating_product'] ) ) {
    $query['meta_query'][] = array(
      array(
          'key' => '_wc_average_rating',
          'value' => esc_attr( $query_vars['average_rating_product'] ),
          'compare' => '>',
          'type' => 'NUMERIC'
      )
    );
  }

  // Product query by branch meta
  if ( ! empty( $query_vars['product_by_branch'] ) ) {
    $query['meta_query'][] = array(
      array(
          'key' => 'foodbookbranch_list',
          'value' => esc_attr( $query_vars['product_by_branch'] ),
          'compare' => 'LIKE'
      )
    );
  }

  // Average rating
  if ( ! empty( $query_vars['total_sales_product'] ) ) {
    $query['meta_query'][] = array(
      array(
          'key' => 'total_sales'
      )
    );
  }
  

  return $query;

}

//
public static function wc_order_status_changed( $order_id ) {
    // Current Time Date
    $time = current_time( "Y-m-d H:i:s" );
    $CurrentDate = current_time( "d-m-Y" );

    $order = wc_get_order( $order_id ); // returns WC_Order object.

    // Get delivery date
    $dDate = get_post_meta( absint($order_id), '_delivery_date', true );

    // Pre order status
    if( strtotime( $dDate ) >  strtotime( $CurrentDate ) ) {
      $order->update_meta_data( '_pre_order_status', sanitize_text_field( 'PO' ) );
    }

    // Update status
    $order->update_meta_data( '_order_tracking_status', sanitize_text_field( 'OP' ) );
    $order->update_meta_data( '_order_tracking_status_time', sanitize_text_field( $time ) );
    $order->save();

}


/**
 * WooCommerce order cancelled callback
 * Update order tracking status
 * @param  int $order_id
 * @return void
 */
public static function wc_cancelled_order( $order_id ) {
  
  $time = current_time( "Y-m-d H:i:s" );
  $order = wc_get_order( $order_id ); // returns WC_Order object.

  $order->update_meta_data( '_order_tracking_status', sanitize_text_field( 'OC' ) );
  $order->update_meta_data( '_order_tracking_status_time', sanitize_text_field( $time ) );
  $order->save();

}

/**
 * wc_failed_order 
 * Update order tracking status when order failed
 * @param  int $order_id and object $order
 * @return void
 */
public static function wc_failed_order( $order_id, $order ) {
  
  $time  = current_time( "Y-m-d H:i:s" );
  $order = wc_get_order( $order_id ); // returns WC_Order object.

  $order->update_meta_data( '_order_tracking_status', sanitize_text_field( 'OF' ) );
  $order->update_meta_data( '_order_tracking_status_time', sanitize_text_field( $time ) );
  $order->save();


}

/**
 * action_woocommerce_cancel_unpaid_orders
 * Update order tracking status to cancelled when unpaid order time limit reached.
 * @param  bool $cby and object $order
 * @return void
 */
public static function action_woocommerce_cancel_unpaid_orders( $cby, $order ) {

  if( $cby ) {
    $order->update_meta_data( '_order_tracking_status', sanitize_text_field( 'OC' ) );
    $order->save();
    $order->update_status( 'cancelled', esc_html__( 'Unpaid order cancelled - time limit reached.', 'foodbook' ) );
  }
  
}

/**
 *
 * Add Fee and Calculate Total
 * 
 */

function checkout_radio_choice_fee( $cart ) {
   
  if ( is_admin() && ! defined( 'DOING_AJAX' ) ) return;

  $options = get_option('foodbook_options');

  //
  if( !empty( $options['location_type'] ) && $options['location_type'] == 'address' ) {
    $deliveryFee = foodbook_radius_based_delivery_fee();
  } else {
    $deliveryFee = foodbook_zipcode_based_delivery_fee();
  }

  $freeShippingAmount = isset( $options['free-shipping-amount'] ) ? $options['free-shipping-amount'] : '';

  // Free shipping
  if( !empty( $freeShippingAmount ) && WC()->cart->get_subtotal() >= $freeShippingAmount ) {
    return;
  }
  //
  $radio = WC()->session->get( 'radio_chosen' );

  if ( $radio == 'Delivery' ) {
      $fee = $deliveryFee;
  }else {
      $fee = 0;
  }

  //
  if ( $fee ) {
    $is_taxable = !empty( $options['delivery_fee_tax_status'] ) && $options['delivery_fee_tax_status'] == 'taxable' ? true : false;
    $cart->add_fee( esc_html__( 'Delivery Fee', 'foodbook' ), esc_html( $fee ), $is_taxable );
  }
   
}

/**
 *
 * Add Radio Choice to Session
 * 
 */
function checkout_radio_choice_set_session( $posted_data ) {
  parse_str( $posted_data, $output );
  if ( isset( $output['fb_delivery_options'] ) ){
      WC()->session->set( 'radio_chosen', $output['fb_delivery_options'] );
  }
}

/**
 * [register_pre_order_status description]
 * @return [type] [description]
 */
public static  function register_pre_order_status() {
 
  register_post_status( 'wc-pre-order', array(

  'label' => esc_html__( 'Pre Order', 'foodbook' ),

  'public' => true,

  'show_in_admin_status_list' => true,

  'show_in_admin_all_list' => true,

  'exclude_from_search' => false,

  'label_count' => _n_noop( 'Pre Order <span class="count">(%s)</span>', 'Pre Order <span

  class="count">(%s)</span>' )

  ) );

}

/**
 * [add_pre_order_statuses description]
 * @param [type] $order_statuses [description]
 */
public static function add_pre_order_statuses( $order_statuses ) {

  $new_order_statuses = array();

  foreach ( $order_statuses as $key => $status ) {

    $new_order_statuses[ $key ] = $status;
    if ( 'wc-processing' === $key ) {

      $new_order_statuses['wc-draft'] = esc_html__( 'Draft Order', 'foodbook' );
    }

  }

  return $new_order_statuses;

}


/**
 * Validate product extra features option
 */
public static function add_to_cart_validation( $passed, $product_id, $quantity, $variation_id=null ) {

  // Check product extra required item selected
  $requiredStatus = !empty( $_POST['required_status'] ) ? $_POST['required_status'] : [];
  if( in_array( 'false' , $requiredStatus ) ) {
    $passed = false;
    wc_add_notice( esc_html__( 'Features items is a required field.', 'foodbook' ), 'error' );
  }
  return $passed;
}

/**
 * [add_product_column description]
 * @param [type] $columns [description]
 */
public static function add_product_column( $columns ) {
    //add branch column 
    
    if( foodbook_is_multi_branch() ) {
      $columns['branch_name'] = esc_html__( 'Branch Name', 'foodbook' );
    }

    return $columns;
}

/**
 * [add_product_column_content description]
 * @param [type] $column [description]
 * @param [type] $postid [description]
 */
public static function add_product_column_content( $column, $postid ) {

    if ( foodbook_is_multi_branch() && $column == 'branch_name' ) {

      $branch = get_post_meta( $postid, 'foodbookbranch_list', true );

      if( !empty( $branch ) ) {
        $BranchName = [];
        foreach( $branch as $branchId ) {
          $BranchName[] = get_the_title( $branchId );
        }

        echo implode(', ', $BranchName);
      }
        
    }

}

/**
 * [hide_shipping_when_free_is_available description]
 * @param  [type] $rates [description]
 * @return [type]        [description]
 */
public static function hide_shipping_when_free_is_available( $rates ) {
  $free = array();

  $options = get_option('foodbook_options');
  $freeShippingAmount = isset( $options['free-shipping-amount'] ) ? $options['free-shipping-amount'] : '';

  // Free shipping
  if( !empty( $freeShippingAmount ) && WC()->cart->get_subtotal() >= $freeShippingAmount ) {

    foreach ( $rates as $rate_id => $rate ) {
      if ( 'free_shipping' == $rate->method_id ) {
        $free[ $rate_id ] = $rate;
        break;
      }
    }

    return ! empty( $free ) ? $free : $rates;
  } else {
    
    return $rates;
  }

}

/**
  * Validate checkout branch select option
  */
public static function checkout_page_validate( $fields, $errors ) {

  $options = get_option('foodbook_options');
  $text = self::getText();

  if( empty( $options['checkout-delivery-option'] ) ) {
    return;
  }

  // Check branch 
  if( foodbook_is_multi_branch() &&  empty( $_POST['fb_pickup_branch'] ) ) {
    $errors->add( 'validation', esc_html( $text['valid_branch_field'] ) );
  }

  // Check delivery pickup type
  if( empty( $_POST['fb_delivery_options'] ) ) {
    $errors->add( 'validation', esc_html( $text['valid_delivery_type_field'] ) );
  }

  // Check inrestaurant table number
  if( $_POST['fb_delivery_options'] == 'In-Restaurant' && empty( $_POST['fb_inrestaurant_table'] ) ) {
    $errors->add( 'validation', esc_html( $text['valid_inrestaurant_table_number_field'] ) );
  }

  // Check delivery time validation 
  if( !empty( $options['pickup-time-switch'] ) && ( !empty( $_POST['fb_delivery_options'] ) && $_POST['fb_delivery_options'] != 'In-Restaurant'  ) ) {

    // delivery time empty check 
    if( empty( $_POST['fb_delivery_time'] ) ) {
      $errors->add( 'validation', esc_html( $text['valid_delivery_time_field'] ) );
    }
    //
    if( !empty( $options['off-current-order'] ) &&  empty( $_POST['fb_delivery_date'] ) ) {
      $errors->add( 'validation', esc_html( $text['valid_delivery_date_field'] ) );
    }
    //
    if( !empty( $_POST['fb_delivery_time'] ) && in_array( 'no', explode(',', $_POST['fb_delivery_time']) ) ) {
      $errors->add( 'validation', esc_html( $text['valid_slot_not_available'] ) );
    }
    //
    if( !empty( $_POST['fb_delivery_time'] ) && in_array( 'true', explode(',', $_POST['fb_delivery_time'] ) ) ) {
      $errors->add( 'validation', esc_html( $text['valid_break_time'] ) );
    }
    

  }

  //
  
  if( !empty( $options['availability-checker-active'] ) && $_POST['fb_delivery_options'] == 'Delivery' ) {

    if( !isset( $_COOKIE['d_availability_status'] ) ) {
        $errors->add( 'validation', esc_html__( 'Please check delivery availability on your location, Before place order', 'foodbook' ) );
    } else {
      if( $_COOKIE['d_availability_status'] == 'no' ) {
        $errors->add( 'validation', esc_html( $text['delivery_available_error'] ) );
      }
    }

  }
  
}

/**
 * Set a minimum order amount for checkout
 * @return [type] [description]
 */
public static function wc_cart_minimum_order_amount() {
    // Set this variable to specify a minimum order value
    $minimum = foodbook_getOptionData( 'minimum-order-amount' );
    $cartSubtotal = WC()->cart->get_subtotal();

    if ( !empty( $minimum ) && $cartSubtotal < $minimum ) {

      $notice = sprintf(
        esc_html__( 'Your current cart total is %s — you must have an order with a minimum of %s to place your order', 'foodbook' ) , 
        wc_price( $cartSubtotal ), 
        wc_price( $minimum )
      );

      wc_print_notice( $notice, 'error' );

    }
}

/**
 * [wc_checkout_minimum_order_amount description]
 * @return [type] [description]
 */
public static function wc_checkout_minimum_order_amount() {
    // Set this variable to specify a minimum order value
    $minimum = foodbook_getOptionData( 'minimum-order-amount' );
    $cartSubtotal = WC()->cart->get_subtotal();

    if ( !empty( $minimum ) && $cartSubtotal < $minimum ) {

      $notice = sprintf(
        esc_html__( 'Your current order sub total is %s — you must have an order with a minimum of %s to place your order', 'foodbook' ) , 
        wc_price( $cartSubtotal ), 
        wc_price( $minimum )
      );

      wc_add_notice( $notice, 'error' );

    }
}


/**
 * [add_cart_item description]
 * @param [type] $cart_item [description]
 */
public static function add_cart_item( $cart_item ) {

  if( !empty( $cart_item['extra_options_price'] ) ) {
    $totalPrice = $cart_item['data']->get_price() + $cart_item['extra_options_price'];
    $cart_item['data']->set_price( sanitize_text_field( $totalPrice ) );
  }
  return $cart_item;

}
/**
 * [get_cart_item_from_session description]
 * @param  [type] $cart_item [description]
 * @param  [type] $values    [description]
 * @return [type]            [description]
 */
public static function get_cart_item_from_session( $cart_item, $values ) {

  if( !empty( $cart_item['extra_options_price'] ) ) {
    $cart_item = self::add_cart_item($cart_item);
  }
  return $cart_item;

}





}//

// Woo_Hooks class init

new Woo_Hooks();