<?php 

// get all the cufons from a folder
function fp_cufon_list() {
	$fonts = array();
	$cufon_list = scandir(FP_DIR."/cufon");
	
	foreach($cufon_list as $font_file){
		if($font_file != '.' && $font_file != '..') {
			$file_content = file_get_contents(FP_DIR."/cufon/".$font_file);
			if(preg_match('/font-family":"(.*?)"/i',$file_content,$match)){
				$fonts[$match[1]] = basename($font_file);
			}
		}
	}
	return $fonts;
}


// get all the cufons from a folder
function fp_font_list() {
	$fonts = array();
	$font_list = scandir(FP_DIR."/fonts");
	
	foreach($font_list as $folder_name) {
		if($folder_name != '.' && $folder_name != '..') {
			$font_name = fp_urlToName($folder_name); // add the dot to mak it works
			$fonts[$font_name] = $folder_name;
		}
	}
	return $fonts;
}


// get the name of a font from a package
function fp_get_zip_fontname($folder) {
	$font_name = false;
	
	$file_list = scandir($folder.'/fonts'); // use fonts subfolder as per http://everythingfonts.com/font-face - Jan 2015	
	foreach($file_list as $file) {
		$ext = strtolower(fp_stringToExt($file));
		if($ext == '.ttf' || $ext == '.otf') {
			require_once(FP_DIR.'/classes/ttf_info.php');
			$fontinfo = getFontInfo($folder.'/fonts/'.$file);
			$font_name = fp_stringToUrl($fontinfo[4]);
			
			break;
		}
	}
	return $font_name;
}


// create definitive fontface font folder
// copy the font files from temp to new folder 
// delete the temp folder
function fp_copy_zip_fontfiles($tmp_folder, $new_folder) {
	// create
	mkdir($new_folder);	
	if(!file_exists($tmp_folder) || !file_exists($new_folder)) {return false;}
	
	// copy
	$file_list = scandir($tmp_folder.'/fonts'); // use fonts subfolder as per http://everythingfonts.com/font-face - Jan 2015	
	foreach($file_list as $file) {
		$ext = strtolower(fp_stringToExt($file));
		if($ext == '.ttf' || $ext == '.otf' || $ext == '.woff' || $ext == '.eot' || $ext == '.svg') {
			if(!copy($tmp_folder.'/fonts/'.$file, $new_folder.'/'.$file)) {return false; break;}
		}	
	}
	
	// delete
	fp_remove_folder($tmp_folder);
	
	return true;
}
	

// get file extension from a filename
function fp_stringToExt($string) {
	$pos = strrpos($string, '.');
	$ext = strtolower(substr($string,$pos));
	return $ext;	
}


// get filename without extension
function fp_stringToFilename($string, $raw_name = false) {
	$pos = strrpos($string, '.');
	$name = substr($string,0 ,$pos);
	if(!$raw_name) {$name = ucwords(str_replace('_', ' ', $name));}
	return $name;	
}


// string to url format
function fp_stringToUrl($string){
	$trans = array("à" => "a", "è" => "e", "é" => "e", "ò" => "o", "ì" => "i", "ù" => "u");
	$string = trim(strtr($string, $trans));
	$string = preg_replace('/[^a-zA-Z0-9-.]/', '_', $string);
	$string = preg_replace('/-+/', "_", $string);
	$string = str_replace(array('.', '#'), '_', $string);
	
	while( substr($string, 0, 1) == '_') {
		$string = substr($string, 1);	
	}
	
	return $string;
}


// normalize a url string
function fp_urlToName($string) {
	$string = ucwords(str_replace('_', ' ', $string));
	return $string;	
}


// remove a folder and its contents
function fp_remove_folder($path) {
	if($objs = @glob($path."/*")){
		foreach($objs as $obj) {
			@is_dir($obj)? fp_remove_folder($obj) : @unlink($obj);
		}
	 }
	@rmdir($path);
	return true;
}


// remove enabled font during deletion
function fp_remove_enabled($to_remove, $opt_name) {
	$array = get_option($opt_name);
	if(!$array) {return false;}
	
	$new_array = $array;
	foreach($array as $key => $data) {
		if($to_remove == $key) {
			unset($new_array[$key]);
			break;	
		}	
	}
	
	if(count($new_array) > 0) {update_option($opt_name, $new_array);}
	else {delete_option($opt_name);}
	
	return true;
}


// convert HEX to RGB - using $alpha will return rgba
function fp_hex2rgb($hex, $alpha = 90) {
   	// if is RGB - return it
   	$pattern = '/^#[a-f0-9]{6}$/i';
	if(empty($hex) || !preg_match($pattern, $hex)) {return $hex;}
  
	$hex = str_replace("#", "", $hex);
   	if(strlen($hex) == 3) {
		$r = hexdec(substr($hex,0,1).substr($hex,0,1));
		$g = hexdec(substr($hex,1,1).substr($hex,1,1));
		$b = hexdec(substr($hex,2,1).substr($hex,2,1));
	} else {
		$r = hexdec(substr($hex,0,2));
		$g = hexdec(substr($hex,2,2));
		$b = hexdec(substr($hex,4,2));
	}
	$rgb = array($r, $g, $b);
  
	return (empty($alpha)) ? 'rgb('. implode(",", $rgb) .')' : 'rgba('. implode(",", $rgb) .', '. ((int)$alpha / 100) .')';
}


// convert RGB to HEX
function fp_rgb2hex($rgb) {
   	// if is hex - return it
   	$pattern = '/^#[a-f0-9]{6}$/i';
	if(empty($rgb) || preg_match($pattern, $rgb)) {return $rgb;}

  	$rgb = explode(',', str_replace(array('rgb(', ')'), '', $rgb));
  	
	$hex = "#";
	$hex .= str_pad(dechex( trim($rgb[0]) ), 2, "0", STR_PAD_LEFT);
	$hex .= str_pad(dechex( trim($rgb[1]) ), 2, "0", STR_PAD_LEFT);
	$hex .= str_pad(dechex( trim($rgb[2]) ), 2, "0", STR_PAD_LEFT);

	return $hex; 
}


// get adobe typekit data through cURL
function fp_get_kit_data($kit_id) {
	
	$request = wp_safe_remote_request('https://typekit.com/api/v1/json/kits/'.$kit_id.'/published');
	$response = wp_remote_retrieve_body($request);
	$response_code = wp_remote_retrieve_response_code($request);

	if($response_code != 200 && $respose_code != 404) {
		return false;	
	} 
	elseif($response_code == 404) {
		return 'not found';		
	} 
	else {
		// check proper format
		$response = json_decode($response, true); 
		if(!is_array($response) || !isset($response['kit'])) {return false;}
		
		$fonts = $response['kit']['families'];
		$data = array();
		
		foreach($fonts as $font) {
			$data[ $font['slug'] ] = array('name' => $font['name'], 'css' => $font['css_stack']);
		}
		return $data;
	}
}


// HTMLify array into select options
function fp_array_to_html_opt($array, $selected = false, $associative_array = true) {
	if(!is_array($array)) {return false;}
	$code = '';
	
	foreach($array as $key => $val) {
		$true_val = ($associative_array) ? $key : $val; 
		$sel = ($true_val == $selected) ? 'selected="selected"' : '';
		
		$code .= '<option value="'. $true_val .'" '. $sel .'>'. $val .'</option>';
	}
	
	return $code;
}


// returns none instead of an empty string
function fp_empty_to_none($val) {
	return (empty($val)) ? 'none' : $val;	
}


////////////////////////////////////////////////////////////


// create the options for enabled fonts select
// @param $type -> cufon, webfonts, fontface
function fp_get_enabled_fonts($type, $selected = false) {
	if($type != 'ad_typekit') {
		$fonts_list = array();
			
		switch($type) {
			case 'cufon' : 
				$fonts_list = get_option('fp'.FP_BID.'_cufon_enabled', array());
				$associative = false;
				break;	
				
			case 'webfonts' : 
				$fonts_list = get_option('fp'.FP_BID.'_webfonts_enabled', array());
				$associative = false;
				break;	
			
			case 'fontface' : 
				$raw = get_option('fp'.FP_BID.'_font_enabled', array());
				
				$fonts_list = array();
				foreach($raw as $font) {
					$fonts_list[$font] = fp_urlToName($font);	
				}
				
				$associative = true;
				break;	
		}
		
		$opts = fp_array_to_html_opt($fonts_list, $selected, $associative);
	}
	
	else {
		$kits_list = (array)unserialize(get_option('fp_adobe_typekits', ''));
		$enabled_kits = get_option('fp'.FP_BID.'_ad_typekits_enabled', array());
		
		if($kits_list) {
			$opts = '';
			
			foreach($kits_list as $id => $data) {
				if(!in_array($id, $enabled_kits)) {continue;}
				
				foreach($data as $slug => $subdata) {
					$sel = ($selected == $slug) ? 'selected="selected"' : '';
					$opts .= '<option value="'.$slug.'" '.$sel.'>'.$subdata['name'].'</cufon>';
				}
			}
		}
	}
	
	return $opts;	
}


// elements array for main setting
function fp_get_elements($return = 'array', $selected = false) {
	$elements = array(
		'body' 				=> __('All page elements', 'fp_ml'),
		'h1,h2,h3,h4,h5,h6'	=> __('Headings', 'fp_ml'),
		'custom'			=> __('Custom Element', 'fp_ml')
	);
	
	return ($return == 'html') ? fp_array_to_html_opt($elements, $selected) : $elements;	
}


// font types array for main setting
function fp_get_fontypes($return = 'array', $selected = false) {
	$elements = array(
		'webfonts'		=> 'Web Fonts',
		'cufon' 		=> 'Cufons',
		'fontface'		=> 'Font-Face Fonts',
		'ad_typekit' 	=> 'Adobe Typekit',
		'css_font' 		=> 'Standard CSS Font', 
		'inherited'		=> __('Inherited Font', 'fp_ml')
	);
	
	return ($return == 'html') ? fp_array_to_html_opt($elements, $selected) : $elements;		
}


// font size types array for main setting
function fp_get_fontsize_types($return = 'array', $selected = false) {
	$elements = array(
		'px' 	=> 'px',
		'em'	=> 'em',
		'%'		=> '%',
		'pt'	=> 'pt',
		'rem'	=> 'rem',
		'ex'	=> 'ex',
		'vw'	=> 'vw',
		'vh'	=> 'vh',
		'vmin'	=> 'vmin',
		'ch'	=> 'ch',
	);
	
	return ($return == 'html') ? fp_array_to_html_opt($elements, $selected) : $elements;	
}


// text decoractions array
function fp_font_weight($return = 'array', $selected = false) {
	$elements = array(
		'inherit'		=> __('inherit', 'fp_ml'),
	);
	
	for($a=1; $a<=9; $a++) {
		$val = $a * 100;
		
		if($val == 400) {
			$elements[$val] = __('normal', 'fp_ml');	
		}
		elseif($val == 700) {
			$elements[$val] = __('bold', 'fp_ml');	
		}
		else {
			$elements[$val] = $val;	
		}
	}
	return ($return == 'html') ? fp_array_to_html_opt($elements, $selected) : $elements;	
}


// text italic options
function fp_font_style($return = 'array', $selected = false) {
	$elements = array(
		'inherit'	=> __('inherit', 'fp_ml'),
		'normal'	=> __('normal', 'fp_ml'),
		'italic'	=> __('italic', 'fp_ml'),
		'oblique'	=> __('oblique', 'fp_ml'),
	);
	
	return ($return == 'html') ? fp_array_to_html_opt($elements, $selected) : $elements;	
}


// text decoractions array
function fp_text_decorations($return = 'array', $selected = false) {
	$elements = array(
		'inherit'		=> __('inherit', 'fp_ml'),
		'underline'		=> __('underline', 'fp_ml'),
		'overline'		=> __('overline', 'fp_ml'),
		'line-through'	=> __('line-through', 'fp_ml'),
		'none'			=> __('none', 'fp_ml'),
	);
	
	return ($return == 'html') ? fp_array_to_html_opt($elements, $selected) : $elements;	
}


// text stroke width
function fp_text_stroke_w($return = 'array', $selected = false) {
	$elements = array();
	
	for($a=0; $a<=20; $a++) {
		$elements[$a] = $a .'px';
	}
	
	return ($return == 'html') ? fp_array_to_html_opt($elements, $selected) : $elements;	
}


// text transform options
function fp_text_transform($return = 'array', $selected = false) {
	$elements = array(
		'inherit'		=> __('inherit', 'fp_ml'),
		'capitalize'	=> __('capitalize', 'fp_ml'),
		'uppercase'		=> __('uppercase', 'fp_ml'),
		'lowercase'		=> __('lowercase', 'fp_ml'),
		'none'			=> __('none', 'fp_ml'),
	);
	
	return ($return == 'html') ? fp_array_to_html_opt($elements, $selected) : $elements;	
}


//////////////////////////////////////////////////////////


// return default values for builders
function fp_default_vals() {
	return array(
		'element'			=> 'body',
		'element_subj'		=> 'body',
		'font_type'			=> 'webfonts',
		'font_name'			=> '',
		'font_size'			=> 0,
		'font_size_type'	=> 'px',
		'line_height'		=> 0,
		'line_height_type'	=> 'px',
		'font_weight'		=> 'inherit',
		'font_style'		=> 'inherit',
		'text_decoration'	=> 'inherit',
		'text_transform'	=> 'inherit',
		'text_color' 		=> 'none',
		'text_stroke_w'		=> 0,
		'text_stroke_col'	=> '#777777',
		'shadow'			=> array(0, 0, 0, 90, '#777777'),
		'highlight'			=> 'none', // use background color
		'unselectable'		=> 0, // use css
		'force_rule'		=> 0,
	);	
}


// general rules - form creator per row
function fp_rule_row_form($data = array(1)) {
	$html = '';
	$off_btn = '<small>OFF</small>';	

	foreach((array)$data as $rules) {

		///////////////////////////////
		// retrocompatibility
		if(isset($rules['shadow_x'])) {
			$shadow_color = (!fp_rgb2hex($rules['shadow_color'])) ? '#555555' : fp_rgb2hex($rules['shadow_color']); 
			
			$rules['shadow'] = array(
				$rules['shadow_x'],
				$rules['shadow_y'],
				$rules['shadow_r'],
				90,
				$shadow_color,
			);	
			
			if(isset($rules['subj'])) {
				$rules['element_subj'] = $rules['subj'];
			}
			
			$rules['force_rule'] = 1;
		}
		////////////////////////////////////////
		
		$rules = wp_parse_args($rules, fp_default_vals());
		$no_font_name_class = ($rules['font_type'] == 'inherit') ? 'fp_no_font_name' : '';
		
		// header
		$html .= '
		<div class="postbox fp_rule_block">
			<header>
				<label>'. __('Selector', 'fp_ml') .'</label>
				<select class="choose_element" name="element[]" autocomplete="off">
					'. fp_get_elements('html', $rules['element']) .'
				</select>
				
				<div class="fp_selector_helper">';
				
					if($rules['element'] == 'custom') {
						$html .= '<input type="text" name="element_subj[]" value="'. esc_attr($rules['element_subj']) .'" placeholder="'. __('insert a CSS selector', 'fp_ml') .'" autocomplete="off" />';	
					} else {
						$html .= '<input type="hidden" name="element_subj[]" value="'. $rules['element_subj'] .'" /><span>'. $rules['element_subj'] .'</span>';		
					}
					
				$html .= '
				</div>    
					
				<div class="fp_rule_cmd">
					<a href="javascript:void(0)">('. __('collapse', 'fp_ml') .')</a>
					<span class="fp_move_rule" title="move rule"></span>
					<span class="fp_del_rule" title="delete rule"></span>
				</div>
			</header>';		
		
		
			// body
			$html .= '
			<section class="fp_rule_font_type">
				<label>'. __('Font type', 'fp_ml') .'</label>
				<select name="font_type[]" class="choose_font_type" autocomplete="off">
				  '.fp_get_fontypes('html', $rules['font_type']).'
				</select>
			</section>
			
			<section class="fp_rule_font_name '. $no_font_name_class .'">
				<label>'. __('Font name', 'fp_ml') .'</label>
				<div>';
				
					if($rules['font_type'] != 'css_font' && $rules['font_type'] != 'inherited') {
					  $html .= '
					  <select name="font_name[]" autocomplete="off">'.
						fp_get_enabled_fonts($rules['font_type'], $rules['font_name']).
					  '</select>';
					  }
					  elseif($rules['font_type'] == 'css_font') {
						  $html .= '<input type="text" name="font_name[]" value="'.$rules['font_name'].'" placeholder="'. __('write font name', 'fp_ml') .'" autocomplete="off" />';
					  }
					  else {
						  $html .= '<span>'. __('inherited', 'fp_ml') .'</span>
						  <input type="hidden" name="font_name[]" value="inherited" autocomplete="off" />';
					  }
				
				$html .= '	  
				</div>
			</section>
			
			<section class="fp_rule_font_size">
				<label>'. __('Font size', 'fp_ml') .' '.$off_btn.'</label>
				<input type="number" min="0" max="100" name="font_size[]" maxlength="3" value="'. (int)$rules['font_size'] .'" autocomplete="off" />
				<select name="font_size_type[]">
					'.fp_get_fontsize_types('html', $rules['font_size_type']).'
				</select>
			</section>
			
			<section class="fp_rule_line_height">
				<label>'. __('Line height', 'fp_ml') .' '.$off_btn.'</label>
				<input type="number" min="0" max="150" name="line_height[]" maxlength="3" value="'. (int)$rules['line_height'] .'" autocomplete="off" />
				<select name="line_height_type[]" autocomplete="off">
					'.fp_get_fontsize_types('html', $rules['font_size_type']).'
				</select>
			</section>
			
			<section class="fp_rule_font_weight">
				<label>'. __('Font weight', 'fp_ml') .'</label>
				<select name="font_weight[]" autocomplete="off">
					'. fp_font_weight('html', $rules['font_weight']) .'
				</select>
			</section>

			<section class="fp_rule_font_style">
				<label>'. __('Font style', 'fp_ml') .'</label>
				<select name="font_style[]" autocomplete="off">
					'. fp_font_style('html', $rules['font_weight']) .'
				</select>
			</section>

			<section class="fp_rule_text_decoration">
				<label>'. __('Text decoration', 'fp_ml') .'</label>
				<select name="text_decoration[]" autocomplete="off">
					'. fp_text_decorations('html', $rules['text_decoration']) .'
				</select>
			</section>
			
			<section class="fp_rule_text_transform">
				<label>'. __('Text transform', 'fp_ml') .'</label>
				<select name="text_transform[]" autocomplete="off">
					'. fp_text_transform('html', $rules['text_transform']) .'
				</select>
			</section>

			<section class="fp_rule_color">
				<label>'. __('Text color', 'fp_ml') .' '.$off_btn.'</label>
				<div class="lcwp_colpick">
					<span class="lcwp_colblock" style="background-color: '.$rules['text_color'].';"></span>
					<input type="text" name="text_color[]" maxlength="7" value="'. fp_empty_to_none($rules['text_color']) .'" autocomplete="off" />
			  </div>
			</section>
			
			<section class="fp_rule_stroke">
				<label>'. __('Outline', 'fp_ml') .'</label>
				<select name="text_stroke_w[]" autocomplete="off" style="width: 62px;">
					'. fp_text_stroke_w('html', $rules['text_stroke_w']) .'
				</select>
				<div class="lcwp_colpick">
					<span class="lcwp_colblock" style="background-color: '.$rules['text_stroke_col'].';"></span>
					<input type="text" name="text_stroke_col[]" value="'. $rules['text_stroke_col'] .'" autocomplete="off" />
			  </div>
			</section>

			<section class="fp_rule_shadow">
				<label>'. __('Shadow', 'fp_ml') .' '.$off_btn.'</label>
				
				<label title="X axis offset">X</label>
				<input type="number" min="0" max="30" maxlength="2" name="shadow_x[]" value="'. (int)$rules['shadow'][0] .'" autocomplete="off" />
				
				<label title="Y axis offset">Y</label>
				<input type="number" min="0" max="30" maxlength="2" name="shadow_y[]" value="'. (int)$rules['shadow'][1] .'" autocomplete="off" />
				
				<label title="shadow radius">R</label>
				<input type="number" min="0" max="30" maxlength="2" name="shadow_r[]" value="'. (int)$rules['shadow'][2] .'" autocomplete="off" /><br/>
				
				<label title="shadow opacity">%</label>
				<input type="number" min="10" max="100" maxlength="3" name="shadow_perc[]" class="fp_shadow_perc" value="'. (int)$rules['shadow'][3] .'" autocomplete="off" />
				
				<div class="lcwp_colpick">
					  <span class="lcwp_colblock" style="background-color: '.$rules['shadow'][4].';"></span>
					  <input type="text" name="shadow_color[]" maxlength="7" value="'.fp_rgb2hex($rules['shadow'][4]).'" autocomplete="off" />
				</div>
			</section>
			
			<section class="fp_rule_highlight">
				<label>'. __('Highlight text', 'fp_ml') .' '.$off_btn.'</label>
				<div class="lcwp_colpick">
					<span class="lcwp_colblock" style="background-color: '.$rules['highlight'].';"></span>
					<input type="text" name="highlight[]" maxlength="7" value="'. fp_empty_to_none($rules['highlight']) .'" autocomplete="off" />
				</div>
			</section>
			
			<section class="fp_rule_unselectable">
				<label>'. __('Unselectable text?', 'fp_ml') .'</label>
				<input type="checkbox" name="unselectable[]" value="1" '. checked($rules['unselectable'], 1, false) .' autocomplete="off" />
			</section>
			
			<section class="fp_rule_use_important">
				<label>'. __('Force rule?', 'fp_ml') .'</label>
				<input type="checkbox" name="force_rule[]" value="1" '. checked($rules['force_rule'], 1, false) .' autocomplete="off" />
			</section>
		</div>';	
	}
	
	return $html;	
}


//////////////////////////////////////////////////////////


// enqueue enabled cufon for the frontend
function fp_enqueue_enabled_cufon() {
	$enabled_list = get_option('fp'.FP_BID.'_cufon_enabled');
	$cufon_list = fp_cufon_list();
	
	if($enabled_list) {
		wp_enqueue_script( 'fp-cufon', FP_URL.'/js/cufon-yui.js' );	
		
		foreach($enabled_list as $enabled) {
			$enabled_id = fp_stringToUrl($enabled);
			wp_enqueue_script('fp_cufon_'.$enabled_id, FP_URL . '/cufon/'.$cufon_list[$enabled]);	
		}
	}
	return true;
}


// enqueue enabled cufon for the frontend
function fp_enqueue_enabled_webfont() {
	$enabled_list = get_option('fp'.FP_BID.'_webfonts_enabled');
	$webfont_list = get_option('fp_webfonts');
	
	if($enabled_list) {
		foreach($enabled_list as $enabled) {
			$enabled_id = fp_stringToUrl($enabled);
			
			// remove the HTTP/HTTPS for SSL compatibility
			$fixed_enabled = str_replace(array('http:', 'https:', 'HTTP:', 'HTTPS:'), '', $webfont_list[$enabled]);
			
			// enqueue for fifferent types
			if(strpos($webfont_list[$enabled], '/css?') === false) {
				wp_enqueue_script('fp_EdgeFont_'.$enabled_id, $fixed_enabled);
			} else {
				wp_enqueue_style('fp_webfont_'.$enabled_id, $fixed_enabled);
			}
		}
	}
	return true;
}


// fontface CSS code creator
function fp_fontface_css_creator($folder_name) {
	$path = FP_DIR . '/fonts/'.$folder_name;
	$font_name = fp_urlToName($folder_name);
	
	// remove the HTTP/HTTPS for SSL compatibility
	$fixed_enabled = str_replace(array('http:', 'https:', 'HTTP:', 'HTTPS:'), '', FP_URL);
	$font_baseurl = $fixed_enabled . '/fonts/'.$folder_name.'/';
	
	// get files name
	$file_list = scandir($path);	
	$file_name = fp_stringToFilename($file_list[2], true);	

	$css = "
@font-face {
	font-family: '".$font_name."';
	src: url('".$font_baseurl.$file_name.".eot');
	src: url('".$font_baseurl.$file_name.".eot?#iefix') format('embedded-opentype'),
		 url('".$font_baseurl.$file_name.".woff') format('woff'),
		 url('".$font_baseurl.$file_name.".ttf') format('truetype'),
		 url('".$font_baseurl.$file_name.".svg#".$file_name."') format('svg');
}
	";
	return $css;
}


// enqueue enabled adobe typekit for the frontend
function fp_enqueue_enabled_typekit() {
	$kits_list = (array)unserialize(get_option('fp_adobe_typekits', ''));
	$enabled_kits = get_option('fp'.FP_BID.'_ad_typekits_enabled', array());
	
	if($enabled_kits && $kits_list) {
		foreach($kits_list as $id => $data) {
			if(!in_array($id, $enabled_kits)) {continue;}
			wp_enqueue_script('fp_typekit_'.$id, "//use.typekit.net/". $id .".js");
		}
	}

	return true;
}


// adobe typekit get CSS name from slug
function fp_typekit_css_name($font_slug) {
	$kits_list = (array)unserialize(get_option('fp_adobe_typekits', ''));

	if($kits_list) {
		foreach($kits_list as $id => $data) {
			foreach($data as $slug => $subdata) {
				if($font_slug == $slug) {
					return $subdata['css'];
					break;	
				}
			}
		}
	}
}



/////////////////////////////////////////////////////////////////



// create the frontend css and js
function fp_create_frontend_files() {
	
	// css
	ob_start();
	require_once(FP_DIR.'/frontend_css.php');
	$css = ob_get_clean();
	
	if(trim($css) != '') {
		if(!file_put_contents(FP_DIR.'/custom_files/frontend'.FP_BID.'.css', $css, LOCK_EX)) {$error = true;}
	} else {
		if(file_exists(FP_DIR.'/custom_files/frontend'.FP_BID.'.css'))	{unlink(FP_DIR.'/custom_files/frontend'.FP_BID.'.css');}
	}
	
	// js
	ob_start();
	require_once(FP_DIR.'/frontend_js.php');
	$js = ob_get_clean();
	
	if(trim($js) != '') {
		if(!file_put_contents(FP_DIR.'/custom_files/frontend'.FP_BID.'.js', $js, LOCK_EX)) {$error = true;}
	} else {
		if(file_exists(FP_DIR.'/custom_files/frontend'.FP_BID.'.js'))	{unlink(FP_DIR.'/custom_files/frontend'.FP_BID.'.js');}
	}
	
	
	if(isset($error)) {
		update_option('fp_inline_code', 1);
		return false;
	}
	else {
		delete_option('fp_inline_code');
		return true;
	}
}
