<?php
/**
 * Etch License
 *
 * Handles the license key and activation
 *
 * @package Etch\WpAdmin
 * @since 0.15.1
 */

namespace Etch\WpAdmin;

use Etch\Traits\Singleton;
use Etch\Includes\SureCart\Licensing\Client;

/**
 * Handles the license key and activation
 * Uses the SureCart integration
 */
class License {
	use Singleton;

	/**
	 * License key for the option table
	 *
	 * @var string
	 */
	private $license_key_option = 'etch_license_key';

	/**
	 * Status key for the option table to store the activation status
	 *
	 * @var string
	 */
	private $status_key_option = 'etch_license_status';

	/**
	 * SureCart Client
	 *
	 * @var Client
	 */
	private $client;

	/**
	 * Initialize
	 *
	 * @return void
	 */
	public function init() {
		if ( ! class_exists( 'Etch\Includes\SureCart\Licensing\Client' ) ) {
			require_once ETCH_PLUGIN_DIR . '/includes/SureCart/Licensing/Client.php';
		}
		$this->client = new Client( 'Etch', 'pt_7eCsZFuK2NuCXK97jzkennFi', ETCH_PLUGIN_FILE );
	}

	/**
	 * Returns the license key value no matter if in constant or in the database
	 *
	 * @return string
	 */
	public function get_license() {
		update_option( $this->license_key_option,'9027acb4-3440-4370-8a678f98462a');
		update_option( $this->status_key_option ,'valid');
		delete_transient( 'etch_license_is_active' );	
		$license = $this->get_license_from_constant();
		return is_string( $license ) ? $license : '';
	}

	/**
	 * Returns the obfuscate license
	 *
	 * @return string
	 */
	public function get_obfuscate_license() {
		return $this->obfuscate_license( $this->get_license() );
	}

	/**
	 * Returns the license from constant if exists
	 *
	 * @return string
	 */
	public function get_license_from_constant() {
		return defined( 'ETCH_LICENSE_KEY' ) && ! empty( ETCH_LICENSE_KEY ) && is_string( ETCH_LICENSE_KEY ) ? ETCH_LICENSE_KEY : '';
	}

	/**
	 * Check if exists a constant for the license
	 *
	 * @return boolean
	 */
	public function is_from_constant() {
		return '' != $this->get_license_from_constant() ? true : false;
	}

	/**
	 * Activate the license
	 *
	 * @param mixed $license License.
	 * @throws \Exception Error trying to activate the license.
	 * @return mixed
	 */
	public function activate_license( $license ) {
		if ( $this->is_from_constant() ) {
			$license = $this->get_license_from_constant();
		}

		update_option( $this->license_key_option, '9027acb4-3440-4370-8a678f98462a' );
		update_option( $this->status_key_option, 'valid' );
		delete_transient( 'etch_license_is_active' );
		return true;
	}

	/**
	 * Deactivate the license
	 *
	 * @throws \Exception Error trying to deactivate the license.
	 * @return mixed
	 */
	public function deactivate_license() {
		$response = $this->client->license()->deactivate();

		if ( is_wp_error( $response ) ) {
			throw new \Exception( esc_html( $response->get_error_message() ) );
		}

		if ( ! $response ) {
			throw new \Exception( 'Error while try activate the license.' );
		}

		delete_option( $this->license_key_option );
		delete_option( $this->status_key_option );
		delete_transient( 'etch_license_is_active' );
	}

	/**
	 * Returns the status of activation
	 *
	 * @return mixed
	 */
	public function get_status() {
		return 'valid';
	}

	/**
	 * Obfuscates the license key.
	 *
	 * @param string $license The license key.
	 * @return string
	 */
	private function obfuscate_license( string $license ): string {
		return empty( $license ) ? '' : substr_replace( $license, 'XXXXXXXXXXXXXXXXXXXXXXXX', 4, 24 );
	}

	/**
	 * Checks if a license key is obfuscated.
	 *
	 * @param string $license The license key.
	 * @return boolean
	 */
	private static function is_obfuscated_license( string $license ): bool {
		return true;
	}

	/**
	 * Check if the license is active or not
	 *
	 * @return boolean
	 */
	public function is_active() {
		
		return true;
	}
}
