<?php
namespace WeDevs\ERP_PRO\CRM\Deals;

// don't call the file directly
use WeDevs\Deals\Admin;
use WeDevs\Deals\Frontend;
use WeDevs\Deals\Hooks;
use WeDevs\Deals\Log;

if ( !defined( 'ABSPATH' ) ) exit;

/**
 * Deals main class
 *
 * @since 1.0.0
 */
class Module {

	/**
	 * Add-on Version
	 *
	 * @var  string
	 */
	public $version = '1.3.0';

    /**
     * Initializes the class
     *
     * Checks for an existing instance
     * and if it doesn't find one, creates it.
     *
     * @since 1.0.0
     *
     * @return object Class instance
     */
    public static function init() {
        static $instance = false;

        if ( ! $instance ) {
            $instance = new self();
        }

        return $instance;
    }

	/**
     * Constructor for the class
     *
     * Sets up all the appropriate hooks and actions
     *
     * @since 1.0.0
     *
     * @return void
     */
    private function __construct() {
        // on activate plugin register hook
        add_action( 'erp_pro_activated_module_deals', array( $this, 'activate' ) );

        // on register deactivation hook
        add_action( 'erp_pro_deactivated_module_deals', array( $this, 'deactivate' ) );

        // crm loaded hook
        add_action( 'erp_crm_loaded', [ $this, 'erp_crm_loaded' ] );
    }

    /**
     * Executes during plugin activation
     *
     * @since 1.0.0
     *
     * @return void
     */
    public function activate() {
        $this->create_table();
        $this->insert_initial_table_data();
        $this->create_files();
    }

    /**
     * Executes during plugin deactivation
     *
     * @since 1.0.0
     *
     * @return void
     */
    public function deactivate() {

    }

    /**
     * Placeholder for creating tables while activating plugin
     *
     * @since 1.0.0
     *
     * @return void
     */
    private function create_table() {

        global $wpdb;

        $collate = $wpdb->get_charset_collate();

        $table_schema = include_once dirname( __FILE__ ) . '/table-schema.php';

        require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );

        foreach ( $table_schema as $table ) {

            dbDelta( $table );
        }
    }

    /**
     * Insert default data for the plugin during installation
     *
     * @since 1.0.0
     *
     * @return void
     */
    private function insert_initial_table_data() {
        include_once dirname( __FILE__ ) . '/table-data.php';
    }

    /**
     * Install files and folders for uploading files and prevent hotlinking
     *
     * @since 1.0.0
     *
     * @return void
     */
    private function create_files() {
        $upload_dir      = wp_upload_dir();

        $files = array(
            array(
                'base'      => $upload_dir['basedir'] . '/erp-deals-uploads',
                'file'      => 'index.html',
                'content'   => ''
            ),
            array(
                'base'      => $upload_dir['basedir'] . '/erp-deals-uploads',
                'file'      => '.htaccess',
                'content'   => 'deny from all'
            ),
        );

        foreach ( $files as $file ) {
            if ( wp_mkdir_p( $file['base'] ) && !file_exists( trailingslashit( $file['base'] ) . $file['file'] ) ) {
                if ( $file_handle = @fopen( trailingslashit( $file['base'] ) . $file['file'], 'w' ) ) {
                    fwrite( $file_handle, $file['content'] );
                    fclose( $file_handle );
                }
            }
        }
    }

    /**
     * Executes if CRM is installed
     *
     * @since 1.0.0
     *
     * @return void
     */
    public function erp_crm_loaded() {
        $this->define_constants();
        $this->includes();
    }

    /**
     * Define Add-on constants
     *
     * @since 1.0.0
     *
     * @return void
     */
    private function define_constants() {
        define( 'WPERP_DEALS_VERSION', $this->version );
        define( 'WPERP_DEALS_FILE', __FILE__ );
        define( 'WPERP_DEALS_PATH', dirname( WPERP_DEALS_FILE ) );
        define( 'WPERP_DEALS_INCLUDES', WPERP_DEALS_PATH . '/includes' );
        define( 'WPERP_DEALS_URL', plugins_url( '', WPERP_DEALS_FILE ) );
        define( 'WPERP_DEALS_ASSETS', WPERP_DEALS_URL . '/assets' );
        define( 'WPERP_DEALS_VIEWS', WPERP_DEALS_PATH . '/views' );
    }

    /**
     * Include required files
     *
     * @since 1.0.0
     *
     * @return void
     */
    private function includes() {
        include_once WPERP_DEALS_INCLUDES . '/Models/Deal.php';
        include_once WPERP_DEALS_INCLUDES . '/Models/Activity.php';
        include_once WPERP_DEALS_INCLUDES . '/Models/ActivityType.php';
        include_once WPERP_DEALS_INCLUDES . '/Models/Pipeline.php';
        include_once WPERP_DEALS_INCLUDES . '/Models/PipelineStage.php';
        include_once WPERP_DEALS_INCLUDES . '/Models/StageHistory.php';
        include_once WPERP_DEALS_INCLUDES . '/Models/LostReason.php';
        include_once WPERP_DEALS_INCLUDES . '/Models/Participant.php';
        include_once WPERP_DEALS_INCLUDES . '/Models/Agent.php';
        include_once WPERP_DEALS_INCLUDES . '/Models/Note.php';
        include_once WPERP_DEALS_INCLUDES . '/Models/Attachment.php';
        include_once WPERP_DEALS_INCLUDES . '/Models/Competitor.php';
        include_once WPERP_DEALS_INCLUDES . '/Models/Email.php';
        include_once WPERP_DEALS_INCLUDES . '/erp-helper.php';
        include_once WPERP_DEALS_INCLUDES . '/Shortcodes.php';
        include_once WPERP_DEALS_INCLUDES . '/Helpers.php';
        include_once WPERP_DEALS_INCLUDES . '/Hooks.php';
        include_once WPERP_DEALS_INCLUDES . '/Log.php';
        include_once WPERP_DEALS_INCLUDES . '/Deals.php';
        new Hooks();
        Log::instance();

        // admin functionalities
        add_action( 'init', function () {
            if ( current_user_can( 'administrator' ) || erp_crm_is_current_user_manager() || erp_crm_is_current_user_crm_agent() ) {
                new Admin();
            }

            if ( !is_admin() ) {
                new Frontend();
            }
        } );


    }

}
